/**
 * TicketPayGo Calendar Script
 */
document.addEventListener('DOMContentLoaded', function() {
    // Find all calendars on the page
    const calendars = document.querySelectorAll('.ticketpaygo-calendar');
    
    calendars.forEach(function(calendarEl) {
        if (!calendarEl || typeof FullCalendar === 'undefined') {
            console.error('FullCalendar not loaded or calendar element not found');
            return;
        }
        
        const calendarId = calendarEl.id;
        const initialView = calendarEl.dataset.view || 'dayGridMonth';
        const category = calendarEl.dataset.category || '';
        
        // Settings from localized script
        const locale = ticketpaygo_calendar.locale || 'en_US';
        const eventsUrl = ticketpaygo_calendar.events_url;
        
        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: initialView,
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listMonth'
            },
            locale: locale,
            firstDay: 1,
            eventTimeFormat: {
                hour: '2-digit',
                minute: '2-digit',
                hour12: false
            },
            events: {
                url: eventsUrl,
                method: 'GET',
                extraParams: function() {
                    return {
                        category: category,
                        format: 'calendar'
                    };
                },
                failure: function() {
                    console.error('Error fetching events');
                },
                success: function(data) {
                    return data.map(function(event) {
                        return {
                            id: event.id,
                            title: event.title,
                            start: event.start_date,
                            end: event.end_date,
                            url: event.url,
                            extendedProps: {
                                location: event.location,
                                price: event.price,
                                currency: event.currency,
                                description: event.description,
                                image: event.featured_image,
                                tickets_remaining: event.tickets_remaining
                            }
                        };
                    });
                }
            },
            eventClick: function(info) {
                info.jsEvent.preventDefault();
                showEventPopup(info.event);
            },
            loading: function(isLoading) {
                if (isLoading) {
                    calendarEl.classList.add('loading');
                } else {
                    calendarEl.classList.remove('loading');
                }
            }
        });
        
        calendar.render();
        
        // Store calendar instance
        calendarEl.calendarInstance = calendar;
        
        // View selector change handler
        const viewSelect = document.querySelector('.calendar-view-select[data-calendar="' + calendarId + '"]');
        if (viewSelect) {
            viewSelect.addEventListener('change', function() {
                calendar.changeView(this.value);
            });
        }
    });
    
    // Event popup functionality
    function showEventPopup(event) {
        const popup = document.getElementById('calendar-event-popup');
        if (!popup) return;
        
        const props = event.extendedProps;
        
        // Set image
        const imageEl = popup.querySelector('.popup-event-image');
        if (props.image) {
            imageEl.style.backgroundImage = 'url(' + props.image + ')';
        } else {
            imageEl.style.backgroundImage = '';
        }
        
        // Set details
        popup.querySelector('.popup-event-title').textContent = event.title;
        popup.querySelector('.popup-event-date span').textContent = event.start.toLocaleDateString(ticketpaygo_calendar.locale, { 
            weekday: 'long', 
            year: 'numeric', 
            month: 'long', 
            day: 'numeric' 
        });
        popup.querySelector('.popup-event-time span').textContent = event.start.toLocaleTimeString(ticketpaygo_calendar.locale, { 
            hour: '2-digit', 
            minute: '2-digit' 
        });
        
        const locationText = props.location || ticketpaygo_calendar.strings.location_tba || 'Location TBA';
        popup.querySelector('.popup-event-location span').textContent = locationText;
        
        let priceText = '';
        const currency = props.currency || 'EUR';
        const currencySymbol = currency === 'EUR' ? '€' : (currency === 'USD' ? '$' : '£');
        
        if (props.price) {
            priceText = currencySymbol + parseFloat(props.price).toFixed(2);
        }
        popup.querySelector('.popup-event-price span').textContent = priceText;
        
        popup.querySelector('.popup-event-description').textContent = props.description ? props.description.substring(0, 200) + '...' : '';
        
        // Set links
        popup.querySelector('.btn-book-tickets').href = event.url || '#';
        popup.querySelector('.btn-view-details').href = event.url || '#';
        
        // Show popup
        popup.style.display = 'flex';
        document.body.style.overflow = 'hidden';
    }
    
    // Close popup handlers
    const popup = document.getElementById('calendar-event-popup');
    if (popup) {
        const closeBtn = popup.querySelector('.popup-close');
        const overlay = popup.querySelector('.popup-overlay');
        
        if (closeBtn) closeBtn.addEventListener('click', closePopup);
        if (overlay) overlay.addEventListener('click', closePopup);
        
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') closePopup();
        });
    }
    
    function closePopup() {
        const popup = document.getElementById('calendar-event-popup');
        if (popup) {
            popup.style.display = 'none';
            document.body.style.overflow = '';
        }
    }
});
