/**
 * TicketPayGo Admin JavaScript
 */

(function($) {
    'use strict';

    // Admin object
    const TicketPayGoAdmin = {
        
        init: function() {
            this.initDashboard();
            this.initEventForm();
            this.initSettings();
            this.initAnalytics();
            this.bindEvents();
        },

        initDashboard: function() {
            // Initialize sales chart if canvas exists
            const chartCanvas = document.getElementById('salesChart');
            if (chartCanvas && typeof Chart !== 'undefined') {
                this.initSalesChart(chartCanvas);
            }

            // Auto-refresh stats every 30 seconds
            this.startStatsRefresh();
        },

        initSalesChart: function(canvas) {
            const ctx = canvas.getContext('2d');
            
            // Get data from localized script
            if (typeof ticketpaygo_chart_data !== 'undefined') {
                const salesData = ticketpaygo_chart_data;
                
                new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: salesData.map(item => item.period),
                        datasets: [{
                            label: ticketpaygo_admin.strings.revenue || 'Revenue',
                            data: salesData.map(item => parseFloat(item.revenue)),
                            borderColor: '#0073aa',
                            backgroundColor: 'rgba(0, 115, 170, 0.1)',
                            tension: 0.4,
                            fill: true
                        }, {
                            label: ticketpaygo_admin.strings.orders || 'Orders',
                            data: salesData.map(item => parseInt(item.orders)),
                            borderColor: '#00a32a',
                            backgroundColor: 'rgba(0, 163, 42, 0.1)',
                            tension: 0.4,
                            yAxisID: 'y1'
                        }]
                    },
                    options: {
                        responsive: true,
                        interaction: {
                            mode: 'index',
                            intersect: false,
                        },
                        scales: {
                            y: {
                                type: 'linear',
                                display: true,
                                position: 'left',
                                title: {
                                    display: true,
                                    text: ticketpaygo_admin.strings.revenue || 'Revenue'
                                }
                            },
                            y1: {
                                type: 'linear',
                                display: true,
                                position: 'right',
                                title: {
                                    display: true,
                                    text: ticketpaygo_admin.strings.orders || 'Orders'
                                },
                                grid: {
                                    drawOnChartArea: false,
                                }
                            }
                        },
                        plugins: {
                            legend: {
                                display: true,
                                position: 'top'
                            },
                            tooltip: {
                                mode: 'index',
                                intersect: false
                            }
                        }
                    }
                });
            }
        },

        startStatsRefresh: function() {
            setInterval(() => {
                this.refreshRealtimeStats();
            }, 30000); // 30 seconds
        },

        refreshRealtimeStats: function() {
            $.ajax({
                url: ticketpaygo_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'ticketpaygo_admin_action',
                    admin_action: 'get_realtime_stats',
                    nonce: ticketpaygo_admin.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.updateStatsDisplay(response.data);
                    }
                },
                error: (xhr, status, error) => {
                    console.error('Stats refresh failed:', error);
                }
            });
        },

        updateStatsDisplay: function(data) {
            // Update today's stats
            const todayStatsGrid = $('.today-stats-grid');
            if (todayStatsGrid.length) {
                todayStatsGrid.find('.today-stat:nth-child(1) .value').text(data.today_orders);
                todayStatsGrid.find('.today-stat:nth-child(2) .value').text(
                    (ticketpaygo_admin.currency_symbol || '€') + parseFloat(data.today_revenue).toFixed(2)
                );
                todayStatsGrid.find('.today-stat:nth-child(3) .value').text(data.pending_orders);
            }
        },

        initEventForm: function() {
            // Featured Image Media Uploader
            var featuredImageUploader;
            
            $('#upload_featured_image').off('click').on('click', function(e) {
                e.preventDefault();
                
                if (!featuredImageUploader) {
                    featuredImageUploader = wp.media({
                        title: ticketpaygo_admin.strings.select_image || 'Select Featured Image',
                        button: { text: ticketpaygo_admin.strings.use_image || 'Use this image' },
                        multiple: false,
                        library: { type: 'image' }
                    });
                    
                    featuredImageUploader.on('select', function() {
                        var attachment = featuredImageUploader.state().get('selection').first().toJSON();
                        $('#featured_image').val(attachment.url);
                        $('#featured_image_preview').html('<img src="' + attachment.url + '" alt="Featured Image" style="max-width: 300px; height: auto; border-radius: 8px;" />').show();
                        $('#remove_featured_image').show();
                    });
                }
                
                featuredImageUploader.open();
            });
            
            $('#remove_featured_image').off('click').on('click', function(e) {
                e.preventDefault();
                $('#featured_image').val('');
                $('#featured_image_preview').hide().html('');
                $(this).hide();
            });
            
            // Date/time validation - compare full datetime not just date
            $('#start_date, #end_date').on('change', function() {
                const startDateVal = $('#start_date').val();
                const endDateVal = $('#end_date').val();
                
                if (startDateVal && endDateVal) {
                    const startDate = new Date(startDateVal);
                    const endDate = new Date(endDateVal);
                    
                    // Only show error if end datetime is actually before or equal to start datetime
                    if (endDate <= startDate) {
                        alert(ticketpaygo_admin.strings.invalid_dates || 'End date must be after start date');
                        $(this).focus();
                    }
                }
            });

            // Price validation
            $('#price').on('input', function() {
                const price = parseFloat($(this).val());
                if (price < 0) {
                    $(this).val(0);
                }
            });

            // Max tickets validation
            $('#max_tickets').on('input', function() {
                const maxTickets = parseInt($(this).val());
                if (maxTickets < 0) {
                    $(this).val(0);
                }
            });

            // Auto-populate organizer info from WordPress user
            if ($('#organizer_name').val() === '' && typeof ticketpaygo_admin.current_user !== 'undefined') {
                $('#organizer_name').val(ticketpaygo_admin.current_user.display_name);
                $('#organizer_email').val(ticketpaygo_admin.current_user.email);
            }

            // Initialize location map
            this.initLocationMap();
        },

        initLocationMap: function() {
            const mapElement = document.getElementById('event_location_map');
            if (!mapElement || typeof L === 'undefined') {
                return;
            }

            // Check if map is already initialized
            if (mapElement._leaflet_id) {
                return;
            }

            // Fix Leaflet marker icon paths
            delete L.Icon.Default.prototype._getIconUrl;
            L.Icon.Default.mergeOptions({
                iconRetinaUrl: 'https://unpkg.com/leaflet@1.9.4/dist/images/marker-icon-2x.png',
                iconUrl: 'https://unpkg.com/leaflet@1.9.4/dist/images/marker-icon.png',
                shadowUrl: 'https://unpkg.com/leaflet@1.9.4/dist/images/marker-shadow.png'
            });

            // Get initial coordinates (default to Amsterdam if not set)
            const lat = parseFloat($('#latitude').val()) || 52.3676;
            const lng = parseFloat($('#longitude').val()) || 4.9041;

            // Initialize map
            const map = L.map('event_location_map').setView([lat, lng], 13);

            // Add OpenStreetMap tiles (free, no API key needed)
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                maxZoom: 19
            }).addTo(map);

            // Add marker
            let marker = L.marker([lat, lng], {
                draggable: true
            }).addTo(map);

            // Update coordinates when marker is dragged
            marker.on('dragend', function(e) {
                const position = marker.getLatLng();
                $('#latitude').val(position.lat.toFixed(6));
                $('#longitude').val(position.lng.toFixed(6));
                
                // Reverse geocode to get address
                TicketPayGoAdmin.reverseGeocode(position.lat, position.lng);
            });

            // Click on map to place marker
            map.on('click', function(e) {
                marker.setLatLng(e.latlng);
                $('#latitude').val(e.latlng.lat.toFixed(6));
                $('#longitude').val(e.latlng.lng.toFixed(6));
                
                // Reverse geocode to get address
                TicketPayGoAdmin.reverseGeocode(e.latlng.lat, e.latlng.lng);
            });

            // Search location button
            $('#search_location_btn').on('click', function(e) {
                e.preventDefault();
                const searchQuery = $('#map_search').val().trim();
                
                if (!searchQuery) {
                    alert('Please enter a location to search');
                    return;
                }

                // Show loading state
                $(this).prop('disabled', true).text('Searching...');

                // Use Nominatim (OpenStreetMap) geocoding service (free, no API key)
                $.ajax({
                    url: 'https://nominatim.openstreetmap.org/search',
                    data: {
                        q: searchQuery,
                        format: 'json',
                        limit: 1
                    },
                    success: function(data) {
                        $('#search_location_btn').prop('disabled', false).text('Search');
                        
                        if (data && data.length > 0) {
                            const result = data[0];
                            const lat = parseFloat(result.lat);
                            const lng = parseFloat(result.lon);
                            
                            // Update map and marker
                            map.setView([lat, lng], 15);
                            marker.setLatLng([lat, lng]);
                            
                            // Update hidden fields
                            $('#latitude').val(lat.toFixed(6));
                            $('#longitude').val(lng.toFixed(6));
                            
                            // Update address field if empty
                            if (!$('#address').val()) {
                                $('#address').val(result.display_name);
                            }
                        } else {
                            alert('Location not found. Please try a different search term.');
                        }
                    },
                    error: function() {
                        $('#search_location_btn').prop('disabled', false).text('Search');
                        alert('Error searching for location. Please try again.');
                    }
                });
            });

            // Allow search on Enter key
            $('#map_search').on('keypress', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    $('#search_location_btn').click();
                }
            });
        },

        reverseGeocode: function(lat, lng) {
            // Use Nominatim reverse geocoding to get address from coordinates
            $.ajax({
                url: 'https://nominatim.openstreetmap.org/reverse',
                data: {
                    lat: lat,
                    lon: lng,
                    format: 'json'
                },
                success: function(data) {
                    if (data && data.display_name) {
                        // Only update if address field is empty
                        if (!$('#address').val()) {
                            $('#address').val(data.display_name);
                        }
                    }
                }
            });
        },

        initSettings: function() {
            // Payment method toggles
            $('.payment-method-toggle').on('change', function() {
                const method = $(this).data('method');
                const isEnabled = $(this).is(':checked');
                $(`.${method}-settings`).toggle(isEnabled);
            });

            // Test payment connection buttons
            $('.test-payment-connection').on('click', function(e) {
                e.preventDefault();
                const method = $(this).data('method');
                this.testPaymentConnection(method);
            });

            // Currency symbol auto-update
            $('#currency').on('change', function() {
                const currency = $(this).val();
                const symbols = {
                    'EUR': '€',
                    'USD': '$',
                    'GBP': '£',
                    'CAD': 'C$',
                    'AUD': 'A$'
                };
                
                if (symbols[currency]) {
                    $('#currency_symbol').val(symbols[currency]);
                }
            });
        },

        testPaymentConnection: function(method) {
            const button = $(`.test-payment-connection[data-method="${method}"]`);
            const originalText = button.text();
            
            button.text(ticketpaygo_admin.strings.testing || 'Testing...').prop('disabled', true);

            $.ajax({
                url: ticketpaygo_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'ticketpaygo_admin_action',
                    admin_action: 'test_payment_connection',
                    payment_method: method,
                    nonce: ticketpaygo_admin.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice('success', response.data.message);
                    } else {
                        this.showNotice('error', response.data.message);
                    }
                },
                error: () => {
                    this.showNotice('error', ticketpaygo_admin.strings.connection_failed || 'Connection test failed');
                },
                complete: () => {
                    button.text(originalText).prop('disabled', false);
                }
            });
        },

        initAnalytics: function() {
            // Export buttons
            $('.export-data-btn').on('click', function(e) {
                e.preventDefault();
                const exportType = $(this).data('export-type');
                const eventId = $(this).data('event-id') || '';
                this.exportData(exportType, eventId);
            });

            // Date range picker for analytics
            if ($.fn.datepicker) {
                $('.analytics-date-range').datepicker({
                    dateFormat: 'yy-mm-dd',
                    onSelect: function() {
                        // Refresh analytics data
                        TicketPayGoAdmin.refreshAnalytics();
                    }
                });
            }
        },

        exportData: function(type, eventId) {
            const form = $('<form>', {
                method: 'POST',
                action: ticketpaygo_admin.ajax_url
            });

            form.append($('<input>', {
                type: 'hidden',
                name: 'action',
                value: 'ticketpaygo_admin_action'
            }));

            form.append($('<input>', {
                type: 'hidden',
                name: 'admin_action',
                value: 'export_data'
            }));

            form.append($('<input>', {
                type: 'hidden',
                name: 'export_type',
                value: type
            }));

            if (eventId) {
                form.append($('<input>', {
                    type: 'hidden',
                    name: 'event_id',
                    value: eventId
                }));
            }

            form.append($('<input>', {
                type: 'hidden',
                name: 'nonce',
                value: ticketpaygo_admin.nonce
            }));

            $('body').append(form);
            form.submit();
            form.remove();
        },

        refreshAnalytics: function() {
            const analyticsContainer = $('.analytics-container');
            if (analyticsContainer.length) {
                analyticsContainer.addClass('loading');
                
                $.ajax({
                    url: ticketpaygo_admin.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'ticketpaygo_admin_action',
                        admin_action: 'refresh_analytics',
                        nonce: ticketpaygo_admin.nonce,
                        date_from: $('#analytics_date_from').val(),
                        date_to: $('#analytics_date_to').val()
                    },
                    success: (response) => {
                        if (response.success) {
                            analyticsContainer.html(response.data.html);
                        }
                    },
                    complete: () => {
                        analyticsContainer.removeClass('loading');
                    }
                });
            }
        },

        bindEvents: function() {
            // Bulk actions
            $('.bulkactions select').on('change', function() {
                const action = $(this).val();
                const applyButton = $(this).siblings('.button.action');
                
                if (action !== '-1') {
                    applyButton.prop('disabled', false);
                } else {
                    applyButton.prop('disabled', true);
                }
            });

            // Confirm delete actions
            $('.delete a, .submitdelete').on('click', function(e) {
                if (!confirm(ticketpaygo_admin.strings.confirm_delete || 'Are you sure you want to delete this item?')) {
                    e.preventDefault();
                    return false;
                }
            });

            // Send reminder emails
            $('.send-reminder-btn').on('click', function(e) {
                e.preventDefault();
                const eventId = $(this).data('event-id');
                this.sendReminderEmails(eventId);
            });

            // Scan ticket modal
            $('.scan-ticket-btn').on('click', function(e) {
                e.preventDefault();
                this.openScanTicketModal();
            });

            // Form validation
            $('form').on('submit', function(e) {
                return this.validateForm($(this));
            });
        },

        sendReminderEmails: function(eventId) {
            const button = $(`.send-reminder-btn[data-event-id="${eventId}"]`);
            const originalText = button.text();
            
            button.text(ticketpaygo_admin.strings.sending || 'Sending...').prop('disabled', true);

            $.ajax({
                url: ticketpaygo_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'ticketpaygo_admin_action',
                    admin_action: 'send_reminder',
                    event_id: eventId,
                    nonce: ticketpaygo_admin.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice('success', response.data.message);
                    } else {
                        this.showNotice('error', response.data.message);
                    }
                },
                error: () => {
                    this.showNotice('error', ticketpaygo_admin.strings.send_failed || 'Failed to send reminders');
                },
                complete: () => {
                    button.text(originalText).prop('disabled', false);
                }
            });
        },

        openScanTicketModal: function() {
            // Create modal HTML
            const modalHtml = `
                <div id="scan-ticket-modal" class="ticketpaygo-modal">
                    <div class="modal-content">
                        <span class="close">&times;</span>
                        <h2>${ticketpaygo_admin.strings.scan_ticket || 'Scan Ticket'}</h2>
                        <form id="scan-ticket-form">
                            <p>
                                <label for="ticket_number">${ticketpaygo_admin.strings.ticket_number || 'Ticket Number'}:</label>
                                <input type="text" id="ticket_number" name="ticket_number" required>
                            </p>
                            <p>
                                <button type="submit" class="button-primary">${ticketpaygo_admin.strings.scan || 'Scan'}</button>
                            </p>
                        </form>
                        <div id="scan-result"></div>
                    </div>
                </div>
            `;

            $('body').append(modalHtml);
            $('#scan-ticket-modal').show();

            // Bind modal events
            $('.close, .ticketpaygo-modal').on('click', function(e) {
                if (e.target === this) {
                    $('#scan-ticket-modal').remove();
                }
            });

            $('#scan-ticket-form').on('submit', (e) => {
                e.preventDefault();
                this.scanTicket($('#ticket_number').val());
            });
        },

        scanTicket: function(ticketNumber) {
            const resultDiv = $('#scan-result');
            resultDiv.html('<div class="ticketpaygo-loading"></div>');

            $.ajax({
                url: ticketpaygo_admin.ajax_url,
                type: 'POST',
                data: {
                    action: 'ticketpaygo_admin_action',
                    admin_action: 'scan_ticket',
                    ticket_number: ticketNumber,
                    nonce: ticketpaygo_admin.nonce
                },
                success: (response) => {
                    if (response.success) {
                        resultDiv.html(`<div class="notice notice-success"><p>${response.data.message}</p></div>`);
                    } else {
                        resultDiv.html(`<div class="notice notice-error"><p>${response.data}</p></div>`);
                    }
                },
                error: () => {
                    resultDiv.html(`<div class="notice notice-error"><p>${ticketpaygo_admin.strings.scan_failed || 'Scan failed'}</p></div>`);
                }
            });
        },

        validateForm: function($form) {
            let isValid = true;
            
            // Check required fields
            $form.find('[required]').each(function() {
                if (!$(this).val().trim()) {
                    $(this).addClass('error');
                    isValid = false;
                } else {
                    $(this).removeClass('error');
                }
            });

            // Email validation
            $form.find('input[type="email"]').each(function() {
                const email = $(this).val();
                if (email && !this.isValidEmail(email)) {
                    $(this).addClass('error');
                    isValid = false;
                } else {
                    $(this).removeClass('error');
                }
            });

            if (!isValid) {
                this.showNotice('error', ticketpaygo_admin.strings.form_validation_failed || 'Please fill in all required fields correctly');
            }

            return isValid;
        },

        isValidEmail: function(email) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        },

        showNotice: function(type, message) {
            const noticeHtml = `<div class="notice notice-${type} is-dismissible"><p>${message}</p></div>`;
            $('.wrap h1').after(noticeHtml);
            
            // Auto-dismiss after 5 seconds
            setTimeout(() => {
                $('.notice.is-dismissible').fadeOut();
            }, 5000);
        },

        // Utility functions
        formatCurrency: function(amount) {
            const symbol = ticketpaygo_admin.currency_symbol || '€';
            const position = ticketpaygo_admin.currency_position || 'before';
            const formatted = parseFloat(amount).toFixed(2);
            
            return position === 'before' ? symbol + formatted : formatted + symbol;
        },

        formatDate: function(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
        }
    };

    // Customization page functionality
    function initializeCustomizationPage() {
        if (!document.getElementById('card_preview')) return;
        
        // Range input updates
        const ranges = ['border_radius', 'shadow_intensity', 'overlay_opacity', 'card_font_size', 'title_font_size', 'button_font_size'];
        ranges.forEach(function(range) {
            const input = document.getElementById(range);
            const display = document.getElementById(range + '_value');
            if (input && display) {
                input.addEventListener('input', function() {
                    let value = this.value;
                    if (range.includes('font_size') || range === 'border_radius') {
                        value += 'px';
                    }
                    display.textContent = value;
                    updatePreview();
                });
            }
        });
        
        // Color and text input updates
        const inputs = ['card_background', 'card_text_color', 'title_color', 'button_background', 'button_text_color', 'button_text', 'card_font_family', 'card_layout', 'card_size'];
        inputs.forEach(function(inputId) {
            const input = document.getElementById(inputId);
            if (input) {
                input.addEventListener('input', updatePreview);
                input.addEventListener('change', updatePreview);
            }
        });
        
        // Reset defaults
        const resetBtn = document.getElementById('reset_defaults');
        if (resetBtn) {
            resetBtn.addEventListener('click', function() {
                if (confirm('Reset all customization to defaults?')) {
                    const defaults = {
                        'card_background': '#ffffff',
                        'card_text_color': '#333333',
                        'card_font_family': 'Inter, system-ui, sans-serif',
                        'card_font_size': 14,
                        'title_color': '#ffffff',
                        'title_font_size': 24,
                        'button_background': '#667eea',
                        'button_text_color': '#ffffff',
                        'button_text': 'Book Tickets Now',
                        'button_font_size': 16,
                        'overlay_opacity': 0.8,
                        'border_radius': 16,
                        'shadow_intensity': 12,
                        'card_layout': 'horizontal',
                        'card_size': 'medium'
                    };
                    
                    Object.keys(defaults).forEach(function(key) {
                        const input = document.getElementById(key);
                        if (input) {
                            input.value = defaults[key];
                            if (input.type === 'range') {
                                const display = document.getElementById(key + '_value');
                                if (display) {
                                    let value = defaults[key];
                                    if (key.includes('font_size') || key === 'border_radius') {
                                        value += 'px';
                                    }
                                    display.textContent = value;
                                }
                            }
                        }
                    });
                    updatePreview();
                }
            });
        }
        
        function updatePreview() {
            const preview = document.getElementById('card_preview');
            if (!preview) return;
            
            const settings = {};
            
            // Get all current values
            ranges.concat(inputs).forEach(function(id) {
                const input = document.getElementById(id);
                if (input) {
                    settings[id] = input.value;
                }
            });
            
            // Calculate size multipliers based on card_size
            const sizeMultipliers = {
                'small': 0.8,
                'medium': 1.0,
                'large': 1.3
            };
            const multiplier = sizeMultipliers[settings.card_size] || 1.0;
            
            // Calculate dimensions based on layout and size
            const isVertical = settings.card_layout === 'vertical';
            const previewWidth = isVertical ? (300 * multiplier) : (600 * multiplier);
            const aspectRatio = isVertical ? '3/4' : '16/9';
            
            // Update preview styles
            preview.style.borderRadius = settings.border_radius + 'px';
            preview.style.boxShadow = `0 8px ${settings.shadow_intensity * 2}px rgba(0,0,0,0.12)`;
            preview.style.fontFamily = settings.card_font_family;
            preview.style.maxWidth = previewWidth + 'px';
            preview.style.aspectRatio = aspectRatio;
            
            // Update preview content to match frontend template
            preview.innerHTML = `
                <div class="event-image-container" style="position: relative; width: 100%; height: 100%; background: url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjAwIiBoZWlnaHQ9IjMzOCIgdmlld0JveD0iMCAwIDYwMCAzMzgiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CjxyZWN0IHdpZHRoPSI2MDAiIGhlaWdodD0iMzM4IiBmaWxsPSJ1cmwoI3BhaW50MF9saW5lYXJfMF8xKSIvPgo8cGF0aCBkPSJNMjUwIDEwMEMzMDUuMjI4IDEwMCAzNTAgMTQ0Ljc3MiAzNTAgMjAwQzM1MCAyNTUuMjI4IDMwNS4yMjggMzAwIDI1MCAzMDBDMTk0Ljc3MiAzMDAgMTUwIDI1NS4yMjggMTUwIDIwMEMxNTAgMTQ0Ljc3MiAxOTQuNzcyIDEwMCAyNTAgMTAwWiIgZmlsbD0id2hpdGUiIGZpbGwtb3BhY2l0eT0iMC4xIi8+CjxwYXRoIGQ9Ik0yNzUgMTc1SDIyNVYyMjVIMjc1VjE3NVoiIGZpbGw9IndoaXRlIiBmaWxsLW9wYWNpdHk9IjAuMyIvPgo8ZGVmcz4KPGxpbmVhckdyYWRpZW50IGlkPSJwYWludDBfbGluZWFyXzBfMSIgeDE9IjAiIHkxPSIwIiB4Mj0iNjAwIiB5Mj0iMzM4IiBncmFkaWVudFVuaXRzPSJ1c2VyU3BhY2VPblVzZSI+CjxzdG9wIHN0b3AtY29sb3I9IiM2NjdlZWEiLz4KPHN0b3Agb2Zmc2V0PSIxIiBzdG9wLWNvbG9yPSIjNzY0YmEyIi8+CjwvbGluZWFyR3JhZGllbnQ+CjwvZGVmcz4KPC9zdmc+') center/cover; border-radius: ${settings.border_radius}px; overflow: hidden;">
                    <div class="event-overlay" style="position: absolute; bottom: 0; left: 0; right: 0; background: linear-gradient(transparent, rgba(0,0,0,${settings.overlay_opacity})); color: white; padding: 20px; font-family: ${settings.card_font_family};">
                        <h3 class="event-title" style="font-size: ${settings.title_font_size}px; color: ${settings.title_color}; margin: 0 0 10px 0; font-weight: 700; line-height: 1.2;">Sample Event Title</h3>
                        
                        <div class="event-quick-info" style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 8px; margin-bottom: 12px; font-size: 13px;">
                            <div class="info-item" style="display: flex; align-items: center; gap: 6px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M19 3H18V1H16V3H8V1H6V3H5C3.89 3 3.01 3.9 3.01 5L3 19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V8H19V19Z" fill="currentColor"/>
                                </svg>
                                <span>Dec 25, 2024</span>
                            </div>
                            <div class="info-item" style="display: flex; align-items: center; gap: 6px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12 2C8.13 2 5 5.13 5 9C5 14.25 12 22 12 22S19 14.25 19 9C19 5.13 15.87 2 12 2ZM12 11.5C10.62 11.5 9.5 10.38 9.5 9S10.62 6.5 12 6.5S14.5 7.62 14.5 9S13.38 11.5 12 11.5Z" fill="currentColor"/>
                                </svg>
                                <span>Sample Venue</span>
                            </div>
                            <div class="info-item" style="display: flex; align-items: center; gap: 6px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12 2C6.48 2 2 6.48 2 12S6.48 22 12 22S22 17.52 22 12S17.52 2 12 2ZM13.41 18.09L13.41 20H10.59V18.09C9.59 17.89 8.79 17.42 8.2 16.7L9.66 15.24C10.1 15.68 10.65 15.9 11.3 15.9C12.1 15.9 12.5 15.5 12.5 14.9C12.5 14.3 12.1 13.9 11.3 13.9H10.7C9.3 13.9 8.6 13.2 8.6 11.8C8.6 10.4 9.6 9.7 10.59 9.51V8H13.41V9.51C14.41 9.71 15.21 10.18 15.8 10.9L14.34 12.36C13.9 11.92 13.35 11.7 12.7 11.7C11.9 11.7 11.5 12.1 11.5 12.7C11.5 13.3 11.9 13.7 12.7 13.7H13.3C14.7 13.7 15.4 14.4 15.4 15.8C15.4 17.2 14.4 17.89 13.41 18.09Z" fill="currentColor"/>
                                </svg>
                                <span>EUR 25.00</span>
                            </div>
                            <div class="info-item" style="display: flex; align-items: center; gap: 6px;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M22 10V6C22 4.9 21.1 4 20 4H4C2.9 4 2 4.9 2 6V10C3.1 10 4 10.9 4 12S3.1 14 2 14V18C2 19.1 2.9 20 4 20H20C21.1 20 22 19.1 22 18V14C20.9 14 20 13.1 20 12S20.9 10 22 10ZM13 17.5H11V16.5H13V17.5ZM13 15.5H11V14.5H13V15.5ZM13 13.5H11V12.5H13V13.5ZM13 11.5H11V10.5H13V11.5ZM13 9.5H11V8.5H13V9.5ZM13 7.5H11V6.5H13V7.5Z" fill="currentColor"/>
                                </svg>
                                <span>100 tickets</span>
                            </div>
                        </div>
                        
                        <button class="book-button" style="background: ${settings.button_background}; color: ${settings.button_text_color}; border: none; padding: 12px 24px; border-radius: 8px; cursor: pointer; font-weight: 600; font-size: ${settings.button_font_size}px; font-family: ${settings.card_font_family}; margin-top: 10px; display: inline-flex; align-items: center; gap: 8px; transition: all 0.3s ease;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M22 10V6C22 4.9 21.1 4 20 4H4C2.9 4 2 4.9 2 6V10C3.1 10 4 10.9 4 12S3.1 14 2 14V18C2 19.1 2.9 20 4 20H20C21.1 20 22 19.1 22 18V14C20.9 14 20 13.1 20 12S20.9 10 22 10Z" fill="currentColor"/>
                            </svg>
                            ${settings.button_text}
                        </button>
                    </div>
                </div>
            `;
        }
        
        // Initial preview update
        updatePreview();
    }

    // Initialize when DOM is ready
    $(document).ready(function() {
        TicketPayGoAdmin.init();
    });

    // Make functions globally available
    window.TicketPayGoAdmin = TicketPayGoAdmin;

})(jQuery);
