<?php
/**
 * Admin Settings Page (Lite Version)
 * Tabbed settings with General, Email Settings, and Customization
 * 
 * phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
 */

if (!defined('ABSPATH')) {
    exit;
}

// Verify user capabilities
if (!current_user_can('manage_options')) {
    wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
}

// Show success message after PayPal disconnect redirect
if (isset($_GET['disconnected']) && $_GET['disconnected'] === '1') {
    echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('PayPal credentials have been disconnected and cleared.', 'ticketpaygo-lite') . '</p></div>';
}

// Handle table creation
if (isset($_GET['tpgl_create_tables']) && current_user_can('manage_options')) {
    if (isset($_GET['_wpnonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'tpgl_create_tables')) {
        require_once TPGL_PLUGIN_PATH . 'includes/class-database.php';
        $results = TicketPayGo_Database::create_tables();
        
        if (TicketPayGo_Database::tables_exist()) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Database tables created successfully!', 'ticketpaygo-lite') . '</p></div>';
        } else {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Failed to create some database tables. Please check error logs.', 'ticketpaygo-lite') . '</p></div>';
        }
    }
}

// Handle PayPal disconnect
if (isset($_POST['disconnect_paypal_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['disconnect_paypal_nonce'])), 'disconnect_paypal_action')) {
    TicketPayGo_Database::update_setting('paypal_enabled', '0');
    TicketPayGo_Database::update_setting('paypal_client_id', '');
    TicketPayGo_Database::update_setting('paypal_client_secret', '');
    TicketPayGo_Database::update_setting('paypal_sandbox', '1');

    // Redirect to prevent form resubmission and ensure fresh data
    wp_safe_redirect(add_query_arg(array('page' => 'ticketpaygo-lite-settings', 'disconnected' => '1'), admin_url('admin.php')));
    exit;
}

// Handle form submission for General tab
if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['tpgl_settings_nonce'])) {
    if (wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['tpgl_settings_nonce'])), 'tpgl_settings_action')) {
        
        // General Settings
        TicketPayGo_Database::update_setting('currency', isset($_POST['currency']) ? sanitize_text_field(wp_unslash($_POST['currency'])) : 'EUR');
        TicketPayGo_Database::update_setting('currency_symbol', isset($_POST['currency_symbol']) ? sanitize_text_field(wp_unslash($_POST['currency_symbol'])) : '€');
        TicketPayGo_Database::update_setting('currency_position', isset($_POST['currency_position']) ? sanitize_text_field(wp_unslash($_POST['currency_position'])) : 'before');
        TicketPayGo_Database::update_setting('timezone', isset($_POST['timezone']) ? sanitize_text_field(wp_unslash($_POST['timezone'])) : 'UTC');
        
        // Free Event Registration
        TicketPayGo_Database::update_setting('free_event_registration', isset($_POST['free_event_registration']) ? sanitize_text_field(wp_unslash($_POST['free_event_registration'])) : 'required');
        
        // Terms & Privacy URLs
        TicketPayGo_Database::update_setting('terms_url', isset($_POST['terms_url']) ? esc_url_raw(wp_unslash($_POST['terms_url'])) : '');
        TicketPayGo_Database::update_setting('privacy_url', isset($_POST['privacy_url']) ? esc_url_raw(wp_unslash($_POST['privacy_url'])) : '');
        
        // PayPal Settings
        TicketPayGo_Database::update_setting('paypal_enabled', isset($_POST['paypal_enabled']) ? '1' : '0');
        TicketPayGo_Database::update_setting('paypal_client_id', isset($_POST['paypal_client_id']) ? sanitize_text_field(wp_unslash($_POST['paypal_client_id'])) : '');
        TicketPayGo_Database::update_setting('paypal_client_secret', isset($_POST['paypal_client_secret']) ? sanitize_text_field(wp_unslash($_POST['paypal_client_secret'])) : '');
        TicketPayGo_Database::update_setting('paypal_sandbox', isset($_POST['paypal_sandbox']) ? '1' : '0');
        
        echo '<div class="notice notice-success"><p>' . esc_html__('Settings saved successfully!', 'ticketpaygo-lite') . '</p></div>';
    }
}

// Get current settings
$settings = array(
    'currency' => TicketPayGo_Database::get_setting('currency', 'EUR'),
    'currency_symbol' => TicketPayGo_Database::get_setting('currency_symbol', '€'),
    'currency_position' => TicketPayGo_Database::get_setting('currency_position', 'before'),
    'timezone' => TicketPayGo_Database::get_setting('timezone', 'Europe/Amsterdam'),
    'free_event_registration' => TicketPayGo_Database::get_setting('free_event_registration', 'required'),
    'terms_url' => TicketPayGo_Database::get_setting('terms_url', ''),
    'privacy_url' => TicketPayGo_Database::get_setting('privacy_url', ''),
    'paypal_enabled' => TicketPayGo_Database::get_setting('paypal_enabled', '0'),
    'paypal_client_id' => TicketPayGo_Database::get_setting('paypal_client_id', ''),
    'paypal_client_secret' => TicketPayGo_Database::get_setting('paypal_client_secret', ''),
    'paypal_sandbox' => TicketPayGo_Database::get_setting('paypal_sandbox', '1')
);

// Determine current tab
$current_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'general';
$tabs = array(
    'general' => __('General', 'ticketpaygo-lite'),
    'email' => __('Email Settings', 'ticketpaygo-lite'),
    'customization' => __('Customization', 'ticketpaygo-lite'),
);

?>

<div class="wrap">
    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
        <h1 style="margin: 0;"><?php esc_html_e('TicketPayGo Lite Settings', 'ticketpaygo-lite'); ?></h1>
        <a href="https://ticketpaygo.upkyk.com/docs#setup" target="_blank" class="button button-secondary">
            <span class="dashicons dashicons-book-alt" style="margin-top: 3px;"></span> <?php esc_html_e('Documentation', 'ticketpaygo-lite'); ?>
        </a>
    </div>
    
    <nav class="nav-tab-wrapper">
        <?php foreach ($tabs as $tab_id => $tab_name): ?>
            <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-settings&tab=' . $tab_id)); ?>" 
               class="nav-tab <?php echo $current_tab === $tab_id ? 'nav-tab-active' : ''; ?>">
                <?php echo esc_html($tab_name); ?>
            </a>
        <?php endforeach; ?>
    </nav>
    
    <div class="tab-content" style="margin-top: 20px;">
    
    <?php if ($current_tab === 'email'): ?>
        <?php include TPGL_PLUGIN_PATH . 'admin/email-settings.php'; ?>
    <?php elseif ($current_tab === 'customization'): ?>
        <?php include TPGL_PLUGIN_PATH . 'admin/customization.php'; ?>
    <?php else: ?>
    
    <form method="post" action="">
        <?php wp_nonce_field('tpgl_settings_action', 'tpgl_settings_nonce'); ?>
        
        <div class="tpgl-settings-container">
            
            <!-- General Settings -->
            <div class="card" style="max-width: 800px; padding: 20px; margin-bottom: 20px;">
                <h2 style="margin-top: 0;"><?php esc_html_e('General Settings', 'ticketpaygo-lite'); ?></h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="currency"><?php esc_html_e('Default Currency', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <select id="currency" name="currency">
                                <option value="EUR" <?php selected($settings['currency'], 'EUR'); ?>>EUR (€)</option>
                                <option value="USD" <?php selected($settings['currency'], 'USD'); ?>>USD ($)</option>
                                <option value="GBP" <?php selected($settings['currency'], 'GBP'); ?>>GBP (£)</option>
                                <option value="CAD" <?php selected($settings['currency'], 'CAD'); ?>>CAD (C$)</option>
                                <option value="AUD" <?php selected($settings['currency'], 'AUD'); ?>>AUD (A$)</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="currency_symbol"><?php esc_html_e('Currency Symbol', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="currency_symbol" name="currency_symbol" class="regular-text" 
                                   value="<?php echo esc_attr($settings['currency_symbol']); ?>">
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="currency_position"><?php esc_html_e('Currency Position', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <select id="currency_position" name="currency_position">
                                <option value="before" <?php selected($settings['currency_position'], 'before'); ?>><?php esc_html_e('Before amount (€10.00)', 'ticketpaygo-lite'); ?></option>
                                <option value="after" <?php selected($settings['currency_position'], 'after'); ?>><?php esc_html_e('After amount (10.00€)', 'ticketpaygo-lite'); ?></option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="timezone"><?php esc_html_e('Timezone', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <select id="timezone" name="timezone">
                                <option value="Europe/Amsterdam" <?php selected($settings['timezone'], 'Europe/Amsterdam'); ?>>Europe/Amsterdam</option>
                                <option value="Europe/London" <?php selected($settings['timezone'], 'Europe/London'); ?>>Europe/London</option>
                                <option value="Europe/Paris" <?php selected($settings['timezone'], 'Europe/Paris'); ?>>Europe/Paris</option>
                                <option value="Europe/Berlin" <?php selected($settings['timezone'], 'Europe/Berlin'); ?>>Europe/Berlin</option>
                                <option value="America/New_York" <?php selected($settings['timezone'], 'America/New_York'); ?>>America/New_York</option>
                                <option value="America/Chicago" <?php selected($settings['timezone'], 'America/Chicago'); ?>>America/Chicago</option>
                                <option value="America/Los_Angeles" <?php selected($settings['timezone'], 'America/Los_Angeles'); ?>>America/Los_Angeles</option>
                                <option value="Asia/Tokyo" <?php selected($settings['timezone'], 'Asia/Tokyo'); ?>>Asia/Tokyo</option>
                                <option value="Australia/Sydney" <?php selected($settings['timezone'], 'Australia/Sydney'); ?>>Australia/Sydney</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="free_event_registration"><?php esc_html_e('Free Event Registration', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <select id="free_event_registration" name="free_event_registration">
                                <option value="required" <?php selected($settings['free_event_registration'], 'required'); ?>><?php esc_html_e('Require registration (send emails & tickets)', 'ticketpaygo-lite'); ?></option>
                                <option value="optional" <?php selected($settings['free_event_registration'], 'optional'); ?>><?php esc_html_e('No registration needed (walk-ins allowed)', 'ticketpaygo-lite'); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e('Choose whether free events require registration or allow walk-ins without tickets. "No registration" saves on emails and prevents abuse.', 'ticketpaygo-lite'); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="terms_url"><?php esc_html_e('Terms & Conditions URL', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <input type="url" id="terms_url" name="terms_url" class="regular-text" 
                                   value="<?php echo esc_attr($settings['terms_url']); ?>" placeholder="https://yoursite.com/terms">
                            <p class="description"><?php esc_html_e('URL to your Terms & Conditions page. This will be shown in the booking modal.', 'ticketpaygo-lite'); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="privacy_url"><?php esc_html_e('Privacy Policy URL', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <input type="url" id="privacy_url" name="privacy_url" class="regular-text" 
                                   value="<?php echo esc_attr($settings['privacy_url']); ?>" placeholder="https://yoursite.com/privacy">
                            <p class="description"><?php esc_html_e('URL to your Privacy Policy page. This will be shown in the booking modal.', 'ticketpaygo-lite'); ?></p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- PayPal Settings -->
            <div class="card" style="max-width: 800px; padding: 20px; margin-bottom: 20px;">
                <h2 style="margin-top: 0;"><?php esc_html_e('PayPal Payment Gateway', 'ticketpaygo-lite'); ?></h2>
                <p style="color: #666; margin-bottom: 20px;"><?php esc_html_e('Accept payments via PayPal for your ticket sales.', 'ticketpaygo-lite'); ?></p>
                
                <?php
                // Test PayPal connection if credentials are set
                $paypal_status = null;
                $paypal_error = '';
                $paypal_debug = '';
                if (!empty($settings['paypal_client_id']) && !empty($settings['paypal_client_secret'])) {
                    $sandbox = $settings['paypal_sandbox'] === '1';
                    $base_url = $sandbox ? 'https://api-m.sandbox.paypal.com' : 'https://api-m.paypal.com';
                    
                    $client_id = trim($settings['paypal_client_id']);
                    $client_secret = trim($settings['paypal_client_secret']);
                    
                    $auth_string = $client_id . ':' . $client_secret;
                    $auth_base64 = base64_encode($auth_string);
                    
                    $token_response = wp_remote_post($base_url . '/v1/oauth2/token', array(
                        'timeout' => 30,
                        'sslverify' => true,
                        'headers' => array(
                            'Accept' => 'application/json',
                            'Accept-Language' => 'en_US',
                            'Content-Type' => 'application/x-www-form-urlencoded',
                            'Authorization' => 'Basic ' . $auth_base64
                        ),
                        'body' => 'grant_type=client_credentials'
                    ));
                    
                    if (!is_wp_error($token_response)) {
                        $token_code = wp_remote_retrieve_response_code($token_response);
                        $token_body_raw = wp_remote_retrieve_body($token_response);
                        $token_body = json_decode($token_body_raw, true);
                        
                        if ($token_code === 200 && !empty($token_body['access_token'])) {
                            $paypal_status = true;
                        } else {
                            $paypal_status = false;
                            $paypal_error = isset($token_body['error_description']) ? $token_body['error_description'] : 'Authentication failed (HTTP ' . $token_code . ')';
                            $id_preview = substr($client_id, 0, 4) . '...' . substr($client_id, -4);
                            $secret_preview = substr($client_secret, 0, 4) . '...' . substr($client_secret, -4);
                            $paypal_debug = 'Mode: ' . ($sandbox ? 'Sandbox' : 'Live') . ' | Client ID: ' . $id_preview . ' (len:' . strlen($client_id) . ') | Secret: ' . $secret_preview . ' (len:' . strlen($client_secret) . ') | HTTP: ' . $token_code;
                        }
                    } else {
                        $paypal_status = false;
                        $paypal_error = 'Connection error: ' . $token_response->get_error_message();
                    }
                }
                ?>
                
                <?php if ($paypal_status === true): ?>
                    <div style="background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 12px 16px; border-radius: 6px; margin-bottom: 20px; display: flex; align-items: center; justify-content: space-between;">
                        <div style="display: flex; align-items: center; gap: 10px;">
                            <span style="font-size: 20px;">✓</span>
                            <div>
                                <strong><?php esc_html_e('PayPal Connected', 'ticketpaygo-lite'); ?></strong>
                                <span style="margin-left: 10px; font-size: 13px;"><?php echo $settings['paypal_sandbox'] === '1' ? esc_html__('(Sandbox Mode)', 'ticketpaygo-lite') : esc_html__('(Live Mode)', 'ticketpaygo-lite'); ?></span>
                            </div>
                        </div>
                        <form method="post" style="margin: 0;" onsubmit="return confirm('<?php esc_attr_e('Are you sure you want to disconnect PayPal? This will clear all credentials.', 'ticketpaygo-lite'); ?>');">
                            <?php wp_nonce_field('disconnect_paypal_action', 'disconnect_paypal_nonce'); ?>
                            <button type="submit" class="button" style="background: #721c24; color: white; border: none; padding: 6px 12px; border-radius: 4px; cursor: pointer; font-size: 13px;">
                                <?php esc_html_e('Disconnect', 'ticketpaygo-lite'); ?>
                            </button>
                        </form>
                    </div>
                <?php elseif ($paypal_status === false): ?>
                    <div style="background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 12px 16px; border-radius: 6px; margin-bottom: 20px;">
                        <div style="display: flex; align-items: center; gap: 10px;">
                            <span style="font-size: 20px;">✗</span>
                            <div>
                                <strong><?php esc_html_e('PayPal Connection Failed', 'ticketpaygo-lite'); ?></strong>
                                <p style="margin: 5px 0 0 0; font-size: 13px;"><?php echo esc_html($paypal_error); ?></p>
                            </div>
                        </div>
                        <?php if (!empty($paypal_debug)): ?>
                            <details style="margin-top: 10px; font-size: 12px;">
                                <summary style="cursor: pointer;"><?php esc_html_e('Debug Info', 'ticketpaygo-lite'); ?></summary>
                                <pre style="margin: 5px 0 0 0; padding: 8px; background: #fff; border-radius: 4px; overflow-x: auto;"><?php echo esc_html($paypal_debug); ?></pre>
                            </details>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
                
                <input type="text" name="fake_username" style="position: absolute; left: -9999px;" tabindex="-1" autocomplete="off">
                <input type="password" name="fake_password" style="position: absolute; left: -9999px;" tabindex="-1" autocomplete="new-password">
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="paypal_enabled"><?php esc_html_e('Enable PayPal', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" id="paypal_enabled" name="paypal_enabled" value="1" <?php checked($settings['paypal_enabled'], '1'); ?>>
                                <?php esc_html_e('Enable PayPal payments', 'ticketpaygo-lite'); ?>
                            </label>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="paypal_client_id"><?php esc_html_e('Client ID', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="paypal_client_id" name="paypal_client_id" value="<?php echo esc_attr($settings['paypal_client_id']); ?>" class="regular-text" autocomplete="off" data-lpignore="true" data-form-type="other">
                            <p class="description"><?php esc_html_e('Your PayPal REST API Client ID', 'ticketpaygo-lite'); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="paypal_client_secret"><?php esc_html_e('Client Secret', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="paypal_client_secret" name="paypal_client_secret" value="<?php echo esc_attr($settings['paypal_client_secret']); ?>" class="regular-text" autocomplete="off" data-lpignore="true" data-form-type="other" style="-webkit-text-security: disc;">
                            <p class="description"><?php esc_html_e('Your PayPal REST API Client Secret', 'ticketpaygo-lite'); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="paypal_sandbox"><?php esc_html_e('Sandbox Mode', 'ticketpaygo-lite'); ?></label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" id="paypal_sandbox" name="paypal_sandbox" value="1" <?php checked($settings['paypal_sandbox'], '1'); ?>>
                                <?php esc_html_e('Enable sandbox/test mode', 'ticketpaygo-lite'); ?>
                            </label>
                            <p class="description"><?php esc_html_e('Use PayPal sandbox for testing. Disable for live payments.', 'ticketpaygo-lite'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <div style="background: #f0f6fc; padding: 15px; border-radius: 6px; margin-top: 15px;">
                    <strong><?php esc_html_e('How to get PayPal API credentials:', 'ticketpaygo-lite'); ?></strong>
                    <ol style="margin: 10px 0 0 20px;">
                        <li><?php esc_html_e('Go to developer.paypal.com and log in', 'ticketpaygo-lite'); ?></li>
                        <li><?php esc_html_e('Navigate to Apps & Credentials', 'ticketpaygo-lite'); ?></li>
                        <li><?php esc_html_e('Create a new app or use an existing one', 'ticketpaygo-lite'); ?></li>
                        <li><?php esc_html_e('Copy the Client ID and Secret', 'ticketpaygo-lite'); ?></li>
                    </ol>
                </div>
            </div>
            
            <!-- Premium Features Upsell -->
            <div class="card" style="max-width: 800px; padding: 25px; margin-bottom: 20px; background: linear-gradient(135deg, #0A4974 0%, #42963B 100%); color: white; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
                <h2 style="margin-top: 0; margin-bottom: 12px; color: white; font-size: 22px;"><?php esc_html_e('Need More Features?', 'ticketpaygo-lite'); ?></h2>
                <p style="margin-bottom: 16px; opacity: 0.95;"><?php esc_html_e('Additional features available with the TicketPayGo add-on:', 'ticketpaygo-lite'); ?></p>
                <ul style="margin-left: 20px; margin-bottom: 20px; line-height: 1.8; opacity: 0.95;">
                    <li><?php esc_html_e('Multiple payment gateways (Stripe, PayPal, Mollie)', 'ticketpaygo-lite'); ?></li>
                    <li><?php esc_html_e('QR Code tickets & Apple Wallet integration', 'ticketpaygo-lite'); ?></li>
                    <li><?php esc_html_e('Advanced ticket customization', 'ticketpaygo-lite'); ?></li>
                    <li><?php esc_html_e('Detailed analytics and reports', 'ticketpaygo-lite'); ?></li>
                    <li><?php esc_html_e('Mobile QR scanner app', 'ticketpaygo-lite'); ?></li>
                </ul>
                <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-upgrade')); ?>" class="button" style="background: white; color: #0A4974; border: none; font-weight: 600; padding: 10px 24px; border-radius: 6px; box-shadow: 0 2px 8px rgba(0,0,0,0.15);">
                    <?php esc_html_e('Learn More', 'ticketpaygo-lite'); ?>
                </a>
            </div>
            
        </div>
        
        <p class="submit">
            <input type="submit" class="button-primary" value="<?php esc_attr_e('Save Settings', 'ticketpaygo-lite'); ?>" style="background: linear-gradient(135deg, #0A4974 0%, #42963B 100%) !important; color: white !important; border: none !important; padding: 8px 20px !important; font-weight: 600 !important;">
        </p>
    </form>
    
    <!-- System Status -->
    <div class="card" style="max-width: 800px; padding: 20px; margin-top: 30px;">
        <h2 style="margin-top: 0;"><?php esc_html_e('System Status', 'ticketpaygo-lite'); ?></h2>
        
        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Plugin Version', 'ticketpaygo-lite'); ?></th>
                <td><?php echo esc_html(TPGL_VERSION); ?></td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('WordPress Version', 'ticketpaygo-lite'); ?></th>
                <td><?php echo esc_html(get_bloginfo('version')); ?></td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('PHP Version', 'ticketpaygo-lite'); ?></th>
                <td><?php echo esc_html(PHP_VERSION); ?></td>
            </tr>
            
            <tr>
                <th scope="row"><?php esc_html_e('Database Tables', 'ticketpaygo-lite'); ?></th>
                <td>
                    <?php
                    global $wpdb;
                    $tables = array(
                        $wpdb->prefix . 'ticketpaygo_lite_events',
                        $wpdb->prefix . 'ticketpaygo_lite_orders',
                        $wpdb->prefix . 'ticketpaygo_lite_tickets',
                        $wpdb->prefix . 'ticketpaygo_lite_settings'
                    );
                    
                    $missing_tables = array();
                    foreach ($tables as $table) {
                        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                        if ($table_exists != $table) {
                            $missing_tables[] = str_replace($wpdb->prefix, '', $table);
                        }
                    }
                    
                    if (empty($missing_tables)) {
                        echo '<span style="color: #00a32a;">' . esc_html__('All tables exist', 'ticketpaygo-lite') . '</span>';
                    } else {
                        /* translators: %s: comma-separated list of missing database table names */
                        echo '<span style="color: #d63638;">' . sprintf(esc_html__('Missing tables: %s', 'ticketpaygo-lite'), esc_html(implode(', ', $missing_tables))) . '</span>';
                        echo '<br><br>';
                        $create_url = wp_nonce_url(
                            admin_url('admin.php?page=ticketpaygo-lite-settings&tpgl_create_tables=1'),
                            'tpgl_create_tables'
                        );
                        echo '<a href="' . esc_url($create_url) . '" class="button button-primary" style="background: linear-gradient(135deg, #0A4974 0%, #42963B 100%) !important; color: white !important; border: none !important;">' . esc_html__('Create Missing Tables', 'ticketpaygo-lite') . '</a>';
                    }
                    ?>
                </td>
            </tr>
        </table>
    </div>
    
    <?php endif; ?>
    
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    // WordPress color picker
    if ($.fn.wpColorPicker) {
        $('.color-picker').wpColorPicker();
    }
    
    // Logo upload
    var mediaUploader;
    $('#upload_logo_button').on('click', function(e) {
        e.preventDefault();
        
        if (mediaUploader) {
            mediaUploader.open();
            return;
        }
        
        mediaUploader = wp.media({
            title: '<?php echo esc_js(__('Choose Logo', 'ticketpaygo-lite')); ?>',
            button: { text: '<?php echo esc_js(__('Use this logo', 'ticketpaygo-lite')); ?>' },
            multiple: false,
            library: { type: 'image' }
        });
        
        mediaUploader.on('select', function() {
            var attachment = mediaUploader.state().get('selection').first().toJSON();
            $('#ticket_organizer_logo').val(attachment.url);
        });
        
        mediaUploader.open();
    });
});
</script>
