<?php
/**
 * Admin Orders Page
 * 
 * phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
 * phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter
 */

/**
 * Admin Orders Page
 *
 * @package TicketPayGo
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Check user capabilities
if (!current_user_can('manage_options')) {
    wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
}

global $wpdb;
$orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
$events_table = $wpdb->prefix . 'ticketpaygo_lite_events';
$tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';

// Handle single order delete via GET
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['order_id'])) {
    check_admin_referer('ticketpaygo_lite_delete_order_' . intval($_GET['order_id']));
    $del_id = intval($_GET['order_id']);
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    $wpdb->delete($tickets_table, array('order_id' => $del_id), array('%d'));
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    $wpdb->delete($orders_table, array('id' => $del_id), array('%d'));
    echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Order deleted.', 'ticketpaygo-lite') . '</p></div>';
}

// Handle bulk actions
if (isset($_POST['action']) && $_POST['action'] !== '-1') {
    check_admin_referer('ticketpaygo_lite_orders_bulk');
    
    $action = sanitize_text_field(wp_unslash($_POST['action'])); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Input is sanitized
    $order_ids = array_map('intval', isset($_POST['order_ids']) ? $_POST['order_ids'] : array());
    
    if (!empty($order_ids)) {
        switch ($action) {
            case 'mark_completed':
                foreach ($order_ids as $order_id) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Admin bulk action on custom table
                    $wpdb->update(
                        $orders_table,
                        ['status' => 'completed'],
                        ['id' => $order_id],
                        ['%s'],
                        ['%d']
                    );
                }
                echo '<div class="notice notice-success"><p>' . esc_html__('Orders marked as completed.', 'ticketpaygo-lite') . '</p></div>';
                break;
                
            case 'mark_cancelled':
                foreach ($order_ids as $order_id) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Admin bulk action on custom table
                    $wpdb->update(
                        $orders_table,
                        ['status' => 'cancelled'],
                        ['id' => $order_id],
                        ['%s'],
                        ['%d']
                    );
                }
                echo '<div class="notice notice-success"><p>' . esc_html__('Orders marked as cancelled.', 'ticketpaygo-lite') . '</p></div>';
                break;
                
            case 'delete':
                foreach ($order_ids as $order_id) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Admin bulk action on custom table
                    $wpdb->delete($tickets_table, array('order_id' => $order_id), array('%d'));
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Admin bulk action on custom table
                    $wpdb->delete($orders_table, array('id' => $order_id), array('%d'));
                }
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Orders deleted.', 'ticketpaygo-lite') . '</p></div>';
                break;
        }
    }
}

// Get filter parameters
// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only filtering in admin page with manage_options capability check
$status_filter = isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : '';
$event_filter = isset($_GET['event_id']) ? intval($_GET['event_id']) : 0;
$date_from = isset($_GET['date_from']) ? sanitize_text_field(wp_unslash($_GET['date_from'])) : '';
$date_to = isset($_GET['date_to']) ? sanitize_text_field(wp_unslash($_GET['date_to'])) : '';
$search = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
// phpcs:enable WordPress.Security.NonceVerification.Recommended

// Build query
$where_conditions = ['1=1'];
$where_values = [];

if ($status_filter) {
    $where_conditions[] = 'o.status = %s';
    $where_values[] = $status_filter;
}

if ($event_filter) {
    $where_conditions[] = 'o.event_id = %d';
    $where_values[] = $event_filter;
}

if ($date_from) {
    $where_conditions[] = 'DATE(o.created_at) >= %s';
    $where_values[] = $date_from;
}

if ($date_to) {
    $where_conditions[] = 'DATE(o.created_at) <= %s';
    $where_values[] = $date_to;
}

if ($search) {
    $where_conditions[] = '(o.order_number LIKE %s OR o.customer_name LIKE %s OR o.customer_email LIKE %s)';
    $search_term = '%' . $wpdb->esc_like($search) . '%';
    $where_values[] = $search_term;
    $where_values[] = $search_term;
    $where_values[] = $search_term;
}

$where_clause = implode(' AND ', $where_conditions);

// Pagination
$per_page = 20;
$current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
$offset = ($current_page - 1) * $per_page;

// Get total count
$count_query = "SELECT COUNT(*) FROM {$orders_table} o WHERE {$where_clause}";
if (!empty($where_values)) {
    $count_query = $wpdb->prepare($count_query, $where_values); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is dynamically built with proper placeholders
}
// Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_orders'
// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table, dynamically built query with proper placeholders
$total_orders = $wpdb->get_var($count_query);
$total_pages = ceil($total_orders / $per_page);

// Get orders
$orders_query = "
    SELECT o.*, e.title as event_title 
    FROM {$orders_table} o 
    LEFT JOIN {$events_table} e ON o.event_id = e.id 
    WHERE {$where_clause} 
    ORDER BY o.created_at DESC 
    LIMIT %d OFFSET %d
";

$query_values = array_merge($where_values, [$per_page, $offset]);
// Table names are safe: constructed from $wpdb->prefix + hardcoded suffixes
// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom tables, dynamically built query with proper placeholders
$orders = $wpdb->get_results($wpdb->prepare($orders_query, $query_values));

// Get events for filter dropdown
// Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_events'
// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table, table name is safely constructed from prefix + hardcoded suffix
$events = $wpdb->get_results("SELECT id, title FROM {$events_table} ORDER BY title");

// Get status counts
// Table name is safe: constructed from $wpdb->prefix + hardcoded 'ticketpaygo_lite_orders'
// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table, table name is safely constructed from prefix + hardcoded suffix
$status_counts = $wpdb->get_results("
    SELECT status, COUNT(*) as count 
    FROM {$orders_table}
    GROUP BY status
", OBJECT_K);
?>

<div class="wrap">
    <h1 class="wp-heading-inline"><?php esc_html_e('Orders', 'ticketpaygo-lite'); ?></h1>
    
    <!-- Filters -->
    <div class="tablenav top">
        <form method="get" class="alignleft actions">
            <input type="hidden" name="page" value="ticketpaygo-lite-orders">
            
            <select name="status">
                <option value=""><?php esc_html_e('All Statuses', 'ticketpaygo-lite'); ?></option>
                <option value="pending" <?php selected($status_filter, 'pending'); ?>><?php esc_html_e('Pending', 'ticketpaygo-lite'); ?></option>
                <option value="completed" <?php selected($status_filter, 'completed'); ?>><?php esc_html_e('Completed', 'ticketpaygo-lite'); ?></option>
                <option value="cancelled" <?php selected($status_filter, 'cancelled'); ?>><?php esc_html_e('Cancelled', 'ticketpaygo-lite'); ?></option>
                <option value="refunded" <?php selected($status_filter, 'refunded'); ?>><?php esc_html_e('Refunded', 'ticketpaygo-lite'); ?></option>
            </select>
            
            <select name="event_id">
                <option value=""><?php esc_html_e('All Events', 'ticketpaygo-lite'); ?></option>
                <?php foreach ($events as $event): ?>
                    <option value="<?php echo esc_attr($event->id); ?>" <?php selected($event_filter, $event->id); ?>>
                        <?php echo esc_html($event->title); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            
            <input type="date" name="date_from" value="<?php echo esc_attr($date_from); ?>" placeholder="<?php esc_html_e('From Date', 'ticketpaygo-lite'); ?>">
            <input type="date" name="date_to" value="<?php echo esc_attr($date_to); ?>" placeholder="<?php esc_html_e('To Date', 'ticketpaygo-lite'); ?>">
            
            <?php submit_button(__('Filter', 'ticketpaygo-lite'), 'secondary', '', false); ?>
            
            <?php if ($status_filter || $event_filter || $date_from || $date_to || $search): ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-orders')); ?>" class="button">
                    <?php esc_html_e('Clear Filters', 'ticketpaygo-lite'); ?>
                </a>
            <?php endif; ?>
        </form>
        
        <form method="get" class="search-box">
            <input type="hidden" name="page" value="ticketpaygo-lite-orders">
            <?php if ($status_filter): ?><input type="hidden" name="status" value="<?php echo esc_attr($status_filter); ?>"><?php endif; ?>
            <?php if ($event_filter): ?><input type="hidden" name="event_id" value="<?php echo esc_attr($event_filter); ?>"><?php endif; ?>
            <input type="search" name="s" value="<?php echo esc_attr($search); ?>" placeholder="<?php esc_html_e('Search orders...', 'ticketpaygo-lite'); ?>">
            <?php submit_button(__('Search', 'ticketpaygo-lite'), 'secondary', '', false); ?>
        </form>
    </div>

    <!-- Status Summary -->
    <ul class="subsubsub">
        <li class="all">
            <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-orders')); ?>" <?php echo !$status_filter ? 'class="current"' : ''; ?>>
                <?php esc_html_e('All', 'ticketpaygo-lite'); ?> <span class="count">(<?php echo intval($total_orders); ?>)</span>
            </a> |
        </li>
        <?php foreach (['pending', 'completed', 'cancelled', 'refunded'] as $status): ?>
            <?php $count = isset($status_counts[$status]) ? $status_counts[$status]->count : 0; ?>
            <li class="<?php echo esc_attr($status); ?>">
                <a href="<?php echo esc_url(add_query_arg('status', $status, admin_url('admin.php?page=ticketpaygo-lite-orders'))); ?>" 
                   <?php echo $status_filter === $status ? 'class="current"' : ''; ?>>
                    <?php echo esc_html(ucfirst($status)); ?> <span class="count">(<?php echo intval($count); ?>)</span>
                </a>
                <?php if ($status !== 'refunded'): ?> | <?php endif; ?>
            </li>
        <?php endforeach; ?>
    </ul>

    <!-- Orders Table -->
    <form method="post">
        <?php wp_nonce_field('ticketpaygo_lite_orders_bulk'); ?>
        
        <div class="tablenav top">
            <div class="alignleft actions bulkactions">
                <select name="action">
                    <option value="-1"><?php esc_html_e('Bulk Actions', 'ticketpaygo-lite'); ?></option>
                    <option value="mark_completed"><?php esc_html_e('Mark as Completed', 'ticketpaygo-lite'); ?></option>
                    <option value="mark_cancelled"><?php esc_html_e('Mark as Cancelled', 'ticketpaygo-lite'); ?></option>
                    <option value="delete"><?php esc_html_e('Delete', 'ticketpaygo-lite'); ?></option>
                </select>
                <?php submit_button(__('Apply', 'ticketpaygo-lite'), 'action', '', false); ?>
            </div>
            
            <div class="tablenav-pages">
                <?php if ($total_pages > 1): ?>
                    <?php /* translators: %s: total number of orders */ ?>
                    <span class="displaying-num"><?php printf(esc_html__('%s items', 'ticketpaygo-lite'), esc_html(number_format_i18n($total_orders))); ?></span>
                    <?php
                    echo wp_kses_post(paginate_links([
                        'base' => add_query_arg('paged', '%#%'),
                        'format' => '',
                        'prev_text' => __('&laquo;', 'ticketpaygo-lite'),
                        'next_text' => __('&raquo;', 'ticketpaygo-lite'),
                        'total' => $total_pages,
                        'current' => $current_page
                    ]));
                    ?>
                <?php endif; ?>
            </div>
        </div>

        <table class="wp-list-table widefat fixed striped orders">
            <thead>
                <tr>
                    <td class="manage-column column-cb check-column">
                        <input type="checkbox" id="cb-select-all-1">
                    </td>
                    <th class="manage-column column-order"><?php esc_html_e('Order', 'ticketpaygo-lite'); ?></th>
                    <th class="manage-column column-event"><?php esc_html_e('Event', 'ticketpaygo-lite'); ?></th>
                    <th class="manage-column column-customer"><?php esc_html_e('Customer', 'ticketpaygo-lite'); ?></th>
                    <th class="manage-column column-quantity"><?php esc_html_e('Quantity', 'ticketpaygo-lite'); ?></th>
                    <th class="manage-column column-amount"><?php esc_html_e('Amount', 'ticketpaygo-lite'); ?></th>
                    <th class="manage-column column-status"><?php esc_html_e('Status', 'ticketpaygo-lite'); ?></th>
                    <th class="manage-column column-date"><?php esc_html_e('Date', 'ticketpaygo-lite'); ?></th>
                    <th class="manage-column column-actions"><?php esc_html_e('Actions', 'ticketpaygo-lite'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($orders)): ?>
                    <tr>
                        <td colspan="9" class="no-orders">
                            <div class="no-orders-message">
                                <span class="dashicons dashicons-cart"></span>
                                <h3><?php esc_html_e('No orders found', 'ticketpaygo-lite'); ?></h3>
                                <p><?php esc_html_e('Orders will appear here once customers start purchasing tickets.', 'ticketpaygo-lite'); ?></p>
                            </div>
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($orders as $order): ?>
                        <tr>
                            <th class="check-column">
                                <input type="checkbox" name="order_ids[]" value="<?php echo esc_attr($order->id); ?>">
                            </th>
                            <td class="column-order">
                                <strong>#<?php echo esc_html($order->order_number); ?></strong>
                                <div class="row-actions">
                                    <span class="view">
                                        <a href="#" onclick="viewOrderDetails(<?php echo intval($order->id); ?>); return false;"><?php esc_html_e('View', 'ticketpaygo-lite'); ?></a> |
                                    </span>
                                    <span class="edit">
                                        <a href="#" onclick="editOrder(<?php echo intval($order->id); ?>); return false;"><?php esc_html_e('Edit', 'ticketpaygo-lite'); ?></a> |
                                    </span>
                                    <span class="delete">
                                        <a href="<?php echo esc_url(wp_nonce_url(add_query_arg(array('action' => 'delete', 'order_id' => $order->id), admin_url('admin.php?page=ticketpaygo-lite-orders')), 'ticketpaygo_lite_delete_order_' . $order->id)); ?>" onclick="return confirm('<?php esc_attr_e('Delete this order? This cannot be undone.', 'ticketpaygo-lite'); ?>')" style="color:#b32d2e;"><?php esc_html_e('Delete', 'ticketpaygo-lite'); ?></a>
                                    </span>
                                </div>
                            </td>
                            <td class="column-event">
                                <?php echo esc_html($order->event_title ?: __('Unknown Event', 'ticketpaygo-lite')); ?>
                            </td>
                            <td class="column-customer">
                                <strong><?php echo esc_html($order->customer_name); ?></strong><br>
                                <a href="mailto:<?php echo esc_attr($order->customer_email); ?>"><?php echo esc_html($order->customer_email); ?></a>
                            </td>
                            <td class="column-quantity">
                                <?php echo intval($order->quantity); ?>
                            </td>
                            <td class="column-amount">
                                <strong><?php echo esc_html($order->currency); ?> <?php echo number_format($order->total_amount, 2); ?></strong>
                                <?php if ($order->payment_method): ?>
                                    <br><small><?php echo esc_html(ucfirst($order->payment_method)); ?></small>
                                <?php endif; ?>
                            </td>
                            <td class="column-status">
                                <span class="status-badge status-<?php echo esc_attr($order->status); ?>">
                                    <?php echo esc_html(ucfirst($order->status)); ?>
                                </span>
                            </td>
                            <td class="column-date">
                                <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($order->created_at))); ?>
                            </td>
                            <td class="column-actions">
                                <div class="order-actions">
                                    <?php if ($order->status === 'completed'): ?>
                                        <a href="#" class="button button-small" onclick="resendTickets(<?php echo intval($order->id); ?>)">
                                            <?php esc_html_e('Resend Tickets', 'ticketpaygo-lite'); ?>
                                        </a>
                                    <?php endif; ?>
                                    <?php if (in_array($order->status, ['completed', 'pending'])): ?>
                                        <a href="#" class="button button-small" onclick="refundOrder(<?php echo intval($order->id); ?>)">
                                            <?php esc_html_e('Refund', 'ticketpaygo-lite'); ?>
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>

        <div class="tablenav bottom">
            <div class="alignleft actions bulkactions">
                <select name="action2">
                    <option value="-1"><?php esc_html_e('Bulk Actions', 'ticketpaygo-lite'); ?></option>
                    <option value="mark_completed"><?php esc_html_e('Mark as Completed', 'ticketpaygo-lite'); ?></option>
                    <option value="mark_cancelled"><?php esc_html_e('Mark as Cancelled', 'ticketpaygo-lite'); ?></option>
                    <option value="delete"><?php esc_html_e('Delete', 'ticketpaygo-lite'); ?></option>
                </select>
                <?php submit_button(__('Apply', 'ticketpaygo-lite'), 'action', '', false); ?>
            </div>
            
            <div class="tablenav-pages">
                <?php if ($total_pages > 1): ?>
                    <?php
                    echo wp_kses_post(paginate_links([
                        'base' => add_query_arg('paged', '%#%'),
                        'format' => '',
                        'prev_text' => __('&laquo;', 'ticketpaygo-lite'),
                        'next_text' => __('&raquo;', 'ticketpaygo-lite'),
                        'total' => $total_pages,
                        'current' => $current_page
                    ]));
                    ?>
                <?php endif; ?>
            </div>
        </div>
    </form>
</div>

<!-- Order View/Edit Modal -->
<div id="tpg-order-modal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.6); z-index:100000; overflow-y:auto;">
    <div style="background:#fff; margin:40px auto; max-width:700px; border-radius:6px; box-shadow:0 4px 20px rgba(0,0,0,0.3); position:relative;">
        <div style="background:#1d2327; color:#fff; padding:16px 20px; border-radius:6px 6px 0 0; display:flex; justify-content:space-between; align-items:center;">
            <h2 id="tpg-modal-title" style="margin:0; font-size:16px;"></h2>
            <button onclick="closeTpgModal()" style="background:none; border:none; color:#fff; font-size:20px; cursor:pointer; line-height:1;">&times;</button>
        </div>
        <div id="tpg-modal-body" style="padding:20px;">
            <p><?php esc_html_e('Loading...', 'ticketpaygo-lite'); ?></p>
        </div>
    </div>
</div>

<script>
var tpgAdminNonce = '<?php echo esc_js(wp_create_nonce('ticketpaygo_admin_nonce')); ?>';

function closeTpgModal() {
    document.getElementById('tpg-order-modal').style.display = 'none';
}

document.getElementById('tpg-order-modal').addEventListener('click', function(e) {
    if (e.target === this) closeTpgModal();
});

function viewOrderDetails(orderId) {
    document.getElementById('tpg-modal-title').textContent = '<?php esc_attr_e('Order Details', 'ticketpaygo-lite'); ?>';
    document.getElementById('tpg-modal-body').innerHTML = '<p><?php esc_attr_e('Loading...', 'ticketpaygo-lite'); ?></p>';
    document.getElementById('tpg-order-modal').style.display = 'block';
    jQuery.post(ajaxurl, {
        action: 'ticketpaygo_get_order_details',
        order_id: orderId,
        nonce: tpgAdminNonce
    }, function(response) {
        if (!response.success) {
            document.getElementById('tpg-modal-body').innerHTML = '<p style="color:red;">' + (response.data || 'Error') + '</p>';
            return;
        }
        var o = response.data.order;
        var tickets = response.data.tickets;
        var statusColors = {pending:'#b45309', completed:'#166534', cancelled:'#991b1b', refunded:'#1e40af'};
        var sc = statusColors[o.status] || '#374151';
        var html = '<table class="widefat" style="margin-bottom:20px;"><tbody>';
        html += '<tr><th style="width:160px;"><?php esc_attr_e('Order Number', 'ticketpaygo-lite'); ?></th><td><strong>#' + o.order_number + '</strong></td></tr>';
        html += '<tr><th><?php esc_attr_e('Event', 'ticketpaygo-lite'); ?></th><td>' + (o.event_title || '&mdash;') + '</td></tr>';
        html += '<tr><th><?php esc_attr_e('Customer', 'ticketpaygo-lite'); ?></th><td>' + o.customer_name + '<br><a href="mailto:' + o.customer_email + '">' + o.customer_email + '</a>' + (o.customer_phone ? '<br>' + o.customer_phone : '') + '</td></tr>';
        html += '<tr><th><?php esc_attr_e('Amount', 'ticketpaygo-lite'); ?></th><td>' + o.currency + ' ' + parseFloat(o.total_amount).toFixed(2) + ' &times; ' + o.quantity + ' ticket(s)</td></tr>';
        html += '<tr><th><?php esc_attr_e('Payment', 'ticketpaygo-lite'); ?></th><td>' + (o.payment_method || '&mdash;') + (o.payment_id ? ' &mdash; <code>' + o.payment_id + '</code>' : '') + '</td></tr>';
        html += '<tr><th><?php esc_attr_e('Status', 'ticketpaygo-lite'); ?></th><td><span style="background:' + sc + ';color:#fff;padding:2px 8px;border-radius:3px;font-size:12px;">' + o.status.charAt(0).toUpperCase() + o.status.slice(1) + '</span></td></tr>';
        html += '<tr><th><?php esc_attr_e('Date', 'ticketpaygo-lite'); ?></th><td>' + o.created_at + '</td></tr>';
        if (o.notes) html += '<tr><th><?php esc_attr_e('Notes', 'ticketpaygo-lite'); ?></th><td>' + o.notes + '</td></tr>';
        html += '</tbody></table>';
        if (tickets && tickets.length > 0) {
            html += '<h3 style="margin:0 0 10px;"><?php esc_attr_e('Tickets', 'ticketpaygo-lite'); ?> (' + tickets.length + ')</h3>';
            html += '<table class="widefat"><thead><tr><th>#</th><th><?php esc_attr_e('Ticket Number', 'ticketpaygo-lite'); ?></th><th><?php esc_attr_e('Type', 'ticketpaygo-lite'); ?></th><th><?php esc_attr_e('Holder', 'ticketpaygo-lite'); ?></th><th><?php esc_attr_e('Status', 'ticketpaygo-lite'); ?></th></tr></thead><tbody>';
            tickets.forEach(function(t, i) {
                html += '<tr><td>' + (i+1) + '</td><td><code>' + t.ticket_number + '</code></td><td>' + (t.ticket_type || '&mdash;') + '</td><td>' + (t.holder_name || '&mdash;') + '</td><td>' + (t.status || '&mdash;') + '</td></tr>';
            });
            html += '</tbody></table>';
        }
        document.getElementById('tpg-modal-body').innerHTML = html;
    }).fail(function() {
        document.getElementById('tpg-modal-body').innerHTML = '<p style="color:red;"><?php esc_attr_e('Request failed.', 'ticketpaygo-lite'); ?></p>';
    });
}

function editOrder(orderId) {
    document.getElementById('tpg-modal-title').textContent = '<?php esc_attr_e('Edit Order', 'ticketpaygo-lite'); ?>';
    document.getElementById('tpg-modal-body').innerHTML = '<p><?php esc_attr_e('Loading...', 'ticketpaygo-lite'); ?></p>';
    document.getElementById('tpg-order-modal').style.display = 'block';
    jQuery.post(ajaxurl, {
        action: 'ticketpaygo_get_order_details',
        order_id: orderId,
        nonce: tpgAdminNonce
    }, function(response) {
        if (!response.success) {
            document.getElementById('tpg-modal-body').innerHTML = '<p style="color:red;">' + (response.data || 'Error') + '</p>';
            return;
        }
        var o = response.data.order;
        var html = '<form id="tpg-edit-order-form">';
        html += '<table class="form-table"><tbody>';
        html += '<tr><th><?php esc_attr_e('Order', 'ticketpaygo-lite'); ?></th><td><strong>#' + o.order_number + '</strong></td></tr>';
        html += '<tr><th><?php esc_attr_e('Customer', 'ticketpaygo-lite'); ?></th><td>' + o.customer_name + ' &lt;' + o.customer_email + '&gt;</td></tr>';
        html += '<tr><th><label for="tpg-edit-status"><?php esc_attr_e('Status', 'ticketpaygo-lite'); ?></label></th><td>';
        html += '<select id="tpg-edit-status" name="status" style="min-width:160px;">';
        ['pending','completed','cancelled','refunded'].forEach(function(s) {
            html += '<option value="' + s + '"' + (o.status === s ? ' selected' : '') + '>' + s.charAt(0).toUpperCase() + s.slice(1) + '</option>';
        });
        html += '</select></td></tr>';
        html += '<tr><th><label for="tpg-edit-notes"><?php esc_attr_e('Notes', 'ticketpaygo-lite'); ?></label></th><td><textarea id="tpg-edit-notes" name="notes" rows="4" style="width:100%;">' + (o.notes || '') + '</textarea></td></tr>';
        html += '</tbody></table>';
        html += '<p><button type="button" class="button button-primary" onclick="saveOrder(' + orderId + ')"><?php esc_attr_e('Save Changes', 'ticketpaygo-lite'); ?></button> ';
        html += '<button type="button" class="button" onclick="closeTpgModal()"><?php esc_attr_e('Cancel', 'ticketpaygo-lite'); ?></button></p>';
        html += '</form>';
        document.getElementById('tpg-modal-body').innerHTML = html;
    }).fail(function() {
        document.getElementById('tpg-modal-body').innerHTML = '<p style="color:red;"><?php esc_attr_e('Request failed.', 'ticketpaygo-lite'); ?></p>';
    });
}

function saveOrder(orderId) {
    var status = document.getElementById('tpg-edit-status').value;
    var notes  = document.getElementById('tpg-edit-notes').value;
    var btn = document.querySelector('#tpg-edit-order-form .button-primary');
    btn.disabled = true;
    btn.textContent = '<?php esc_attr_e('Saving...', 'ticketpaygo-lite'); ?>';
    jQuery.post(ajaxurl, {
        action: 'ticketpaygo_update_order',
        order_id: orderId,
        status: status,
        notes: notes,
        nonce: tpgAdminNonce
    }, function(response) {
        if (response.success) {
            closeTpgModal();
            location.reload();
        } else {
            alert('<?php esc_attr_e('Save failed: ', 'ticketpaygo-lite'); ?>' + (response.data || ''));
            btn.disabled = false;
            btn.textContent = '<?php esc_attr_e('Save Changes', 'ticketpaygo-lite'); ?>';
        }
    }).fail(function() {
        alert('<?php esc_attr_e('Request failed.', 'ticketpaygo-lite'); ?>');
        btn.disabled = false;
        btn.textContent = '<?php esc_attr_e('Save Changes', 'ticketpaygo-lite'); ?>';
    });
}

function resendTickets(orderId) {
    if (confirm('<?php esc_attr_e('Are you sure you want to resend tickets for this order?', 'ticketpaygo-lite'); ?>')) {
        jQuery.post(ajaxurl, {
            action: 'ticketpaygo_resend_tickets',
            order_id: orderId,
            nonce: '<?php echo esc_js(wp_create_nonce('ticketpaygo_resend_tickets')); ?>'
        }, function(response) {
            if (response.success) {
                alert('<?php esc_attr_e('Tickets resent successfully!', 'ticketpaygo-lite'); ?>');
            } else {
                alert('<?php esc_attr_e('Failed to resend tickets: ', 'ticketpaygo-lite'); ?>' + (response.data || ''));
            }
        }).fail(function() {
            alert('<?php esc_attr_e('Error sending request', 'ticketpaygo-lite'); ?>');
        });
    }
}

// Select all checkbox
document.getElementById('cb-select-all-1').addEventListener('change', function() {
    document.querySelectorAll('input[name="order_ids[]"]').forEach(function(cb) {
        cb.checked = this.checked;
    }, this);
});
</script>

