<?php
/**
 * Admin Events Page
 */

if (!defined('ABSPATH')) {
    exit;
}

function tpgl_render_events_page() {
    // Handle actions
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters for display logic, properly sanitized
    $tpgl_action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : 'list';
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameters for display logic, properly sanitized
    $tpgl_event_id = isset($_GET['id']) ? intval($_GET['id']) : (isset($_GET['event_id']) ? intval($_GET['event_id']) : 0);

    // Handle bulk actions
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_action_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['bulk_action_nonce'])), 'bulk_action')) {
        $tpgl_bulk_action = isset($_POST['action']) && $_POST['action'] !== '-1' ? sanitize_text_field(wp_unslash($_POST['action'])) : '';
        $tpgl_event_ids = isset($_POST['event']) ? array_map('intval', $_POST['event']) : array();
    
        if (!empty($tpgl_bulk_action) && !empty($tpgl_event_ids)) {
            $tpgl_success_count = 0;
            $tpgl_error_count = 0;
            
            foreach ($tpgl_event_ids as $tpgl_id) {
                $tpgl_event = new TicketPayGo_Event($tpgl_id);
                
                if ($tpgl_bulk_action === 'delete') {
                    if ($tpgl_event->delete()) {
                        $tpgl_success_count++;
                    } else {
                        $tpgl_error_count++;
                    }
                } elseif ($tpgl_bulk_action === 'activate') {
                    if ($tpgl_event->update(array_merge($tpgl_event->get_data(), array('status' => 'active')))) {
                        $tpgl_success_count++;
                    } else {
                        $tpgl_error_count++;
                    }
                } elseif ($tpgl_bulk_action === 'deactivate') {
                    if ($tpgl_event->update(array_merge($tpgl_event->get_data(), array('status' => 'inactive')))) {
                        $tpgl_success_count++;
                    } else {
                        $tpgl_error_count++;
                    }
                }
            }
            
            if ($tpgl_success_count > 0) {
                $tpgl_action_label = $tpgl_bulk_action === 'delete' ? esc_html__('deleted', 'ticketpaygo-lite') : ($tpgl_bulk_action === 'activate' ? esc_html__('activated', 'ticketpaygo-lite') : esc_html__('deactivated', 'ticketpaygo-lite'));
                /* translators: %1$d: number of events, %2$s: action label (deleted/activated/deactivated) */
                echo '<div class="notice notice-success"><p>' . sprintf(esc_html__('%1$d event(s) %2$s successfully!', 'ticketpaygo-lite'), intval($tpgl_success_count), esc_html($tpgl_action_label)) . '</p></div>';
            }
            if ($tpgl_error_count > 0) {
                /* translators: %d: number of events that failed to process */
                echo '<div class="notice notice-error"><p>' . sprintf(esc_html__('Failed to process %d event(s).', 'ticketpaygo-lite'), intval($tpgl_error_count)) . '</p></div>';
            }
        }
    }

    // Process form submissions
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['ticketpaygo_event_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['ticketpaygo_event_nonce'])), 'ticketpaygo_event_action')) {
            $tpgl_event = new TicketPayGo_Event($tpgl_event_id);
        
            // Process ticket types if enabled
            $tpgl_ticket_types = array();
            $tpgl_total_max_tickets = 0;
            $tpgl_base_price = isset($_POST['price']) ? floatval(wp_unslash($_POST['price'])) : 0;
        
            if (isset($_POST['enable_ticket_types']) && $_POST['enable_ticket_types'] === '1' && !empty($_POST['ticket_types'])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Array elements sanitized individually below
                $tpgl_ticket_types_raw = isset($_POST['ticket_types']) ? wp_unslash($_POST['ticket_types']) : array();
                foreach ($tpgl_ticket_types_raw as $tpgl_index => $tpgl_type) {
                    if (!empty($tpgl_type['name'])) {
                        $tpgl_ticket_type = array(
                            'name' => sanitize_text_field($tpgl_type['name']),
                            'price' => isset($tpgl_type['price']) ? floatval($tpgl_type['price']) : 0,
                            'quantity' => isset($tpgl_type['quantity']) ? intval($tpgl_type['quantity']) : 0,
                            'description' => isset($tpgl_type['description']) ? sanitize_text_field($tpgl_type['description']) : '',
                            'sale_start' => isset($tpgl_type['sale_start']) ? sanitize_text_field($tpgl_type['sale_start']) : '',
                            'sale_end' => isset($tpgl_type['sale_end']) ? sanitize_text_field($tpgl_type['sale_end']) : '',
                            'max_per_order' => isset($tpgl_type['max_per_order']) ? intval($tpgl_type['max_per_order']) : 10,
                            'sold' => 0
                        );
                        $tpgl_ticket_types[] = $tpgl_ticket_type;
                        $tpgl_total_max_tickets += $tpgl_ticket_type['quantity'];
                        
                        // Use lowest price as base price for display
                        if ($tpgl_base_price === 0 || $tpgl_ticket_type['price'] < $tpgl_base_price) {
                            $tpgl_base_price = $tpgl_ticket_type['price'];
                        }
                    }
                }
            }
        
            // If using multiple ticket types, calculate total max from types
            $tpgl_max_tickets = !empty($tpgl_ticket_types) ? $tpgl_total_max_tickets : (isset($_POST['max_tickets']) ? intval(wp_unslash($_POST['max_tickets'])) : 100);
        
            // Process recurrence data
            $tpgl_recurrence = array();
            if (isset($_POST['recurrence']['enabled']) && $_POST['recurrence']['enabled'] === '1') {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Array elements sanitized individually below
                $tpgl_recurrence_data = isset($_POST['recurrence']) ? wp_unslash($_POST['recurrence']) : array();
                $tpgl_recurrence = array(
                    'enabled' => true,
                    'pattern' => isset($tpgl_recurrence_data['pattern']) ? sanitize_text_field($tpgl_recurrence_data['pattern']) : 'weekly',
                    'interval' => isset($tpgl_recurrence_data['interval']) ? intval($tpgl_recurrence_data['interval']) : 1,
                    'days' => isset($tpgl_recurrence_data['days']) ? array_map('sanitize_text_field', (array) $tpgl_recurrence_data['days']) : array(),
                    'end_date' => isset($tpgl_recurrence_data['end_date']) ? sanitize_text_field($tpgl_recurrence_data['end_date']) : '',
                    'max_occurrences' => isset($_POST['recurrence']['max_occurrences']) ? intval(wp_unslash($_POST['recurrence']['max_occurrences'])) : 10
                );
            
                // Clear end_date or max_occurrences based on selected end type
                if (isset($_POST['recurrence_end_type'])) {
                    if (sanitize_text_field(wp_unslash($_POST['recurrence_end_type'])) === 'date') {
                        $tpgl_recurrence['max_occurrences'] = 0;
                    } else {
                        $tpgl_recurrence['end_date'] = '';
                    }
                }
            }
        
            $tpgl_event_data = array(
                'title' => isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '',
                'description' => isset($_POST['description']) ? wp_kses_post(wp_unslash($_POST['description'])) : '',
                'start_date' => isset($_POST['start_date']) ? sanitize_text_field(wp_unslash($_POST['start_date'])) : '',
                'end_date' => isset($_POST['end_date']) ? sanitize_text_field(wp_unslash($_POST['end_date'])) : '',
                'location' => isset($_POST['location']) ? sanitize_text_field(wp_unslash($_POST['location'])) : '',
                'address' => isset($_POST['address']) ? sanitize_textarea_field(wp_unslash($_POST['address'])) : '',
                'latitude' => isset($_POST['latitude']) ? floatval($_POST['latitude']) : 0,
                'longitude' => isset($_POST['longitude']) ? floatval($_POST['longitude']) : 0,
                'max_tickets' => $tpgl_max_tickets,
                'price' => $tpgl_base_price,
                'currency' => isset($_POST['currency']) ? sanitize_text_field(wp_unslash($_POST['currency'])) : 'EUR',
                'status' => isset($_POST['status']) ? sanitize_text_field(wp_unslash($_POST['status'])) : 'active',
                'featured_image' => isset($_POST['featured_image']) ? sanitize_url(wp_unslash($_POST['featured_image'])) : '',
                'image_fit' => isset($_POST['image_fit']) ? sanitize_text_field(wp_unslash($_POST['image_fit'])) : '',
                'show_title_on_card' => isset($_POST['show_title_on_card']) ? 1 : 0,
                'ticket_name' => isset($_POST['ticket_name']) ? sanitize_text_field(wp_unslash($_POST['ticket_name'])) : '',
                'gallery' => array(),
                'organizer_name' => isset($_POST['organizer_name']) ? sanitize_text_field(wp_unslash($_POST['organizer_name'])) : '',
                'organizer_email' => isset($_POST['organizer_email']) ? sanitize_email(wp_unslash($_POST['organizer_email'])) : '',
                'organizer_phone' => isset($_POST['organizer_phone']) ? sanitize_text_field(wp_unslash($_POST['organizer_phone'])) : '',
                'ticket_types' => $tpgl_ticket_types,
                'recurrence' => $tpgl_recurrence,
                'custom_fields' => array(),
                'seo_title' => isset($_POST['seo_title']) ? sanitize_text_field(wp_unslash($_POST['seo_title'])) : '',
                'seo_description' => isset($_POST['seo_description']) ? sanitize_textarea_field(wp_unslash($_POST['seo_description'])) : ''
            );
        
            if ($tpgl_action === 'add') {
                // Validate required fields
                $tpgl_errors = array();
                if (empty($_POST['title'])) {
                    $tpgl_errors[] = __('Event title is required.', 'ticketpaygo-lite');
                }
                if (empty($_POST['start_date'])) {
                    $tpgl_errors[] = __('Start date is required.', 'ticketpaygo-lite');
                }
                if (empty($_POST['end_date'])) {
                    $tpgl_errors[] = __('End date is required.', 'ticketpaygo-lite');
                }
            
                if (empty($tpgl_errors)) {
                    $tpgl_result = $tpgl_event->create($tpgl_event_data);
                    if ($tpgl_result) {
                        // Redirect to edit page to prevent duplicate submissions on refresh (Post-Redirect-Get pattern)
                        $tpgl_redirect_url = add_query_arg(array(
                            'page' => 'ticketpaygo-lite-events',
                            'action' => 'edit',
                            'id' => $tpgl_result,
                            'created' => '1'
                        ), admin_url('admin.php'));
                        wp_safe_redirect($tpgl_redirect_url);
                        exit;
                    } else {
                        global $wpdb;
                        $tpgl_error_msg = $wpdb->last_error ? $wpdb->last_error : esc_html__('Unknown database error occurred.', 'ticketpaygo-lite');
                        /* translators: %s: error message from database */
                        echo '<div class="notice notice-error"><p>' . sprintf(esc_html__('Failed to create event: %s', 'ticketpaygo-lite'), esc_html($tpgl_error_msg)) . '</p></div>';
                    }
                } else {
                    echo '<div class="notice notice-error"><p>' . wp_kses(implode('<br>', $tpgl_errors), array('br' => array())) . '</p></div>';
                }
            } elseif ($tpgl_action === 'edit' && $tpgl_event->exists()) {
                if ($tpgl_event->update($tpgl_event_data)) {
                    // Redirect to prevent duplicate submissions on refresh
                    $tpgl_redirect_url = add_query_arg(array(
                        'page' => 'ticketpaygo-lite-events',
                        'action' => 'edit',
                        'id' => $tpgl_event_id,
                        'updated' => '1'
                    ), admin_url('admin.php'));
                    wp_safe_redirect($tpgl_redirect_url);
                    exit;
                } else {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Failed to update event.', 'ticketpaygo-lite') . '</p></div>';
                }
            }
        }
    }

    // Handle delete action
    if ($tpgl_action === 'delete' && $tpgl_event_id && isset($_GET['_wpnonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'delete_event_' . $tpgl_event_id)) {
        $tpgl_event = new TicketPayGo_Event($tpgl_event_id);
        if ($tpgl_event->delete()) {
            echo '<div class="notice notice-success"><p>' . esc_html__('Event deleted successfully!', 'ticketpaygo-lite') . '</p></div>';
            $tpgl_action = 'list';
        } else {
            echo '<div class="notice notice-error"><p>' . esc_html__('Failed to delete event.', 'ticketpaygo-lite') . '</p></div>';
        }
    }

?>

<div class="wrap">
    <h1>
        <?php esc_html_e('Events', 'ticketpaygo-lite'); ?>
        <?php if ($tpgl_action === 'list'): ?>
            <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-events&action=add')); ?>" class="page-title-action">
                <?php esc_html_e('Add New Event', 'ticketpaygo-lite'); ?>
            </a>
        <?php endif; ?>
    </h1>
    
    <?php
    // Show success message after redirect from creation
    if (isset($_GET['created']) && $_GET['created'] === '1' && $tpgl_action === 'edit' && $tpgl_event_id):
    ?>
        <div class="notice notice-success is-dismissible">
            <p><?php esc_html_e('Event created successfully!', 'ticketpaygo-lite'); ?></p>
            <div class="shortcode-info" style="background: #f0f8ff; padding: 10px; margin-top: 10px; border-left: 4px solid #0073aa;">
                <h4><?php esc_html_e('How to Display This Event:', 'ticketpaygo-lite'); ?></h4>
                <p><strong><?php esc_html_e('Single Event Shortcode:', 'ticketpaygo-lite'); ?></strong></p>
                <code>[ticketpaygo_lite_event id="<?php echo esc_attr($tpgl_event_id); ?>"]</code>
                <p><strong><?php esc_html_e('All Events List:', 'ticketpaygo-lite'); ?></strong></p>
                <code>[ticketpaygo_lite_events]</code>
                <p><strong><?php esc_html_e('Events Calendar:', 'ticketpaygo-lite'); ?></strong></p>
                <code>[ticketpaygo_lite_calendar]</code>
                <p><em><?php esc_html_e('Copy and paste these shortcodes into any page or post to display your events.', 'ticketpaygo-lite'); ?></em></p>
            </div>
        </div>
    <?php endif; ?>
    
    <?php
    // Show success message after redirect from update
    if (isset($_GET['updated']) && $_GET['updated'] === '1' && $tpgl_action === 'edit' && $tpgl_event_id):
    ?>
        <div class="notice notice-success is-dismissible">
            <p><?php esc_html_e('Event updated successfully!', 'ticketpaygo-lite'); ?></p>
        </div>
    <?php endif; ?>
    
    <?php if ($tpgl_action === 'list'): ?>
        <!-- Events List -->
        <?php
        $tpgl_events = TicketPayGo_Event::get_all_events(array('limit' => 50));
        ?>
        
        <form method="post" action="">
            <?php wp_nonce_field('bulk_action', 'bulk_action_nonce'); ?>
            
            <div class="tablenav top">
                <div class="alignleft actions">
                    <select name="action" id="bulk-action-selector-top">
                        <option value="-1"><?php esc_html_e('Bulk Actions', 'ticketpaygo-lite'); ?></option>
                        <option value="delete"><?php esc_html_e('Delete', 'ticketpaygo-lite'); ?></option>
                        <option value="activate"><?php esc_html_e('Activate', 'ticketpaygo-lite'); ?></option>
                        <option value="deactivate"><?php esc_html_e('Deactivate', 'ticketpaygo-lite'); ?></option>
                    </select>
                    <input type="submit" class="button action" value="<?php esc_html_e('Apply', 'ticketpaygo-lite'); ?>">
                </div>
            </div>
            
            <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <td class="manage-column column-cb check-column">
                        <input type="checkbox" id="cb-select-all-1">
                    </td>
                    <th><?php esc_html_e('Title', 'ticketpaygo-lite'); ?></th>
                    <th><?php esc_html_e('Date', 'ticketpaygo-lite'); ?></th>
                    <th><?php esc_html_e('Location', 'ticketpaygo-lite'); ?></th>
                    <th><?php esc_html_e('Price', 'ticketpaygo-lite'); ?></th>
                    <th><?php esc_html_e('Tickets Sold', 'ticketpaygo-lite'); ?></th>
                    <th><?php esc_html_e('Status', 'ticketpaygo-lite'); ?></th>
                    <th><?php esc_html_e('Actions', 'ticketpaygo-lite'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php if (!empty($tpgl_events)): ?>
                    <?php foreach ($tpgl_events as $tpgl_event_data): ?>
                        <?php $tpgl_event = new TicketPayGo_Event($tpgl_event_data['id']); ?>
                        <tr>
                            <th scope="row" class="check-column">
                                <input type="checkbox" name="event[]" value="<?php echo esc_attr($tpgl_event_data['id']); ?>">
                            </th>
                            <td>
                                <strong>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-events&action=edit&event_id=' . $tpgl_event_data['id'])); ?>">
                                        <?php echo esc_html($tpgl_event_data['title']); ?>
                                    </a>
                                </strong>
                                <div class="row-actions">
                                    <span class="edit">
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-events&action=edit&event_id=' . $tpgl_event_data['id'])); ?>">
                                            <?php esc_html_e('Edit', 'ticketpaygo-lite'); ?>
                                        </a> |
                                    </span>
                                    <span class="view">
                                        <a href="<?php echo esc_url(home_url('/event/' . $tpgl_event_data['id'] . '/')); ?>" target="_blank">
                                            <?php esc_html_e('View', 'ticketpaygo-lite'); ?>
                                        </a> |
                                    </span>
                                    <span class="shortcode">
                                        <a href="#" onclick="copyShortcode('ticketpaygo_lite_event', <?php echo intval($tpgl_event_data['id']); ?>); return false;">
                                            <?php esc_html_e('Copy Shortcode', 'ticketpaygo-lite'); ?>
                                        </a> |
                                    </span>
                                    <span class="delete">
                                        <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=ticketpaygo-lite-events&action=delete&event_id=' . $tpgl_event_data['id']), 'delete_event_' . $tpgl_event_data['id'])); ?>">
                                            <?php esc_html_e('Delete', 'ticketpaygo-lite'); ?>
                                        </a>
                                    </span>
                                </div>
                            </td>
                            <td>
                                <?php echo esc_html(gmdate('M j, Y g:i A', strtotime($tpgl_event_data['start_date']))); ?>
                            </td>
                            <td><?php echo esc_html($tpgl_event_data['location']); ?></td>
                            <td><?php echo esc_html($tpgl_event->get_formatted_price()); ?></td>
                            <td>
                                <?php 
                                $tpgl_tickets_sold = $tpgl_event->get_tickets_sold();
                                $tpgl_max_tickets = $tpgl_event->get('max_tickets');
                                echo intval($tpgl_tickets_sold);
                                if ($tpgl_max_tickets > 0) {
                                    echo ' / ' . intval($tpgl_max_tickets);
                                }
                                ?>
                            </td>
                            <td>
                                <?php 
                                $tpgl_display_status = $tpgl_event->get_display_status();
                                $tpgl_status_labels = array(
                                    'upcoming' => __('Upcoming', 'ticketpaygo-lite'),
                                    'ongoing' => __('Ongoing', 'ticketpaygo-lite'),
                                    'ended' => __('Ended', 'ticketpaygo-lite'),
                                    'active' => __('Active', 'ticketpaygo-lite'),
                                    'inactive' => __('Inactive', 'ticketpaygo-lite')
                                );
                                ?>
                                <span class="status-<?php echo esc_attr($tpgl_display_status); ?>">
                                    <?php echo esc_html($tpgl_status_labels[$tpgl_display_status] ?? ucfirst($tpgl_display_status)); ?>
                                </span>
                            </td>
                            <td>
                                <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-events&action=edit&event_id=' . $tpgl_event_data['id'])); ?>" class="button button-small">
                                    <?php esc_html_e('Edit', 'ticketpaygo-lite'); ?>
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="8"><?php esc_html_e('No events found.', 'ticketpaygo-lite'); ?></td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
        </form>
    <?php elseif ($tpgl_action === 'add' || $tpgl_action === 'edit'): ?>
        <!-- Event Form -->
        <?php
        $tpgl_event_data = array();
        if ($tpgl_action === 'edit' && $tpgl_event_id) {
            $tpgl_event = new TicketPayGo_Event($tpgl_event_id);
            if ($tpgl_event->exists()) {
                $tpgl_event_data = $tpgl_event->get_data();
            } else {
                echo '<div class="notice notice-error"><p>' . esc_html__('Event not found.', 'ticketpaygo-lite') . '</p></div>';
                return;
            }
        }
        ?>
        
        <form method="post" action="">
            <?php wp_nonce_field('ticketpaygo_event_action', 'ticketpaygo_event_nonce'); ?>
            
            <div class="ticketpaygo-form-container">
                <div class="form-section">
                    <h2><?php esc_html_e('Basic Information', 'ticketpaygo-lite'); ?></h2>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="title"><?php esc_html_e('Event Title', 'ticketpaygo-lite'); ?> *</label>
                            </th>
                            <td>
                                <input type="text" id="title" name="title" class="regular-text" 
                                       value="<?php echo esc_attr($tpgl_event_data['title'] ?? ''); ?>" required>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="description"><?php esc_html_e('Description', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <?php
                                wp_editor(
                                    $tpgl_event_data['description'] ?? '',
                                    'description',
                                    array(
                                        'textarea_name' => 'description',
                                        'media_buttons' => true,
                                        'textarea_rows' => 10,
                                        'tinymce' => array(
                                            'content_css' => false,
                                            'body_class' => 'tpgl-editor-content'
                                        ),
                                        'quicktags' => true
                                    )
                                );
                                ?>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="start_date"><?php esc_html_e('Start Date & Time', 'ticketpaygo-lite'); ?> *</label>
                            </th>
                            <td>
                                <input type="datetime-local" id="start_date" name="start_date" 
                                       value="<?php echo esc_attr(isset($tpgl_event_data['start_date']) ? gmdate('Y-m-d\TH:i', strtotime($tpgl_event_data['start_date'])) : ''); ?>" required>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="end_date"><?php esc_html_e('End Date & Time', 'ticketpaygo-lite'); ?> *</label>
                            </th>
                            <td>
                                <input type="datetime-local" id="end_date" name="end_date" 
                                       value="<?php echo esc_attr(isset($tpgl_event_data['end_date']) ? gmdate('Y-m-d\TH:i', strtotime($tpgl_event_data['end_date'])) : ''); ?>" required>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="form-section">
                    <h2><?php esc_html_e('Location', 'ticketpaygo-lite'); ?></h2>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="location"><?php esc_html_e('Venue Name', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <input type="text" id="location" name="location" class="regular-text" 
                                       value="<?php echo esc_attr($tpgl_event_data['location'] ?? ''); ?>">
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="address"><?php esc_html_e('Address', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <textarea id="address" name="address" rows="3" class="large-text"><?php echo esc_textarea($tpgl_event_data['address'] ?? ''); ?></textarea>
                                <p class="description"><?php esc_html_e('Enter the full address or use the map below to set the location', 'ticketpaygo-lite'); ?></p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e('Location on Map', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <div style="margin-bottom: 10px;">
                                    <input type="text" id="map_search" class="regular-text" placeholder="<?php esc_html_e('Search for a location...', 'ticketpaygo-lite'); ?>" style="width: 70%; margin-right: 10px;">
                                    <button type="button" id="search_location_btn" class="button"><?php esc_html_e('Search', 'ticketpaygo-lite'); ?></button>
                                </div>
                                <div id="event_location_map" style="width: 100%; height: 400px; border: 1px solid #ddd; border-radius: 4px; margin-bottom: 10px;"></div>
                                <p class="description">
                                    <span class="dashicons dashicons-info"></span>
                                    <?php esc_html_e('Click on the map to set the exact location, or search for an address above. The marker shows where your event will appear on maps.', 'ticketpaygo-lite'); ?>
                                </p>
                                
                                <!-- Hidden fields for coordinates -->
                                <input type="hidden" id="latitude" name="latitude" value="<?php echo esc_attr($tpgl_event_data['latitude'] ?? '52.3676'); ?>">
                                <input type="hidden" id="longitude" name="longitude" value="<?php echo esc_attr($tpgl_event_data['longitude'] ?? '4.9041'); ?>">
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="form-section">
                    <h2><?php esc_html_e('Ticketing', 'ticketpaygo-lite'); ?></h2>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="currency"><?php esc_html_e('Currency', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <select id="currency" name="currency">
                                    <option value="EUR" <?php selected($tpgl_event_data['currency'] ?? 'EUR', 'EUR'); ?>>EUR (€)</option>
                                    <option value="USD" <?php selected($tpgl_event_data['currency'] ?? 'EUR', 'USD'); ?>>USD ($)</option>
                                    <option value="GBP" <?php selected($tpgl_event_data['currency'] ?? 'EUR', 'GBP'); ?>>GBP (£)</option>
                                </select>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e('Enable Multiple Ticket Types', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <?php 
                                $tpgl_ticket_types = isset($tpgl_event_data['ticket_types']) ? (is_string($tpgl_event_data['ticket_types']) ? json_decode($tpgl_event_data['ticket_types'], true) : $tpgl_event_data['ticket_types']) : array();
                                $tpgl_has_multiple_types = !empty($tpgl_ticket_types) && count($tpgl_ticket_types) > 0;
                                ?>
                                <label style="color: #999; cursor: not-allowed;">
                                    <input type="checkbox" id="enable_ticket_types" name="enable_ticket_types" value="1" disabled>
                                    <?php esc_html_e('Use multiple ticket types (VIP, Early Bird, etc.)', 'ticketpaygo-lite'); ?>
                                </label>
                                <p class="description">
                                    <?php 
                                    printf(
                                        /* translators: %s: link to upgrade page */
                                        esc_html__('Multiple ticket types are available in %s.', 'ticketpaygo-lite'),
                                        '<a href="' . esc_url(admin_url('admin.php?page=ticketpaygo-lite-upgrade')) . '">' . esc_html__('TicketPayGo', 'ticketpaygo-lite') . '</a>'
                                    );
                                    ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    
                    <!-- Simple Pricing (shown when multiple types disabled) -->
                    <div id="simple-pricing" style="<?php echo $tpgl_has_multiple_types ? 'display:none;' : ''; ?>">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="ticket_name"><?php esc_html_e('Ticket Name', 'ticketpaygo-lite'); ?></label>
                                </th>
                                <td>
                                    <input type="text" id="ticket_name" name="ticket_name" class="regular-text" 
                                           value="<?php echo esc_attr($tpgl_event_data['ticket_name'] ?? ''); ?>"
                                           placeholder="<?php esc_html_e('e.g., General Admission, Entry Ticket', 'ticketpaygo-lite'); ?>">
                                    <p class="description"><?php esc_html_e('Leave empty to use "General Admission"', 'ticketpaygo-lite'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="price"><?php esc_html_e('Ticket Price', 'ticketpaygo-lite'); ?> *</label>
                                </th>
                                <td>
                                    <input type="number" id="price" name="price" step="0.01" min="0" class="regular-text" 
                                           value="<?php echo esc_attr($tpgl_event_data['price'] ?? '0'); ?>">
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="max_tickets"><?php esc_html_e('Maximum Tickets', 'ticketpaygo-lite'); ?></label>
                                </th>
                                <td>
                                    <input type="number" id="max_tickets" name="max_tickets" min="0" class="regular-text" 
                                           value="<?php echo esc_attr($tpgl_event_data['max_tickets'] ?? '0'); ?>">
                                    <p class="description"><?php esc_html_e('Set to 0 for unlimited tickets.', 'ticketpaygo-lite'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- Multiple Ticket Types (shown when enabled) -->
                    <div id="ticket-types-section" style="<?php echo $tpgl_has_multiple_types ? '' : 'display:none;'; ?>">
                        <div class="ticket-types-header" style="display: flex; justify-content: space-between; align-items: center; margin: 20px 0 10px;">
                            <h3 style="margin: 0;"><?php esc_html_e('Ticket Types', 'ticketpaygo-lite'); ?></h3>
                            <button type="button" id="add-ticket-type" class="button button-secondary">
                                <span class="dashicons dashicons-plus-alt" style="vertical-align: middle;"></span>
                                <?php esc_html_e('Add Ticket Type', 'ticketpaygo-lite'); ?>
                            </button>
                        </div>
                        
                        <div id="ticket-types-container">
                            <?php 
                            if (!empty($tpgl_ticket_types)):
                                foreach ($tpgl_ticket_types as $tpgl_index => $tpgl_type): 
                            ?>
                                <div class="ticket-type-row" data-index="<?php echo esc_attr($tpgl_index); ?>">
                                    <div class="ticket-type-header">
                                        <span class="ticket-type-title"><?php echo esc_html($tpgl_type['name'] ?? __('Ticket Type', 'ticketpaygo-lite')); ?></span>
                                        <button type="button" class="button-link ticket-type-toggle">
                                            <span class="dashicons dashicons-arrow-down-alt2"></span>
                                        </button>
                                        <button type="button" class="button-link ticket-type-remove" title="<?php esc_html_e('Remove', 'ticketpaygo-lite'); ?>">
                                            <span class="dashicons dashicons-trash"></span>
                                        </button>
                                    </div>
                                    <div class="ticket-type-fields">
                                        <div class="ticket-type-field-row">
                                            <div class="ticket-type-field">
                                                <label><?php esc_html_e('Name', 'ticketpaygo-lite'); ?></label>
                                                <input type="text" name="ticket_types[<?php echo esc_attr($tpgl_index); ?>][name]" 
                                                       value="<?php echo esc_attr($tpgl_type['name'] ?? ''); ?>" 
                                                       placeholder="<?php esc_html_e('e.g., General Admission', 'ticketpaygo-lite'); ?>">
                                            </div>
                                            <div class="ticket-type-field">
                                                <label><?php esc_html_e('Price', 'ticketpaygo-lite'); ?></label>
                                                <input type="number" name="ticket_types[<?php echo esc_attr($tpgl_index); ?>][price]" 
                                                       value="<?php echo esc_attr($tpgl_type['price'] ?? '0'); ?>" 
                                                       step="0.01" min="0" placeholder="0 = free">
                                            </div>
                                            <div class="ticket-type-field">
                                                <label><?php esc_html_e('Quantity', 'ticketpaygo-lite'); ?></label>
                                                <input type="number" name="ticket_types[<?php echo esc_attr($tpgl_index); ?>][quantity]" 
                                                       value="<?php echo esc_attr($tpgl_type['quantity'] ?? '0'); ?>" 
                                                       min="0" placeholder="0 = unlimited">
                                            </div>
                                        </div>
                                        <div class="ticket-type-field-row">
                                            <div class="ticket-type-field">
                                                <label><?php esc_html_e('Description', 'ticketpaygo-lite'); ?></label>
                                                <input type="text" name="ticket_types[<?php echo esc_attr($tpgl_index); ?>][description]" 
                                                       value="<?php echo esc_attr($tpgl_type['description'] ?? ''); ?>" 
                                                       placeholder="<?php esc_html_e('Brief description of what\'s included', 'ticketpaygo-lite'); ?>">
                                            </div>
                                        </div>
                                        <div class="ticket-type-field-row">
                                            <div class="ticket-type-field">
                                                <label><?php esc_html_e('Sale Start', 'ticketpaygo-lite'); ?></label>
                                                <input type="datetime-local" name="ticket_types[<?php echo esc_attr($tpgl_index); ?>][sale_start]" 
                                                       value="<?php echo esc_attr(isset($tpgl_type['sale_start']) && $tpgl_type['sale_start'] ? gmdate('Y-m-d\TH:i', strtotime($tpgl_type['sale_start'])) : ''); ?>">
                                            </div>
                                            <div class="ticket-type-field">
                                                <label><?php esc_html_e('Sale End', 'ticketpaygo-lite'); ?></label>
                                                <input type="datetime-local" name="ticket_types[<?php echo esc_attr($tpgl_index); ?>][sale_end]" 
                                                       value="<?php echo esc_attr(isset($tpgl_type['sale_end']) && $tpgl_type['sale_end'] ? gmdate('Y-m-d\TH:i', strtotime($tpgl_type['sale_end'])) : ''); ?>">
                                            </div>
                                            <div class="ticket-type-field">
                                                <label><?php esc_html_e('Max Per Order', 'ticketpaygo-lite'); ?></label>
                                                <input type="number" name="ticket_types[<?php echo esc_attr($tpgl_index); ?>][max_per_order]" 
                                                       value="<?php echo esc_attr($tpgl_type['max_per_order'] ?? '10'); ?>" 
                                                       min="1" max="100">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php 
                                endforeach;
                            endif; 
                            ?>
                        </div>
                        
                        <p class="description" style="margin-top: 15px;">
                            <?php esc_html_e('Create different ticket types with varying prices and availability. Each type can have its own quantity limit and sale period.', 'ticketpaygo-lite'); ?>
                        </p>
                    </div>
                </div>
                
                <!-- Recurring Events - Premium Feature -->
                <div class="form-section" style="background: #f8f9fa; border: 1px dashed #ccc;">
                    <h2><?php esc_html_e('Recurring Events', 'ticketpaygo-lite'); ?></h2>
                    <p style="color: #666;">
                        <?php esc_html_e('Create weekly, monthly, or custom recurring events using the TicketPayGo add-on.', 'ticketpaygo-lite'); ?>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-upgrade')); ?>"><?php esc_html_e('Learn more', 'ticketpaygo-lite'); ?></a>
                    </p>
                </div>
                
                <div class="form-section">
                    <h2><?php esc_html_e('Organizer Information', 'ticketpaygo-lite'); ?></h2>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="organizer_name"><?php esc_html_e('Organizer Name', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <input type="text" id="organizer_name" name="organizer_name" class="regular-text" 
                                       value="<?php echo esc_attr($tpgl_event_data['organizer_name'] ?? ''); ?>">
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="organizer_email"><?php esc_html_e('Organizer Email', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <input type="email" id="organizer_email" name="organizer_email" class="regular-text" 
                                       value="<?php echo esc_attr($tpgl_event_data['organizer_email'] ?? ''); ?>">
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="organizer_phone"><?php esc_html_e('Organizer Phone', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <input type="tel" id="organizer_phone" name="organizer_phone" class="regular-text" 
                                       value="<?php echo esc_attr($tpgl_event_data['organizer_phone'] ?? ''); ?>">
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="form-section">
                    <h2><?php esc_html_e('Settings', 'ticketpaygo-lite'); ?></h2>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="status"><?php esc_html_e('Status', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <select id="status" name="status">
                                    <option value="active" <?php selected($tpgl_event_data['status'] ?? 'active', 'active'); ?>><?php esc_html_e('Active', 'ticketpaygo-lite'); ?></option>
                                    <option value="inactive" <?php selected($tpgl_event_data['status'] ?? 'active', 'inactive'); ?>><?php esc_html_e('Inactive', 'ticketpaygo-lite'); ?></option>
                                    <option value="draft" <?php selected($tpgl_event_data['status'] ?? 'active', 'draft'); ?>><?php esc_html_e('Draft', 'ticketpaygo-lite'); ?></option>
                                </select>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="featured_image"><?php esc_html_e('Featured Image', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <div class="featured-image-field">
                                    <input type="hidden" id="featured_image" name="featured_image" value="<?php echo esc_attr($tpgl_event_data['featured_image'] ?? ''); ?>">
                                    <div id="featured_image_preview" class="image-preview" style="<?php echo empty($tpgl_event_data['featured_image']) ? 'display:none;' : ''; ?>">
                                        <?php if (!empty($tpgl_event_data['featured_image'])): ?>
                                            <img src="<?php echo esc_url($tpgl_event_data['featured_image']); ?>" alt="Featured Image" style="max-width: 300px; height: auto; border-radius: 8px;" />
                                        <?php endif; ?>
                                    </div>
                                    <div class="image-buttons" style="margin-top: 10px;">
                                        <button type="button" id="upload_featured_image" class="button button-secondary">
                                            <span class="dashicons dashicons-upload" style="vertical-align: middle;"></span>
                                            <?php esc_html_e('Select Image', 'ticketpaygo-lite'); ?>
                                        </button>
                                        <button type="button" id="remove_featured_image" class="button" style="<?php echo empty($tpgl_event_data['featured_image']) ? 'display:none;' : ''; ?>">
                                            <span class="dashicons dashicons-trash" style="vertical-align: middle;"></span>
                                            <?php esc_html_e('Remove', 'ticketpaygo-lite'); ?>
                                        </button>
                                    </div>
                                    <p class="description"><?php esc_html_e('Upload or select an image from your media library. This image will be displayed on the event card.', 'ticketpaygo-lite'); ?></p>
                                </div>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="image_fit"><?php esc_html_e('Image Fit', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <select id="image_fit" name="image_fit">
                                    <option value="cover" <?php selected($tpgl_event_data['image_fit'] ?? 'cover', 'cover'); ?>><?php esc_html_e('Cover (fill area, crop if needed)', 'ticketpaygo-lite'); ?></option>
                                    <option value="contain" <?php selected($tpgl_event_data['image_fit'] ?? '', 'contain'); ?>><?php esc_html_e('Contain (fit entire image)', 'ticketpaygo-lite'); ?></option>
                                    <option value="contain-padded" <?php selected($tpgl_event_data['image_fit'] ?? '', 'contain-padded'); ?>><?php esc_html_e('Contain with Padding', 'ticketpaygo-lite'); ?></option>
                                    <option value="fill" <?php selected($tpgl_event_data['image_fit'] ?? '', 'fill'); ?>><?php esc_html_e('Fill (stretch to fit)', 'ticketpaygo-lite'); ?></option>
                                </select>
                                <p class="description"><?php esc_html_e('How the featured image should fit in the card', 'ticketpaygo-lite'); ?></p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="show_title_on_card"><?php esc_html_e('Show Title on Card', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="show_title_on_card" name="show_title_on_card" value="1" <?php checked(isset($tpgl_event_data['show_title_on_card']) ? $tpgl_event_data['show_title_on_card'] : 1, 1); ?> />
                                    <?php esc_html_e('Display event title overlay on the card image', 'ticketpaygo-lite'); ?>
                                </label>
                                <p class="description"><?php esc_html_e('Uncheck if your image already contains the event title', 'ticketpaygo-lite'); ?></p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="seo_title"><?php esc_html_e('SEO Title', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <input type="text" id="seo_title" name="seo_title" class="regular-text" 
                                       value="<?php echo esc_attr($tpgl_event_data['seo_title'] ?? ''); ?>">
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="seo_description"><?php esc_html_e('SEO Description', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <textarea id="seo_description" name="seo_description" rows="3" class="large-text"><?php echo esc_textarea($tpgl_event_data['seo_description'] ?? ''); ?></textarea>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <p class="submit">
                <input type="submit" class="button-primary" value="<?php echo $tpgl_action === 'add' ? esc_attr__('Create Event', 'ticketpaygo-lite') : esc_attr__('Update Event', 'ticketpaygo-lite'); ?>" style="background: linear-gradient(135deg, #0A4974 0%, #42963B 100%) !important; color: white !important; border: none !important; padding: 8px 20px !important; font-weight: 600 !important;">
                <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-events')); ?>" class="button">
                    <?php esc_html_e('Cancel', 'ticketpaygo-lite'); ?>
                </a>
            </p>
        </form>
        
        <?php if ($tpgl_action === 'edit' && $tpgl_event_id): ?>
            <!-- Shortcode Information Box -->
            <div class="shortcode-info-box" style="background: #f0f8ff; border: 1px solid #0073aa; border-radius: 6px; padding: 20px; margin-top: 20px;">
                <h3 style="margin-top: 0; color: #0073aa;"><span class="dashicons dashicons-clipboard"></span> <?php esc_html_e('Shortcode Information', 'ticketpaygo-lite'); ?></h3>
                <p><?php esc_html_e('Use these shortcodes to display this event on your website:', 'ticketpaygo-lite'); ?></p>
                
                <div class="shortcode-item" style="margin: 15px 0;">
                    <strong><?php esc_html_e('Single Event Display:', 'ticketpaygo-lite'); ?></strong>
                    <div style="display: flex; align-items: center; gap: 10px; margin-top: 5px;">
                        <input type="text" readonly value="[ticketpaygo_lite_event id=&quot;<?php echo intval($tpgl_event_id); ?>&quot;]" 
                               id="shortcode-single" style="flex: 1; padding: 8px; background: #fff; border: 1px solid #ddd;">
                        <button type="button" class="button" onclick="copyToClipboard('shortcode-single')"><?php esc_html_e('Copy', 'ticketpaygo-lite'); ?></button>
                    </div>
                </div>
                
                <div class="shortcode-item" style="margin: 15px 0;">
                    <strong><?php esc_html_e('All Events List:', 'ticketpaygo-lite'); ?></strong>
                    <div style="display: flex; align-items: center; gap: 10px; margin-top: 5px;">
                        <input type="text" readonly value="[ticketpaygo_lite_events]" 
                               id="shortcode-list" style="flex: 1; padding: 8px; background: #fff; border: 1px solid #ddd;">
                        <button type="button" class="button" onclick="copyToClipboard('shortcode-list')"><?php esc_html_e('Copy', 'ticketpaygo-lite'); ?></button>
                    </div>
                </div>
                
                <div class="shortcode-item" style="margin: 15px 0;">
                    <strong><?php esc_html_e('Events Calendar:', 'ticketpaygo-lite'); ?></strong>
                    <div style="display: flex; align-items: center; gap: 10px; margin-top: 5px;">
                        <input type="text" readonly value="[ticketpaygo_lite_calendar]" 
                               id="shortcode-calendar" style="flex: 1; padding: 8px; background: #fff; border: 1px solid #ddd;">
                        <button type="button" class="button" onclick="copyToClipboard('shortcode-calendar')"><?php esc_html_e('Copy', 'ticketpaygo-lite'); ?></button>
                    </div>
                    <p class="description" style="margin-top: 5px; color: #666;"><?php esc_html_e('Displays all events in an interactive calendar view with month, week, and list options.', 'ticketpaygo-lite'); ?></p>
                </div>
                
                <p style="margin-bottom: 0; font-style: italic; color: #666;">
                    <?php esc_html_e('Paste these shortcodes into any page or post to display your events.', 'ticketpaygo-lite'); ?>
                </p>
            </div>
        <?php endif; ?>
        
    <?php endif; ?>
</div>

<?php
}
tpgl_render_events_page();
