<?php
/**
 * Admin Check-in Page (Lite Version)
 * Simple wp-admin based ticket check-in
 * 
 * phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
 * phpcs:disable WordPress.DB.DirectDatabaseQuery
 */

if (!defined('ABSPATH')) {
    exit;
}

// Verify user capabilities
if (!current_user_can('manage_options')) {
    wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ticketpaygo-lite'));
}

// Handle check-in action
if (isset($_POST['tpgl_checkin_action']) && isset($_POST['tpgl_checkin_nonce'])) {
    if (wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['tpgl_checkin_nonce'])), 'tpgl_checkin_action')) {
        $ticket_number = isset($_POST['ticket_number']) ? trim(sanitize_text_field(wp_unslash($_POST['ticket_number']))) : '';
        $event_id = isset($_POST['event_id']) ? intval($_POST['event_id']) : 0;
        
        if (!empty($ticket_number)) {
            global $wpdb;
            $tickets_table = $wpdb->prefix . 'ticketpaygo_lite_tickets';
            $orders_table = $wpdb->prefix . 'ticketpaygo_lite_orders';
            
            // Clean up the input - remove # prefix if present
            $search_number = ltrim($ticket_number, '#');
            
            // 1. Search by ticket_number in tickets table
            $ticket = $wpdb->get_row($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                "SELECT t.*, o.customer_name, o.customer_email, o.payment_status, o.status as order_status, o.order_number
                 FROM {$wpdb->prefix}ticketpaygo_lite_tickets t 
                 LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id 
                 WHERE t.ticket_number = %s",
                $search_number
            ), ARRAY_A);
            
            // 2. Search by order_number (tickets joined with orders)
            if (!$ticket) {
                $ticket = $wpdb->get_row($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                    "SELECT t.*, o.customer_name, o.customer_email, o.payment_status, o.status as order_status, o.order_number
                     FROM {$wpdb->prefix}ticketpaygo_lite_tickets t 
                     LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id 
                     WHERE o.order_number = %s",
                    $search_number
                ), ARRAY_A);
            }
            
            // 3. Search orders directly (in case tickets weren't generated)
            if (!$ticket) {
                $order_data = $wpdb->get_row($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                    "SELECT *, status as order_status FROM {$wpdb->prefix}ticketpaygo_lite_orders WHERE order_number = %s",
                    $search_number
                ), ARRAY_A);
                
                if ($order_data) {
                    // Order exists but no ticket record
                    $ticket = array_merge($order_data, array(
                        'ticket_number' => $order_data['order_number'],
                        'id' => $order_data['id'],
                        'status' => ($order_data['order_status'] === 'completed') ? 'active' : $order_data['order_status'],
                        'no_ticket_record' => true
                    ));
                }
            }
            
            // 4. Partial match as last resort
            if (!$ticket) {
                $ticket = $wpdb->get_row($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                    "SELECT t.*, o.customer_name, o.customer_email, o.payment_status, o.status as order_status, o.order_number
                     FROM {$wpdb->prefix}ticketpaygo_lite_tickets t 
                     LEFT JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id 
                     WHERE t.ticket_number LIKE %s OR o.order_number LIKE %s",
                    '%' . $wpdb->esc_like($search_number) . '%',
                    '%' . $wpdb->esc_like($search_number) . '%'
                ), ARRAY_A);
            }
            
            if ($ticket) {
                // Check if event matches (if specified)
                $has_event_id = isset($ticket['event_id']) && $ticket['event_id'];
                if ($has_event_id && $event_id > 0 && $ticket['event_id'] != $event_id) {
                    $checkin_message = array('type' => 'error', 'text' => __('This ticket is for a different event.', 'ticketpaygo-lite'));
                } elseif ($ticket['payment_status'] !== 'completed' && $ticket['payment_status'] !== 'paid' && $ticket['order_status'] !== 'completed') {
                    $checkin_message = array('type' => 'error', 'text' => sprintf(
                        /* translators: %s: payment or order status */
                        esc_html__('Payment not completed for this order. Status: %s', 'ticketpaygo-lite'),
                        esc_html($ticket['payment_status'] ?: $ticket['order_status'])
                    ));
                } elseif ($ticket['status'] === 'checked_in') {
                    $checked_in_time = isset($ticket['scanned_at']) && $ticket['scanned_at'] 
                        ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($ticket['scanned_at']))
                        : __('previously', 'ticketpaygo-lite');
                    $checkin_message = array('type' => 'warning', 'text' => sprintf(
                        /* translators: %s: date and time when ticket was checked in */
                        esc_html__('This ticket was already checked in on %s.', 'ticketpaygo-lite'),
                        esc_html($checked_in_time)
                    ));
                } else {
                    // Perform check-in
                    $update_success = false;
                    $update_error = '';
                    
                    if (isset($ticket['no_ticket_record']) && $ticket['no_ticket_record']) {
                        // No ticket record - update order status directly
                        $order_id = intval($ticket['id']);
                        $updated = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                            $orders_table,
                            array('status' => 'checked_in'),
                            array('id' => $order_id),
                            array('%s'),
                            array('%d')
                        );
                        
                        if ($updated === false) {
                            $update_error = $wpdb->last_error;
                        } elseif ($updated === 0) {
                            // No rows updated - maybe ID doesn't exist
                            $update_error = sprintf('No order found with ID %d', $order_id);
                        } else {
                            $update_success = true;
                        }
                    } else {
                        // Update ticket record (use 'scanned_at' column per DB schema)
                        $ticket_id = intval($ticket['id']);
                        $updated = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                            $tickets_table,
                            array(
                                'status' => 'checked_in',
                                'scanned_at' => current_time('mysql')
                            ),
                            array('id' => $ticket_id),
                            array('%s', '%s'),
                            array('%d')
                        );
                        
                        if ($updated === false) {
                            $update_error = $wpdb->last_error;
                        } elseif ($updated === 0) {
                            $update_error = sprintf('No ticket found with ID %d', $ticket_id);
                        } else {
                            $update_success = true;
                        }
                    }
                    
                    if ($update_success) {
                        $checkin_message = array(
                            'type' => 'success',
                            'text' => sprintf(
                                /* translators: %s: customer name */
                                esc_html__('✓ Check-in successful! Welcome %s.', 'ticketpaygo-lite'),
                                esc_html($ticket['customer_name'])
                            )
                        );
                    } else {
                        $checkin_message = array('type' => 'error', 'text' => sprintf(
                            /* translators: %s: error message */
                            esc_html__('Failed to update status: %s', 'ticketpaygo-lite'),
                            esc_html($update_error ?: 'Unknown error')
                        ));
                    }
                }
            } else {
                // Debug: Check tickets and orders count
                $ticket_count = $wpdb->get_var($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                    "SELECT COUNT(*) FROM {$wpdb->prefix}ticketpaygo_lite_tickets WHERE event_id = %d",
                    $event_id
                )) ?: 0;
                
                $order_count = $wpdb->get_var($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                    "SELECT COUNT(*) FROM {$wpdb->prefix}ticketpaygo_lite_orders WHERE event_id = %d AND status = 'completed'",
                    $event_id
                )) ?: 0;
                
                $checkin_message = array('type' => 'error', 'text' => sprintf(
                    /* translators: %1$s: search number, %2$d: ticket count, %3$d: completed orders count */
                    esc_html__('Not found: %1$s. (Debug: %2$d tickets, %3$d completed orders for this event)', 'ticketpaygo-lite'),
                    esc_html($search_number),
                    intval($ticket_count),
                    intval($order_count)
                ));
            }
        } else {
            $checkin_message = array('type' => 'error', 'text' => __('Please enter a ticket number or order number.', 'ticketpaygo-lite'));
        }
    }
}

// Get events for dropdown
global $wpdb;
$events = $wpdb->get_results($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
    "SELECT id, title, start_date, end_date FROM {$wpdb->prefix}ticketpaygo_lite_events WHERE status = %s ORDER BY start_date DESC",
    'active'
), ARRAY_A);

// Get selected event - auto-select ongoing event if none selected
// Verify nonce for GET parameter if event_id is provided via URL
$selected_event = 0;
if (isset($_GET['event_id'])) {
    // Verify nonce for event selection via GET
    if (isset($_GET['_wpnonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'tpgl_checkin_filter')) {
        $selected_event = intval($_GET['event_id']);
    }
    // Also allow if coming from POST form submission (event_id in form action URL)
    if ($selected_event === 0 && isset($_POST['tpgl_checkin_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['tpgl_checkin_nonce'])), 'tpgl_checkin_action')) {
        $selected_event = intval($_GET['event_id']);
    }
}

// If no event selected, try to auto-select the currently ongoing event
if ($selected_event === 0 && !empty($events)) {
    $current_datetime = current_time('mysql');
    
    foreach ($events as $event) {
        $start_date = $event['start_date'];
        $end_date = $event['end_date'];
        
        // Check if current time is between start and end date
        if ($current_datetime >= $start_date && $current_datetime <= $end_date) {
            $selected_event = intval($event['id']);
            break;
        }
    }
    
    // If no ongoing event, select the most recent/upcoming event
    if ($selected_event === 0 && !empty($events)) {
        $selected_event = intval($events[0]['id']);
    }
}

// Get check-in stats for selected event
$checkin_stats = array('total' => 0, 'checked_in' => 0, 'remaining' => 0);
if ($selected_event > 0) {
    // Count tickets from completed orders
    $checkin_stats['total'] = $wpdb->get_var($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        "SELECT COUNT(*) FROM {$wpdb->prefix}ticketpaygo_lite_tickets t 
         JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id 
         WHERE t.event_id = %d AND (o.payment_status = 'completed' OR o.payment_status = 'paid' OR o.status = 'completed')",
        $selected_event
    )) ?: 0;
    
    $checkin_stats['checked_in'] = $wpdb->get_var($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        "SELECT COUNT(*) FROM {$wpdb->prefix}ticketpaygo_lite_tickets t 
         JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id 
         WHERE t.event_id = %d AND t.status = 'checked_in' AND (o.payment_status = 'completed' OR o.payment_status = 'paid' OR o.status = 'completed')",
        $selected_event
    )) ?: 0;
    
    $checkin_stats['remaining'] = $checkin_stats['total'] - $checkin_stats['checked_in'];
}

// Get recent check-ins
$recent_checkins = array();
if ($selected_event > 0) {
    $recent_checkins = $wpdb->get_results($wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        "SELECT t.ticket_number, t.scanned_at, o.customer_name, o.customer_email, o.quantity, o.total_amount, o.currency
         FROM {$wpdb->prefix}ticketpaygo_lite_tickets t 
         JOIN {$wpdb->prefix}ticketpaygo_lite_orders o ON t.order_id = o.id 
         WHERE t.event_id = %d AND t.status = 'checked_in' 
         ORDER BY t.scanned_at DESC 
         LIMIT 20",
        $selected_event
    ), ARRAY_A);
}
?>

<div class="wrap">
    <h1><?php esc_html_e('Ticket Check-in', 'ticketpaygo-lite'); ?></h1>
    
    <?php if (isset($checkin_message)): ?>
        <div class="notice notice-<?php echo esc_attr($checkin_message['type']); ?> is-dismissible">
            <p><?php echo esc_html($checkin_message['text']); ?></p>
        </div>
    <?php endif; ?>
    
    <div class="tpgl-checkin-container" style="margin-top: 20px;">
        
        <!-- Check-in Form -->
        <div class="tpgl-checkin-form" style="max-width: 500px; margin-bottom: 30px;">
            <div class="card" style="padding: 20px;">
                <h2><?php esc_html_e('Check-in Ticket', 'ticketpaygo-lite'); ?></h2>
                
                <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-checkin' . ($selected_event > 0 ? '&event_id=' . $selected_event : ''))); ?>">
                    <?php wp_nonce_field('tpgl_checkin_action', 'tpgl_checkin_nonce'); ?>
                    <input type="hidden" name="tpgl_checkin_action" value="1">
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="event_id"><?php esc_html_e('Event', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <select name="event_id" id="event_id" class="regular-text" onchange="window.location.href='<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=ticketpaygo-lite-checkin'), 'tpgl_checkin_filter')); ?>&event_id=' + this.value">
                                    <option value="0"><?php esc_html_e('— Select Event —', 'ticketpaygo-lite'); ?></option>
                                    <?php foreach ($events as $event): ?>
                                        <option value="<?php echo esc_attr($event['id']); ?>" <?php selected($selected_event, $event['id']); ?>>
                                            <?php echo esc_html($event['title']); ?> (<?php echo esc_html(date_i18n(get_option('date_format'), strtotime($event['start_date']))); ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="ticket_number"><?php esc_html_e('Order Number', 'ticketpaygo-lite'); ?></label>
                            </th>
                            <td>
                                <input type="text" name="ticket_number" id="ticket_number" class="regular-text" 
                                       placeholder="<?php esc_attr_e('e.g. TPG202512244166', 'ticketpaygo-lite'); ?>" required>
                                <p class="description">
                                    <?php esc_html_e('Enter the order number. Only COMPLETED orders can be checked in.', 'ticketpaygo-lite'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <button type="submit" class="button button-primary button-large" style="background: linear-gradient(135deg, #0A4974 0%, #42963B 100%) !important; color: white !important; border: none !important;">
                            <?php esc_html_e('Check In', 'ticketpaygo-lite'); ?>
                        </button>
                    </p>
                </form>
            </div>
            
            <!-- Upgrade Notice -->
            <div class="card" style="padding: 25px; margin-top: 20px; background: linear-gradient(135deg, #0A4974 0%, #42963B 100%); color: white; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
                <h3 style="color: white; margin-top: 0; margin-bottom: 12px; font-size: 20px;"><?php esc_html_e('Need Mobile Scanning?', 'ticketpaygo-lite'); ?></h3>
                <p style="margin-bottom: 20px; opacity: 0.95; line-height: 1.6;"><?php esc_html_e('Fast mobile QR scanning, multi-device sync, offline mode, and real-time check-in statistics are available with the TicketPayGo add-on.', 'ticketpaygo-lite'); ?></p>
                <a href="<?php echo esc_url(admin_url('admin.php?page=ticketpaygo-lite-upgrade')); ?>" class="button" style="background: white; color: #0A4974; border: none; font-weight: 600; padding: 10px 24px; border-radius: 6px; box-shadow: 0 2px 8px rgba(0,0,0,0.15);">
                    <?php esc_html_e('Learn More', 'ticketpaygo-lite'); ?>
                </a>
            </div>
        </div>
        
        <!-- Stats and Recent Check-ins -->
        <div class="tpgl-checkin-stats">
            <?php if ($selected_event > 0): ?>
                <!-- Stats Cards -->
                <div style="display: flex; gap: 20px; margin-bottom: 30px; max-width: 800px;">
                    <div class="card" style="flex: 1; padding: 24px 20px; text-align: center; border-radius: 12px; border: 1px solid #e2e8f0;">
                        <div style="font-size: 42px; font-weight: 700; color: #2271b1; line-height: 1;"><?php echo esc_html($checkin_stats['total']); ?></div>
                        <div style="color: #666; margin-top: 8px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;"><?php esc_html_e('Total Tickets', 'ticketpaygo-lite'); ?></div>
                    </div>
                    <div class="card" style="flex: 1; padding: 24px 20px; text-align: center; border-radius: 12px; border: 1px solid #e2e8f0;">
                        <div style="font-size: 42px; font-weight: 700; color: #00a32a; line-height: 1;"><?php echo esc_html($checkin_stats['checked_in']); ?></div>
                        <div style="color: #666; margin-top: 8px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;"><?php esc_html_e('Checked In', 'ticketpaygo-lite'); ?></div>
                    </div>
                    <div class="card" style="flex: 1; padding: 24px 20px; text-align: center; border-radius: 12px; border: 1px solid #e2e8f0;">
                        <div style="font-size: 42px; font-weight: 700; color: #dba617; line-height: 1;"><?php echo esc_html($checkin_stats['remaining']); ?></div>
                        <div style="color: #666; margin-top: 8px; font-size: 13px; text-transform: uppercase; letter-spacing: 0.5px;"><?php esc_html_e('Remaining', 'ticketpaygo-lite'); ?></div>
                    </div>
                </div>
                
                <!-- Recent Check-ins - Full Width -->
                <div class="card" style="padding: 20px;">
                    <h3><?php esc_html_e('Recent Check-ins', 'ticketpaygo-lite'); ?></h3>
                    <?php if (!empty($recent_checkins)): ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e('Name', 'ticketpaygo-lite'); ?></th>
                                    <th><?php esc_html_e('Ticket #', 'ticketpaygo-lite'); ?></th>
                                    <th><?php esc_html_e('Qty', 'ticketpaygo-lite'); ?></th>
                                    <th><?php esc_html_e('Amount', 'ticketpaygo-lite'); ?></th>
                                    <th><?php esc_html_e('Time', 'ticketpaygo-lite'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recent_checkins as $checkin): ?>
                                    <tr>
                                        <td><?php echo esc_html($checkin['customer_name']); ?></td>
                                        <td><code><?php echo esc_html($checkin['ticket_number']); ?></code></td>
                                        <td><?php echo esc_html($checkin['quantity']); ?></td>
                                        <td><?php echo esc_html($checkin['currency'] . ' ' . number_format($checkin['total_amount'], 2)); ?></td>
                                        <td><?php echo esc_html(date_i18n(get_option('time_format'), strtotime($checkin['scanned_at']))); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p style="color: #666;"><?php esc_html_e('No check-ins yet for this event.', 'ticketpaygo-lite'); ?></p>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <div class="card" style="padding: 40px; text-align: center;">
                    <span class="dashicons dashicons-tickets-alt" style="font-size: 48px; color: #ccc;"></span>
                    <h3><?php esc_html_e('Select an Event', 'ticketpaygo-lite'); ?></h3>
                    <p style="color: #666;"><?php esc_html_e('Choose an event from the dropdown to start checking in attendees.', 'ticketpaygo-lite'); ?></p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

