<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       third-party-api-authentication
 * @since      1.0.0
 *
 * @package    Third_Party_Api_Authentication
 * @subpackage Third_Party_Api_Authentication/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 *
 * @package    Third_Party_Api_Authentication
 * @subpackage Third_Party_Api_Authentication/admin
 * @author     Edon Sekiraqa <edon@frakton.com>
 */
class Third_Party_Api_Authentication_Admin
{

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $plugin_name The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string $version The current version of this plugin.
     */
    private $version;
    private $headers;

    /**
     * Initialize the class and set its properties.
     *
     * @param string $plugin_name The name of this plugin.
     * @param string $version The version of this plugin.
     * @since    1.0.0
     */
    public function __construct($plugin_name, $version)
    {

        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in Third_Party_Api_Authentication_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The Third_Party_Api_Authentication_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/third-party-api-authentication-admin.css', array(), $this->version, 'all');

    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in Third_Party_Api_Authentication_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The Third_Party_Api_Authentication_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/third-party-api-authentication-admin.js', array('jquery'), $this->version, false);

    }

    public function plugin_settings()
    {
        add_options_page('Settings', 'Third Party API Settings', 'manage_options', $this->plugin_name, [$this, 'render_settings']);
    }

    public function register_plugin_settings()
    {
        register_setting($this->plugin_name . '_plugin_options', $this->plugin_name . '_plugin_options');
        add_settings_section('api_settings', 'API Settings', $this->plugin_name . '_section_text', $this->plugin_name . '_plugin');

        add_settings_field($this->plugin_name . '_setting_api_key', 'API Key', 'plugin_setting_api_key', 'plugin_setting_api_key');
    }

    public function add_settings_page()
    {
        add_options_page('Third Party API Settings', 'Third Party API Settings', 'manage_options', $this->plugin_name, [$this, 'render_plugin_settings_page']);
    }

    public function render_plugin_settings_page()
    {
        ?>
        <h2>Third Party API Plugin Settings</h2>
        <form action="options.php" method="post">
            <?php
            settings_fields($this->plugin_name . '_plugin_options');
            do_settings_sections($this->plugin_name); ?>
            <input name="submit" class="button button-primary" type="submit" value="<?php esc_attr_e('Save'); ?>"/>
        </form>
        <?php
    }

    public function register_settings()
    {
        register_setting($this->plugin_name . '_plugin_options', $this->plugin_name . '_plugin_options', [$this, 'plugin_options_validate']);
        add_settings_section('api_settings', 'API Authentication Endpoint Setup', [$this, 'section_text'], $this->plugin_name);

        add_settings_field($this->plugin_name . '_plugin_setting_api_key', 'API Authentication Endpoint', [$this, 'setting_api_key_field'], $this->plugin_name, 'api_settings');
    }

    public function plugin_options_validate($input)
    {
        return $input;
    }

    public function section_text()
    {
        echo '<p>These settings require the API Endpoint which will be used to authorize the WordPress API from headers. </p>';
        echo '<p>The request will be made via POST request and the json body below:</p>';
        echo 'Request:
<pre>                                  
<code>
{                                                   
    authToken: {The token provided in http headers }
}                                                   
</code></pre>';
        echo 'If the user has a successful authentication then the response below should be returned:
<pre>                                  
<code>
{                                                  
    userId: {The id of the user },                 
    email: {The email of the user },               
    fullName: {The full name of the user }         
}                                                  
</code></pre>';
        echo 'If the user has not a successful authentication then the response below should be returned:
<pre>                                  
<code>
{                                                  
    error: {Any message}                           
}                                                  
</code></pre>';
    }

    public function setting_api_key_field()
    {
        $options = get_option($this->plugin_name . '_plugin_options');
        echo "<input id='" . $this->plugin_name . "_plugin_setting_api_key' style='width: 600px;' name='" . $this->plugin_name . "_plugin_options[api_key]' type='text' value='" . esc_attr($options['api_key']) . "' />";
    }

    private function getRequestHeaders()
    {
        $headers = array();
        foreach ($_SERVER as $key => $value) {
            if (substr($key, 0, 5) <> 'HTTP_') {
                continue;
            }
            $header = str_replace(' ', '-', ucwords(str_replace('_', ' ', strtolower(substr($key, 5)))));
            $headers[$header] = $value;
        }
        $this->headers = $headers;
        return $this->headers;
    }

    public function rest_api_authentication_handler($result)
    {
        // If a previous authentication check was applied,
        // pass that result along without modification.
        if (is_wp_error($result)) {
            return $result;
        }

        if(is_user_logged_in()){
            return $result;
        }

        $authenticationError = new WP_Error(
            'rest_not_logged_in',
            __('You are not currently logged in.'),
            array('status' => 401)
        );

        $headers = $this->getRequestHeaders();

        if (!isset($headers['Cauthorization']) || strlen(trim($headers['Cauthorization'])) == 0) {
            return $authenticationError;
        }

        $user = $this->extractUserFromExternalApi($headers['Cauthorization']);

        if ($user == null || !($user instanceof WP_User)) {
            return $authenticationError;
        }

        wp_set_current_user( $user->ID, $user->user_login );
        wp_set_auth_cookie(  $user->ID );
        do_action( 'wp_login', $user->user_login );

        // No authentication has been performed yet.
        // Return an error if user is not logged in.
        if (!is_user_logged_in()) {
            return $authenticationError;
        }

        // Our custom authentication check should have no effect
        // on logged-in requests
        return $result;
    }

    private function extractUserFromExternalApi($authToken)
    {
        $options = get_option($this->plugin_name . '_plugin_options');
        if (!isset($options['api_key'])) {
            return null;
        }

        $apiEndpoint = $options['api_key'];
        $postData = [
            'authToken' => $authToken
        ];

        $response = wp_remote_post($apiEndpoint, array(
            'headers' => array('Content-Type' => 'application/json; charset=utf-8'),
            'body' => json_encode($postData),
            'method' => 'POST',
            'data_format' => 'body',
        ));

        if (is_wp_error($response)) {
            return null;
        }

        try {
            $jsonData = json_decode($response['body'], true);
            if (empty($jsonData)) {
                return null;
            }

            if (isset($jsonData['userId'])
                && isset($jsonData['email'])
                && isset($jsonData['fullName'])) {

                $userId = $jsonData['userId'];
                $email = $jsonData['email'];
                $fullName = $jsonData['fullName'];

                if ($this->isStringEmpty($userId) ||
                    $this->isStringEmpty($email) ||
                    $this->isStringEmpty($fullName)) {
                    return null;
                }

                $user = get_user_by('email', $email);
                if(is_wp_error($user)){
                    return null;
                }

                $internalUsername = "ext_".$userId;
                if(!empty($user)){
                    if($user->user_login !== $internalUsername){
                        return null;
                    }

                    return $user;
                }else{
                    // create the new user
                    $result = wp_insert_user( array(
                        'user_login' => $internalUsername,
                        'user_pass' =>  md5(time())."_".time(),
                        'user_email' =>$email,
                        'display_name' => $fullName,
                        'user_nicename' => $fullName,
                        'role' => 'subscriber'
                    ));

                    if(is_wp_error($result)){
                        return null;
                    }

                    return get_user_by('id', $result);
                }
            }
        } catch (Exception $e) {
            // silent exception
        }

        return null;
    }

    private function isStringEmpty($value)
    {
        return empty($value) || $value === '';
    }
}