<?php

/**
 * Plugin Name: ThinkRank
 * Plugin URI: https://thinkrank.ai/wordpress-plugin
 * Description: AI-native SEO plugin for WordPress. Automate and enhance your SEO with cutting-edge AI while maintaining editorial control.
 * Version: 1.7.0
 * Author: rupok
 * Author URI: https://thinkrank.ai
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: thinkrank
 * Domain Path: /languages
 * Requires at least: 6.0
 * Tested up to: 6.9
 * Requires PHP: 8.0
 * 
 * @package ThinkRank
 * @version 1.7.0
 * @since 1.0.0
 */

declare(strict_types=1);

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('THINKRANK_VERSION', '1.7.0');
define('THINKRANK_PLUGIN_FILE', __FILE__);
define('THINKRANK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('THINKRANK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('THINKRANK_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Minimum requirements check
if (version_compare(PHP_VERSION, '8.0', '<')) {
    add_action('admin_notices', function () {
        echo '<div class="notice notice-error"><p>';
        echo esc_html__('ThinkRank requires PHP 8.0 or higher. Please upgrade your PHP version.', 'thinkrank');
        echo '</p></div>';
    });
    return;
}

if (version_compare(get_bloginfo('version'), '6.0', '<')) {
    add_action('admin_notices', function () {
        echo '<div class="notice notice-error"><p>';
        echo esc_html__('ThinkRank requires WordPress 6.0 or higher. Please upgrade your WordPress installation.', 'thinkrank');
        echo '</p></div>';
    });
    return;
}

// Autoloader
require_once THINKRANK_PLUGIN_DIR . 'includes/class-autoloader.php';

/**
 * Main ThinkRank Plugin Class
 * 
 * Follows Single Responsibility Principle - only handles plugin initialization
 * 
 * @since 1.0.0
 */
final class ThinkRank {

    /**
     * Plugin instance (Singleton Pattern)
     * 
     * @var ThinkRank|null
     */
    private static ?ThinkRank $instance = null;

    /**
     * Plugin components
     * 
     * @var array
     */
    private array $components = [];

    /**
     * Get plugin instance (Singleton Pattern)
     * 
     * @return ThinkRank
     */
    public static function get_instance(): ThinkRank {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct() {
        $this->init_autoloader();
        $this->init_hooks();
    }

    /**
     * Prevent cloning
     */
    private function __clone() {
    }

    /**
     * Prevent unserialization
     */
    public function __wakeup() {
        throw new \Exception('Cannot unserialize singleton');
    }

    /**
     * Initialize autoloader
     * 
     * @return void
     */
    private function init_autoloader(): void {
        ThinkRank\Core\Autoloader::register();
    }

    /**
     * Initialize WordPress hooks
     * 
     * @return void
     */
    private function init_hooks(): void {
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);

        add_action('plugins_loaded', [$this, 'init']);
    }

    /**
     * Initialize plugin components
     * 
     * @return void
     */
    public function init(): void {
        try {
            $this->load_components();
            $this->init_components();
            $this->load_template_functions();

            do_action('thinkrank_loaded');
        } catch (\Exception $e) {
            $this->handle_error($e);
        }
    }

    /**
     * Load plugin components (Dependency Injection Container pattern)
     * 
     * @return void
     */
    private function load_components(): void {
        $this->components = [
            'database' => new ThinkRank\Core\Database(),
            'settings' => new ThinkRank\Core\Settings(),
            'security_headers' => new ThinkRank\Core\Security_Headers(),
            'asset_optimizer' => new ThinkRank\Core\Asset_Optimizer(),
            'api' => new ThinkRank\API\Manager(),
            'admin' => new ThinkRank\Admin\Manager(),
            'ai' => new ThinkRank\AI\Manager(),
            'frontend_seo' => new ThinkRank\Frontend\SEO_Manager(),
            'seo_notice' => new ThinkRank\Admin\SEO_Notice(),
            'integrations' => new ThinkRank\Integrations\Manager(),
            'performance_collector' => new ThinkRank\SEO\Performance_Data_Collector(),
            'instant_indexing' => new ThinkRank\SEO\Instant_Indexing_Manager(),
            'author_archives' => new ThinkRank\SEO\Author_Archives_Manager(),
        ];
    }

    /**
     * Initialize all components
     *
     * @return void
     */
    private function init_components(): void {
        foreach ($this->components as $component) {
            if (method_exists($component, 'init')) {
                $component->init();
            }
        }
    }

    /**
     * Load template functions for themes
     *
     * @return void
     */
    private function load_template_functions(): void {
        require_once THINKRANK_PLUGIN_DIR . 'includes/frontend/template-functions.php';
    }

    /**
     * Get component instance
     * 
     * @param string $component Component name
     * @return object|null
     */
    public function get_component(string $component): ?object {
        return $this->components[$component] ?? null;
    }

    /**
     * Plugin activation
     * 
     * @return void
     */
    public function activate(): void {
        try {
            $activator = new ThinkRank\Core\Activator();
            $activator->activate();

            // Flush rewrite rules
            flush_rewrite_rules();
        } catch (\Exception $e) {
            $this->handle_error($e);
            wp_die(
                esc_html__('ThinkRank activation failed. Please check your server logs.', 'thinkrank'),
                esc_html__('Plugin Activation Error', 'thinkrank'),
                ['back_link' => true]
            );
        }
    }

    /**
     * Plugin deactivation
     * 
     * @return void
     */
    public function deactivate(): void {
        try {
            $deactivator = new ThinkRank\Core\Deactivator();
            $deactivator->deactivate();

            // Flush rewrite rules
            flush_rewrite_rules();
        } catch (\Exception $e) {
            $this->handle_error($e);
        }
    }

    /**
     * Handle errors consistently
     * 
     * @param \Exception $e Exception to handle
     * @return void
     */
    private function handle_error(\Exception $e): void {
        // Error handling without debug output
        error_log($e->getMessage());
    }

    /**
     * Get plugin version
     * 
     * @return string
     */
    public function get_version(): string {
        return THINKRANK_VERSION;
    }

    /**
     * Get plugin directory path
     * 
     * @return string
     */
    public function get_plugin_dir(): string {
        return THINKRANK_PLUGIN_DIR;
    }

    /**
     * Get plugin URL
     * 
     * @return string
     */
    public function get_plugin_url(): string {
        return THINKRANK_PLUGIN_URL;
    }
}

/**
 * Initialize the plugin
 * 
 * @return ThinkRank
 */
function thinkrank(): ThinkRank {
    return ThinkRank::get_instance();
}

// Start the plugin
thinkrank();
