<?php
/**
 * Sitemap Generator Class
 *
 * XML sitemap generation and management for search engine optimization.
 * Implements 2025 SEO best practices with real sitemap specifications,
 * dynamic content inclusion, and performance optimization.
 *
 * @package ThinkRank
 * @subpackage SEO
 * @since 1.0.0
 */

declare(strict_types=1);

namespace ThinkRank\SEO;

/**
 * Sitemap Generator Class
 *
 * Generates and manages XML sitemaps for search engine optimization.
 * Provides dynamic sitemap generation with content filtering, priority
 * calculation, and change frequency optimization.
 *
 * @since 1.0.0
 */
class Sitemap_Generator extends Abstract_SEO_Manager {

    /**
     * Supported sitemap types
     *
     * @since 1.0.0
     * @var array
     */
    private array $sitemap_types = [
        'posts' => [
            'name' => 'Posts',
            'post_types' => ['post'],
            'priority' => 0.8,
            'changefreq' => 'weekly'
        ],
        'pages' => [
            'name' => 'Pages',
            'post_types' => ['page'],
            'priority' => 0.9,
            'changefreq' => 'monthly'
        ],
        'categories' => [
            'name' => 'Categories',
            'taxonomy' => 'category',
            'priority' => 0.6,
            'changefreq' => 'weekly'
        ],
        'tags' => [
            'name' => 'Tags',
            'taxonomy' => 'post_tag',
            'priority' => 0.4,
            'changefreq' => 'monthly'
        ]
    ];

    /**
     * Constructor
     *
     * @since 1.0.0
     */
    public function __construct() {
        parent::__construct('sitemap');

        // Initialize auto-generation hooks
        $this->init_auto_generation_hooks();
    }

    /**
     * Initialize WordPress hooks for auto-generation
     *
     * @since 1.0.0
     * @return void
     */
    private function init_auto_generation_hooks(): void {
        // Content change hooks - use priority 20 to run after other plugins
        add_action('save_post', [$this, 'handle_content_change'], 20, 2);
        add_action('delete_post', [$this, 'handle_content_deletion'], 20);
        add_action('wp_trash_post', [$this, 'handle_content_deletion'], 20);
        add_action('untrash_post', [$this, 'handle_content_change_by_id'], 20);

        // Taxonomy change hooks
        add_action('created_term', [$this, 'handle_taxonomy_change'], 20, 3);
        add_action('edited_term', [$this, 'handle_taxonomy_change'], 20, 3);
        add_action('delete_term', [$this, 'handle_taxonomy_change'], 20, 3);

        // Debounced regeneration to prevent multiple rapid-fire generations
        add_action('thinkrank_regenerate_sitemap', [$this, 'auto_regenerate_sitemap']);
    }

    /**
     * Generate XML sitemap
     *
     * @since 1.0.0
     *
     * @param array $options Sitemap generation options
     * @return string XML sitemap content
     */
    public function generate_sitemap(array $options = []): string {
        $settings = $this->get_settings('site');

        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";

        // Add XSL stylesheet only if styling is enabled
        if (!empty($settings['enable_styling'])) {
            $xml .= '<?xml-stylesheet type="text/xsl" href="' . home_url('/wp-content/plugins/thinkrank/assets/sitemap.xsl') . '"?>' . "\n";
        }

        // Add image namespace if images are enabled
        if (!empty($settings['include_images'])) {
            $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";
        } else {
            $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        }

        // Add homepage
        $xml .= $this->generate_url_entry(home_url('/'), gmdate('c'), 1.0, 'daily');

        // MEMORY-EFFICIENT: Generate posts using chunked processing
        $enabled_post_types = $this->get_enabled_post_types($settings);
        if (!empty($enabled_post_types)) {
            $xml .= $this->generate_posts_chunked($enabled_post_types, $settings);
        }

        // OPTIMIZED: Get all enabled taxonomies and fetch in single query
        $enabled_taxonomies = $this->get_enabled_taxonomies($settings);
        if (!empty($enabled_taxonomies)) {
            $all_taxonomy_data = $this->fetch_taxonomies_optimized($enabled_taxonomies, $settings);

            // Process each taxonomy's data
            foreach ($enabled_taxonomies as $taxonomy) {
                if (!empty($all_taxonomy_data[$taxonomy])) {
                    $xml .= $this->process_taxonomies_to_xml($all_taxonomy_data[$taxonomy], $taxonomy, $settings);
                }
            }
        }

        $xml .= '</urlset>';

        return $xml;
    }

    /**
     * Validate SEO settings (implements interface)
     *
     * @since 1.0.0
     *
     * @param array $settings Settings array to validate
     * @return array Validation results
     */
    public function validate_settings(array $settings): array {
        $validation = [
            'valid' => true,
            'errors' => [],
            'warnings' => [],
            'suggestions' => []
        ];

        try {
            // Validate exclude_posts
            if (!empty($settings['exclude_posts'])) {
                $this->validate_exclude_posts($settings['exclude_posts']);
            }

            // Validate exclude_terms
            if (!empty($settings['exclude_terms'])) {
                $this->validate_exclude_terms($settings['exclude_terms']);
            }

            // Validate custom_url_pattern
            if (!empty($settings['custom_url_pattern'])) {
                $this->validate_custom_url_pattern($settings['custom_url_pattern']);
            }

            // Validate sitemap_urls
            if (!empty($settings['sitemap_urls']) && is_array($settings['sitemap_urls'])) {
                foreach ($settings['sitemap_urls'] as $sitemap) {
                    if (!empty($sitemap['url'])) {
                        $this->validate_sitemap_url($sitemap['url']);
                    }
                }
            }

            // Validate numeric settings
            if (isset($settings['links_per_sitemap'])) {
                $this->validate_links_per_sitemap($settings['links_per_sitemap']);
            }

        } catch (InvalidArgumentException $e) {
            $validation['valid'] = false;
            $validation['errors'][] = $e->getMessage();
        }

        return $validation;
    }

    /**
     * Get output data for frontend rendering (implements interface)
     *
     * @since 1.0.0
     *
     * @param string   $context_type The context type
     * @param int|null $context_id   Optional. Context ID
     * @return array Output data ready for frontend rendering
     */
    public function get_output_data(string $context_type, ?int $context_id): array {
        $settings = $this->get_settings($context_type, $context_id);

        return [
            'sitemap_url' => home_url('/sitemap.xml'),
            'enabled' => $settings['enabled'] ?? true,
            'last_generated' => $settings['last_generated'] ?? '',
            'total_urls' => $this->count_sitemap_urls($settings)
        ];
    }

    /**
     * Get default settings for a context type (implements interface)
     *
     * @since 1.0.0
     *
     * @param string $context_type The context type to get defaults for
     * @return array Default settings array
     */
    public function get_default_settings(string $context_type): array {
        return [
            // Core settings
            'enabled' => true,

            // Multiple Sitemap URLs
            'sitemap_urls' => [
                [
                    'url' => '/sitemap.xml',
                    'type' => 'general',
                    'enabled' => true,
                    'last_checked' => null,
                    'status' => 'unknown'
                ]
            ],
            'use_sitemap_index' => false,

            // General Settings
            'links_per_sitemap' => 1000,
            'include_images' => true,
            'include_featured_images' => false,
            'auto_generate' => true,
            'ping_search_engines' => true,

            // Content Inclusion (backward compatibility)
            'include_posts' => true,
            'include_pages' => true,
            'include_categories' => true,
            'include_tags' => false,

            // Content Filtering
            'exclude_posts' => '',
            'exclude_terms' => '',
            'exclude_password_protected' => true,
            'exclude_private_posts' => true,

            // Advanced Options
            'enable_styling' => true,
            'custom_url_pattern' => 'sitemap-{type}.xml',

            // Generation tracking
            'last_generated' => ''
        ];
    }

    /**
     * Get settings schema definition (implements interface)
     *
     * @since 1.0.0
     *
     * @param string $context_type The context type to get schema for
     * @return array Settings schema definition
     */
    public function get_settings_schema(string $context_type): array {
        return [
            'enabled' => [
                'type' => 'boolean',
                'title' => 'Enable Sitemap',
                'description' => 'Generate XML sitemap for search engines',
                'default' => true
            ],
            'include_posts' => [
                'type' => 'boolean',
                'title' => 'Include Posts',
                'description' => 'Include blog posts in sitemap',
                'default' => true
            ],
            'include_pages' => [
                'type' => 'boolean',
                'title' => 'Include Pages',
                'description' => 'Include static pages in sitemap',
                'default' => true
            ],
            'include_categories' => [
                'type' => 'boolean',
                'title' => 'Include Categories',
                'description' => 'Include category pages in sitemap',
                'default' => true
            ],
            'include_tags' => [
                'type' => 'boolean',
                'title' => 'Include Tags',
                'description' => 'Include tag pages in sitemap',
                'default' => false
            ],

            'auto_generate' => [
                'type' => 'boolean',
                'title' => 'Auto Generate',
                'description' => 'Automatically regenerate sitemap when content changes',
                'default' => true
            ],
            'ping_search_engines' => [
                'type' => 'boolean',
                'title' => 'Ping Search Engines',
                'description' => 'Notify Google and Bing when sitemap is updated',
                'default' => true
            ],
            'last_generated' => [
                'type' => 'string',
                'title' => 'Last Generated',
                'description' => 'Timestamp of last sitemap generation',
                'default' => ''
            ]
        ];
    }

    /**
     * Generate URL entry for sitemap
     *
     * @since 1.0.0
     *
     * @param string $url        URL
     * @param string $lastmod    Last modification date
     * @param float  $priority   Priority (0.0 to 1.0)
     * @param string $changefreq Change frequency
     * @param array  $images     Optional array of image data
     * @return string XML URL entry
     */
    private function generate_url_entry(string $url, string $lastmod, float $priority, string $changefreq, array $images = []): string {
        $xml = "  <url>\n";
        $xml .= "    <loc>" . esc_url($url) . "</loc>\n";
        $xml .= "    <lastmod>" . esc_html($lastmod) . "</lastmod>\n";
        $xml .= "    <priority>" . number_format($priority, 1) . "</priority>\n";
        $xml .= "    <changefreq>" . esc_html($changefreq) . "</changefreq>\n";

        // Add image entries if provided
        foreach ($images as $image) {
            $xml .= "    <image:image>\n";
            $xml .= "      <image:loc>" . esc_url($image['url']) . "</image:loc>\n";

            if (!empty($image['title'])) {
                $xml .= "      <image:title>" . esc_html($image['title']) . "</image:title>\n";
            }

            if (!empty($image['alt'])) {
                $xml .= "      <image:caption>" . esc_html($image['alt']) . "</image:caption>\n";
            }

            $xml .= "    </image:image>\n";
        }

        $xml .= "  </url>\n";

        return $xml;
    }

    /**
     * Generate post entries for sitemap (MEMORY-EFFICIENT VERSION)
     *
     * @since 1.0.0
     *
     * @param string $post_type Post type
     * @param array  $settings  Sitemap settings
     * @return string XML entries
     */
    private function generate_post_entries(string $post_type, array $settings): string {
        // Use memory-efficient chunked processing for single post type
        return $this->generate_posts_chunked([$post_type], $settings);
    }

    /**
     * Generate posts XML using memory-efficient chunked processing
     *
     * @since 1.0.0
     *
     * @param array $post_types Array of post types to fetch
     * @param array $settings   Sitemap settings
     * @return string XML entries for all posts
     */
    private function generate_posts_chunked(array $post_types, array $settings): string {
        if (empty($post_types)) {
            return '';
        }

        // Parse and validate exclude_posts setting
        $exclude_ids = [];
        if (!empty($settings['exclude_posts'])) {
            try {
                $exclude_ids = $this->validate_exclude_posts($settings['exclude_posts']);
            } catch (InvalidArgumentException $e) {
                // Continue with empty array on validation failure - error details available in exception
            }
        }

        // Use links_per_sitemap setting to limit posts per sitemap
        $links_per_sitemap = !empty($settings['links_per_sitemap']) ?
            intval($settings['links_per_sitemap']) : 1000;

        // Ensure value is within reasonable bounds
        $links_per_sitemap = max(1, min(50000, $links_per_sitemap));

        $xml = '';
        $chunk_size = 500; // Process 500 posts at a time
        $offset = 0;
        $total_processed = 0;

        do {
            // Fetch posts in chunks to prevent memory issues
            $posts = get_posts([
                'post_type' => $post_types,
                'post_status' => 'publish',
                'numberposts' => $chunk_size,
                'offset' => $offset,
                'exclude' => $exclude_ids,
                'orderby' => 'post_type post_date',
                'order' => 'ASC DESC'
            ]);

            // Process each post in the chunk
            foreach ($posts as $post) {
                if ($total_processed >= $links_per_sitemap) {
                    break 2; // Exit both loops when limit reached
                }

                if ($this->should_include_in_sitemap($post, $settings)) {
                    $url = get_permalink($post);
                    $lastmod = gmdate('c', strtotime($post->post_modified_gmt));
                    $priority = $this->calculate_intelligent_priority($post, $post->post_type);
                    $changefreq = $this->calculate_change_frequency($post, $post->post_type);
                    $images = $this->extract_post_images($post, $settings);

                    $xml .= $this->generate_url_entry($url, $lastmod, $priority, $changefreq, $images);
                    $total_processed++;
                }
            }

            $offset += $chunk_size;

            // Store count before freeing memory
            $posts_count = count($posts);

            // Free memory after each chunk
            unset($posts);

        } while ($posts_count === $chunk_size && $total_processed < $links_per_sitemap);

        return $xml;
    }



    /**
     * Extract images from post for sitemap
     *
     * @since 1.0.0
     *
     * @param \WP_Post $post Post object
     * @param array $settings Sitemap settings
     * @return array Array of image data
     */
    private function extract_post_images(\WP_Post $post, array $settings): array {
        $images = [];

        // Skip if images are disabled
        if (empty($settings['include_images'])) {
            return $images;
        }

        // Get featured image if enabled
        if (!empty($settings['include_featured_images'])) {
            $featured_image = $this->get_featured_image($post->ID);
            if ($featured_image) {
                $images[] = $featured_image;
            }
        }

        // Extract images from content
        $content_images = $this->extract_content_images($post->post_content);
        $images = array_merge($images, $content_images);

        // Remove duplicates based on URL
        $unique_images = [];
        $seen_urls = [];
        foreach ($images as $image) {
            if (!in_array($image['url'], $seen_urls)) {
                $unique_images[] = $image;
                $seen_urls[] = $image['url'];
            }
        }

        return $unique_images;
    }

    /**
     * Get featured image data
     *
     * @since 1.0.0
     *
     * @param int $post_id Post ID
     * @return array|null Featured image data or null
     */
    private function get_featured_image(int $post_id): ?array {
        $thumbnail_id = get_post_thumbnail_id($post_id);
        if (!$thumbnail_id) {
            return null;
        }

        $image_url = wp_get_attachment_image_url($thumbnail_id, 'full');
        if (!$image_url) {
            return null;
        }

        $image_title = get_the_title($thumbnail_id);
        $image_alt = get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true);

        return [
            'url' => $image_url,
            'title' => $image_title ?: '',
            'alt' => $image_alt ?: ''
        ];
    }

    /**
     * Extract images from post content
     *
     * @since 1.0.0
     *
     * @param string $content Post content
     * @return array Array of image data
     */
    private function extract_content_images(string $content): array {
        $images = [];

        // Find all img tags in content
        preg_match_all('/<img[^>]+>/i', $content, $img_tags);

        foreach ($img_tags[0] as $img_tag) {
            // Extract src attribute
            if (preg_match('/src=["\']([^"\']+)["\']/', $img_tag, $src_match)) {
                $image_url = $src_match[1];

                // Skip if not a valid URL or external image
                if (!filter_var($image_url, FILTER_VALIDATE_URL)) {
                    continue;
                }

                // Extract title and alt attributes
                $title = '';
                $alt = '';

                if (preg_match('/title=["\']([^"\']*)["\']/', $img_tag, $title_match)) {
                    $title = $title_match[1];
                }

                if (preg_match('/alt=["\']([^"\']*)["\']/', $img_tag, $alt_match)) {
                    $alt = $alt_match[1];
                }

                $images[] = [
                    'url' => $image_url,
                    'title' => $title,
                    'alt' => $alt
                ];
            }
        }

        return $images;
    }

    /**
     * Generate taxonomy entries for sitemap (OPTIMIZED VERSION)
     *
     * @since 1.0.0
     *
     * @param string $taxonomy Taxonomy name
     * @param array  $settings Sitemap settings
     * @return string XML entries
     */
    private function generate_taxonomy_entries(string $taxonomy, array $settings): string {
        // Use optimized batch fetching for single taxonomy
        $taxonomy_data = $this->fetch_taxonomies_optimized([$taxonomy], $settings);

        if (empty($taxonomy_data[$taxonomy])) {
            return '';
        }

        return $this->process_taxonomies_to_xml($taxonomy_data[$taxonomy], $taxonomy, $settings);
    }

    /**
     * Get enabled taxonomies based on settings
     *
     * @since 1.0.0
     * @param array $settings Sitemap settings
     * @return array Array of enabled taxonomies
     */
    private function get_enabled_taxonomies(array $settings): array {
        $taxonomies = [];

        // Core taxonomies based on settings
        if (!empty($settings['include_categories'])) {  // ✅ Evidence-based field name
            $taxonomies[] = 'category';
        }

        if (!empty($settings['include_tags'])) {  // ✅ Evidence-based field name
            $taxonomies[] = 'post_tag';
        }

        // Auto-detect public custom taxonomies
        $custom_taxonomies = get_taxonomies([
            'public' => true,
            '_builtin' => false
        ], 'names');

        foreach ($custom_taxonomies as $taxonomy) {
            if ($this->should_include_taxonomy($taxonomy)) {
                $taxonomies[] = $taxonomy;
            }
        }

        return array_unique($taxonomies);
    }



    /**
     * Fetch taxonomies using optimized combined query
     *
     * @since 1.0.0
     *
     * @param array $taxonomies Array of taxonomy names to fetch
     * @param array $settings   Sitemap settings
     * @return array Grouped terms by taxonomy
     */
    private function fetch_taxonomies_optimized(array $taxonomies, array $settings): array {
        if (empty($taxonomies)) {
            return [];
        }

        // Parse and validate exclude_terms setting
        $exclude_term_ids = [];
        if (!empty($settings['exclude_terms'])) {
            try {
                $exclude_term_ids = $this->validate_exclude_terms($settings['exclude_terms']);
            } catch (InvalidArgumentException $e) {
                // Continue with empty array on validation failure - error details available in exception
            }
        }

        // Determine hide_empty setting based on taxonomies
        $hide_empty = true;
        foreach ($taxonomies as $taxonomy) {
            // For product categories, don't hide empty categories since they might not have products yet
            if ($taxonomy === 'product_cat') {
                $hide_empty = false;
                break;
            }
        }

        // OPTIMIZED: Single query for multiple taxonomies
        $all_terms = get_terms([
            'taxonomy' => $taxonomies,  // ✅ Multiple taxonomies in single query
            'hide_empty' => $hide_empty,
            'exclude' => $exclude_term_ids,
            'orderby' => 'taxonomy count',
            'order' => 'ASC DESC'  // Order by taxonomy ASC, then count DESC
        ]);

        if (is_wp_error($all_terms)) {
            return [];
        }

        // Group terms by taxonomy
        return $this->group_terms_by_taxonomy($all_terms);
    }

    /**
     * Group terms by taxonomy
     *
     * @since 1.0.0
     *
     * @param array $terms Array of term objects
     * @return array Grouped terms by taxonomy
     */
    private function group_terms_by_taxonomy(array $terms): array {
        $grouped = [];

        foreach ($terms as $term) {
            $taxonomy = $term->taxonomy;  // ✅ Evidence-based property name

            if (!isset($grouped[$taxonomy])) {
                $grouped[$taxonomy] = [];
            }

            $grouped[$taxonomy][] = $term;
        }

        return $grouped;
    }

    /**
     * Process taxonomy terms to XML entries
     *
     * @since 1.0.0
     *
     * @param array  $terms    Array of term objects
     * @param string $taxonomy Taxonomy name
     * @param array  $settings Sitemap settings
     * @return string XML entries
     */
    private function process_taxonomies_to_xml(array $terms, string $taxonomy, array $settings): string {
        $xml = '';

        foreach ($terms as $term) {
            $url = get_term_link($term);
            if (!is_wp_error($url)) {
                $lastmod = gmdate('c');
                $priority = $taxonomy === 'category' ? 0.6 : 0.4;
                $changefreq = 'weekly';

                $xml .= $this->generate_url_entry($url, $lastmod, $priority, $changefreq);
            }
        }

        return $xml;
    }

    /**
     * Calculate intelligent priority based on content factors
     *
     * @since 1.0.0
     *
     * @param \WP_Post $post      Post object
     * @param string   $post_type Post type
     * @return float Priority value between 0.1 and 1.0
     */
    private function calculate_intelligent_priority(\WP_Post $post, string $post_type): float {
        $base_priority = $post_type === 'page' ? 0.9 : 0.8;

        // Factors that can adjust priority
        $adjustments = 0;

        // Recent content gets higher priority
        $days_old = (time() - strtotime($post->post_date)) / DAY_IN_SECONDS;
        if ($days_old < 30) {
            $adjustments += 0.1; // Recent content boost
        } elseif ($days_old > 365) {
            $adjustments -= 0.1; // Older content penalty
        }

        // Content length factor
        $content_length = strlen(wp_strip_all_tags($post->post_content));
        if ($content_length > 2000) {
            $adjustments += 0.05; // Comprehensive content boost
        } elseif ($content_length < 500) {
            $adjustments -= 0.1; // Thin content penalty
        }

        // Special page types get higher priority
        if ($post_type === 'page') {
            $page_template = get_page_template_slug($post->ID);
            if (in_array($page_template, ['page-home.php', 'front-page.php']) ||
                $post->ID == get_option('page_on_front')) {
                $base_priority = 1.0; // Homepage gets maximum priority
            } elseif (in_array($page_template, ['page-contact.php', 'page-about.php'])) {
                $adjustments += 0.05; // Important pages boost
            }
        }

        // Ensure priority stays within valid range
        $final_priority = max(0.1, min(1.0, $base_priority + $adjustments));

        return round($final_priority, 1);
    }

    /**
     * Calculate change frequency based on content type and age
     *
     * @since 1.0.0
     *
     * @param \WP_Post $post      Post object
     * @param string   $post_type Post type
     * @return string Change frequency
     */
    private function calculate_change_frequency(\WP_Post $post, string $post_type): string {
        // Pages typically change less frequently
        if ($post_type === 'page') {
            $page_template = get_page_template_slug($post->ID);
            if ($post->ID == get_option('page_on_front')) {
                return 'daily'; // Homepage changes frequently
            } elseif (in_array($page_template, ['page-contact.php', 'page-about.php'])) {
                return 'monthly'; // Static pages change monthly
            }
            return 'yearly'; // Other pages change rarely
        }

        // Posts frequency based on age and type
        $days_old = (time() - strtotime($post->post_date)) / DAY_IN_SECONDS;

        if ($days_old < 7) {
            return 'daily'; // Very recent posts
        } elseif ($days_old < 30) {
            return 'weekly'; // Recent posts
        } elseif ($days_old < 365) {
            return 'monthly'; // Older posts
        }

        return 'yearly'; // Very old posts
    }

    /**
     * Determine if content should be included in sitemap
     *
     * @since 1.0.0
     *
     * @param \WP_Post $post Post object
     * @param array $settings Sitemap settings
     * @return bool Whether to include in sitemap
     */
    private function should_include_in_sitemap(\WP_Post $post, array $settings): bool {
        // Respect user setting for password protected content
        if (!empty($post->post_password) && !empty($settings['exclude_password_protected'])) {
            return false;
        }

        // Respect user setting for private posts
        if ($post->post_status === 'private' && !empty($settings['exclude_private_posts'])) {
            return false;
        }

        // Skip very short content (likely test/placeholder)
        $content_length = strlen(wp_strip_all_tags($post->post_content));
        if ($content_length < 100) {
            return false;
        }

        // Skip posts with test/demo indicators in title (refined filtering)
        $title_indicators = ['test', 'demo', 'sample', 'placeholder'];
        $title_lower = strtolower($post->post_title);

        foreach ($title_indicators as $indicator) {
            if (strpos($title_lower, $indicator) !== false) {
                return false;
            }
        }

        // Skip content with obvious placeholder text
        $content_indicators = ['lorem ipsum'];
        $content_lower = strtolower($post->post_content);

        foreach ($content_indicators as $indicator) {
            if (strpos($content_lower, $indicator) !== false) {
                return false;
            }
        }

        // Skip drafts (but allow private posts if user setting permits)
        if (!in_array($post->post_status, ['publish', 'private'])) {
            return false;
        }

        // Check if post is set to noindex via meta
        $noindex = get_post_meta($post->ID, '_thinkrank_noindex', true);
        if ($noindex === '1' || $noindex === 'true') {
            return false;
        }

        return true;
    }

    /**
     * Count total URLs in sitemap
     *
     * @since 1.0.0
     *
     * @param array $settings Sitemap settings
     * @return int Total URL count
     */
    private function count_sitemap_urls(array $settings): int {
        $count = 1; // Homepage

        if (!empty($settings['include_posts'])) {
            $count += wp_count_posts('post')->publish;
        }

        if (!empty($settings['include_pages'])) {
            $count += wp_count_posts('page')->publish;
        }

        if (!empty($settings['include_categories'])) {
            $count += wp_count_terms(['taxonomy' => 'category', 'hide_empty' => true]);
        }

        if (!empty($settings['include_tags'])) {
            $count += wp_count_terms(['taxonomy' => 'post_tag', 'hide_empty' => true]);
        }

        return $count;
    }

    /**
     * Handle content changes for auto-generation
     *
     * @since 1.0.0
     * @param int      $post_id Post ID
     * @param \WP_Post $post    Post object
     * @return void
     */
    public function handle_content_change(int $post_id, \WP_Post $post): void {
        // Skip if auto-generation is disabled
        if (!$this->should_auto_generate()) {
            return;
        }

        // Skip autosaves and revisions
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }

        // Only process published content
        if ($post->post_status !== 'publish') {
            return;
        }

        // Check if this post type should trigger regeneration
        if (!$this->should_include_post_type($post->post_type)) {
            return;
        }

        // Schedule debounced regeneration
        $this->schedule_debounced_regeneration();
    }

    /**
     * Handle content deletion for auto-generation
     *
     * @since 1.0.0
     * @param int $post_id Post ID
     * @return void
     */
    public function handle_content_deletion(int $post_id): void {
        // Skip if auto-generation is disabled
        if (!$this->should_auto_generate()) {
            return;
        }

        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        // Check if this post type should trigger regeneration
        if (!$this->should_include_post_type($post->post_type)) {
            return;
        }

        // Schedule debounced regeneration
        $this->schedule_debounced_regeneration();
    }

    /**
     * Handle content change by ID (for untrash, etc.)
     *
     * @since 1.0.0
     * @param int $post_id Post ID
     * @return void
     */
    public function handle_content_change_by_id(int $post_id): void {
        $post = get_post($post_id);
        if ($post) {
            $this->handle_content_change($post_id, $post);
        }
    }

    /**
     * Handle taxonomy changes for auto-generation
     *
     * @since 1.0.0
     * @param int    $term_id  Term ID
     * @param int    $tt_id    Term taxonomy ID
     * @param string $taxonomy Taxonomy slug
     * @return void
     */
    public function handle_taxonomy_change(int $term_id, int $tt_id, string $taxonomy): void {
        // Skip if auto-generation is disabled
        if (!$this->should_auto_generate()) {
            return;
        }

        // Check if this taxonomy should trigger regeneration
        if (!$this->should_include_taxonomy($taxonomy)) {
            return;
        }

        // Schedule debounced regeneration
        $this->schedule_debounced_regeneration();
    }

    /**
     * Check if auto-generation is enabled
     *
     * @since 1.0.0
     * @return bool True if auto-generation is enabled
     */
    private function should_auto_generate(): bool {
        $settings = $this->get_settings('site');

        // Check if sitemap is enabled
        if (empty($settings['enabled'])) {
            return false;
        }

        // Check if auto-generation is enabled
        return !empty($settings['auto_generate']);
    }

    /**
     * Get enabled post types based on settings
     *
     * @since 1.0.0
     * @param array $settings Sitemap settings
     * @return array Array of enabled post types
     */
    private function get_enabled_post_types(array $settings): array {
        $post_types = [];

        // Core post types based on settings
        if (!empty($settings['include_posts'])) {
            $post_types[] = 'post';
        }

        if (!empty($settings['include_pages'])) {
            $post_types[] = 'page';
        }

        // Auto-detect public custom post types that should be included
        $custom_post_types = get_post_types([
            'public' => true,
            '_builtin' => false
        ], 'names');

        foreach ($custom_post_types as $post_type) {
            if ($this->should_include_post_type($post_type)) {
                $post_types[] = $post_type;
            }
        }

        return array_unique($post_types);
    }

    /**
     * Check if post type should trigger regeneration
     *
     * @since 1.0.0
     * @param string $post_type Post type
     * @return bool True if post type should trigger regeneration
     */
    private function should_include_post_type(string $post_type): bool {
        // Include all public post types (presets control which sitemaps are created)
        $post_type_obj = get_post_type_object($post_type);
        return $post_type_obj && $post_type_obj->public;
    }

    /**
     * Check if taxonomy should trigger regeneration
     *
     * @since 1.0.0
     * @param string $taxonomy Taxonomy slug
     * @return bool True if taxonomy should trigger regeneration
     */
    private function should_include_taxonomy(string $taxonomy): bool {
        // Include all public taxonomies (presets control which sitemaps are created)
        $taxonomy_obj = get_taxonomy($taxonomy);
        return $taxonomy_obj && $taxonomy_obj->public;
    }

    /**
     * Schedule debounced sitemap regeneration
     *
     * @since 1.0.0
     * @return void
     */
    private function schedule_debounced_regeneration(): void {
        // Clear any existing scheduled regeneration
        wp_clear_scheduled_hook('thinkrank_regenerate_sitemap');

        // Schedule regeneration in 30 seconds to debounce rapid changes
        wp_schedule_single_event(time() + 30, 'thinkrank_regenerate_sitemap');
    }

    /**
     * Auto-regenerate sitemap (called by scheduled action)
     *
     * @since 1.0.0
     * @return void
     */
    public function auto_regenerate_sitemap(): void {
        try {
            // Double-check that auto-generation is still enabled
            if (!$this->should_auto_generate()) {
                return;
            }

            // Generate sitemap with current settings
            $settings = $this->get_settings('site');
            $sitemap_xml = $this->generate_sitemap($settings);

            // Save sitemap to file
            $this->save_sitemap_to_file($sitemap_xml);

            // Update last generated timestamp
            $this->update_setting('last_generated', gmdate('c'), 'site');

            // Ping search engines if enabled
            if (!empty($settings['ping_search_engines'])) {
                $this->ping_search_engines();
            }

            // Sitemap auto-regenerated successfully

        } catch (\Exception $e) {
            // Sitemap auto-regeneration failed - error details available in exception
        }
    }

    /**
     * Save sitemap XML to file
     *
     * @since 1.0.0
     * @param string $sitemap_xml Sitemap XML content
     * @param string $filename Optional. Filename to save (defaults to 'sitemap.xml')
     * @return bool True on success, false on failure
     */
    private function save_sitemap_to_file(string $sitemap_xml, string $filename = 'sitemap.xml'): bool {
        // Validate and sanitize filename for security
        try {
            $filename = $this->validate_sitemap_filename($filename);
        } catch (InvalidArgumentException $e) {
            // File validation failed - error details available in exception
            return false;
        }

        $sitemap_path = ABSPATH . $filename;

        // Use WordPress filesystem API for better security
        global $wp_filesystem;
        if (!$wp_filesystem) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }

        if ($wp_filesystem) {
            return $wp_filesystem->put_contents($sitemap_path, $sitemap_xml, FS_CHMOD_FILE);
        }

        // Fallback to file_put_contents if WP_Filesystem fails
        return file_put_contents($sitemap_path, $sitemap_xml) !== false;
    }

    /**
     * Generate multiple sitemaps based on settings
     *
     * @since 1.0.0
     * @param array $settings Sitemap settings
     * @return array Results of sitemap generation
     */
    public function generate_multiple_sitemaps(array $settings): array {
        $results = [
            'success' => true,
            'sitemaps_generated' => [],
            'errors' => [],
            'total_urls' => 0
        ];

        $sitemap_urls = $settings['sitemap_urls'] ?? [];

        if (empty($sitemap_urls)) {
            $results['success'] = false;
            $results['errors'][] = 'No sitemap URLs configured';
            return $results;
        }

        foreach ($sitemap_urls as $sitemap_config) {
            if (!$sitemap_config['enabled']) {
                continue;
            }

            try {
                $sitemap_xml = '';
                $url_count = 0;

                // Generate sitemap based on type
                switch ($sitemap_config['type']) {
                    case 'index':
                        $sitemap_xml = $this->generate_sitemap_index($sitemap_urls);
                        $url_count = count(array_filter($sitemap_urls, fn($s) => $s['enabled'] && $s['type'] !== 'index'));
                        break;
                    case 'general':
                    case 'wordpress':
                        $sitemap_xml = $this->generate_sitemap($settings);
                        $url_count = $this->count_urls_in_xml($sitemap_xml);
                        break;
                    case 'posts':
                        $sitemap_xml = $this->generate_post_sitemap('post', $settings);
                        $url_count = $this->count_urls_in_xml($sitemap_xml);
                        break;
                    case 'pages':
                        $sitemap_xml = $this->generate_post_sitemap('page', $settings);
                        $url_count = $this->count_urls_in_xml($sitemap_xml);
                        break;
                    case 'categories':
                        $sitemap_xml = $this->generate_taxonomy_sitemap('category', $settings);
                        $url_count = $this->count_urls_in_xml($sitemap_xml);
                        break;
                    case 'products':
                        // Generate product-specific sitemap
                        if (post_type_exists('product')) {
                            $sitemap_xml = $this->generate_post_sitemap('product', $settings);
                        } else {
                            // Create empty sitemap if WooCommerce not installed
                            $sitemap_xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";

                            // Add XSL stylesheet only if styling is enabled
                            if (!empty($settings['enable_styling'])) {
                                $sitemap_xml .= '<?xml-stylesheet type="text/xsl" href="' . home_url('/wp-content/plugins/thinkrank/assets/sitemap.xsl') . '"?>' . "\n";
                            }

                            $sitemap_xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
                            $sitemap_xml .= '</urlset>';
                        }
                        $url_count = $this->count_urls_in_xml($sitemap_xml);
                        break;
                    case 'product_categories':
                        // Generate product category sitemap
                        if (taxonomy_exists('product_cat')) {
                            $sitemap_xml = $this->generate_taxonomy_sitemap('product_cat', $settings);
                        } else {
                            // Create empty sitemap if WooCommerce not installed
                            $sitemap_xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";

                            // Add XSL stylesheet only if styling is enabled
                            if (!empty($settings['enable_styling'])) {
                                $sitemap_xml .= '<?xml-stylesheet type="text/xsl" href="' . home_url('/wp-content/plugins/thinkrank/assets/sitemap.xsl') . '"?>' . "\n";
                            }

                            $sitemap_xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
                            $sitemap_xml .= '</urlset>';
                        }
                        $url_count = $this->count_urls_in_xml($sitemap_xml);
                        break;
                    default:
                        // Check if it's a custom post type
                        $post_types = get_post_types(['public' => true, '_builtin' => false], 'names');
                        if (in_array($sitemap_config['type'], $post_types)) {
                            $sitemap_xml = $this->generate_post_sitemap($sitemap_config['type'], $settings);
                            $url_count = $this->count_urls_in_xml($sitemap_xml);
                        } else {
                            // Fallback to general sitemap
                            $sitemap_xml = $this->generate_sitemap($settings);
                            $url_count = $this->count_urls_in_xml($sitemap_xml);
                        }
                        break;
                }

                // Extract filename from URL
                $filename = basename(wp_parse_url($sitemap_config['url'], PHP_URL_PATH));

                // Save sitemap to file
                if ($this->save_sitemap_to_file($sitemap_xml, $filename)) {
                    $results['sitemaps_generated'][] = [
                        'url' => $sitemap_config['url'],
                        'type' => $sitemap_config['type'],
                        'filename' => $filename,
                        'url_count' => $url_count
                    ];
                    $results['total_urls'] += $url_count;
                } else {
                    $error_message = "Failed to save sitemap: {$filename}";
                    $results['errors'][] = $error_message;
                    $results['success'] = false;

                    // File save error details available in results array
                }

            } catch (\Exception $e) {
                $results['errors'][] = "Error generating {$sitemap_config['type']} sitemap: " . $e->getMessage();
                $results['success'] = false;
            }
        }

        return $results;
    }

    /**
     * Generate sitemap index XML
     *
     * @since 1.0.0
     * @param array $sitemap_urls Array of sitemap configurations
     * @return string Sitemap index XML
     */
    private function generate_sitemap_index(array $sitemap_urls): string {
        $settings = $this->get_settings('site');
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";

        // Add XSL stylesheet only if styling is enabled
        if (!empty($settings['enable_styling'])) {
            $xml .= '<?xml-stylesheet type="text/xsl" href="' . home_url('/wp-content/plugins/thinkrank/assets/sitemap-index.xsl') . '"?>' . "\n";
        }
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        $site_url = home_url();

        foreach ($sitemap_urls as $sitemap_config) {
            if (!$sitemap_config['enabled'] || $sitemap_config['type'] === 'index') {
                continue;
            }

            $sitemap_url = $sitemap_config['url'];
            if (!str_starts_with($sitemap_url, 'http')) {
                $sitemap_url = $site_url . $sitemap_url;
            }

            $xml .= "  <sitemap>\n";
            $xml .= "    <loc>" . esc_url($sitemap_url) . "</loc>\n";
            $xml .= "    <lastmod>" . gmdate('c') . "</lastmod>\n";
            $xml .= "  </sitemap>\n";
        }

        $xml .= '</sitemapindex>';

        return $xml;
    }

    /**
     * Generate sitemap for specific post type
     *
     * @since 1.0.0
     * @param string $post_type Post type to generate sitemap for
     * @param array $settings Sitemap settings
     * @return string Sitemap XML
     */
    private function generate_post_sitemap(string $post_type, array $settings): string {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";

        // Add XSL stylesheet only if styling is enabled
        if (!empty($settings['enable_styling'])) {
            $xml .= '<?xml-stylesheet type="text/xsl" href="' . home_url('/wp-content/plugins/thinkrank/assets/sitemap.xsl') . '"?>' . "\n";
        }

        // Add image namespace if images are enabled
        if (!empty($settings['include_images'])) {
            $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";
        } else {
            $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        }

        // Add homepage only for general sitemap
        if ($post_type === 'page') {
            $xml .= $this->generate_url_entry(home_url('/'), gmdate('c'), 1.0, 'daily');
        }

        // Add posts/pages of specific type
        $xml .= $this->generate_post_entries($post_type, $settings);

        $xml .= '</urlset>';
        return $xml;
    }

    /**
     * Generate sitemap for specific taxonomy
     *
     * @since 1.0.0
     * @param string $taxonomy Taxonomy to generate sitemap for
     * @param array $settings Sitemap settings
     * @return string Sitemap XML
     */
    private function generate_taxonomy_sitemap(string $taxonomy, array $settings): string {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";

        // Add XSL stylesheet only if styling is enabled
        if (!empty($settings['enable_styling'])) {
            $xml .= '<?xml-stylesheet type="text/xsl" href="' . home_url('/wp-content/plugins/thinkrank/assets/sitemap.xsl') . '"?>' . "\n";
        }
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        // Add taxonomy terms
        $xml .= $this->generate_taxonomy_entries($taxonomy, $settings);

        $xml .= '</urlset>';
        return $xml;
    }

    /**
     * Count URLs in sitemap XML content
     *
     * @since 1.0.0
     * @param string $sitemap_xml Sitemap XML content
     * @return int Number of URLs
     */
    private function count_urls_in_xml(string $sitemap_xml): int {
        return substr_count($sitemap_xml, '<url>');
    }

    /**
     * Ping search engines about sitemap update
     *
     * @since 1.0.0
     * @return void
     */
    private function ping_search_engines(): void {
        $sitemap_url = home_url('/sitemap.xml');

        $ping_urls = [
            'google' => 'https://www.google.com/ping?sitemap=' . urlencode($sitemap_url),
            'bing' => 'https://www.bing.com/ping?sitemap=' . urlencode($sitemap_url)
        ];

        foreach ($ping_urls as $engine => $ping_url) {
            // Use wp_remote_get with timeout to prevent hanging
            $response = wp_remote_get($ping_url, [
                'timeout' => 10,
                'blocking' => false // Non-blocking to prevent delays
            ]);

            if (is_wp_error($response)) {
                // Search engine ping failed - error details available in WP_Error response
            }
        }
    }

    /**
     * Validate and sanitize exclude posts input
     *
     * @since 1.0.0
     * @param string $exclude_posts Comma-separated post IDs
     * @return array Validated post IDs
     * @throws InvalidArgumentException If validation fails
     */
    private function validate_exclude_posts(string $exclude_posts): array {
        if (empty($exclude_posts)) {
            return [];
        }

        // Limit input length to prevent DoS
        if (strlen($exclude_posts) > 1000) {
            throw new InvalidArgumentException('Exclude posts list too long (max 1000 characters)');
        }

        // Validate comma-separated integers only
        if (!preg_match('/^[\d,\s]+$/', $exclude_posts)) {
            throw new InvalidArgumentException('Exclude posts must contain only numbers and commas');
        }

        $ids = array_map('intval', array_filter(explode(',', $exclude_posts)));

        // Limit number of exclusions to prevent performance issues
        if (count($ids) > 100) {
            throw new InvalidArgumentException('Too many posts to exclude (max 100)');
        }

        return array_filter($ids, function($id) {
            return $id > 0; // Only positive integers
        });
    }

    /**
     * Validate and sanitize exclude terms input
     *
     * @since 1.0.0
     * @param string $exclude_terms Comma-separated term IDs
     * @return array Validated term IDs
     * @throws InvalidArgumentException If validation fails
     */
    private function validate_exclude_terms(string $exclude_terms): array {
        if (empty($exclude_terms)) {
            return [];
        }

        // Limit input length to prevent DoS
        if (strlen($exclude_terms) > 1000) {
            throw new InvalidArgumentException('Exclude terms list too long (max 1000 characters)');
        }

        // Validate comma-separated integers only
        if (!preg_match('/^[\d,\s]+$/', $exclude_terms)) {
            throw new InvalidArgumentException('Exclude terms must contain only numbers and commas');
        }

        $ids = array_map('intval', array_filter(explode(',', $exclude_terms)));

        // Limit number of exclusions to prevent performance issues
        if (count($ids) > 100) {
            throw new InvalidArgumentException('Too many terms to exclude (max 100)');
        }

        return array_filter($ids, function($id) {
            return $id > 0; // Only positive integers
        });
    }

    /**
     * Validate and sanitize custom URL pattern
     *
     * @since 1.0.0
     * @param string $pattern Custom URL pattern
     * @return string Validated and sanitized pattern
     * @throws InvalidArgumentException If validation fails
     */
    private function validate_custom_url_pattern(string $pattern): string {
        if (empty($pattern)) {
            return 'sitemap-{type}.xml';
        }

        // Remove any HTML/script tags to prevent XSS
        $pattern = wp_strip_all_tags($pattern);

        // Limit pattern length
        if (strlen($pattern) > 100) {
            throw new InvalidArgumentException('URL pattern too long (max 100 characters)');
        }

        // Validate pattern format - only allow safe characters
        if (!preg_match('/^[a-zA-Z0-9\-_{}\.]+$/', $pattern)) {
            throw new InvalidArgumentException('URL pattern contains invalid characters. Only letters, numbers, hyphens, underscores, dots, and {type} are allowed');
        }

        // Ensure it contains {type} placeholder
        if (strpos($pattern, '{type}') === false) {
            throw new InvalidArgumentException('URL pattern must contain {type} placeholder');
        }

        // Ensure it ends with .xml
        if (!str_ends_with($pattern, '.xml')) {
            $pattern .= '.xml';
        }

        return sanitize_file_name($pattern);
    }

    /**
     * Validate sitemap URL
     *
     * @since 1.0.0
     * @param string $url Sitemap URL
     * @return string Validated and sanitized URL
     * @throws InvalidArgumentException If validation fails
     */
    private function validate_sitemap_url(string $url): string {
        if (empty($url)) {
            throw new InvalidArgumentException('Sitemap URL cannot be empty');
        }

        // Remove leading/trailing whitespace
        $url = trim($url);

        // Limit URL length
        if (strlen($url) > 200) {
            throw new InvalidArgumentException('Sitemap URL too long (max 200 characters)');
        }

        // Ensure it starts with /
        if (!str_starts_with($url, '/')) {
            $url = '/' . $url;
        }

        // Validate URL path format
        if (!preg_match('/^\/[a-zA-Z0-9\-_\/\.]+\.xml$/', $url)) {
            throw new InvalidArgumentException('Invalid sitemap URL format. Must be a valid path ending with .xml');
        }

        // Prevent directory traversal
        if (strpos($url, '..') !== false) {
            throw new InvalidArgumentException('Directory traversal not allowed in sitemap URL');
        }

        // Prevent multiple slashes
        $url = preg_replace('/\/+/', '/', $url);

        return sanitize_url($url);
    }

    /**
     * Validate links per sitemap setting
     *
     * @since 1.0.0
     * @param mixed $links_per_sitemap Links per sitemap value
     * @return int Validated links per sitemap
     * @throws InvalidArgumentException If validation fails
     */
    private function validate_links_per_sitemap($links_per_sitemap): int {
        $links = intval($links_per_sitemap);

        if ($links < 1) {
            throw new InvalidArgumentException('Links per sitemap must be at least 1');
        }

        if ($links > 50000) {
            throw new InvalidArgumentException('Links per sitemap cannot exceed 50,000');
        }

        return $links;
    }

    /**
     * Validate sitemap filename for security
     *
     * @since 1.0.0
     * @param string $filename Filename to validate
     * @return string Validated and sanitized filename
     * @throws InvalidArgumentException If validation fails
     */
    private function validate_sitemap_filename(string $filename): string {
        if (empty($filename)) {
            return 'sitemap.xml';
        }

        // Remove any path components to prevent directory traversal
        $filename = basename($filename);

        // Limit filename length
        if (strlen($filename) > 100) {
            throw new InvalidArgumentException('Filename too long (max 100 characters)');
        }

        // Validate filename format - only allow safe characters
        if (!preg_match('/^[a-zA-Z0-9\-_\.]+$/', $filename)) {
            throw new InvalidArgumentException('Filename contains invalid characters. Only letters, numbers, hyphens, underscores, and dots are allowed');
        }

        // Prevent directory traversal attempts
        if (strpos($filename, '..') !== false) {
            throw new InvalidArgumentException('Directory traversal not allowed in filename');
        }

        // Ensure it ends with .xml
        if (!str_ends_with($filename, '.xml')) {
            $filename .= '.xml';
        }

        // Additional sanitization
        $filename = sanitize_file_name($filename);

        // Final security check - ensure it's still a valid XML filename
        if (!preg_match('/^[a-zA-Z0-9\-_]+\.xml$/', $filename)) {
            throw new InvalidArgumentException('Invalid XML filename after sanitization');
        }

        return $filename;
    }
}
