<?php

/**
 * Image SEO Manager Class
 *
 * Manages image-specific SEO settings including automatic
 * ALT and TITLE attribute management.
 *
 * @package ThinkRank
 * @subpackage SEO
 * @since 1.0.0
 */

declare(strict_types=1);

namespace ThinkRank\SEO;

/**
 * Image SEO Manager Class
 *
 * Handles image attribute optimization and settings management.
 *
 * @since 1.0.0
 */
class Image_SEO_Manager extends Abstract_SEO_Manager {

    /**
     * Constructor
     *
     * @since 1.0.0
     */
    public function __construct() {
        parent::__construct('image_seo');
    }

    /**
     * Validate SEO settings (implements interface)
     *
     * @since 1.0.0
     *
     * @param array $settings Settings array to validate
     * @return array Validation results
     */
    public function validate_settings(array $settings): array {
        $validation = [
            'valid' => true,
            'errors' => [],
            'warnings' => [],
            'suggestions' => [],
            'score' => 100
        ];

        $boolean_fields = ['add_missing_alt', 'add_missing_title'];
        foreach ($boolean_fields as $field) {
            if (isset($settings[$field]) && !is_bool($settings[$field])) {
                $validation['errors'][] = sprintf('%s must be a boolean value', $field);
                $validation['valid'] = false;
            }
        }

        $string_fields = ['alt_format', 'title_format'];
        foreach ($string_fields as $field) {
            if (isset($settings[$field]) && !is_string($settings[$field])) {
                $validation['errors'][] = sprintf('%s must be a string', $field);
                $validation['valid'] = false;
            }
        }

        return $validation;
    }

    /**
     * Get output data for frontend rendering (implements interface)
     *
     * @since 1.0.0
     *
     * @param string   $context_type The context type
     * @param int|null $context_id   Optional. Context ID
     * @return array Output data ready for frontend rendering
     */
    public function get_output_data(string $context_type, ?int $context_id): array {
        return $this->get_settings($context_type, $context_id);
    }

    /**
     * Get default settings for a context type (implements interface)
     *
     * @since 1.0.0
     *
     * @param string $context_type The context type to get defaults for
     * @return array Default settings array
     */
    public function get_default_settings(string $context_type): array {
        return [
            'add_missing_alt' => false,
            'alt_format' => '%filename%',
            'add_missing_title' => false,
            'title_format' => '%title% %separator% %sitename%',
        ];
    }

    /**
     * Get settings schema definition (implements interface)
     *
     * @since 1.0.0
     *
     * @param string $context_type The context type to get schema for
     * @return array Settings schema definition
     */
    public function get_settings_schema(string $context_type): array {
        return [
            'add_missing_alt' => [
                'type' => 'boolean',
                'title' => __('Add Missing Alt Attributes', 'thinkrank'),
                'description' => __('Automatically add ALT attributes to images if they are missing.', 'thinkrank'),
                'default' => false
            ],
            'alt_format' => [
                'type' => 'string',
                'title' => __('Alt attribute format', 'thinkrank'),
                'description' => __('The format to use for automatically generated ALT attributes.', 'thinkrank'),
                'default' => '%filename%'
            ],
            'add_missing_title' => [
                'type' => 'boolean',
                'title' => __('Add Missing Title Attributes', 'thinkrank'),
                'description' => __('Automatically add TITLE attributes to images if they are missing.', 'thinkrank'),
                'default' => false
            ],
            'title_format' => [
                'type' => 'string',
                'title' => __('Title attribute format', 'thinkrank'),
                'description' => __('The format to use for automatically generated TITLE attributes.', 'thinkrank'),
                'default' => '%title% %separator% %sitename%'
            ]
        ];
    }

    /**
     * Process content and inject missing image attributes
     *
     * @since 1.0.0
     * @param string $content The content to process
     * @return string Processed content
     */
    public function process_content(string $content, $post_id = null): string {
        $settings = $this->get_settings('site');

        if (!$settings['add_missing_alt'] && !$settings['add_missing_title']) {
            return $content;
        }

        static $count = 0;
        $post_id ??= get_the_ID();
        $id_to_pass = is_int($post_id) ? $post_id : 0;

        // Use regex for high performance, but careful with HTML structure
        return preg_replace_callback('/<img([^>]+)>/i', function ($matches) use ($settings, &$count, $id_to_pass) {
            $count++;
            $img_tag = $matches[0];
            $attributes_str = $matches[1];

            // Parse attributes
            preg_match_all('/([a-z-]+)=["\']([^"\']*)["\']/i', $attributes_str, $attr_matches);
            $attributes = array_combine($attr_matches[1], $attr_matches[2]);

            $src = $attributes['src'] ?? '';
            $attachment_id = $src ? attachment_url_to_postid($src) : 0;

            // Handle ALT attribute
            if ($settings['add_missing_alt'] && (empty($attributes['alt']) || trim($attributes['alt']) === '')) {
                $alt_val = $this->generate_attribute_value($settings['alt_format'] ?? '', $attachment_id, $id_to_pass, $count, $src);
                if ($alt_val) {
                    if (isset($attributes['alt'])) {
                        $img_tag = preg_replace('/alt=["\'][^"\']*["\']/i', 'alt="' . esc_attr($alt_val) . '"', $img_tag);
                    } else {
                        $img_tag = str_replace('<img', '<img alt="' . esc_attr($alt_val) . '"', $img_tag);
                    }
                }
            }

            // Handle TITLE attribute
            if ($settings['add_missing_title'] && (empty($attributes['title']) || trim($attributes['title']) === '')) {
                $title_val = $this->generate_attribute_value($settings['title_format'] ?? '', $attachment_id, $id_to_pass, $count, $src);
                if ($title_val) {
                    if (isset($attributes['title'])) {
                        $img_tag = preg_replace('/title=["\'][^"\']*["\']/i', 'title="' . esc_attr($title_val) . '"', $img_tag);
                    } else {
                        $img_tag = str_replace('<img', '<img title="' . esc_attr($title_val) . '"', $img_tag);
                    }
                }
            }

            return $img_tag;
        }, $content);
    }

    /**
     * Generate attribute value based on format and context
     *
     * @since 1.0.0
     * @param string $format        The format string
     * @param int    $attachment_id Attachment ID
     * @param int    $post_id       Current Post ID
     * @param int    $count         Image counter
     * @param string $src           Image source URL
     * @return string Generated value
     */
    private function generate_attribute_value(string $format, int $attachment_id, int $post_id, int $count, string $src): string {
        $replacements = [
            '%site_title%'    => get_bloginfo('name'),
            '%sitename%'      => get_bloginfo('name'),
            '%title%'         => $post_id > 0 ? get_the_title($post_id) : get_bloginfo('name'),
            '%count%'         => (string) $count,
            '%separator%'     => $this->get_separator(),
            '%sep%'           => $this->get_separator(),
            '%filename%'      => '',
            '%image_title%'   => '',
            '%image_caption%' => '',
        ];

        // Get filename from src
        if ($src) {
            $filename = pathinfo($src, PATHINFO_FILENAME);
            $replacements['%filename%'] = str_replace(['-', '_'], ' ', $filename);
        }

        // Get attachment data if ID exists
        if ($attachment_id) {
            $attachment = get_post($attachment_id);
            if ($attachment) {
                $replacements['%image_title%'] = $attachment->post_title;
                $replacements['%image_caption%'] = $attachment->post_excerpt;
            }
        }

        // Apply replacements
        $value = str_replace(array_keys($replacements), array_values($replacements), $format);

        // Clean up double spaces if any
        $value = preg_replace('/\s+/', ' ', $value);

        return trim($value);
    }

    /**
     * Get site separator
     *
     * @since 1.0.0
     * @return string
     */
    private function get_separator(): string {
        return Site_Identity_Manager::get_active_separator_symbol();
    }
}
