<?php
/**
 * Template Functions for ThinkRank SEO
 * 
 * Helper functions for themes to display ThinkRank SEO elements
 * 
 * @package ThinkRank\Frontend
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Display ThinkRank breadcrumbs
 * 
 * Usage in theme templates:
 * <?php thinkrank_breadcrumbs(); ?>
 * 
 * @since 1.0.0
 * 
 * @param array $args Optional. Arguments to customize breadcrumb display
 * @return void
 */
function thinkrank_breadcrumbs($args = []) {
    /**
     * Fires to display ThinkRank breadcrumbs
     * 
     * @since 1.0.0
     * 
     * @param array $args Breadcrumb display arguments
     */
    do_action('thinkrank_breadcrumbs', $args);
}

/**
 * Get ThinkRank breadcrumbs data without displaying
 *
 * @since 1.0.0
 *
 * @return array|null Breadcrumb data or null if not available
 */
function thinkrank_get_breadcrumbs() {
    // Check if ThinkRank is loaded
    if (!function_exists('thinkrank')) {
        return null;
    }

    $plugin = thinkrank();
    if (!$plugin) {
        return null;
    }

    $seo_manager = $plugin->get_component('frontend_seo');

    if (!$seo_manager || !method_exists($seo_manager, 'get_current_breadcrumbs')) {
        return null;
    }

    return $seo_manager->get_current_breadcrumbs();
}

/**
 * Check if ThinkRank SEO is handling the current page
 *
 * @since 1.0.0
 *
 * @return bool True if ThinkRank is handling SEO for current page
 */
function thinkrank_is_seo_active() {
    // Check if ThinkRank is loaded
    if (!function_exists('thinkrank')) {
        return false;
    }

    $plugin = thinkrank();
    if (!$plugin) {
        return false;
    }

    $seo_manager = $plugin->get_component('frontend_seo');

    if (!$seo_manager || !method_exists($seo_manager, 'has_seo_data')) {
        return false;
    }

    return $seo_manager->has_seo_data();
}

/**
 * Get current page SEO metadata
 *
 * @since 1.0.0
 *
 * @return array Current SEO metadata
 */
function thinkrank_get_seo_metadata() {
    // Check if ThinkRank is loaded
    if (!function_exists('thinkrank')) {
        return [];
    }

    $plugin = thinkrank();
    if (!$plugin) {
        return [];
    }

    $seo_manager = $plugin->get_component('frontend_seo');

    if (!$seo_manager || !method_exists($seo_manager, 'get_current_metadata')) {
        return [];
    }

    return $seo_manager->get_current_metadata();
}

/**
 * Display custom breadcrumb styles
 * 
 * @since 1.0.0
 * 
 * @return void
 */
function thinkrank_breadcrumb_styles() {
    ?>
    <style>
    .thinkrank-breadcrumbs {
        margin: 1rem 0;
        font-size: 0.9rem;
    }
    
    .thinkrank-breadcrumbs .breadcrumb-prefix {
        font-weight: 600;
        margin-right: 0.5rem;
    }
    
    .thinkrank-breadcrumbs .breadcrumb-list {
        list-style: none;
        margin: 0;
        padding: 0;
        display: flex;
        flex-wrap: wrap;
        align-items: center;
    }
    
    .thinkrank-breadcrumbs .breadcrumb-item {
        margin: 0;
        padding: 0;
        display: flex;
        align-items: center;
    }
    
    .thinkrank-breadcrumbs .breadcrumb-item a {
        color: #0073aa;
        text-decoration: none;
    }
    
    .thinkrank-breadcrumbs .breadcrumb-item a:hover {
        text-decoration: underline;
    }
    
    .thinkrank-breadcrumbs .breadcrumb-item.current span {
        color: #666;
        font-weight: 500;
    }
    
    .thinkrank-breadcrumbs .breadcrumb-separator {
        margin: 0 0.5rem;
        color: #999;
    }
    </style>
    <?php
}

// Auto-include breadcrumb styles if breadcrumbs are enabled
add_action('wp_head', function() {
    // Check if ThinkRank is loaded
    if (!function_exists('thinkrank')) {
        return;
    }

    $plugin = thinkrank();
    if (!$plugin) {
        return;
    }

    $seo_manager = $plugin->get_component('frontend_seo');

    if ($seo_manager) {
        // Check if Site Identity Manager exists and breadcrumbs are enabled
        if (class_exists('ThinkRank\\SEO\\Site_Identity_Manager')) {
            $identity_manager = new \ThinkRank\SEO\Site_Identity_Manager();
            $settings = $identity_manager->get_settings('site');

            if (!empty($settings['breadcrumbs_enabled'])) {
                thinkrank_breadcrumb_styles();
            }
        }
    }
}, 100);
