<?php
/**
 * SEO Prompts Class
 *
 * Handles SEO-related AI prompts including metadata generation and content analysis.
 * Preserves exact functionality from original OpenAI and Claude client implementations.
 *
 * @package ThinkRank
 * @subpackage AI\Prompts
 * @since 1.0.0
 */

declare(strict_types=1);

namespace ThinkRank\AI\Prompts;

/**
 * SEO Prompts Class
 *
 * Centralized SEO prompt building for consistent AI optimization across all providers.
 * Handles SEO metadata generation and content analysis prompts.
 *
 * @since 1.0.0
 */
class SEO_Prompts {

    /**
     * Build SEO metadata optimization prompt
     *
     * @since 1.0.0
     *
     * @param string $content Content to optimize
     * @param string $target_keyword Target keyword
     * @param string $content_type Type of content
     * @param string $tone Desired tone
     * @param string $provider AI provider
     * @return string Formatted prompt
     */
    public function build_seo_prompt(string $content, string $target_keyword, string $content_type, string $tone, string $provider = 'openai'): string {
        $keyword_instruction = $target_keyword ? "Focus on the keyword: \"{$target_keyword}\"" : "Identify the main topic";
        
        return "You are an expert SEO copywriter. Analyze the following {$content_type} content and generate optimized SEO metadata.

{$keyword_instruction}

Content:
{$content}

Generate:
1. SEO Title (50-60 characters, compelling and keyword-optimized)
2. Meta Description (150-160 characters, engaging with call-to-action)
3. Focus Keyword (main keyword for this content)

Requirements:
- Use {$tone} tone
- Include target keyword naturally
- Make titles compelling for clicks
- Write descriptions that encourage clicks
- Ensure proper character limits

Format your response as JSON:
{
  \"title\": \"Your SEO title here\",
  \"description\": \"Your meta description here\",
  \"focus_keyword\": \"main keyword\",
  \"suggestions\": [\"improvement suggestion 1\", \"improvement suggestion 2\"]
}";
    }

    /**
     * Build content analysis prompt
     *
     * @since 1.0.0
     *
     * @param string $content Content to analyze
     * @param array $metadata Existing metadata
     * @param string $provider AI provider
     * @return string Formatted prompt
     */
    public function build_analysis_prompt(string $content, array $metadata, string $provider = 'openai'): string {
        $title = $metadata['title'] ?? '';
        $description = $metadata['description'] ?? '';
        $focus_keyword = $metadata['focus_keyword'] ?? '';

        $metadata_section = '';
        if (!empty($title) || !empty($description) || !empty($focus_keyword)) {
            $metadata_section = "
Current SEO Metadata:
- Title: {$title}
- Description: {$description}
- Focus Keyword: {$focus_keyword}
";
        }

        return "You are an expert SEO analyst. Analyze the following content for SEO optimization opportunities.

Content:
{$content}
{$metadata_section}

Provide a comprehensive SEO analysis including:
1. SEO Score (1-100 based on current optimization)
2. Content quality assessment
3. Keyword optimization analysis
4. Readability assessment
5. Specific improvement suggestions

Format your response as JSON:
{
  \"seo_score\": 75,
  \"content_analysis\": {
    \"word_count\": 500,
    \"readability\": \"good\",
    \"keyword_density\": \"optimal\",
    \"structure\": \"needs improvement\"
  },
  \"suggestions\": [
    \"Add more subheadings to improve structure\",
    \"Include the focus keyword in the first paragraph\",
    \"Optimize meta description length\"
  ],
  \"strengths\": [
    \"Good content length\",
    \"Clear writing style\"
  ],
  \"weaknesses\": [
    \"Missing focus keyword in title\",
    \"No internal links\"
  ]
}";
    }
}
