<?php
/**
 * SEO Plugin Notice Manager
 * 
 * Handles admin notices for SEO plugin conflicts
 * 
 * @package ThinkRank\Admin
 * @since 1.0.0
 */

declare(strict_types=1);

namespace ThinkRank\Admin;

use ThinkRank\Core\SEO_Plugin_Detector;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * SEO Notice Manager Class
 * 
 * Single Responsibility: Display admin notices for SEO plugin conflicts
 * 
 * @since 1.0.0
 */
class SEO_Notice {
    
    /**
     * Initialize notice manager
     *
     * @return void
     */
    public function init(): void {
        add_action('admin_notices', [$this, 'display_seo_plugin_notice']);
        add_action('wp_ajax_thinkrank_dismiss_seo_notice', [$this, 'ajax_dismiss_notice']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_notice_scripts']);
    }
    
    /**
     * Display SEO plugin conflict notice
     *
     * @return void
     */
    public function display_seo_plugin_notice(): void {
        if (!SEO_Plugin_Detector::should_show_notice()) {
            return;
        }

        $message = SEO_Plugin_Detector::get_notice_message();
        if (empty($message)) {
            return;
        }
        
        ?>
        <div class="notice notice-warning is-dismissible thinkrank-seo-notice">
            <h3><?php esc_html_e('ThinkRank SEO Priority', 'thinkrank'); ?></h3>
            <p><?php echo esc_html($message); ?></p>

            <p>
                <strong><?php esc_html_e('How it works:', 'thinkrank'); ?></strong><br>
                • <?php esc_html_e('Posts with ThinkRank AI metadata → ThinkRank takes priority', 'thinkrank'); ?><br>
                • <?php esc_html_e('Posts without ThinkRank metadata → Other SEO plugins work normally', 'thinkrank'); ?><br>
                • <?php esc_html_e('You have full control per post', 'thinkrank'); ?>
            </p>

            <p>
                <a href="#" class="button button-secondary thinkrank-dismiss-seo-notice">
                    <?php esc_html_e('Got it, dismiss this notice', 'thinkrank'); ?>
                </a>
            </p>
        </div>
        <?php
    }
    
    /**
     * Enqueue notice scripts
     * 
     * @param string $hook Current admin page hook
     * @return void
     */
    public function enqueue_notice_scripts(string $hook): void {
        if (!SEO_Plugin_Detector::has_seo_plugins()) {
            return;
        }
        
        wp_add_inline_script('jquery', '
            jQuery(document).ready(function($) {
                $(".thinkrank-dismiss-seo-notice").on("click", function(e) {
                    e.preventDefault();
                    
                    $.post(ajaxurl, {
                        action: "thinkrank_dismiss_seo_notice",
                        nonce: "' . wp_create_nonce('thinkrank_seo_notice_nonce') . '"
                    }, function(response) {
                        if (response.success) {
                            $(".thinkrank-seo-notice").fadeOut();
                        }
                    });
                });
            });
        ');
    }
    
    /**
     * AJAX handler to dismiss SEO notice
     * 
     * @return void
     */
    public function ajax_dismiss_notice(): void {
        $nonce = sanitize_text_field(wp_unslash($_POST['nonce'] ?? ''));
        if (!wp_verify_nonce($nonce, 'thinkrank_seo_notice_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        SEO_Plugin_Detector::dismiss_notice();
        wp_send_json_success();
    }
}
