<?php
/*
Plugin Name: TheNoXport
Description: A simple plugin to display selected post type fields and taxonomies in a table.
Version: 1.9.2
Author: Hasan
Author website: https://hasansyed.com/
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if (!defined('ABSPATH')) exit;

class TNX_TheNoXport {

    public function __construct() {
        add_action('admin_menu', [$this, 'tnx_add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'tnx_enqueue_scripts']);
        add_action('wp_ajax_tnx_fetch_fields', [$this, 'tnx_fetch_fields_via_ajax']);
        add_action('wp_ajax_tnx_generate_table', [$this, 'tnx_fetch_post_data']);
    }

    public function tnx_add_admin_menu() {
        add_menu_page(
            'TheNoXport',
            'TheNoXport',
            'manage_options',
            'thenoxport',
            [$this, 'tnx_display_admin_page'],
            'dashicons-admin-generic',
            20
        );
    }

    public function tnx_enqueue_scripts() {
        wp_enqueue_style('thenoxport-css', plugin_dir_url(__FILE__) . 'css/thenoxport.css', [], filemtime(plugin_dir_path(__FILE__) . 'css/thenoxport.css'));
        $version = filemtime(plugin_dir_path(__FILE__) . 'js/thenoxport.js');
        wp_enqueue_script('thenoxport-js', plugin_dir_url(__FILE__) . 'js/thenoxport.js', ['jquery'], $version, true);

        wp_localize_script('thenoxport-js', 'thenoxport_ajax', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('thenoxport_ajax_nonce')
        ]);
    }

    public function tnx_display_admin_page() {
        $post_types = get_post_types(['public' => true], 'objects');
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">TheNoXport</h1>
            <form id="thenoxport-form" method="post" class="mb-4">
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="post_type">Select Post Type:</label></th>
                        <td>
                            <select id="post_type" name="post_type" class="regular-text">
                                <option value="">Select Post Type</option>
                                <?php foreach ($post_types as $post_type): ?>
                                    <option value="<?php echo esc_attr($post_type->name); ?>"><?php echo esc_html($post_type->label); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <button type="button" id="go-button" class="button button-primary">Go</button>
                        </td>
                    </tr>
                </table>
                <div id="fields-container" class="form-wrap"></div>
                <div id="actions" style="display:none;">
                    <button type="submit" class="button">Generate Table</button>
                    <button type="button" id="html-this-table" class="button">HTML This Table</button>
                    <button type="button" id="start-over-button" class="button button-secondary">Start Over</button>
                </div>
            </form>
        </div>

        <!-- Move the loading message outside of the .wrap div and below the form -->
        <div id="loading-table" class="notice notice-info" style="display:none;">Generating table, please wait...</div>

        <!-- Table and Pagination -->
        <div id="thenoxport-table" class="wp-list-table widefat fixed striped"></div>
        <div id="thenoxport-pagination" class="pagination-container" style="text-align: center; margin: 20px 0;"></div>
        <?php
    }

    public function tnx_fetch_fields_via_ajax() {
        $nonce = isset($_POST['_ajax_nonce']) ? sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])) : '';

        if (!wp_verify_nonce($nonce, 'thenoxport_ajax_nonce')) {
            wp_send_json_error('Invalid AJAX nonce.');
        }

        if (isset($_POST['post_type'])) {
            $post_type = sanitize_text_field(wp_unslash($_POST['post_type']));

            $fields = [
                'post_id'     => 'Post ID',
                'post_title'  => 'Title',
                'post_author' => 'Author',
            ];

            $args = [
                'post_type'      => $post_type,
                'posts_per_page' => 1,
                'post_status'    => 'publish'
            ];

            $posts = get_posts($args);
            $custom_fields = [];

            if ($posts) {
                $post_id = $posts[0]->ID;
                $custom_meta = get_post_custom($post_id);

                foreach ($custom_meta as $meta_key => $meta_value) {
                    if (substr($meta_key, 0, 1) !== '_') {
                        $custom_fields[$meta_key] = ucwords(str_replace('_', ' ', $meta_key));
                    }
                }
            }

            $taxonomies = get_object_taxonomies($post_type, 'objects');

            ob_start();
            echo '<h3>Select Fields to Display</h3>';
            echo '<label><input type="checkbox" id="select-all"> Select All / Deselect All</label><br>';

            foreach ($fields as $field_key => $field_label) {
                echo '<label><input type="checkbox" name="fields[]" value="' . esc_attr($field_key) . '" checked> ' . esc_html($field_label) . '</label><br>';
            }

            if (!empty($custom_fields)) {
                foreach ($custom_fields as $custom_key => $custom_label) {
                    echo '<label><input type="checkbox" name="fields[]" value="' . esc_attr($custom_key) . '"> ' . esc_html($custom_label) . '</label><br>';
                }
            } else {
                echo '<p>No custom fields found for this post type.</p>';
            }

            if (!empty($taxonomies)) {
                foreach ($taxonomies as $taxonomy) {
                    echo '<label><input type="checkbox" name="taxonomies[]" value="' . esc_attr($taxonomy->name) . '"> ' . esc_html($taxonomy->label) . '</label><br>';
                }
            }

            $output = ob_get_clean();
            wp_send_json_success($output);
        } else {
            wp_send_json_error('No post type selected.');
        }
    }

    public function tnx_fetch_post_data() {
    $nonce = isset($_POST['_ajax_nonce']) ? sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])) : '';

    if (!wp_verify_nonce($nonce, 'thenoxport_ajax_nonce')) {
        wp_send_json_error('Invalid AJAX nonce.');
    }

    $paged = isset($_POST['paged']) ? intval($_POST['paged']) : 1;
    $fetch_all = isset($_POST['fetch_all']) && $_POST['fetch_all'] === 'true';

    if (isset($_POST['post_type']) && !empty($_POST['fields'])) {
        $post_type = sanitize_text_field(wp_unslash($_POST['post_type']));
        $fields = array_map('sanitize_text_field', wp_unslash($_POST['fields']));
        $taxonomies = isset($_POST['taxonomies']) ? array_map('sanitize_text_field', wp_unslash($_POST['taxonomies'])) : [];

        $args = [
            'post_type' => $post_type,
            'posts_per_page' => $fetch_all ? -1 : 10, // Fetch all posts if fetching for HTML export
            'paged' => $paged,
        ];

        $query = new WP_Query($args);

        if ($query->have_posts()) {
            ob_start();
            echo '<table class="wp-list-table widefat fixed striped"><thead><tr>';

            foreach ($fields as $field) {
                echo '<th>' . esc_html($this->tnx_get_field_label($field)) . '</th>';
            }

            foreach ($taxonomies as $taxonomy) {
                echo '<th>' . esc_html(get_taxonomy($taxonomy)->label) . '</th>';
            }

            echo '</tr></thead><tbody>';

            while ($query->have_posts()) {
                $query->the_post();
                echo '<tr>';

                foreach ($fields as $field) {
                    if ($field === 'post_id') {
                        echo '<td>' . esc_html(get_the_ID()) . '</td>';
                    } elseif ($field === 'post_title') {
                        echo '<td>' . esc_html(get_the_title()) . '</td>';
                    } elseif ($field === 'post_author') {
                        echo '<td>' . esc_html(get_the_author()) . '</td>';
                    } else {
                        echo '<td>' . esc_html(get_post_meta(get_the_ID(), $field, true)) . '</td>';
                    }
                }

                foreach ($taxonomies as $taxonomy) {
                    $terms = get_the_terms(get_the_ID(), $taxonomy);
                    if (!empty($terms) && !is_wp_error($terms)) {
                        $term_names = wp_list_pluck($terms, 'name');
                        $escaped_terms = array_map('esc_html', $term_names);
                        echo '<td>' . implode(', ', $escaped_terms) . '</td>';
                    } else {
                        echo '<td></td>';
                    }
                }

                echo '</tr>';
            }

            echo '</tbody></table>';

            // Add pagination links if not fetching all data for export
            $pagination = '';
            if (!$fetch_all) {
                $total_pages = $query->max_num_pages;
                if ($total_pages > 1) {
                    $pagination .= '<div class="pagination-container">';
                    for ($i = 1; $i <= $total_pages; $i++) {
                        $pagination .= '<a href="#" class="pagination-link" data-page="' . $i . '">' . $i . '</a>';
                    }
                    $pagination .= '</div>';
                }
            }

            $output = ob_get_clean();

            wp_send_json_success([
                'table' => $output,
                'pagination' => $pagination,
                'total' => $query->found_posts, // Send the total number of posts
            ]);
        } else {
            wp_send_json_error('No posts found.');
        }

        wp_reset_postdata();
    } else {
        wp_send_json_error('Post type or fields missing.');
    }
}


    private function tnx_get_field_label($field_name) {
        return ucwords(str_replace('_', ' ', $field_name));
    }
}

new TNX_TheNoXport();
