<?php
/**
 * Plugin Name: Popup Coupon Generator for WooCommerce
 * Description: Popups and automatic coupons for your WooCommerce store. Create effective popups, send automated emails with personalized coupons and analyze results.
 * Version: 1.0.0
 * Author: marcodedo, themeinthebox
 * Author URI: https://www.themeinthebox.com/
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: themeinthebox-popup-coupons-for-woocommerce
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * WC requires at least: 4.0
 * WC tested up to: 7.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Declare compatibility with WooCommerce High-Performance Order Storage (HPOS).
add_action( 'before_woocommerce_init', function () {
	if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
	}
} );

// Define plugin constants
define( 'POPUCOGE_VERSION', '1.0.0' );
define( 'POPUCOGE_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'POPUCOGE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'POPUCOGE_PLUGIN_FILE', __FILE__ );

/**
 * Main plugin class
 */
class Popucoge_Plugin {
	
	/**
	 * Instance of this class
	 */
	private static $instance = null;
	
	/**
	 * Get instance of this class
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor
	 */
	private function __construct() {
		$this->check_dependencies();
		$this->includes();
		$this->init_hooks();
	}
	
	/**
	 * Check if WooCommerce is active
	 */
	private function check_dependencies() {
		add_action( 'admin_notices', array( $this, 'check_woocommerce' ) );
	}
	
	/**
	 * Check WooCommerce dependency
	 */
	public function check_woocommerce() {
		if ( ! class_exists( 'WooCommerce' ) ) {
			?>
			<div class="error">
				<p><?php esc_html_e( 'Popup Coupon Generator for WooCommerce requires WooCommerce to function. Please install and activate WooCommerce.', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
			</div>
			<?php
		}
	}
	
	/**
	 * Include required files
	 */
	private function includes() {
		require_once POPUCOGE_PLUGIN_DIR . 'includes/class-pcg-database.php';
		require_once POPUCOGE_PLUGIN_DIR . 'includes/class-pcg-admin.php';
		require_once POPUCOGE_PLUGIN_DIR . 'includes/class-pcg-popup.php';
		require_once POPUCOGE_PLUGIN_DIR . 'includes/class-pcg-automation.php';
		require_once POPUCOGE_PLUGIN_DIR . 'includes/class-pcg-statistics.php';
		require_once POPUCOGE_PLUGIN_DIR . 'includes/class-pcg-frontend.php';
	}
	
	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		register_activation_hook( POPUCOGE_PLUGIN_FILE, array( $this, 'activate' ) );
		register_deactivation_hook( POPUCOGE_PLUGIN_FILE, array( $this, 'deactivate' ) );
		
		add_action( 'plugins_loaded', array( $this, 'init' ) );
	}
	
	/**
	 * Initialize plugin
	 */
	public function init() {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return;
		}
		
		// Initialize components
		Popucoge_Admin::get_instance();
		Popucoge_Popup::get_instance();
		Popucoge_Automation::get_instance();
		Popucoge_Statistics::get_instance();
		Popucoge_Frontend::get_instance();
	}
	
	/**
	 * Activation hook
	 */
	public function activate() {
		// Create database tables
		Popucoge_Database::create_tables();
		
		// Set default options
		$this->set_default_options();
		
		// Flush rewrite rules
		flush_rewrite_rules();
	}
	
	/**
	 * Deactivation hook
	 */
	public function deactivate() {
		flush_rewrite_rules();
	}
	
	/**
	 * Set default options
	 */
	private function set_default_options() {
		if ( ! get_option( 'popucoge_settings' ) ) {
			add_option( 'popucoge_settings', array(
				'enable_analytics' => 'no',
				'analytics_id' => '',
				'email_from_name' => get_bloginfo( 'name' ),
				'email_from_email' => get_option( 'admin_email' ),
			) );
		}
	}
}

/**
 * Initialize the plugin
 */
function popucoge_plugin() {
	return Popucoge_Plugin::get_instance();
}

// Start the plugin
popucoge_plugin();
