<?php
/**
 * Statistics class for Popup Coupon Generator for WooCommerce
 *
 * @package Popup_Coupon_Generator
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Statistics management class
 */
class Popucoge_Statistics {
	
	/**
	 * Instance of this class
	 */
	private static $instance = null;
	
	/**
	 * Get instance of this class
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor
	 */
	private function __construct() {
		// Statistics tracking handled in other classes
	}
	
	/**
	 * Get statistics for automation
	 */
	public function get_automation_stats( $automation_id ) {
		global $wpdb;
		
		$stats = array(
			'emails_sent' => 0,
			'emails_opened' => 0,
			'emails_clicked' => 0,
			'coupons_used' => 0,
			'orders' => 0,
			'revenue' => 0,
		);
		
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$wpdb->prefix}popucoge_email_stats WHERE automation_id = %s",
			$automation_id
		) );
		
		foreach ( $results as $result ) {
			$stats['emails_sent']++;
			
			if ( ! empty( $result->email_opened_date ) ) {
				$stats['emails_opened']++;
			}
			
			if ( ! empty( $result->email_clicked_date ) ) {
				$stats['emails_clicked']++;
			}
			
			if ( ! empty( $result->order_id ) ) {
				$stats['orders']++;
				$order = wc_get_order( $result->order_id );
				if ( $order ) {
					$stats['revenue'] += $order->get_total();
					
					// Check if coupon was used
					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
					$coupon_codes = $wpdb->get_col( $wpdb->prepare(
						"SELECT coupon_code FROM {$wpdb->prefix}popucoge_coupons WHERE automation_id = %s AND subscriber_id = %d",
						$automation_id,
						$result->subscriber_id
					) );
					
					$used_coupons = $order->get_coupon_codes();
					foreach ( $used_coupons as $used_code ) {
						if ( in_array( $used_code, $coupon_codes ) ) {
							$stats['coupons_used']++;
							break;
						}
					}
				}
			}
		}
		
		return $stats;
	}
	
	/**
	 * Get overall statistics
	 */
	public function get_overall_stats() {
		global $wpdb;
		
		$stats = array(
			'total_subscribers' => 0,
			'total_coupons' => 0,
			'total_emails_sent' => 0,
			'total_orders' => 0,
			'total_revenue' => 0,
		);
		
		// Total subscribers
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$stats['total_subscribers'] = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->prefix}popucoge_subscribers WHERE status = 'active'"
		);
		
		// Total coupons
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$stats['total_coupons'] = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->prefix}popucoge_coupons WHERE status = 'active'"
		);
		
		// Total emails sent
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$stats['total_emails_sent'] = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->prefix}popucoge_email_stats"
		);
		
		// Orders and revenue
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$orders = $wpdb->get_results(
			"SELECT DISTINCT order_id FROM {$wpdb->prefix}popucoge_email_stats WHERE order_id IS NOT NULL AND order_id > 0"
		);
		
		foreach ( $orders as $order_row ) {
			$order = wc_get_order( $order_row->order_id );
			if ( $order ) {
				$stats['total_orders']++;
				$stats['total_revenue'] += $order->get_total();
			}
		}
		
		return $stats;
	}
	
	/**
	 * Render admin page
	 */
	public function render_admin_page() {
		$overall_stats = $this->get_overall_stats();
		$automations = array();
		if ( class_exists( 'Popucoge_Automation' ) ) {
			$automations = Popucoge_Automation::get_instance()->get_automations();
		}
		
		?>
		<div class="wrap popucoge-statistics-page">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<div class="popucoge-stats-overview">
				<h2><?php esc_html_e( 'General Overview', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></h2>
				
				<div class="popucoge-stats-grid">
					<div class="popucoge-stat-box">
						<h3><?php esc_html_e( 'Total Subscribers', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></h3>
						<div class="popucoge-stat-value"><?php echo esc_html( number_format_i18n( $overall_stats['total_subscribers'] ) ); ?></div>
					</div>
					
					<div class="popucoge-stat-box">
						<h3><?php esc_html_e( 'Coupons Generated', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></h3>
						<div class="popucoge-stat-value"><?php echo esc_html( number_format_i18n( $overall_stats['total_coupons'] ) ); ?></div>
					</div>
					
					<div class="popucoge-stat-box">
						<h3><?php esc_html_e( 'Emails Sent', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></h3>
						<div class="popucoge-stat-value"><?php echo esc_html( number_format_i18n( $overall_stats['total_emails_sent'] ) ); ?></div>
					</div>
					
					<div class="popucoge-stat-box">
						<h3><?php esc_html_e( 'Total Orders', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></h3>
						<div class="popucoge-stat-value"><?php echo esc_html( number_format_i18n( $overall_stats['total_orders'] ) ); ?></div>
					</div>
					
					<div class="popucoge-stat-box">
						<h3><?php esc_html_e( 'Revenue', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></h3>
						<div class="popucoge-stat-value"><?php echo wp_kses_post( wc_price( $overall_stats['total_revenue'] ) ); ?></div>
					</div>
				</div>
			</div>
			
			<div class="popucoge-stats-automations">
				<h2><?php esc_html_e( 'Statistics by Automation', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></h2>
				
				<?php if ( empty( $automations ) ) : ?>
					<p><?php esc_html_e( 'No automations found.', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
				<?php else : ?>
					<table class="wp-list-table widefat fixed striped">
						<thead>
							<tr>
								<th><?php esc_html_e( 'Automation', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
								<th><?php esc_html_e( 'Emails Sent', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
								<th><?php esc_html_e( 'Opens', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
								<th><?php esc_html_e( 'Clicks', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
								<th><?php esc_html_e( 'Coupons Used', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
								<th><?php esc_html_e( 'Orders', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
								<th><?php esc_html_e( 'Revenue', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<?php foreach ( $automations as $auto_id => $automation ) : ?>
								<?php
								$auto_stats = $this->get_automation_stats( $auto_id );
								$open_rate = $auto_stats['emails_sent'] > 0 ? ( $auto_stats['emails_opened'] / $auto_stats['emails_sent'] ) * 100 : 0;
								$click_rate = $auto_stats['emails_sent'] > 0 ? ( $auto_stats['emails_clicked'] / $auto_stats['emails_sent'] ) * 100 : 0;
								$conversion_rate = $auto_stats['emails_sent'] > 0 ? ( $auto_stats['orders'] / $auto_stats['emails_sent'] ) * 100 : 0;
								?>
								<tr>
									<td><strong><?php echo esc_html( isset( $automation['name'] ) ? $automation['name'] : $auto_id ); ?></strong></td>
									<td><?php echo esc_html( number_format_i18n( $auto_stats['emails_sent'] ) ); ?></td>
									<td>
										<?php echo esc_html( number_format_i18n( $auto_stats['emails_opened'] ) ); ?>
										<small style="color: #666;">(<?php echo esc_html( number_format( $open_rate, 1 ) ); ?>%)</small>
									</td>
									<td>
										<?php echo esc_html( number_format_i18n( $auto_stats['emails_clicked'] ) ); ?>
										<small style="color: #666;">(<?php echo esc_html( number_format( $click_rate, 1 ) ); ?>%)</small>
									</td>
									<td><?php echo esc_html( number_format_i18n( $auto_stats['coupons_used'] ) ); ?></td>
									<td>
										<?php echo esc_html( number_format_i18n( $auto_stats['orders'] ) ); ?>
										<small style="color: #666;">(<?php echo esc_html( number_format( $conversion_rate, 1 ) ); ?>%)</small>
									</td>
									<td><strong><?php echo wp_kses_post( wc_price( $auto_stats['revenue'] ) ); ?></strong></td>
								</tr>
							<?php endforeach; ?>
						</tbody>
					</table>
				<?php endif; ?>
			</div>
		</div>
		<?php
	}
}

