<?php
/**
 * Popup class for Popup Coupon Generator for WooCommerce
 *
 * @package Popup_Coupon_Generator
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Popup management class
 */
class Popucoge_Popup {
	
	/**
	 * Instance of this class
	 */
	private static $instance = null;
	
	/**
	 * Get instance of this class
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'wp_ajax_popucoge_delete_popup', array( $this, 'ajax_delete_popup' ) );
		add_action( 'wp_ajax_popucoge_toggle_popup', array( $this, 'ajax_toggle_popup' ) );
	}
	
	/**
	 * Get all popups
	 */
	public function get_popups() {
		$popups = get_option( 'popucoge_popups', array() );
		return $popups;
	}
	
	/**
	 * Get popup by ID
	 */
	public function get_popup( $popup_id ) {
		$popups = $this->get_popups();
		return isset( $popups[ $popup_id ] ) ? $popups[ $popup_id ] : false;
	}
	
	/**
	 * Save popup
	 */
	public function save_popup( $popup_data ) {
		$popups = $this->get_popups();
		
		if ( ! isset( $popup_data['id'] ) || empty( $popup_data['id'] ) ) {
			$popup_data['id'] = uniqid( 'popup_' );
			$popup_data['created'] = current_time( 'mysql' );
		}
		
		$popup_data['updated'] = current_time( 'mysql' );
		
		$popups[ $popup_data['id'] ] = $popup_data;
		
		update_option( 'popucoge_popups', $popups );
		
		return $popup_data['id'];
	}
	
	/**
	 * Delete popup
	 */
	public function delete_popup( $popup_id ) {
		$popups = $this->get_popups();
		
		if ( isset( $popups[ $popup_id ] ) ) {
			unset( $popups[ $popup_id ] );
			update_option( 'popucoge_popups', $popups );
			return true;
		}
		
		return false;
	}
	
	/**
	 * Render admin page
	 */
	public function render_admin_page() {
		$this->handle_form_submit();
		
		$popups = $this->get_popups();
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading GET parameters for display purposes only
		$editing_popup_id = isset( $_GET['edit'] ) ? sanitize_text_field( wp_unslash( $_GET['edit'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reading GET parameters for display purposes only
		$action_new = isset( $_GET['action'] ) && sanitize_text_field( wp_unslash( $_GET['action'] ) ) === 'new';
		$editing_popup = ( $editing_popup_id && ! $action_new ) ? $this->get_popup( $editing_popup_id ) : null;
		
		// If action=new, show form with empty popup
		if ( $action_new ) {
			$editing_popup = array();
		}
		
		?>
		<div class="wrap popucoge-popup-page">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<?php if ( ! $editing_popup && ! $action_new ) : ?>
				<div class="popucoge-header-actions">
					<a href="<?php echo esc_url( add_query_arg( 'action', 'new', admin_url( 'admin.php?page=themeinthebox-popup-coupons-for-woocommerce' ) ) ); ?>" class="button button-primary">
						<?php esc_html_e( '+ New Popup', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
					</a>
				</div>
				
				<table class="wp-list-table widefat fixed striped">
					<thead>
						<tr>
							<th><?php esc_html_e( 'Name', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
							<th><?php esc_html_e( 'Display Type', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
							<th><?php esc_html_e( 'Status', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
							<th><?php esc_html_e( 'Actions', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php if ( empty( $popups ) ) : ?>
							<tr>
								<td colspan="4"><?php esc_html_e( 'No popups found. Create your first popup!', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></td>
							</tr>
						<?php else : ?>
							<?php foreach ( $popups as $popup_id => $popup ) : ?>
								<tr>
									<td><strong><?php echo esc_html( isset( $popup['name'] ) ? $popup['name'] : __( 'Untitled', 'themeinthebox-popup-coupons-for-woocommerce' ) ); ?></strong></td>
									<td><?php echo esc_html( $this->get_display_type_label( isset( $popup['display_type'] ) ? $popup['display_type'] : 'exit_intent' ) ); ?></td>
									<td>
										<span class="popucoge-status-badge status-<?php echo esc_attr( isset( $popup['status'] ) && $popup['status'] === 'active' ? 'active' : 'inactive' ); ?>">
											<?php echo esc_html( isset( $popup['status'] ) && $popup['status'] === 'active' ? __( 'Active', 'themeinthebox-popup-coupons-for-woocommerce' ) : __( 'Inactive', 'themeinthebox-popup-coupons-for-woocommerce' ) ); ?>
										</span>
									</td>
									<td>
										<a href="<?php echo esc_url( add_query_arg( array( 'page' => 'themeinthebox-popup-coupons-for-woocommerce', 'edit' => $popup_id ), admin_url( 'admin.php' ) ) ); ?>" class="button button-small">
											<?php esc_html_e( 'Edit', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
										</a>
										<a href="#" class="button button-small popucoge-toggle-popup" data-popup-id="<?php echo esc_attr( $popup_id ); ?>" data-status="<?php echo esc_attr( isset( $popup['status'] ) && $popup['status'] === 'active' ? 'active' : 'inactive' ); ?>">
											<?php echo esc_html( isset( $popup['status'] ) && $popup['status'] === 'active' ? __( 'Deactivate', 'themeinthebox-popup-coupons-for-woocommerce' ) : __( 'Activate', 'themeinthebox-popup-coupons-for-woocommerce' ) ); ?>
										</a>
										<a href="#" class="button button-small popucoge-delete-popup" data-popup-id="<?php echo esc_attr( $popup_id ); ?>">
											<?php esc_html_e( 'Delete', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
										</a>
									</td>
								</tr>
							<?php endforeach; ?>
						<?php endif; ?>
					</tbody>
				</table>
			<?php else : ?>
				<?php $this->render_popup_form( $editing_popup ); ?>
			<?php endif; ?>
		</div>
		<?php
	}
	
	/**
	 * Render popup form
	 */
	private function render_popup_form( $popup = null ) {
		$popup = $popup ? $popup : array();
		?>
		<form method="post" action="" id="popucoge-popup-form">
			<?php wp_nonce_field( 'popucoge-save-popup', 'popucoge_popup_nonce' ); ?>
			
			<input type="hidden" name="popup_id" value="<?php echo esc_attr( isset( $popup['id'] ) ? $popup['id'] : '' ); ?>" />
			
			<table class="form-table">
				<tbody>
					<tr>
						<th scope="row">
							<label for="popup_name"><?php esc_html_e( 'Popup Name', 'themeinthebox-popup-coupons-for-woocommerce' ); ?> <span class="required">*</span></label>
						</th>
						<td>
							<input type="text" id="popup_name" name="popup_name" value="<?php echo esc_attr( isset( $popup['name'] ) ? $popup['name'] : '' ); ?>" class="regular-text" required />
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="display_type"><?php esc_html_e( 'Display Type', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<select id="display_type" name="display_type" class="regular-text">
								<option value="exit_intent" <?php selected( isset( $popup['display_type'] ) ? $popup['display_type'] : 'exit_intent', 'exit_intent' ); ?>>
									<?php esc_html_e( 'Exit Intent (when mouse leaves window)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
								</option>
								<option value="time_delay" <?php selected( isset( $popup['display_type'] ) ? $popup['display_type'] : '', 'time_delay' ); ?>>
									<?php esc_html_e( 'After a certain time', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
								</option>
								<option value="scroll" <?php selected( isset( $popup['display_type'] ) ? $popup['display_type'] : '', 'scroll' ); ?>>
									<?php esc_html_e( 'After scrolling a percentage', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
								</option>
								<option value="page_load" <?php selected( isset( $popup['display_type'] ) ? $popup['display_type'] : '', 'page_load' ); ?>>
									<?php esc_html_e( 'On page load', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
								</option>
							</select>
						</td>
					</tr>
					
					<tr class="display-condition" id="condition-time_delay" style="display: none;">
						<th scope="row">
							<label for="time_delay"><?php esc_html_e( 'Delay (seconds)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<input type="number" id="time_delay" name="time_delay" value="<?php echo esc_attr( isset( $popup['time_delay'] ) ? $popup['time_delay'] : '5' ); ?>" min="1" class="small-text" />
						</td>
					</tr>
					
					<tr class="display-condition" id="condition-scroll" style="display: none;">
						<th scope="row">
							<label for="scroll_percentage"><?php esc_html_e( 'Scroll Percentage', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<input type="number" id="scroll_percentage" name="scroll_percentage" value="<?php echo esc_attr( isset( $popup['scroll_percentage'] ) ? $popup['scroll_percentage'] : '50' ); ?>" min="1" max="100" class="small-text" /> %
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="popup_content"><?php esc_html_e( 'Popup Content (HTML)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<?php
							$content = isset( $popup['content'] ) ? $popup['content'] : '';
							wp_editor( $content, 'popup_content', array(
								'textarea_name' => 'popup_content',
								'textarea_rows' => 10,
								'media_buttons' => true,
								'tinymce' => true,
							) );
							?>
							<p class="description"><?php esc_html_e( 'Enter the popup content. You can include a form for newsletter subscription.', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="form_shortcode"><?php esc_html_e( 'Shortcode Form Newsletter', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<input type="text" id="form_shortcode" name="form_shortcode" value="<?php echo esc_attr( isset( $popup['form_shortcode'] ) ? $popup['form_shortcode'] : '' ); ?>" class="regular-text" placeholder="[contact-form-7 id=&quot;123&quot;]" />
							<p class="description"><?php esc_html_e( 'Enter the newsletter form shortcode (e.g. Contact Form 7, MailPoet, etc.)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="automation_id"><?php esc_html_e( 'Email Automation', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<?php
							$automations = array();
							if ( class_exists( 'Popucoge_Automation' ) ) {
								$automations = Popucoge_Automation::get_instance()->get_automations();
							}
							?>
							<select id="automation_id" name="automation_id" class="regular-text">
								<option value=""><?php esc_html_e( 'No automation', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></option>
								<?php foreach ( $automations as $auto_id => $automation ) : ?>
									<option value="<?php echo esc_attr( $auto_id ); ?>" <?php selected( isset( $popup['automation_id'] ) ? $popup['automation_id'] : '', $auto_id ); ?>>
										<?php echo esc_html( isset( $automation['name'] ) ? $automation['name'] : $auto_id ); ?>
									</option>
								<?php endforeach; ?>
							</select>
							<p class="description"><?php esc_html_e( 'Select the email automation to trigger when a user subscribes through this popup', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="popup_status"><?php esc_html_e( 'Status', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<label>
								<input type="checkbox" id="popup_status" name="popup_status" value="active" <?php checked( ! isset( $popup['status'] ) || $popup['status'] === 'active' ); ?> />
								<?php esc_html_e( 'Active', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
							</label>
						</td>
					</tr>
				</tbody>
			</table>
			
			<p class="submit">
				<?php submit_button( __( 'Save Popup', 'themeinthebox-popup-coupons-for-woocommerce' ), 'primary', 'save_popup', false ); ?>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=themeinthebox-popup-coupons-for-woocommerce' ) ); ?>" class="button">
					<?php esc_html_e( 'Cancel', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
				</a>
			</p>
		</form>
		<?php
	}
	
	/**
	 * Handle form submit
	 */
	private function handle_form_submit() {
		if ( ! isset( $_POST['popucoge_popup_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['popucoge_popup_nonce'] ) ), 'popucoge-save-popup' ) ) {
			return;
		}
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}
		
		if ( isset( $_POST['save_popup'] ) ) {
			$popup_data = array(
				'id' => isset( $_POST['popup_id'] ) ? sanitize_text_field( wp_unslash( $_POST['popup_id'] ) ) : '',
				'name' => isset( $_POST['popup_name'] ) ? sanitize_text_field( wp_unslash( $_POST['popup_name'] ) ) : '',
				'display_type' => isset( $_POST['display_type'] ) ? sanitize_text_field( wp_unslash( $_POST['display_type'] ) ) : 'exit_intent',
				'time_delay' => isset( $_POST['time_delay'] ) ? intval( wp_unslash( $_POST['time_delay'] ) ) : 5,
				'scroll_percentage' => isset( $_POST['scroll_percentage'] ) ? intval( wp_unslash( $_POST['scroll_percentage'] ) ) : 50,
				'content' => isset( $_POST['popup_content'] ) ? wp_kses_post( wp_unslash( $_POST['popup_content'] ) ) : '',
				'form_shortcode' => isset( $_POST['form_shortcode'] ) ? sanitize_text_field( wp_unslash( $_POST['form_shortcode'] ) ) : '',
				'automation_id' => isset( $_POST['automation_id'] ) ? sanitize_text_field( wp_unslash( $_POST['automation_id'] ) ) : '',
				'status' => isset( $_POST['popup_status'] ) ? 'active' : 'inactive',
			);
			
			$this->save_popup( $popup_data );
			
			wp_safe_redirect( admin_url( 'admin.php?page=themeinthebox-popup-coupons-for-woocommerce' ) );
			exit;
		}
	}
	
	/**
	 * Get display type label
	 */
	private function get_display_type_label( $type ) {
		$labels = array(
			'exit_intent' => __( 'Exit Intent', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'time_delay' => __( 'Time Delay', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'scroll' => __( 'After Scroll', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'page_load' => __( 'On Load', 'themeinthebox-popup-coupons-for-woocommerce' ),
		);
		
		return isset( $labels[ $type ] ) ? $labels[ $type ] : $type;
	}
	
	/**
	 * AJAX: Delete popup
	 */
	public function ajax_delete_popup() {
		check_ajax_referer( 'popucoge-admin-nonce', 'nonce' );
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		$popup_id = isset( $_POST['popup_id'] ) ? sanitize_text_field( wp_unslash( $_POST['popup_id'] ) ) : '';
		
		if ( empty( $popup_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Missing popup ID', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		if ( $this->delete_popup( $popup_id ) ) {
			wp_send_json_success( array( 'message' => __( 'Popup deleted successfully', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		wp_send_json_error( array( 'message' => __( 'Error during deletion', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
	}
	
	/**
	 * AJAX: Toggle popup status
	 */
	public function ajax_toggle_popup() {
		check_ajax_referer( 'popucoge-admin-nonce', 'nonce' );
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		$popup_id = isset( $_POST['popup_id'] ) ? sanitize_text_field( wp_unslash( $_POST['popup_id'] ) ) : '';
		$status = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : 'inactive';
		
		if ( empty( $popup_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Missing popup ID', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		$popup = $this->get_popup( $popup_id );
		if ( ! $popup ) {
			wp_send_json_error( array( 'message' => __( 'Popup not found', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		$popup['status'] = $status === 'active' ? 'inactive' : 'active';
		$this->save_popup( $popup );
		
		wp_send_json_success( array( 
			'message' => __( 'Status updated', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'status' => $popup['status']
		) );
	}
}
