<?php
/**
 * Frontend class for Popup Coupon Generator for WooCommerce
 *
 * @package Popup_Coupon_Generator
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Frontend management class
 */
class Popucoge_Frontend {
	
	/**
	 * Instance of this class
	 */
	private static $instance = null;
	
	/**
	 * Get instance of this class
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'wp_footer', array( $this, 'render_popups' ) );
		add_action( 'wp_ajax_popucoge_subscribe', array( $this, 'ajax_subscribe' ) );
		add_action( 'wp_ajax_nopriv_popucoge_subscribe', array( $this, 'ajax_subscribe' ) );
		add_action( 'woocommerce_thankyou', array( $this, 'track_order_coupon' ), 10, 1 );
		add_filter( 'woocommerce_email_tracking_pixel', array( $this, 'add_email_tracking' ), 10, 2 );
	}
	
	/**
	 * Enqueue frontend scripts and styles
	 */
	public function enqueue_scripts() {
		wp_enqueue_style(
			'popucoge-frontend',
			POPUCOGE_PLUGIN_URL . 'assets/css/frontend.css',
			array(),
			POPUCOGE_VERSION
		);
		
		wp_enqueue_script(
			'popucoge-frontend',
			POPUCOGE_PLUGIN_URL . 'assets/js/frontend.js',
			array( 'jquery' ),
			POPUCOGE_VERSION,
			true
		);
		
		$popups = Popucoge_Popup::get_instance()->get_popups();
		$active_popups = array();
		
		foreach ( $popups as $popup_id => $popup ) {
			if ( isset( $popup['status'] ) && $popup['status'] === 'active' ) {
				$active_popups[ $popup_id ] = array(
					'display_type' => isset( $popup['display_type'] ) ? $popup['display_type'] : 'exit_intent',
					'time_delay' => isset( $popup['time_delay'] ) ? intval( $popup['time_delay'] ) : 5,
					'scroll_percentage' => isset( $popup['scroll_percentage'] ) ? intval( $popup['scroll_percentage'] ) : 50,
				);
			}
		}
		
		$settings = get_option( 'popucoge_settings', array() );
		
		wp_localize_script( 'popucoge-frontend', 'popucogeFrontend', array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce' => wp_create_nonce( 'popucoge-frontend-nonce' ),
			'popups' => $active_popups,
			'analytics_enabled' => isset( $settings['enable_analytics'] ) && $settings['enable_analytics'] === 'yes',
			'analytics_id' => isset( $settings['analytics_id'] ) ? $settings['analytics_id'] : '',
		) );
	}
	
	/**
	 * Render popups in footer
	 */
	public function render_popups() {
		$popups = Popucoge_Popup::get_instance()->get_popups();
		
		foreach ( $popups as $popup_id => $popup ) {
			if ( isset( $popup['status'] ) && $popup['status'] === 'active' ) {
				$this->render_popup( $popup_id, $popup );
			}
		}
	}
	
	/**
	 * Render single popup
	 */
	private function render_popup( $popup_id, $popup ) {
		$content = isset( $popup['content'] ) ? $popup['content'] : '';
		$form_shortcode = isset( $popup['form_shortcode'] ) ? $popup['form_shortcode'] : '';
		$display_type = isset( $popup['display_type'] ) ? $popup['display_type'] : 'exit_intent';
		?>
		<div id="popucoge-popup-<?php echo esc_attr( $popup_id ); ?>" class="popucoge-popup" data-popup-id="<?php echo esc_attr( $popup_id ); ?>" data-display-type="<?php echo esc_attr( $display_type ); ?>" style="display: none;">
			<div class="popucoge-popup-overlay"></div>
			<div class="popucoge-popup-container">
				<button class="popucoge-popup-close" aria-label="<?php esc_attr_e( 'Close', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>">&times;</button>
				<div class="popucoge-popup-content">
					<?php echo wp_kses_post( do_shortcode( $content ) ); ?>
					<?php if ( ! empty( $form_shortcode ) ) : ?>
						<div class="popucoge-popup-form">
							<?php echo do_shortcode( $form_shortcode ); ?>
						</div>
					<?php endif; ?>
					
					<?php if ( empty( $form_shortcode ) ) : ?>
						<div class="popucoge-popup-form">
							<form id="popucoge-subscribe-form-<?php echo esc_attr( $popup_id ); ?>" class="popucoge-subscribe-form">
								<?php wp_nonce_field( 'popucoge-subscribe', 'popucoge_subscribe_nonce' ); ?>
								<input type="hidden" name="popup_id" value="<?php echo esc_attr( $popup_id ); ?>" />
								<p>
									<label for="popucoge-email-<?php echo esc_attr( $popup_id ); ?>"><?php esc_html_e( 'Email', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
									<input type="email" id="popucoge-email-<?php echo esc_attr( $popup_id ); ?>" name="email" required />
								</p>
								<p>
									<label for="popucoge-name-<?php echo esc_attr( $popup_id ); ?>"><?php esc_html_e( 'Name (optional)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
									<input type="text" id="popucoge-name-<?php echo esc_attr( $popup_id ); ?>" name="name" />
								</p>
								<p>
									<button type="submit" class="popucoge-subscribe-button"><?php esc_html_e( 'Subscribe', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></button>
								</p>
								<div class="popucoge-form-message"></div>
							</form>
						</div>
					<?php endif; ?>
				</div>
			</div>
		</div>
		<?php
	}
	
	/**
	 * AJAX: Subscribe user
	 */
	public function ajax_subscribe() {
		check_ajax_referer( 'popucoge-frontend-nonce', 'nonce' );
		
		$email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		$name = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
		$popup_id = isset( $_POST['popup_id'] ) ? sanitize_text_field( wp_unslash( $_POST['popup_id'] ) ) : '';
		
		if ( empty( $email ) || ! is_email( $email ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid email address', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		if ( empty( $popup_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Missing popup ID', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		global $wpdb;
		
		// Check if subscriber already exists
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$existing = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$wpdb->prefix}popucoge_subscribers WHERE email = %s",
			$email
		) );
		
		$subscriber_id = null;
		
		if ( $existing ) {
			// Update existing subscriber
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->update(
				$wpdb->prefix . 'popucoge_subscribers',
				array(
					'name' => $name,
					'popup_id' => $popup_id,
				),
				array( 'id' => $existing->id ),
				array( '%s', '%s' ),
				array( '%d' )
			);
			$subscriber_id = $existing->id;
		} else {
			// Create new subscriber
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->insert(
				$wpdb->prefix . 'popucoge_subscribers',
				array(
					'email' => $email,
					'name' => $name,
					'popup_id' => $popup_id,
					'subscription_date' => current_time( 'mysql' ),
					'status' => 'active',
				),
				array( '%s', '%s', '%s', '%s', '%s' )
			);
			$subscriber_id = $wpdb->insert_id;
		}
		
		if ( $subscriber_id ) {
			// Trigger automation
			do_action( 'popucoge_subscriber_registered', $subscriber_id, $popup_id );
			
			// Track Google Analytics event if enabled
			$settings = get_option( 'popucoge_settings', array() );
			if ( isset( $settings['enable_analytics'] ) && $settings['enable_analytics'] === 'yes' ) {
				// Event tracked in frontend.js
			}
			
			wp_send_json_success( array( 
				'message' => __( 'Subscription completed successfully! Check your email.', 'themeinthebox-popup-coupons-for-woocommerce' )
			) );
		}
		
		wp_send_json_error( array( 'message' => __( 'Error during subscription', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
	}
	
	/**
	 * Track order coupon usage
	 */
	public function track_order_coupon( $order_id ) {
		if ( ! $order_id ) {
			return;
		}
		
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}
		
		$coupon_codes = $order->get_coupon_codes();
		if ( empty( $coupon_codes ) ) {
			return;
		}
		
		global $wpdb;
		
		foreach ( $coupon_codes as $coupon_code ) {
			// Find coupon in our database
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$coupon_data = $wpdb->get_row( $wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}popucoge_coupons WHERE coupon_code = %s",
				$coupon_code
			) );
			
			if ( $coupon_data ) {
				// Update coupon as used
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->update(
					$wpdb->prefix . 'popucoge_coupons',
					array(
						'used_date' => current_time( 'mysql' ),
						'status' => 'used',
					),
					array( 'id' => $coupon_data->id ),
					array( '%s', '%s' ),
					array( '%d' )
				);
				
				// Update email stats with order
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->update(
					$wpdb->prefix . 'popucoge_email_stats',
					array( 'order_id' => $order_id ),
					array(
						'subscriber_id' => $coupon_data->subscriber_id,
						'automation_id' => $coupon_data->automation_id,
					),
					array( '%d' ),
					array( '%d', '%s' )
				);
			}
		}
	}
	
	/**
	 * Add email tracking pixel (placeholder for future implementation)
	 */
	public function add_email_tracking( $tracking_url, $email ) {
		// Future implementation for email open tracking
		return $tracking_url;
	}
}

