<?php
/**
 * Database class for Popup Coupon Generator for WooCommerce
 *
 * @package Popup_Coupon_Generator
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Database management class
 */
class Popucoge_Database {
	
	/**
	 * Create database tables and migrate from old prefix if needed
	 */
	public static function create_tables() {
		global $wpdb;
		
		self::maybe_migrate_from_pcg();
		
		$charset_collate = $wpdb->get_charset_collate();
		
		// Table for popup subscribers
		$table_subscribers = $wpdb->prefix . 'popucoge_subscribers';
		$sql_subscribers = "CREATE TABLE IF NOT EXISTS $table_subscribers (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			email varchar(255) NOT NULL,
			name varchar(255) DEFAULT NULL,
			popup_id bigint(20) DEFAULT NULL,
			subscription_date datetime DEFAULT CURRENT_TIMESTAMP,
			status varchar(20) DEFAULT 'active',
			PRIMARY KEY (id),
			UNIQUE KEY email (email),
			KEY popup_id (popup_id)
		) $charset_collate;";
		
		// Table for coupons linked to subscribers
		$table_coupons = $wpdb->prefix . 'popucoge_coupons';
		$sql_coupons = "CREATE TABLE IF NOT EXISTS $table_coupons (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			subscriber_id bigint(20) NOT NULL,
			coupon_code varchar(100) NOT NULL,
			automation_id bigint(20) DEFAULT NULL,
			created_date datetime DEFAULT CURRENT_TIMESTAMP,
			used_date datetime DEFAULT NULL,
			status varchar(20) DEFAULT 'active',
			PRIMARY KEY (id),
			UNIQUE KEY coupon_code (coupon_code),
			KEY subscriber_id (subscriber_id),
			KEY automation_id (automation_id)
		) $charset_collate;";
		
		// Table for email automation stats
		$table_email_stats = $wpdb->prefix . 'popucoge_email_stats';
		$sql_email_stats = "CREATE TABLE IF NOT EXISTS $table_email_stats (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			subscriber_id bigint(20) NOT NULL,
			automation_id bigint(20) NOT NULL,
			email_sent_date datetime DEFAULT CURRENT_TIMESTAMP,
			email_opened_date datetime DEFAULT NULL,
			email_clicked_date datetime DEFAULT NULL,
			order_id bigint(20) DEFAULT NULL,
			PRIMARY KEY (id),
			KEY subscriber_id (subscriber_id),
			KEY automation_id (automation_id),
			KEY order_id (order_id)
		) $charset_collate;";
		
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql_subscribers );
		dbDelta( $sql_coupons );
		dbDelta( $sql_email_stats );
	}
	
	/**
	 * One-time migration from old pcg_ prefix to popucoge_
	 */
	private static function maybe_migrate_from_pcg() {
		global $wpdb;
		
		// Migrate options
		$options_to_migrate = array( 'pcg_settings', 'pcg_popups', 'pcg_automations' );
		$new_prefix = 'popucoge_';
		foreach ( $options_to_migrate as $old_key ) {
			$new_key = str_replace( 'pcg_', $new_prefix, $old_key );
			if ( get_option( $old_key ) !== false && get_option( $new_key ) === false ) {
				update_option( $new_key, get_option( $old_key ) );
				delete_option( $old_key );
			}
		}
		
		// Migrate tables (rename if old exist and new don't).
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration: no WP API for SHOW TABLES or RENAME TABLE.
		$tables = array( 'subscribers', 'coupons', 'email_stats' );
		foreach ( $tables as $table_suffix ) {
			$old_table = $wpdb->prefix . 'pcg_' . $table_suffix;
			$new_table = $wpdb->prefix . 'popucoge_' . $table_suffix;
			if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $old_table ) ) === $old_table
				&& $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new_table ) ) !== $new_table ) {
				$wpdb->query( "RENAME TABLE `{$old_table}` TO `{$new_table}`" );
			}
		}
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	}
}
