<?php
/**
 * Automation class for Popup Coupon Generator for WooCommerce
 *
 * @package Popup_Coupon_Generator
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Automation management class
 */
class Popucoge_Automation {
	
	/**
	 * Instance of this class
	 */
	private static $instance = null;
	
	/**
	 * Get instance of this class
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'wp_ajax_popucoge_delete_automation', array( $this, 'ajax_delete_automation' ) );
		add_action( 'wp_ajax_popucoge_toggle_automation', array( $this, 'ajax_toggle_automation' ) );
		add_action( 'popucoge_subscriber_registered', array( $this, 'trigger_automation' ), 10, 2 );
	}
	
	/**
	 * Get all automations
	 */
	public function get_automations() {
		$automations = get_option( 'popucoge_automations', array() );
		return $automations;
	}
	
	/**
	 * Get automation by ID
	 */
	public function get_automation( $automation_id ) {
		$automations = $this->get_automations();
		return isset( $automations[ $automation_id ] ) ? $automations[ $automation_id ] : false;
	}
	
	/**
	 * Save automation
	 */
	public function save_automation( $automation_data ) {
		$automations = $this->get_automations();
		
		if ( ! isset( $automation_data['id'] ) || empty( $automation_data['id'] ) ) {
			$automation_data['id'] = uniqid( 'auto_' );
			$automation_data['created'] = current_time( 'mysql' );
		}
		
		$automation_data['updated'] = current_time( 'mysql' );
		
		$automations[ $automation_data['id'] ] = $automation_data;
		
		update_option( 'popucoge_automations', $automations );
		
		return $automation_data['id'];
	}
	
	/**
	 * Delete automation
	 */
	public function delete_automation( $automation_id ) {
		$automations = $this->get_automations();
		
		if ( isset( $automations[ $automation_id ] ) ) {
			unset( $automations[ $automation_id ] );
			update_option( 'popucoge_automations', $automations );
			return true;
		}
		
		return false;
	}
	
	/**
	 * Trigger automation for subscriber
	 */
	public function trigger_automation( $subscriber_id, $popup_id ) {
		global $wpdb;
		
		// Get popup to find associated automation
		if ( ! class_exists( 'Popucoge_Popup' ) ) {
			return;
		}
		
		$popup = Popucoge_Popup::get_instance()->get_popup( $popup_id );
		
		if ( ! $popup || empty( $popup['automation_id'] ) ) {
			return;
		}
		
		$automation = $this->get_automation( $popup['automation_id'] );
		
		if ( ! $automation || ( isset( $automation['status'] ) && $automation['status'] !== 'active' ) ) {
			return;
		}
		
		// Get subscriber data
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$subscriber = $wpdb->get_row( $wpdb->prepare(
			"SELECT * FROM {$wpdb->prefix}popucoge_subscribers WHERE id = %d",
			$subscriber_id
		) );
		
		if ( ! $subscriber ) {
			return;
		}
		
		// Generate unique coupon
		$coupon_code = $this->generate_coupon_code( $automation, $subscriber );
		
		if ( ! $coupon_code ) {
			return;
		}
		
		// Create WooCommerce coupon
		$wc_coupon_id = $this->create_wc_coupon( $coupon_code, $automation, $subscriber );
		
		if ( ! $wc_coupon_id ) {
			return;
		}
		
		// Store coupon link
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->insert(
			$wpdb->prefix . 'popucoge_coupons',
			array(
				'subscriber_id' => $subscriber_id,
				'coupon_code' => $coupon_code,
				'automation_id' => $automation['id'],
				'created_date' => current_time( 'mysql' ),
				'status' => 'active',
			),
			array( '%d', '%s', '%s', '%s', '%s' )
		);
		
		// Send email
		$this->send_automation_email( $subscriber, $automation, $coupon_code );
	}
	
	/**
	 * Generate unique coupon code
	 */
	private function generate_coupon_code( $automation, $subscriber ) {
		$prefix = isset( $automation['coupon_prefix'] ) ? $automation['coupon_prefix'] : 'PCG';
		$email = sanitize_user( str_replace( '@', '', $subscriber->email ), true );
		$email = substr( $email, 0, 8 );
		$random = substr( md5( $subscriber->email . time() ), 0, 6 );
		$code = strtoupper( $prefix . '-' . $email . '-' . $random );
		
		// Check if code already exists
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$exists = $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*) FROM {$wpdb->prefix}popucoge_coupons WHERE coupon_code = %s",
			$code
		) );
		
		if ( $exists > 0 ) {
			$code = strtoupper( $prefix . '-' . $email . '-' . substr( md5( $subscriber->email . time() . wp_rand() ), 0, 6 ) );
		}
		
		return $code;
	}
	
	/**
	 * Create WooCommerce coupon
	 */
	private function create_wc_coupon( $coupon_code, $automation, $subscriber ) {
		if ( ! class_exists( 'WC_Coupon' ) ) {
			return false;
		}
		
		$coupon = new WC_Coupon();
		$coupon->set_code( $coupon_code );
		$coupon->set_discount_type( isset( $automation['coupon_type'] ) ? $automation['coupon_type'] : 'percent' );
		$coupon->set_amount( isset( $automation['coupon_amount'] ) ? floatval( $automation['coupon_amount'] ) : 10 );
		$coupon->set_individual_use( true );
		$coupon->set_usage_limit( 1 );
		$coupon->set_usage_limit_per_user( 1 );
		$coupon->set_email_restrictions( array( $subscriber->email ) );
		
		if ( isset( $automation['coupon_expiry'] ) && ! empty( $automation['coupon_expiry'] ) && intval( $automation['coupon_expiry'] ) > 0 ) {
			$coupon->set_date_expires( strtotime( '+' . intval( $automation['coupon_expiry'] ) . ' days' ) );
		}
		
		if ( isset( $automation['free_shipping'] ) && $automation['free_shipping'] === 'yes' ) {
			$coupon->set_free_shipping( true );
		}
		
		$coupon->save();
		
		return $coupon->get_id();
	}
	
	/**
	 * Send automation email
	 */
	private function send_automation_email( $subscriber, $automation, $coupon_code ) {
		global $wpdb;
		
		$settings = get_option( 'popucoge_settings', array() );
		$from_name = isset( $settings['email_from_name'] ) ? $settings['email_from_name'] : get_bloginfo( 'name' );
		$from_email = isset( $settings['email_from_email'] ) ? $settings['email_from_email'] : get_option( 'admin_email' );
		
		$subject = isset( $automation['email_subject'] ) ? $automation['email_subject'] : __( 'Welcome! Here is your discount code', 'themeinthebox-popup-coupons-for-woocommerce' );
		$message = isset( $automation['email_content'] ) ? $automation['email_content'] : '';
		
		// Replace placeholders
		$message = str_replace( '{coupon_code}', $coupon_code, $message );
		$message = str_replace( '{customer_name}', $subscriber->name ? $subscriber->name : '', $message );
		$message = str_replace( '{site_name}', get_bloginfo( 'name' ), $message );
		$message = str_replace( '{shop_url}', wc_get_page_permalink( 'shop' ), $message );
		
		// Add coupon code to message if not present
		if ( strpos( $message, $coupon_code ) === false ) {
			$message .= "\n\n" . __( 'Your discount code:', 'themeinthebox-popup-coupons-for-woocommerce' ) . ' ' . $coupon_code;
		}
		
		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
			'From: ' . $from_name . ' <' . $from_email . '>',
		);
		
		$email_content = $this->get_email_template( $message, $coupon_code );
		
		$sent = wp_mail( $subscriber->email, $subject, $email_content, $headers );
		
		if ( $sent ) {
			// Track email sent
			$automation_id = $automation['id'];
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->insert(
				$wpdb->prefix . 'popucoge_email_stats',
				array(
					'subscriber_id' => $subscriber->id,
					'automation_id' => $automation_id,
					'email_sent_date' => current_time( 'mysql' ),
				),
				array( '%d', '%s', '%s' )
			);
		}
		
		return $sent;
	}
	
	/**
	 * Get email template
	 */
	private function get_email_template( $content, $coupon_code ) {
		ob_start();
		?>
		<!DOCTYPE html>
		<html>
		<head>
			<meta charset="UTF-8">
			<meta name="viewport" content="width=device-width, initial-scale=1.0">
		</head>
		<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f4f4f4;">
			<table width="100%" cellpadding="0" cellspacing="0" style="background-color: #f4f4f4; padding: 20px;">
				<tr>
					<td align="center">
						<table width="600" cellpadding="0" cellspacing="0" style="background-color: #ffffff; border-radius: 8px; overflow: hidden;">
							<tr>
								<td style="padding: 40px; text-align: center; background-color: #ffffff;">
									<?php echo wp_kses_post( wpautop( $content ) ); ?>
									<?php if ( ! empty( $coupon_code ) ) : ?>
										<div style="margin: 30px 0; padding: 20px; background-color: #f0f0f0; border-radius: 5px; text-align: center;">
											<h2 style="margin: 0 0 10px; color: #333; font-size: 24px;"><?php echo esc_html( $coupon_code ); ?></h2>
											<p style="margin: 0; color: #666;"><?php esc_html_e( 'Use this code at checkout', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
										</div>
									<?php endif; ?>
									<p style="margin-top: 30px;">
										<a href="<?php echo esc_url( wc_get_page_permalink( 'shop' ) ); ?>" style="display: inline-block; padding: 12px 30px; background-color: #0073aa; color: #ffffff; text-decoration: none; border-radius: 5px;"><?php esc_html_e( 'Go to shop', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></a>
									</p>
								</td>
							</tr>
						</table>
					</td>
				</tr>
			</table>
		</body>
		</html>
		<?php
		return ob_get_clean();
	}
	
	/**
	 * Render admin page
	 */
	public function render_admin_page() {
		$this->handle_form_submit();
		
		$automations = $this->get_automations();
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading GET parameters for display purposes only
		$editing_automation_id = isset( $_GET['edit'] ) ? sanitize_text_field( wp_unslash( $_GET['edit'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reading GET parameters for display purposes only
		$action_new = isset( $_GET['action'] ) && sanitize_text_field( wp_unslash( $_GET['action'] ) ) === 'new';
		$editing_automation = ( $editing_automation_id && ! $action_new ) ? $this->get_automation( $editing_automation_id ) : null;
		
		// If action=new, show form with empty automation
		if ( $action_new ) {
			$editing_automation = array();
		}
		
		?>
		<div class="wrap popucoge-automation-page">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<?php if ( ! $editing_automation && ! $action_new ) : ?>
				<div class="popucoge-header-actions">
					<a href="<?php echo esc_url( add_query_arg( 'action', 'new', admin_url( 'admin.php?page=popucoge-automations' ) ) ); ?>" class="button button-primary">
						<?php esc_html_e( '+ New Automation', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
					</a>
				</div>
				
				<table class="wp-list-table widefat fixed striped">
					<thead>
						<tr>
							<th><?php esc_html_e( 'Name', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
							<th><?php esc_html_e( 'Coupon Type', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
							<th><?php esc_html_e( 'Discount', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
							<th><?php esc_html_e( 'Status', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
							<th><?php esc_html_e( 'Actions', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php if ( empty( $automations ) ) : ?>
							<tr>
								<td colspan="5"><?php esc_html_e( 'No automations found. Create your first automation!', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></td>
							</tr>
						<?php else : ?>
							<?php foreach ( $automations as $auto_id => $automation ) : ?>
								<tr>
									<td><strong><?php echo esc_html( isset( $automation['name'] ) ? $automation['name'] : __( 'Untitled', 'themeinthebox-popup-coupons-for-woocommerce' ) ); ?></strong></td>
									<td><?php echo esc_html( isset( $automation['coupon_type'] ) && $automation['coupon_type'] === 'fixed_cart' ? __( 'Fixed Amount', 'themeinthebox-popup-coupons-for-woocommerce' ) : __( 'Percentage', 'themeinthebox-popup-coupons-for-woocommerce' ) ); ?></td>
									<td><?php echo esc_html( isset( $automation['coupon_amount'] ) ? $automation['coupon_amount'] : '0' ); ?><?php echo ( isset( $automation['coupon_type'] ) && $automation['coupon_type'] === 'fixed_cart' ) ? ' €' : '%'; ?></td>
									<td>
										<span class="popucoge-status-badge status-<?php echo esc_attr( isset( $automation['status'] ) && $automation['status'] === 'active' ? 'active' : 'inactive' ); ?>">
											<?php echo esc_html( isset( $automation['status'] ) && $automation['status'] === 'active' ? __( 'Active', 'themeinthebox-popup-coupons-for-woocommerce' ) : __( 'Inactive', 'themeinthebox-popup-coupons-for-woocommerce' ) ); ?>
										</span>
									</td>
									<td>
										<a href="<?php echo esc_url( add_query_arg( array( 'page' => 'popucoge-automations', 'edit' => $auto_id ), admin_url( 'admin.php' ) ) ); ?>" class="button button-small">
											<?php esc_html_e( 'Edit', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
										</a>
										<a href="#" class="button button-small popucoge-toggle-automation" data-automation-id="<?php echo esc_attr( $auto_id ); ?>" data-status="<?php echo esc_attr( isset( $automation['status'] ) && $automation['status'] === 'active' ? 'active' : 'inactive' ); ?>">
											<?php echo esc_html( isset( $automation['status'] ) && $automation['status'] === 'active' ? __( 'Deactivate', 'themeinthebox-popup-coupons-for-woocommerce' ) : __( 'Active', 'themeinthebox-popup-coupons-for-woocommerce' ) ); ?>
										</a>
										<a href="#" class="button button-small popucoge-delete-automation" data-automation-id="<?php echo esc_attr( $auto_id ); ?>">
											<?php esc_html_e( 'Delete', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
										</a>
									</td>
								</tr>
							<?php endforeach; ?>
						<?php endif; ?>
					</tbody>
				</table>
			<?php else : ?>
				<?php $this->render_automation_form( $editing_automation ); ?>
			<?php endif; ?>
		</div>
		<?php
	}
	
	/**
	 * Render automation form
	 */
	private function render_automation_form( $automation = null ) {
		$automation = $automation ? $automation : array();
		?>
		<form method="post" action="" id="popucoge-automation-form">
			<?php wp_nonce_field( 'popucoge-save-automation', 'popucoge_automation_nonce' ); ?>
			
			<input type="hidden" name="automation_id" value="<?php echo esc_attr( isset( $automation['id'] ) ? $automation['id'] : '' ); ?>" />
			
			<table class="form-table">
				<tbody>
					<tr>
						<th scope="row">
							<label for="automation_name"><?php esc_html_e( 'Automation Name', 'themeinthebox-popup-coupons-for-woocommerce' ); ?> <span class="required">*</span></label>
						</th>
						<td>
							<input type="text" id="automation_name" name="automation_name" value="<?php echo esc_attr( isset( $automation['name'] ) ? $automation['name'] : '' ); ?>" class="regular-text" required />
							<p class="description"><?php esc_html_e( 'E.g. Welcome email, 10% discount coupon', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="email_subject"><?php esc_html_e( 'Email Subject', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<input type="text" id="email_subject" name="email_subject" value="<?php echo esc_attr( isset( $automation['email_subject'] ) ? $automation['email_subject'] : __( 'Welcome! Here is your discount code', 'themeinthebox-popup-coupons-for-woocommerce' ) ); ?>" class="regular-text" />
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="email_content"><?php esc_html_e( 'Email Content', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<?php
							$content = isset( $automation['email_content'] ) ? $automation['email_content'] : __( 'Thank you for subscribing! Here is your special discount code: {coupon_code}', 'themeinthebox-popup-coupons-for-woocommerce' );
							wp_editor( $content, 'email_content', array(
								'textarea_name' => 'email_content',
								'textarea_rows' => 10,
								'media_buttons' => false,
								'tinymce' => true,
							) );
							?>
							<p class="description">
								<?php esc_html_e( 'Available placeholders:', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
								<code>{coupon_code}</code>, <code>{customer_name}</code>, <code>{site_name}</code>, <code>{shop_url}</code>
							</p>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="coupon_type"><?php esc_html_e( 'Discount Type', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<select id="coupon_type" name="coupon_type" class="regular-text">
								<option value="percent" <?php selected( isset( $automation['coupon_type'] ) ? $automation['coupon_type'] : 'percent', 'percent' ); ?>>
									<?php esc_html_e( 'Percentage', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
								</option>
								<option value="fixed_cart" <?php selected( isset( $automation['coupon_type'] ) ? $automation['coupon_type'] : '', 'fixed_cart' ); ?>>
									<?php esc_html_e( 'Fixed Amount', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
								</option>
							</select>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="coupon_amount"><?php esc_html_e( 'Discount Value', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<input type="number" id="coupon_amount" name="coupon_amount" value="<?php echo esc_attr( isset( $automation['coupon_amount'] ) ? $automation['coupon_amount'] : '10' ); ?>" step="0.01" min="0" class="small-text" />
							<span id="coupon_amount_type">%</span>
							<p class="description"><?php esc_html_e( 'Discount value to apply', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="coupon_prefix"><?php esc_html_e( 'Coupon Code Prefix', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<input type="text" id="coupon_prefix" name="coupon_prefix" value="<?php echo esc_attr( isset( $automation['coupon_prefix'] ) ? $automation['coupon_prefix'] : 'PCG' ); ?>" class="small-text" maxlength="10" />
							<p class="description"><?php esc_html_e( 'Prefix for generated coupon codes (max 10 characters)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="coupon_expiry"><?php esc_html_e( 'Coupon Expiry (days)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<input type="number" id="coupon_expiry" name="coupon_expiry" value="<?php echo esc_attr( isset( $automation['coupon_expiry'] ) ? $automation['coupon_expiry'] : '30' ); ?>" min="0" class="small-text" />
							<p class="description"><?php esc_html_e( 'Number of days before expiry (0 = no expiry)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="free_shipping"><?php esc_html_e( 'Free Shipping', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<label>
								<input type="checkbox" id="free_shipping" name="free_shipping" value="yes" <?php checked( isset( $automation['free_shipping'] ) && $automation['free_shipping'] === 'yes' ); ?> />
								<?php esc_html_e( 'Enable free shipping with this coupon', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
							</label>
						</td>
					</tr>
					
					<tr>
						<th scope="row">
							<label for="automation_status"><?php esc_html_e( 'Status', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
						</th>
						<td>
							<label>
								<input type="checkbox" id="automation_status" name="automation_status" value="active" <?php checked( ! isset( $automation['status'] ) || $automation['status'] === 'active' ); ?> />
								<?php esc_html_e( 'Active', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
							</label>
						</td>
					</tr>
				</tbody>
			</table>
			
			<p class="submit">
				<?php submit_button( __( 'Save Automation', 'themeinthebox-popup-coupons-for-woocommerce' ), 'primary', 'save_automation', false ); ?>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=popucoge-automations' ) ); ?>" class="button">
					<?php esc_html_e( 'Cancel', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
				</a>
			</p>
		</form>
		<?php
	}
	
	/**
	 * Handle form submit
	 */
	private function handle_form_submit() {
		if ( ! isset( $_POST['popucoge_automation_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['popucoge_automation_nonce'] ) ), 'popucoge-save-automation' ) ) {
			return;
		}
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}
		
		if ( isset( $_POST['save_automation'] ) ) {
			$automation_data = array(
				'id' => isset( $_POST['automation_id'] ) ? sanitize_text_field( wp_unslash( $_POST['automation_id'] ) ) : '',
				'name' => isset( $_POST['automation_name'] ) ? sanitize_text_field( wp_unslash( $_POST['automation_name'] ) ) : '',
				'email_subject' => isset( $_POST['email_subject'] ) ? sanitize_text_field( wp_unslash( $_POST['email_subject'] ) ) : '',
				'email_content' => isset( $_POST['email_content'] ) ? wp_kses_post( wp_unslash( $_POST['email_content'] ) ) : '',
				'coupon_type' => isset( $_POST['coupon_type'] ) ? sanitize_text_field( wp_unslash( $_POST['coupon_type'] ) ) : 'percent',
				'coupon_amount' => isset( $_POST['coupon_amount'] ) ? floatval( wp_unslash( $_POST['coupon_amount'] ) ) : 10,
				'coupon_prefix' => isset( $_POST['coupon_prefix'] ) ? sanitize_text_field( wp_unslash( $_POST['coupon_prefix'] ) ) : 'PCG',
				'coupon_expiry' => isset( $_POST['coupon_expiry'] ) ? intval( wp_unslash( $_POST['coupon_expiry'] ) ) : 30,
				'free_shipping' => isset( $_POST['free_shipping'] ) ? 'yes' : 'no',
				'status' => isset( $_POST['automation_status'] ) ? 'active' : 'inactive',
			);
			
			$this->save_automation( $automation_data );
			
			wp_safe_redirect( admin_url( 'admin.php?page=popucoge-automations' ) );
			exit;
		}
	}
	
	/**
	 * AJAX: Delete automation
	 */
	public function ajax_delete_automation() {
		check_ajax_referer( 'popucoge-admin-nonce', 'nonce' );
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		$automation_id = isset( $_POST['automation_id'] ) ? sanitize_text_field( wp_unslash( $_POST['automation_id'] ) ) : '';
		
		if ( empty( $automation_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Missing automation ID', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		if ( $this->delete_automation( $automation_id ) ) {
			wp_send_json_success( array( 'message' => __( 'Automation deleted successfully', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		wp_send_json_error( array( 'message' => __( 'Error during deletion', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
	}
	
	/**
	 * AJAX: Toggle automation status
	 */
	public function ajax_toggle_automation() {
		check_ajax_referer( 'popucoge-admin-nonce', 'nonce' );
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		$automation_id = isset( $_POST['automation_id'] ) ? sanitize_text_field( wp_unslash( $_POST['automation_id'] ) ) : '';
		$status = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : 'inactive';
		
		if ( empty( $automation_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Missing automation ID', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		$automation = $this->get_automation( $automation_id );
		if ( ! $automation ) {
			wp_send_json_error( array( 'message' => __( 'Automation not found', 'themeinthebox-popup-coupons-for-woocommerce' ) ) );
		}
		
		$automation['status'] = $status === 'active' ? 'inactive' : 'active';
		$this->save_automation( $automation );
		
		wp_send_json_success( array( 
			'message' => __( 'Stato aggiornato', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'status' => $automation['status']
		) );
	}
}

