<?php
/**
 * Admin class for Popup Coupon Generator for WooCommerce
 *
 * @package Popup_Coupon_Generator
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin management class
 */
class Popucoge_Admin {
	
	/**
	 * Instance of this class
	 */
	private static $instance = null;
	
	/**
	 * Get instance of this class
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
	}
	
	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		// Main menu
		add_menu_page(
			__( 'Popup Coupon Generator', 'themeinthebox-popup-coupons-for-woocommerce' ),
			__( 'Popup Coupon', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'manage_woocommerce',
			'themeinthebox-popup-coupons-for-woocommerce',
			array( $this, 'render_popup_page' ),
			'dashicons-email-alt',
			56
		);
		
		// Submenu: Popup
		add_submenu_page(
			'themeinthebox-popup-coupons-for-woocommerce',
			__( 'Popup', 'themeinthebox-popup-coupons-for-woocommerce' ),
			__( 'Popup', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'manage_woocommerce',
			'themeinthebox-popup-coupons-for-woocommerce',
			array( $this, 'render_popup_page' )
		);
		
		// Submenu: Automations
		add_submenu_page(
			'themeinthebox-popup-coupons-for-woocommerce',
			__( 'Automations', 'themeinthebox-popup-coupons-for-woocommerce' ),
			__( 'Automations', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'manage_woocommerce',
			'popucoge-automations',
			array( $this, 'render_automations_page' )
		);
		
		// Submenu: Statistics
		add_submenu_page(
			'themeinthebox-popup-coupons-for-woocommerce',
			__( 'Statistics', 'themeinthebox-popup-coupons-for-woocommerce' ),
			__( 'Statistics', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'manage_woocommerce',
			'popucoge-statistics',
			array( $this, 'render_statistics_page' )
		);
		
		// Submenu: Settings
		add_submenu_page(
			'themeinthebox-popup-coupons-for-woocommerce',
			__( 'Settings', 'themeinthebox-popup-coupons-for-woocommerce' ),
			__( 'Settings', 'themeinthebox-popup-coupons-for-woocommerce' ),
			'manage_woocommerce',
			'popucoge-settings',
			array( $this, 'render_settings_page' )
		);
	}
	
	/**
	 * Enqueue admin scripts and styles
	 */
	public function enqueue_admin_scripts( $hook ) {
		if ( strpos( $hook, 'themeinthebox-popup-coupons-for-woocommerce' ) === false && strpos( $hook, 'popucoge-' ) === false ) {
			return;
		}
		
		wp_enqueue_style(
			'popucoge-admin',
			POPUCOGE_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			POPUCOGE_VERSION
		);
		
		wp_enqueue_script(
			'popucoge-admin',
			POPUCOGE_PLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			POPUCOGE_VERSION,
			true
		);
		
		wp_localize_script( 'popucoge-admin', 'popucogeAdmin', array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce' => wp_create_nonce( 'popucoge-admin-nonce' ),
		) );
		
		// Add inline scripts based on current page
		$this->add_inline_scripts( $hook );
	}
	
	/**
	 * Add inline scripts for specific admin pages
	 */
	private function add_inline_scripts( $hook ) {
		// Settings page: Analytics toggle script
		if ( strpos( $hook, 'popucoge-settings' ) !== false ) {
			$inline_script = 'jQuery(document).ready(function($) {
				$("#enable_analytics").on("change", function() {
					if ($(this).is(":checked")) {
						$(".analytics-setting").show();
					} else {
						$(".analytics-setting").hide();
					}
				});
			});';
			wp_add_inline_script( 'popucoge-admin', $inline_script );
		}
		
		// Popup page: Display conditions toggle script (only when editing)
		if ( strpos( $hook, 'themeinthebox-popup-coupons-for-woocommerce' ) !== false ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking GET parameter for conditional script loading
			$is_editing = isset( $_GET['edit'] ) || ( isset( $_GET['action'] ) && sanitize_text_field( wp_unslash( $_GET['action'] ) ) === 'new' );
			if ( $is_editing ) {
				$inline_script = 'jQuery(document).ready(function($) {
					function toggleConditions() {
						var displayType = $("#display_type").val();
						$(".display-condition").hide();
						$("#condition-" + displayType).show();
					}
					$("#display_type").on("change", toggleConditions);
					toggleConditions();
				});';
				wp_add_inline_script( 'popucoge-admin', $inline_script );
			}
		}
		
		// Automations page: Coupon type toggle script (only when editing)
		if ( strpos( $hook, 'popucoge-automations' ) !== false ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking GET parameter for conditional script loading
			$is_editing = isset( $_GET['edit'] ) || ( isset( $_GET['action'] ) && sanitize_text_field( wp_unslash( $_GET['action'] ) ) === 'new' );
			if ( $is_editing ) {
				$inline_script = 'jQuery(document).ready(function($) {
					$("#coupon_type").on("change", function() {
						if ($(this).val() === "fixed_cart") {
							$("#coupon_amount_type").text("€");
						} else {
							$("#coupon_amount_type").text("%");
						}
					}).trigger("change");
				});';
				wp_add_inline_script( 'popucoge-admin', $inline_script );
			}
		}
	}
	
	/**
	 * Render Popup page
	 */
	public function render_popup_page() {
		Popucoge_Popup::get_instance()->render_admin_page();
	}
	
	/**
	 * Render Automations page
	 */
	public function render_automations_page() {
		Popucoge_Automation::get_instance()->render_admin_page();
	}
	
	/**
	 * Render Statistics page
	 */
	public function render_statistics_page() {
		Popucoge_Statistics::get_instance()->render_admin_page();
	}
	
	/**
	 * Render Settings page
	 */
	public function render_settings_page() {
		$this->handle_settings_save();
		
		$settings = get_option( 'popucoge_settings', array() );
		?>
		<div class="wrap popucoge-settings">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<form method="post" action="">
				<?php wp_nonce_field( 'popucoge-settings', 'popucoge_settings_nonce' ); ?>
				
				<table class="form-table">
					<tbody>
						<tr>
							<th scope="row">
								<label for="email_from_name"><?php esc_html_e( 'Email From Name', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
							</th>
							<td>
								<input type="text" id="email_from_name" name="email_from_name" value="<?php echo esc_attr( isset( $settings['email_from_name'] ) ? $settings['email_from_name'] : get_bloginfo( 'name' ) ); ?>" class="regular-text" />
								<p class="description"><?php esc_html_e( 'Name displayed as sender of automated emails', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
							</td>
						</tr>
						
						<tr>
							<th scope="row">
								<label for="email_from_email"><?php esc_html_e( 'Email From Address', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
							</th>
							<td>
								<input type="email" id="email_from_email" name="email_from_email" value="<?php echo esc_attr( isset( $settings['email_from_email'] ) ? $settings['email_from_email'] : get_option( 'admin_email' ) ); ?>" class="regular-text" />
								<p class="description"><?php esc_html_e( 'Email address used to send automated emails', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
							</td>
						</tr>
						
						<tr>
							<th scope="row">
								<label for="enable_analytics"><?php esc_html_e( 'Enable Google Analytics', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
							</th>
							<td>
								<label>
									<input type="checkbox" id="enable_analytics" name="enable_analytics" value="yes" <?php checked( isset( $settings['enable_analytics'] ) && $settings['enable_analytics'] === 'yes' ); ?> />
									<?php esc_html_e( 'Enable event tracking with Google Analytics', 'themeinthebox-popup-coupons-for-woocommerce' ); ?>
								</label>
							</td>
						</tr>
						
						<tr class="analytics-setting" style="<?php echo ( isset( $settings['enable_analytics'] ) && $settings['enable_analytics'] === 'yes' ) ? '' : 'display:none;'; ?>">
							<th scope="row">
								<label for="analytics_id"><?php esc_html_e( 'Google Analytics ID', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></label>
							</th>
							<td>
								<input type="text" id="analytics_id" name="analytics_id" value="<?php echo esc_attr( isset( $settings['analytics_id'] ) ? $settings['analytics_id'] : '' ); ?>" class="regular-text" placeholder="G-XXXXXXXXXX or UA-XXXXXXXXX-X" />
								<p class="description"><?php esc_html_e( 'Enter your Google Analytics ID (GA4 or Universal Analytics)', 'themeinthebox-popup-coupons-for-woocommerce' ); ?></p>
							</td>
						</tr>
					</tbody>
				</table>
				
				<?php submit_button(); ?>
			</form>
		</div>
		<?php
	}
	
	/**
	 * Handle settings save
	 */
	private function handle_settings_save() {
		if ( ! isset( $_POST['popucoge_settings_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['popucoge_settings_nonce'] ) ), 'popucoge-settings' ) ) {
			return;
		}
		
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}
		
		$settings = array(
			'email_from_name' => isset( $_POST['email_from_name'] ) ? sanitize_text_field( wp_unslash( $_POST['email_from_name'] ) ) : get_bloginfo( 'name' ),
			'email_from_email' => isset( $_POST['email_from_email'] ) ? sanitize_email( wp_unslash( $_POST['email_from_email'] ) ) : get_option( 'admin_email' ),
			'enable_analytics' => isset( $_POST['enable_analytics'] ) ? 'yes' : 'no',
			'analytics_id' => isset( $_POST['analytics_id'] ) ? sanitize_text_field( wp_unslash( $_POST['analytics_id'] ) ) : '',
		);
		
		update_option( 'popucoge_settings', $settings );
		
		echo '<div class="notice notice-success is-dismissible"><p>' . esc_html( __( 'Settings saved successfully.', 'themeinthebox-popup-coupons-for-woocommerce' ) ) . '</p></div>';
	}
}
