/**
 * Frontend JavaScript for Popup Coupon Generator for WooCommerce
 */

(function($) {
	'use strict';
	
	var Popucoge = {
		popups: {},
		shownPopups: {},
		init: function() {
			this.popups = popucogeFrontend.popups || {};
			this.bindEvents();
			this.initPopups();
		},
		
		initPopups: function() {
			var self = this;
			
			$.each(this.popups, function(popupId, popupData) {
				self.setupPopup(popupId, popupData);
			});
		},
		
		setupPopup: function(popupId, popupData) {
			var $popup = $('#popucoge-popup-' + popupId);
			var displayType = popupData.display_type || 'exit_intent';
			
			if ($popup.length === 0) {
				return;
			}
			
			// Check if already shown in this session
			if (this.shownPopups[popupId]) {
				return;
			}
			
			switch(displayType) {
				case 'exit_intent':
					this.setupExitIntent($popup, popupId);
					break;
				case 'time_delay':
					this.setupTimeDelay($popup, popupId, popupData.time_delay || 5);
					break;
				case 'scroll':
					this.setupScroll($popup, popupId, popupData.scroll_percentage || 50);
					break;
				case 'page_load':
					this.setupPageLoad($popup, popupId);
					break;
			}
		},
		
		setupExitIntent: function($popup, popupId) {
			var self = this;
			var exitIntentTriggered = false;
			
			$(document).on('mouseleave', function(e) {
				if (exitIntentTriggered || self.shownPopups[popupId]) {
					return;
				}
				
				// Check if mouse is leaving the top of the page
				if (e.clientY < 0) {
					exitIntentTriggered = true;
					self.showPopup($popup, popupId);
					
					// Track Google Analytics event
					if (popucogeFrontend.analytics_enabled) {
						self.trackAnalyticsEvent('exit_intent', 'popup_shown', popupId);
					}
				}
			});
		},
		
		setupTimeDelay: function($popup, popupId, delay) {
			var self = this;
			
			setTimeout(function() {
				if (!self.shownPopups[popupId]) {
					self.showPopup($popup, popupId);
					
					// Track Google Analytics event
					if (popucogeFrontend.analytics_enabled) {
						self.trackAnalyticsEvent('time_delay', 'popup_shown', popupId);
					}
				}
			}, delay * 1000);
		},
		
		setupScroll: function($popup, popupId, percentage) {
			var self = this;
			var scrollTriggered = false;
			
			$(window).on('scroll', function() {
				if (scrollTriggered || self.shownPopups[popupId]) {
					return;
				}
				
				var scrollTop = $(window).scrollTop();
				var documentHeight = $(document).height();
				var windowHeight = $(window).height();
				var scrollPercent = (scrollTop / (documentHeight - windowHeight)) * 100;
				
				if (scrollPercent >= percentage) {
					scrollTriggered = true;
					self.showPopup($popup, popupId);
					
					// Track Google Analytics event
					if (popucogeFrontend.analytics_enabled) {
						self.trackAnalyticsEvent('scroll', 'popup_shown', popupId);
					}
				}
			});
		},
		
		setupPageLoad: function($popup, popupId) {
			var self = this;
			
			$(document).ready(function() {
				setTimeout(function() {
					self.showPopup($popup, popupId);
					
					// Track Google Analytics event
					if (popucogeFrontend.analytics_enabled) {
						self.trackAnalyticsEvent('page_load', 'popup_shown', popupId);
					}
				}, 500);
			});
		},
		
		showPopup: function($popup, popupId) {
			if (this.shownPopups[popupId]) {
				return;
			}
			
			this.shownPopups[popupId] = true;
			$popup.fadeIn(300);
			$('body').addClass('popucoge-popup-open');
		},
		
		hidePopup: function($popup, popupId) {
			$popup.fadeOut(300);
			$('body').removeClass('popucoge-popup-open');
		},
		
		bindEvents: function() {
			var self = this;
			
			// Close button
			$(document).on('click', '.popucoge-popup-close', function(e) {
				e.preventDefault();
				var $popup = $(this).closest('.popucoge-popup');
				var popupId = $popup.data('popup-id');
				self.hidePopup($popup, popupId);
			});
			
			// Close on overlay click
			$(document).on('click', '.popucoge-popup-overlay', function(e) {
				var $popup = $(this).closest('.popucoge-popup');
				var popupId = $popup.data('popup-id');
				self.hidePopup($popup, popupId);
			});
			
			// Prevent closing on popup content click
			$(document).on('click', '.popucoge-popup-container', function(e) {
				e.stopPropagation();
			});
			
			// Subscribe form
			$(document).on('submit', '.popucoge-subscribe-form', function(e) {
				e.preventDefault();
				self.handleSubscribe($(this));
			});
		},
		
		handleSubscribe: function($form) {
			var self = this;
			var $button = $form.find('.popucoge-subscribe-button');
			var $message = $form.find('.popucoge-form-message');
			var originalText = $button.text();
			
			$button.prop('disabled', true).text('Invio in corso...');
			$message.removeClass('success error').hide();
			
			$.ajax({
				url: popucogeFrontend.ajax_url,
				type: 'POST',
				data: {
					action: 'popucoge_subscribe',
					nonce: popucogeFrontend.nonce,
					email: $form.find('input[name="email"]').val(),
					name: $form.find('input[name="name"]').val(),
					popup_id: $form.find('input[name="popup_id"]').val(),
				},
				success: function(response) {
					if (response.success) {
						$message.addClass('success').text(response.data.message).show();
						$form[0].reset();
						
						// Track Google Analytics event
						if (popucogeFrontend.analytics_enabled) {
							var popupId = $form.find('input[name="popup_id"]').val();
							self.trackAnalyticsEvent('subscription', 'subscribed', popupId);
						}
						
						// Hide popup after 2 seconds
						setTimeout(function() {
							var $popup = $form.closest('.popucoge-popup');
							var popupId = $popup.data('popup-id');
							self.hidePopup($popup, popupId);
						}, 2000);
					} else {
						$message.addClass('error').text(response.data.message || 'Errore durante l\'iscrizione').show();
					}
					
					$button.prop('disabled', false).text(originalText);
				},
				error: function() {
					$message.addClass('error').text('Errore di connessione. Riprova più tardi.').show();
					$button.prop('disabled', false).text(originalText);
				}
			});
		},
		
		trackAnalyticsEvent: function(category, action, label) {
			if (!popucogeFrontend.analytics_enabled || !popucogeFrontend.analytics_id) {
				return;
			}
			
			var analyticsId = popucogeFrontend.analytics_id;
			
			// Check if gtag is available (GA4)
			if (typeof gtag !== 'undefined') {
				gtag('event', action, {
					'event_category': category,
					'event_label': label,
					'send_to': analyticsId
				});
			}
			// Check if ga is available (Universal Analytics)
			else if (typeof ga !== 'undefined') {
				ga('send', 'event', category, action, label);
			}
			// Fallback: create analytics pixel
			else {
				// Analytics might load later, so we'll just log for now
				console.log('Analytics event:', category, action, label);
			}
		}
	};
	
	$(document).ready(function() {
		Popucoge.init();
	});
	
})(jQuery);

