(function() {
    // Supabase client is loaded from CDN and available as window.supabase
    const { createClient } = supabase;

    class CoreSDK {
        constructor(config = {}) {
            this.supabaseUrl = config.supabaseUrl || '';
            this.supabaseKey = config.supabaseKey || '';
            this.supabase = null;
            this._initializeSupabaseClient();

            this.entryCache = new Map();
            this.cache = { characters: null, arguments: null, proverbs: null, chapters: null };
            this.selectedItems = { characters: new Map(), arguments: new Map(), proverbs: new Map(), chapter: { id: 'all', name: '全部章节' } };
            this.pagination = { pageSize: 10, currentPage: 0, totalLoaded: 0, isLoading: false, hasMore: true, totalCount: 0, loadedIds: new Set() };
            this.currentFilters = {};
            this.isSearchInitialized = false;
            this.scrollListener = null;
            this.isAutoLoadingEnabled = false;
            this.searchConditionsChanged = false;
            this.searchEventsInitialized = false;
            this.latestSearchId = 0;
            this._handleGlobalClick = this._handleGlobalClick.bind(this);
        }

        _initializeSupabaseClient() {
            if (this.supabaseUrl && this.supabaseKey) {
                this.supabase = createClient(this.supabaseUrl, this.supabaseKey);
            }
        }

        init() {
            this._initializeGlobalEventListeners();
        }

        _initializeGlobalEventListeners() {
            document.body.removeEventListener('click', this._handleGlobalClick);
            document.body.addEventListener('click', this._handleGlobalClick);
        }

        _handleGlobalClick(event) {
            const moreOptionsButton = event.target.closest('.more-options-btn');
            if (moreOptionsButton) {
                event.stopPropagation();
                const dropdown = moreOptionsButton.nextElementSibling;
                if (!dropdown) return;
                const wasActive = dropdown.classList.contains('active');
                document.querySelectorAll('.card-actions-dropdown.active').forEach(d => d.classList.remove('active'));
                if (!wasActive) {
                    dropdown.classList.add('active');
                }
                return;
            }
            const activeCardDropdown = document.querySelector('.card-actions-dropdown.active');
            if (activeCardDropdown && !event.target.closest('.card-actions-container')) {
                activeCardDropdown.classList.remove('active');
            }
        }

        validateConfig() {
            if (!this.supabaseUrl || this.supabaseUrl.includes('your-project') || !this.supabase) {
                throw new Error('Please configure valid Supabase URL and Key in WordPress settings.');
            }
        }

        async apiRequest(endpoint, params = {}) {
            this.validateConfig();
            const { count, select, ...restParams } = params;
            let query = this.supabase.from(endpoint).select(select || '*', { count: count ? 'exact' : null });

            for (const [key, value] of Object.entries(restParams)) {
                if (key === 'order') {
                    const [col, dir] = value.split('.');
                    query = query.order(col, { ascending: dir === 'asc' });
                } else if (key === 'limit') {
                    query = query.limit(parseInt(value, 10));
                } else if (key === 'offset') {
                    const start = parseInt(value, 10);
                    const end = start + (parseInt(restParams.limit || 10, 10)) - 1;
                    query = query.range(start, end);
                } else {
                    const [op, ...valParts] = value.split('.');
                    const val = valParts.join('.');
                    query = query.filter(key, op, val);
                }
            }

            const { data, error, count: resultCount } = await query;
            if (error) {
                console.error('API Error:', error);
                throw new Error(`API Request Failed: ${error.message}`);
            }
            if (count) {
              return { data, count: resultCount || 0 };
            }
            return data;
        }

        async fetchAnalects(filters = {}, page = 0, pageSize = 10) {
            try {
                const params = this.buildQueryParams(filters, page, pageSize);
                
                let result = await this.apiRequest('analects_entries_expanded', params);
                let data = Array.isArray(result) ? result : result.data;


                const uniqueData = this.removeDuplicates(data);
                if (page === 0) {
                    this.pagination.totalCount = await this.getTotalCount(filters);
                }
                const totalLoadedSoFar = page * pageSize + uniqueData.length;
                const hasMore = totalLoadedSoFar < this.pagination.totalCount;
                return { data: uniqueData, hasMore, total: uniqueData.length, totalCount: this.pagination.totalCount };
            } catch (error) {
                console.error('Failed to fetch Analects data:', error);
                throw error;
            }
        }

        buildQueryParams(filters, page, pageSize) {
            const params = {
              select: `*,entry_characters(character_id,characters(id,name)),entry_arguments(argument_id,arguments(id,title)),entry_proverbs(proverbs(*))`,
              order: 'id.asc',
              limit: pageSize,
              offset: page * pageSize
            };

            if (filters.chapter && filters.chapter !== 'all' && filters.chapter.trim()) {
              params.chapter = `eq.${filters.chapter.trim()}`;
            }
            if (filters.characterIds && filters.characterIds.length > 0) {
              params.character_ids = `cs.{${filters.characterIds.join(',')}}`;
            }
            if (filters.argumentIds && filters.argumentIds.length > 0) {
              params.argument_ids = `cs.{${filters.argumentIds.join(',')}}`;
            }
            if (filters.proverbIds && filters.proverbIds.length > 0) {
              params.proverb_ids = `cs.{${filters.proverbIds.join(',')}}`;
            }

            if (filters.keyword && filters.keyword.trim()) {
              const rawKeyword = filters.keyword.trim().replace(/%/g, '');
              const keywords = rawKeyword.split(/\s+/).filter(k => k.length > 0);
              if (keywords.length > 0) {
                const keywordGroups = keywords.map(kw => {
                  const searchFields = [
                    `original_text.ilike.*${kw}*`,
                    `translation.ilike.*${kw}*`,
                    `annotation.ilike.*${kw}*`,
                    `personal_insight.ilike.*${kw}*`
                  ];
                  return `or(${searchFields.join(',')})`;
                });
                params.and = `(${keywordGroups.join(',')})`;
              }
            }
            return params;
        }

        highlightKeywords(text, keyword) {
            if (!text || !keyword) return this.escapeHtml(text || '');
            let escapedText = this.escapeHtml(text);
            const keywords = keyword.trim().split(/\s+/).filter(k => k.length > 0);
            if (keywords.length === 0) return escapedText;
            keywords.forEach(kw => {
                if (kw.length > 0) {
                    const escapedKeyword = this.escapeHtml(kw);
                    const regex = new RegExp(`(${this.escapeRegExp(escapedKeyword)})`, 'gi');
                    escapedText = escapedText.replace(regex, '<mark class="keyword-highlight">$1</mark>');
                }
            });
            return escapedText;
        }

        escapeHtml(text) {
            if (!text) return '';
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        escapeRegExp(string) {
            return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        }

        removeDuplicates(data) {
            const uniqueData = [];
            const seenIds = new Set();
            for (const entry of data) {
                if (entry.id && !seenIds.has(entry.id)) {
                    seenIds.add(entry.id);
                    uniqueData.push(entry);
                }
            }
            return uniqueData;
        }

        async getTotalCount(filters) {
            try {
                const params = { select: 'id', count: 'exact', limit: 0 };
                 if (filters.chapter && filters.chapter !== 'all' && filters.chapter.trim()) {
                    params.chapter = `eq.${filters.chapter.trim()}`;
                }
                if (filters.characterIds && filters.characterIds.length > 0) {
                    params.character_ids = `cs.{${filters.characterIds.join(',')}}`;
                }
                if (filters.argumentIds && filters.argumentIds.length > 0) {
                    params.argument_ids = `cs.{${filters.argumentIds.join(',')}}`;
                }
                if (filters.proverbIds && filters.proverbIds.length > 0) {
                    params.proverb_ids = `cs.{${filters.proverbIds.join(',')}}`;
                }
                if (filters.keyword && filters.keyword.trim()) {
                    const rawKeyword = filters.keyword.trim().replace(/%/g, '');
                    const keywords = rawKeyword.split(/\s+/).filter(k => k.length > 0);
                    if (keywords.length > 0) {
                        const keywordGroups = keywords.map(kw => {
                            const searchFields = [`original_text.ilike.*${kw}*`, `translation.ilike.*${kw}*`, `annotation.ilike.*${kw}*`, `personal_insight.ilike.*${kw}*`];
                            return `or(${searchFields.join(',')})`;
                        });
                        params.and = `(${keywordGroups.join(',')})`;
                    }
                }

                const result = await this.apiRequest('analects_entries_expanded', params);
                return result.count || 0;
            } catch (error) {
                console.error('Failed to get total count:', error);
                return 0;
            }
        }
        
        // **FIXED**: Added sorting helper functions.
        sortByPinyin(items, field) {
            return items.sort((a, b) => {
                const textA = a[field] || '';
                const textB = b[field] || '';
                return textA.localeCompare(textB, 'zh-CN', { numeric: true, sensitivity: 'base' });
            });
        }
        
        sortChaptersByNumber(chapters) {
            return chapters.sort((a, b) => {
                const getChapterNumber = (chapterName) => {
                    const match = (chapterName || '').match(/第([一二三四五六七八九十百千万]+|[0-9]+)/);
                    if (!match) return 0;
                    const numStr = match[1];
                    if (/^\d+$/.test(numStr)) return parseInt(numStr, 10);
                    return this.chineseToNumber(numStr);
                };
                return getChapterNumber(a.name) - getChapterNumber(b.name);
            });
        }

        chineseToNumber(chineseNum) {
            const chineseNumbers = { '零': 0, '一': 1, '二': 2, '三': 3, '四': 4, '五': 5, '六': 6, '七': 7, '八': 8, '九': 9, '十': 10, '百': 100, '千': 1000, '万': 10000 };
            if (chineseNumbers[chineseNum]) return chineseNumbers[chineseNum];
            let result = 0, temp = 0;
            if (chineseNum.startsWith('十')) { result = 10; chineseNum = chineseNum.substring(1); }
            for (let i = 0; i < chineseNum.length; i++) {
                const char = chineseNum[i], num = chineseNumbers[char];
                if (num < 10) temp = num;
                else if (num >= 10) {
                    if (temp === 0) temp = 1;
                    if (num === 10000) { result = (result + temp) * num; temp = 0; }
                    else { result += temp * num; temp = 0; }
                }
            }
            return result + temp;
        }

        async getData(type) {
            if (this.cache[type]) return this.cache[type];
            const endpoints = {
                characters: { table: 'characters', params: { order: 'name.asc' } },
                arguments: { table: 'arguments', params: { order: 'title.asc' } },
                proverbs: { table: 'proverbs', params: { order: 'content.asc' } },
                chapters: { table: 'analects_entries', params: { select: 'chapter', order: 'chapter.asc' } }
            };
            try {
                let data = await this.apiRequest(endpoints[type].table, endpoints[type].params);
                
                // **FIXED**: Apply sorting logic after fetching data.
                if (type === 'chapters') {
                    const uniqueChapters = [...new Set(data.map(item => item.chapter))].filter(Boolean).map(chapter => ({ id: chapter, name: chapter }));
                    data = this.sortChaptersByNumber(uniqueChapters);
                } else {
                    const sortField = type === 'characters' ? 'name' : type === 'arguments' ? 'title' : type === 'proverbs' ? 'content' : null;
                    if (sortField) {
                        data = this.sortByPinyin(data, sortField);
                    }
                }
                
                this.cache[type] = data;
                return data;
            } catch (error) {
                console.error(`Failed to get ${type} list:`, error);
                throw error;
            }
        }

        async getCharacters() { return this.getData('characters'); }
        async getArguments() { return this.getData('arguments'); }
        async getProverbs() { return this.getData('proverbs'); }
        async getChapters() { return this.getData('chapters'); }
        
        // ======================= MODIFICATION START =======================
        // The getDailyAnalect function is updated to use Supabase RPC.
        async getDailyAnalect() {
            try {
                // Directly call the 'get_daily_analect' function from Supabase.
                const { data: dailyEntries, error } = await this.supabase.rpc('get_daily_analect');
                if (error) {
                    throw error;
                }

                // If data is returned, use the first entry.
                if (dailyEntries && dailyEntries.length > 0) {
                    return dailyEntries[0];
                }
                
                // Return null if no entries are found.
                return null;
            } catch (error) {
                console.error('Failed to get daily analect:', error);
                throw error;
            }
        }
        // ======================= MODIFICATION END =======================

        formatDate(date = new Date()) {
            const year = date.getFullYear();
            const month = date.getMonth() + 1;
            const day = date.getDate();
            const weekdays = ['星期日', '星期一', '星期二', '星期三', '星期四', '星期五', '星期六'];
            return {
                full: `${year}年${month}月${day}日`,
                weekday: weekdays[date.getDay()],
                iso: date.toISOString().split('T')[0]
            };
        }

        generateShareLinks(entry, currentUrl = window.location.href) {
            const text = `每日论语：${entry.original_text}`;
            return {
                twitter: `https://twitter.com/intent/tweet?text=${encodeURIComponent(text)}&url=${encodeURIComponent(currentUrl)}`,
                facebook: `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(currentUrl)}`,
                copy: `${text} - ${currentUrl}`,
                email: `mailto:?subject=${encodeURIComponent('每日论语分享')}&body=${encodeURIComponent(text + ' - ' + currentUrl)}`
            };
        }

        renderAnalectCard(entry, container) {
            if (!container) return;
            this.entryCache.set(entry.id, entry);
            const card = document.createElement('div');
            card.className = 'verse-card';
            card.setAttribute('data-entry-id', entry.id);
            card.innerHTML = this.generateResultCardHTML(entry, { showTags: true });
            container.appendChild(card);
            setTimeout(() => card.classList.add('animate-in'), 10);
        }

        generateResultCardHTML(entry, options = {}) {
            if (!entry) return '';
            const { showTags = false } = options;
            const currentKeyword = this.currentFilters?.keyword || '';
        
            const getRelatedData = (items, field) => (items || []).map(item => item[field]?.name || item[field]?.title || item[field]?.content).filter(Boolean);
            const characters = getRelatedData(entry.entry_characters, 'characters');
            const argumentsList = getRelatedData(entry.entry_arguments, 'arguments');
            const proverbs = getRelatedData(entry.entry_proverbs, 'proverbs');
        
            const verseHeaderHTML = `<div class="verse-header"><div class="verse-header-left"><span class="verse-chapter">${this.escapeHtml(entry.chapter)}</span><span class="verse-section">第${this.escapeHtml(entry.section_number)}节</span></div><div class="verse-header-right"></div></div>`;
            
            const highlightedOriginal = this.highlightKeywords(entry.original_text, currentKeyword);
            const highlightedTranslation = entry.translation ? this.highlightKeywords(entry.translation, currentKeyword) : '';
            const highlightedAnnotation = entry.annotation ? this.highlightKeywords(entry.annotation, currentKeyword) : '';
            const contentHTML = `<blockquote class="verse-original">${highlightedOriginal}</blockquote>${highlightedTranslation ? `<p class="verse-translation">【译文】${highlightedTranslation}</p>` : ''}${highlightedAnnotation ? `<div class="verse-annotation-wrapper"><div class="verse-annotation-title">【注释】</div><div class="verse-annotation">${highlightedAnnotation}</div></div>` : ''}`;
            
            let tagsHTML = '';
            if (showTags) {
                const icons = {
                    users: '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M22 21v-2a4 4 0 0 0-3-3.87"/><path d="M16 3.13a4 4 0 0 1 0 7.75"/></svg>',
                    target: '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><circle cx="12" cy="12" r="6"/><circle cx="12" cy="12" r="2"/></svg>',
                    'message-square-quote': '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/></svg>'
                };
                const createTagGroup = (label, items, icon, type) => items.length > 0 ? `<div class="card-tag-group"><div class="card-tag-label">${icons[icon] || ''}<span>${label}</span></div><div class="card-tag-items">${items.map(item => `<span class="card-tag ${type}">${this.escapeHtml(item)}</span>`).join('')}</div></div>` : '';
                tagsHTML = (characters.length > 0 || argumentsList.length > 0 || proverbs.length > 0) ? `<div class="card-tags-section">${createTagGroup('人物', characters, 'users', 'character')}${createTagGroup('论点', argumentsList, 'target', 'argument')}${createTagGroup('谚语', proverbs, 'message-square-quote', 'proverb')}</div>` : '';
            }
        
            const footerHTML = this._generateCoreCardFooterHTML(entry);
            return verseHeaderHTML + contentHTML + tagsHTML + footerHTML;
        }

        _generateCoreCardFooterHTML(entry) {
            const shareLinks = this.generateShareLinks(entry);
            const escapedCopyText = this.escapeHtml(shareLinks.copy).replace(/'/g, "\\'").replace(/\n/g, '\\n');
            const shareMenuHTML = `
                <div class="card-actions-container">
                  <button class="more-options-btn" title="更多选项">
                     <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="1"/><circle cx="19" cy="12" r="1"/><circle cx="5" cy="12" r="1"/></svg>
                  </button>
                  <div class="card-actions-dropdown">
                    <a href="${shareLinks.twitter}" target="_blank" rel="noopener noreferrer" class="dropdown-share-item">
                      <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M23.953 4.57a10 10 0 01-2.825.775 4.958 4.958 0 002.163-2.723c-.951.555-2.005.959-3.127 1.184a4.92 4.92 0 00-8.384 4.482C7.69 8.095 4.067 6.13 1.64 3.162a4.822 4.822 0 00-.666 2.475c0 1.71.87 3.213 2.188 4.096a4.904 4.904 0 01-2.228-.616v.06a4.923 4.923 0 003.946 4.827 4.996 4.996 0 01-2.212.085 4.936 4.936 0 004.604 3.417 9.867 9.867 0 01-6.102 2.105c-.39 0-.779-.023-1.17-.067a13.995 13.995 0 007.557 2.209c9.053 0 13.998-7.496 13.998-13.985 0-.21 0-.42-.015-.63A9.935 9.935 0 0024 4.59z"/></svg>
                      <span>分享到 Twitter</span>
                    </a>
                    <a href="${shareLinks.facebook}" target="_blank" rel="noopener noreferrer" class="dropdown-share-item" onclick="return window.open(this.href, 'facebook-share', 'width=626,height=436')">
                      <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>
                      <span>分享到 Facebook</span>
                    </a>
                    <button onclick="window.AnalectsSDK.copyText('${escapedCopyText}', this)" class="dropdown-share-item">
                      <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M16 1H4c-1.1 0-2 .9-2 2v14h2V3h12V1zm3 4H8c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h11c1.1 0 2-.9 2-2V7c0-1.1-.9-2-2-2zm0 16H8V7h11v14z"/></svg>
                      <span>复制内容</span>
                    </button>
                    <a href="${shareLinks.email}" class="dropdown-share-item">
                      <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor"><path d="M20 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zm0 4l-8 5-8-5V6l8 5 8-5v2z"/></svg>
                      <span>邮件分享</span>
                    </a>
                  </div>
                </div>`;
            return `<div class="analects-card-footer"><div class="footer-actions"><div class="flex items-center gap-2"></div>${shareMenuHTML}</div></div>`;
        }
        
        renderSearchInterface(container) {
            if (!container) return;
            container.innerHTML = this.getSearchInterfaceHTML();
            this.initializeSearchEvents();
            this.loadSearchOptions();
            this.initializeScrollFeatures();
            this.isSearchInitialized = true;
        }

        getSearchInterfaceHTML() {
            return `
            <div class="analects-search">
                <div class="analects-search-header">
                <h2 class="analects-search-title">论语搜索</h2>
                <p class="analects-search-subtitle">Search in the Analects of Confucius</p>
                </div>
                <div class="analects-search-form">
                <div class="analects-keyword-section">
                    <label class="analects-keyword-label" for="analects-keyword">🔍 关键词搜索</label>
                    <input type="text" id="analects-keyword" class="analects-input" placeholder="输入关键词搜索论语原文、翻译或注释...">
                    <div class="analects-selected-items" id="selected-items-container" style="display: none;">
                    <h4>已选择的搜索条件</h4>
                    <div class="analects-selected-tags" id="selected-tags-container"></div>
                    </div>
                </div>
                <div class="analects-advanced-filters">
                    <h3 class="analects-advanced-title">高级搜索</h3>
                    <div class="analects-filters-grid single-column">${this.createSearchSection('chapter', '章节')}</div>
                    <div class="analects-filters-grid">${this.createSearchSection('character', '人物')}${this.createSearchSection('argument', '论点')}</div>
                    <div class="analects-filters-grid">${this.createSearchSection('proverb', '谚语', 'full-width')}</div>
                </div>
                <div class="analects-search-actions">
                    <button id="analects-search-btn" class="analects-btn">🔍 开始搜索</button>
                    <button id="analects-reset-btn" class="analects-btn analects-btn-clear">🔄 重置搜索</button>
                </div>
                </div>
                <div id="analects-search-status" class="analects-search-status"></div>
                <div class="analects-results">
                <div id="analects-results-container" class="analects-results-container"></div>
                <div id="analects-loading-more" style="display:none; text-align:center; margin:10px; color:#666;">加载中...</div>
                <div id="analects-load-complete" class="analects-load-complete" style="display:none;"><span class="analects-load-complete-text">—— ✨ 已全部显示完毕 ✨ ——</span></div>
                </div>
                <div id="scroll-to-top" class="analects-scroll-indicator">↑</div>
            </div>`;
        }

        createSearchSection(type, title, className = '') {
            const icons = {
                chapter: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"/><path d="M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"/></svg>',
                character: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M22 21v-2a4 4 0 0 0-3-3.87"/><path d="M16 3.13a4 4 0 0 1 0 7.75"/></svg>',
                argument: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><circle cx="12" cy="12" r="6"/><circle cx="12" cy="12" r="2"/></svg>',
                proverb: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/></svg>'
            };
            const wrapperStyle = className === 'full-width' ? 'style="grid-column: 1 / -1;"' : '';
            return `<div class="analects-filter-section" ${wrapperStyle}><div class="analects-filter-header"><h4 class="analects-filter-title">${icons[type] || ''}<span>${title}</span></h4><span class="analects-filter-count" id="${type}-count">0</span></div><div class="analects-filter-search"><input type="text" id="${type}-search" placeholder="搜索${title.replace('搜索', '')}..."></div><div id="${type}-filters" class="analects-filter-options"><div class="analects-loading">加载中...</div></div></div>`;
        }
        
        renderDailyAnalect(container) {
            if (!container) return;
            this.getDailyAnalect().then(entry => {
                if (!entry) {
                    container.innerHTML = '<div class="analects-daily-empty">暂无每日论语</div>';
                    return;
                }
                const dateInfo = this.formatDate();
                const shareLinks = this.generateShareLinks(entry);
                container.innerHTML = this.getDailyAnalectHTML(entry, dateInfo, shareLinks);
            }).catch(error => {
                console.error('渲染每日论语失败:', error);
                container.innerHTML = '<div class="analects-daily-error">加载每日论语失败，请检查配置</div>';
            });
        }

        getDailyAnalectHTML(entry, dateInfo, shareLinks) {
             return `
            <div class="analects-daily">
                <div class="analects-daily-header">
                <h2 class="analects-daily-title">每日论语</h2>
                <p class="analects-daily-subtitle">Daily Analects of Confucius</p>
                <div class="analects-daily-date">${dateInfo.full} · ${dateInfo.weekday}</div>
                </div>
                <div class="analects-daily-card">
                <div class="analects-daily-original">${entry.original_text}</div>
                ${entry.translation ? `<div class="analects-daily-translation">${entry.translation}</div>` : ''}
                <div class="analects-daily-reference-wrapper">
                    <div class="analects-daily-reference">${entry.chapter} · 第${entry.section_number || ''}节</div>
                </div>
                </div>
                <div class="analects-daily-share">
                <span class="analects-daily-share-label">分享：</span>
                <div class="analects-share-buttons">
                    ${this.createShareButton('twitter', shareLinks.twitter, '分享到 Twitter')}
                    ${this.createShareButton('facebook', shareLinks.facebook, '分享到 Facebook', true)}
                    ${this.createShareButton('copy', null, '复制分享内容', false, shareLinks.copy)}
                    ${this.createShareButton('email', shareLinks.email, '通过邮件分享')}
                </div>
                </div>
            </div>`;
        }

        createShareButton(type, href, title, popup = false, copyText = null) {
            const icons = {
                twitter: '<path d="M23.953 4.57a10 10 0 01-2.825.775 4.958 4.958 0 002.163-2.723c-.951.555-2.005.959-3.127 1.184a4.92 4.92 0 00-8.384 4.482C7.69 8.095 4.067 6.13 1.64 3.162a4.822 4.822 0 00-.666 2.475c0 1.71.87 3.213 2.188 4.096a4.904 4.904 0 01-2.228-.616v.06a4.923 4.923 0 003.946 4.827 4.996 4.996 0 01-2.212.085 4.936 4.936 0 004.604 3.417 9.867 9.867 0 01-6.102 2.105c-.39 0-.779-.023-1.17-.067a13.995 13.995 0 007.557 2.209c9.053 0 13.998-7.496 13.998-13.985 0-.21 0-.42-.015-.63A9.935 9.935 0 0024 4.59z"/>',
                facebook: '<path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/>',
                copy: '<path d="M16 1H4c-1.1 0-2 .9-2 2v14h2V3h12V1zm3 4H8c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h11c1.1 0 2-.9 2-2V7c0-1.1-.9-2-2-2zm0 16H8V7h11v14z"/>',
                email: '<path d="M20 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zm0 4l-8 5-8-5V6l8 5 8-5v2z"/>'
            };
            if (type === 'copy') {
                return `<button class="analects-share-btn copy" onclick="window.AnalectsSDK.copyText('${copyText.replace(/'/g, "\\'")}', this)" title="${title}"><svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">${icons[type]}</svg></button>`;
            }
            const popupHandler = popup ? `onclick="return window.open(this.href, '${type}-share', 'width=626,height=436')"` : '';
            return `<a href="${href}" class="analects-share-btn ${type}" target="_blank" rel="noopener noreferrer" ${popupHandler} title="${title}"><svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">${icons[type]}</svg></a>`;
        }

        initializeSearchEvents() {
            if (this.searchEventsInitialized) return;
            document.getElementById('analects-search-btn')?.addEventListener('click', () => this.performSearch());
            document.getElementById('analects-reset-btn')?.addEventListener('click', () => this.resetSearch());
            const keywordInput = document.getElementById('analects-keyword');
            if (keywordInput) {
                keywordInput.addEventListener('keydown', e => {
                    if (e.key === 'Enter') {
                        e.preventDefault();
                        document.getElementById('analects-search-btn')?.click();
                    }
                });
                keywordInput.addEventListener('input', () => this.markSearchConditionsChanged());
            }
            ['character', 'argument', 'proverb', 'chapter'].forEach(type => {
                document.getElementById(`${type}-search`)?.addEventListener('input', e => this.filterOptions(type, e.target.value));
            });
            this.searchEventsInitialized = true;
        }

        markSearchConditionsChanged() {
            this.searchConditionsChanged = true;
            document.getElementById('analects-search-btn')?.classList.add('search-changed');
        }

        initializeScrollFeatures() {
            const scrollToTopBtn = document.getElementById('scroll-to-top');
            if (scrollToTopBtn) {
                scrollToTopBtn.addEventListener('click', () => window.scrollTo({ top: 0, behavior: 'smooth' }));
                window.addEventListener('scroll', () => scrollToTopBtn.classList.toggle('visible', window.scrollY > 300));
            }
            this.setupAutoLoad();
        }

        setupAutoLoad() {
            if (this.scrollListener) window.removeEventListener('scroll', this.scrollListener);
            this.scrollListener = () => {
                if (!this.isAutoLoadingEnabled || this.pagination.isLoading || !this.pagination.hasMore) return;
                const resultsContainer = document.getElementById('analects-results-container');
                if (!resultsContainer) return;
                const rect = resultsContainer.getBoundingClientRect();
                if (rect.bottom <= window.innerHeight + 200) {
                    this.loadMoreResults();
                }
            };
            window.addEventListener('scroll', this.scrollListener, { passive: true });
        }

        async loadSearchOptions() {
            try {
                this.showStatusMessage('📄 加载搜索选项...', 'info', 0);
                const [characters, argumentsList, proverbs, chapters] = await Promise.all([this.getCharacters(), this.getArguments(), this.getProverbs(), this.getChapters()]);
                const searchData = { character: characters, argument: argumentsList, proverb: proverbs, chapter: chapters };
                Object.entries(searchData).forEach(([type, data]) => {
                    this.updateSearchCount(type, data.length);
                    this.renderSearchOptions(`${type}-filters`, data, type);
                });
                this.showStatusMessage('✅ 搜索选项加载完成', 'success');
            } catch (error) {
                console.error('加载搜索选项失败:', error);
                this.showSearchLoadError(error);
            }
        }
        
        showSearchLoadError(error) {
            this.showStatusMessage('❌ 加载搜索选项失败，请检查配置', 'error');
            ['character', 'argument', 'proverb', 'chapter'].forEach(type => {
                const container = document.getElementById(`${type}-filters`);
                if(container) container.innerHTML = '<div class="analects-no-options">数据加载失败</div>';
            });
        }
        
        updateSearchCount(type, count) {
            const el = document.getElementById(`${type}-count`);
            if (el) el.textContent = count;
        }

        renderSearchOptions(containerId, options, type) {
            const container = document.getElementById(containerId);
            if (!container) return;
            if (!options || options.length === 0) { container.innerHTML = '<div class="analects-no-options">暂无选项</div>'; return; }
            const tagCloud = document.createElement('div');
            tagCloud.className = 'analects-tag-cloud';
            tagCloud.dataset.type = type;
            if (type === 'chapter') {
                const allTag = this.createOptionTag('all', '全部章节', type, true);
                tagCloud.appendChild(allTag);
                this.selectedItems.chapter = { id: 'all', name: '全部章节' };
            }
            options.forEach(option => {
                const displayName = option.name || option.title || option.content || `选项${option.id}`;
                tagCloud.appendChild(this.createOptionTag(option.id, displayName, type, false));
            });
            container.innerHTML = '';
            container.appendChild(tagCloud);
        }

        createOptionTag(id, displayName, type, isSelected = false) {
            const tag = document.createElement('label');
            tag.className = `analects-option-tag ${isSelected ? 'selected' : ''}`;
            tag.dataset.id = id;
            tag.dataset.name = displayName.toLowerCase();
            tag.innerHTML = `<input type="checkbox" value="${id}" data-type="${type}" ${isSelected ? 'checked' : ''}><span>${displayName}</span>`;
            tag.addEventListener('click', (e) => { e.preventDefault(); e.stopPropagation(); this.handleOptionClick(tag, type, id, displayName); });
            return tag;
        }

        handleOptionClick(tag, type, id, displayName) {
            const checkbox = tag.querySelector('input[type="checkbox"]');
            const tagCloud = tag.closest('.analects-tag-cloud');
            if (type === 'chapter') {
                tagCloud.querySelectorAll('.selected').forEach(el => { el.classList.remove('selected'); el.querySelector('input').checked = false; });
                checkbox.checked = true;
                tag.classList.add('selected');
                this.selectedItems.chapter = { id, name: displayName };
            } else {
                const wasChecked = checkbox.checked;
                checkbox.checked = !wasChecked;
                tag.classList.toggle('selected', checkbox.checked);
                if (checkbox.checked) this.selectedItems[type + 's'].set(parseInt(id), displayName);
                else this.selectedItems[type + 's'].delete(parseInt(id));
            }
            this.markSearchConditionsChanged();
            this.renderSelectedItems();
        }

        renderSelectedItems() {
            const selectedContainer = document.getElementById('selected-items-container');
            const tagsContainer = document.getElementById('selected-tags-container');
            if (!selectedContainer || !tagsContainer) return;
            const totalSelected = this.selectedItems.characters.size + this.selectedItems.arguments.size + this.selectedItems.proverbs.size + (this.selectedItems.chapter.id !== 'all' ? 1 : 0);
            if (totalSelected === 0) { selectedContainer.style.display = 'none'; return; }
            selectedContainer.style.display = 'block';
            tagsContainer.innerHTML = '';
            if (this.selectedItems.chapter.id !== 'all') tagsContainer.appendChild(this.createSelectedTag('chapter', this.selectedItems.chapter.id, this.selectedItems.chapter.name));
            ['characters', 'arguments', 'proverbs'].forEach(type => {
                this.selectedItems[type].forEach((name, id) => tagsContainer.appendChild(this.createSelectedTag(type.slice(0, -1), id, name)));
            });
        }

        createSelectedTag(type, id, name) {
            const tag = document.createElement('div');
            tag.className = `analects-selected-tag ${type}`;
            tag.innerHTML = `<span>${name}</span><span class="remove-tag" data-type="${type}" data-id="${id}">×</span>`;
            tag.querySelector('.remove-tag').addEventListener('click', (e) => { e.stopPropagation(); this.removeSelectedItemById(type, id); });
            return tag;
        }

        removeSelectedItemById(type, id) {
            if (type === 'chapter') {
                this.selectedItems.chapter = { id: 'all', name: '全部章节' };
                const chapterContainer = document.querySelector('.analects-tag-cloud[data-type="chapter"]');
                if (chapterContainer) {
                    chapterContainer.querySelector(`.selected`)?.classList.remove('selected');
                    const currentInput = chapterContainer.querySelector(`[data-id="${id}"] input`);
                    if(currentInput) currentInput.checked = false;
                    const allTag = chapterContainer.querySelector(`[data-id="all"]`);
                    allTag.classList.add('selected');
                    allTag.querySelector('input').checked = true;
                }
            } else {
                this.selectedItems[type + 's'].delete(parseInt(id));
                const checkbox = document.querySelector(`input[data-type="${type}"][value="${id}"]`);
                if (checkbox) {
                    checkbox.checked = false;
                    checkbox.closest('.analects-option-tag')?.classList.remove('selected');
                }
            }
            this.markSearchConditionsChanged();
            this.renderSelectedItems();
        }

        filterOptions(type, searchTerm) {
            const container = document.getElementById(`${type}-filters`);
            if (!container) return;
            const term = searchTerm.toLowerCase().trim();
            let visibleCount = 0;
            const tags = container.querySelectorAll('.analects-option-tag');
            tags.forEach(tag => {
                const shouldShow = !term || tag.dataset.name.includes(term);
                tag.style.display = shouldShow ? 'inline-flex' : 'none';
                if (shouldShow) visibleCount++;
            });
            const countElement = document.getElementById(`${type}-count`);
            if (countElement) countElement.textContent = term ? `${visibleCount}/${tags.length}` : tags.length;
        }

        async performSearch() {
            if (!this.isSearchInitialized || this.pagination.isLoading) return;
            const searchId = ++this.latestSearchId;
            const resultsContainer = document.getElementById('analects-results-container');
            if (!resultsContainer) return;
            this.pagination.isLoading = true;
            this.currentFilters = {
                keyword: document.getElementById('analects-keyword')?.value.trim(),
                characterIds: Array.from(this.selectedItems.characters.keys()),
                argumentIds: Array.from(this.selectedItems.arguments.keys()),
                proverbIds: Array.from(this.selectedItems.proverbs.keys()),
                chapter: this.selectedItems.chapter.id
            };
            this.resetPagination();
            this.searchConditionsChanged = false;
            this.isAutoLoadingEnabled = false;
            document.getElementById('analects-search-btn')?.classList.remove('search-changed');
            document.getElementById('analects-load-complete').style.display = 'none';
            const skeletonCardHTML = `<div class="favorite-card-skeleton"><div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 24px;"><div class="skeleton-block" style="height: 20px; width: 30%;"></div><div class="skeleton-block" style="height: 24px; width: 24px; border-radius: 50%;"></div></div><div class="skeleton-block" style="height: 18px; width: 90%; margin-bottom: 12px;"></div><div class="skeleton-block" style="height: 18px; width: 70%;"></div></div>`;
            resultsContainer.innerHTML = Array(1).fill(skeletonCardHTML).join('');
            try {
                const result = await this.fetchAnalects(this.currentFilters, 0, this.pagination.pageSize);
                if (searchId === this.latestSearchId) {
                    this.isAutoLoadingEnabled = true;
                    this.handleFirstSearchResult(result, resultsContainer);
                }
            } catch (error) {
                if (searchId === this.latestSearchId) this.handleSearchError(error, resultsContainer);
            } finally {
                if (searchId === this.latestSearchId) this.pagination.isLoading = false;
            }
        }

        handleFirstSearchResult(result, resultsContainer) {
            resultsContainer.innerHTML = '';
            if (result.data.length === 0) {
                this.showStatusMessage('⚠️ 未找到匹配的章节', 'warning');
                this.isAutoLoadingEnabled = false;
                return;
            }
            const uniqueResults = result.data.filter(entry => {
                if (!entry.id || this.pagination.loadedIds.has(entry.id)) return false;
                this.pagination.loadedIds.add(entry.id);
                return true;
            });
            this.pagination.currentPage = 1;
            this.pagination.totalLoaded = uniqueResults.length;
            this.pagination.hasMore = result.hasMore;
            this.updateScrollStatus();
            uniqueResults.forEach((entry, index) => {
                setTimeout(() => this.renderAnalectCard(entry, resultsContainer), index * 50);
            });
        }
        
        resetPagination() {
            Object.assign(this.pagination, { currentPage: 0, totalLoaded: 0, hasMore: true, isLoading: false, totalCount: 0 });
            this.pagination.loadedIds.clear();
        }

        handleSearchError(error, resultsContainer) {
            this.showStatusMessage(`❌ 搜索失败：${error.message}`, 'error');
            resultsContainer.innerHTML = `<div class="analects-error"><h3>搜索失败</h3><p>${error.message}</p><p style="font-size: 14px; color: #666; margin-top: 12px;">请检查Supabase配置是否正确，或查看控制台了解详细错误信息。</p></div>`;
            this.isAutoLoadingEnabled = false;
        }

        async loadMoreResults() {
            if (this.pagination.isLoading || !this.pagination.hasMore || !this.isAutoLoadingEnabled) return;
            const searchId = this.latestSearchId;
            this.pagination.isLoading = true;
            document.getElementById('analects-loading-more').style.display = 'block';
            try {
                const result = await this.fetchAnalects(this.currentFilters, this.pagination.currentPage, this.pagination.pageSize);
                if (searchId === this.latestSearchId) {
                    const uniqueResults = result.data.filter(entry => !this.pagination.loadedIds.has(entry.id));
                    const resultsContainer = document.getElementById('analects-results-container');
                    uniqueResults.forEach(entry => {
                        this.pagination.loadedIds.add(entry.id);
                        this.renderAnalectCard(entry, resultsContainer)
                    });
                    this.pagination.currentPage++;
                    this.pagination.totalLoaded += uniqueResults.length;
                    this.pagination.hasMore = this.pagination.totalLoaded < this.pagination.totalCount;
                    this.updateScrollStatus();
                }
            } catch (error) {
                console.error('加载更多结果失败:', error);
                this.showStatusMessage('❌ 加载更多结果失败，请稍后重试', 'error');
            } finally {
                if (searchId === this.latestSearchId) this.pagination.isLoading = false;
                document.getElementById('analects-loading-more').style.display = 'none';
            }
        }

        updateScrollStatus() {
            const statusDiv = document.getElementById('analects-search-status');
            if (!statusDiv || this.pagination.totalLoaded === 0) return;
            const displayedCount = this.pagination.totalLoaded, totalCount = this.pagination.totalCount;
            if (this.pagination.hasMore) {
                statusDiv.innerHTML = `<div class="analects-success">✅ 找到 ${totalCount} 条结果，已显示 ${displayedCount} 条，滚动自动加载更多</div>`;
            } else {
                statusDiv.innerHTML = `<div class="analects-success">✅ 共找到 ${totalCount} 条结果，已全部显示完毕</div>`;
                this.isAutoLoadingEnabled = false;
                document.getElementById('analects-load-complete').style.display = 'block';
            }
        }
        
        resetSearch() {
            document.getElementById('analects-keyword').value = '';
            ['character-search', 'argument-search', 'proverb-search', 'chapter-search'].forEach(type => {
                const searchInput = document.getElementById(`${type}-search`);
                if (searchInput) { searchInput.value = ''; this.filterOptions(type, ''); }
                const tagContainer = document.querySelector(`.analects-tag-cloud[data-type="${type}"]`);
                if (tagContainer) {
                    tagContainer.querySelectorAll('.selected').forEach(el => { el.classList.remove('selected'); el.querySelector('input').checked = false; });
                    if (type === 'chapter') { const allTag = tagContainer.querySelector(`[data-id="all"]`); allTag.classList.add('selected'); allTag.querySelector('input').checked = true; }
                }
            });
            this.selectedItems.characters.clear();
            this.selectedItems.arguments.clear();
            this.selectedItems.proverbs.clear();
            this.selectedItems.chapter = { id: 'all', name: '全部章节' };
            document.getElementById('analects-results-container').innerHTML = '';
            document.getElementById('analects-load-complete').style.display = 'none';
            this.resetPagination();
            this.isAutoLoadingEnabled = false;
            document.getElementById('analects-search-btn')?.classList.remove('search-changed');
            this.renderSelectedItems();
            this.showStatusMessage('🔄 已重置所有搜索条件', 'success');
        }

        showStatusMessage(message, type = 'info', duration = 2000) {
            const statusDiv = document.getElementById('analects-search-status');
            if (statusDiv) {
                const className = `analects-${type}`;
                statusDiv.innerHTML = `<div class="${className}">${message}</div>`;
                if (duration > 0) setTimeout(() => statusDiv.innerHTML = '', duration);
            }
        }
    }

    // --- Global Initialization ---
    document.addEventListener('DOMContentLoaded', function() {
        if (typeof analectsSdkSettings !== 'undefined') {
            const sdk = new CoreSDK(analectsSdkSettings.config);

            // **FIXED**: Call init() to attach event listeners.
            sdk.init();

            const searchContainer = document.getElementById(analectsSdkSettings.targets.search);
            if (searchContainer) {
                sdk.renderSearchInterface(searchContainer);
            }

            const dailyContainer = document.getElementById(analectsSdkSettings.targets.daily);
            if (dailyContainer) {
                sdk.renderDailyAnalect(dailyContainer);
            }
            
            const dailyWidgetContainers = document.querySelectorAll(analectsSdkSettings.targets.dailyWidgetSelector);
            dailyWidgetContainers.forEach(container => {
                if (container) {
                   sdk.renderDailyAnalect(container);
               }
           });

            window.AnalectsSDK = {
                copyText: async function(text, button) {
                    const originalTitle = button.title;
                    button.title = '复制中...';
                    try {
                        await navigator.clipboard.writeText(text);
                        button.title = '已复制！';
                    } catch (err) {
                        console.error('复制失败:', err);
                        button.title = '复制失败';
                    } finally {
                        setTimeout(() => { button.title = originalTitle; }, 2000);
                    }
                }
            };
        }
    });

})();