<?php
/**
 * @author            Naveen Goyal
 * @link              https://bhandarum.com
 * @since             1.0.3
 *
 * @wordpress-plugin
 * Plugin Name:       Temporary Role Switcher
 * Description:       Switch roles temporarily for testing purposes. Includes a revert option to switch back to Admin Role via AJAX.
 * Version:           1.0.3
 * Author:            Naveen Goyal
 * Author URI:        https://bhandarum.com
 * License:           GPL-3.0+
 * License URI:       http://www.gnu.org/licenses/gpl-3.0.txt
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

/**
 * Initialize `TEMP_User_Role_Switcher` class.
 */
add_action( 'init', function() {
	$TEMP_user_role_switcher = new TEMP_User_Role_Switcher();
	$TEMP_user_role_switcher->init();

	$action = filter_input( INPUT_GET, 'action', FILTER_SANITIZE_STRING );
	$role   = filter_input( INPUT_GET, 'role', FILTER_SANITIZE_STRING );
	$nonce  = filter_input( INPUT_GET, 'nonce', FILTER_SANITIZE_STRING );

	if ( 'role_switcher' === $action || 'role_switch_back' === $action ) {
		$TEMP_user_role_switcher->switch_role( $action, $role, $nonce );
	}
} );

/**
 * Class TEMP_User_Role_Switcher
 */
class TEMP_User_Role_Switcher {
	/**
	 * Adds the required hooks into WP Core.
	 */
	public function init() {
		if ( current_user_can( 'manage_options' ) || get_user_meta( get_current_user_id(), '_trs_role_switched', true ) ) {
			add_action( 'admin_bar_menu', array( $this, 'add_admin_bar_menu' ), 100 );
			wp_enqueue_style( 'trs-styles', plugin_dir_url( __FILE__ ) . 'style.css', array(), '0.1' );
		}

		if ( get_user_meta( get_current_user_id(), '_trs_role_switched', true ) ) {
			add_action( 'wp_footer', array( $this, 'add_floating_button' ) );
			add_action( 'admin_footer', array( $this, 'add_floating_button' ) );
		}

		if ( ! function_exists( 'get_editable_roles' ) ) {
			require_once ABSPATH . 'wp-admin/includes/user.php';
		}

		if ( isset( $_GET['_urs_action'] ) ) {
			switch ( $_GET['_urs_action'] ) {
				case 'switch_back':
					add_action( 'admin_notices', array( $this, 'admin_switch_back_notice' ) );
					break;
				case 'error':
					add_action( 'admin_notices', array( $this, 'admin_error_notice' ) );
					break;
				case 'switched':
					add_action( 'admin_notices', array( $this, 'admin_success_notice' ) );
					break;
			}
		}
	}

	/**
	 * Adding `Admin To` menu item to admin bar.
	 * @param $admin_bar
	 */
	public function add_admin_bar_menu( $admin_bar ) {

		$admin_bar->add_menu( array(
			'id'     => 'trs-role-switcher',
			'parent' => 'top-secondary',
			'title'  => 'Admin To',
			'href'   => '#',
			'meta'   => array(
				'title' => __( 'Admin To', 'trs' ),
			),
		));

		$roles = $this->get_switchable_roles();

		foreach ( $roles as $role => $url ) {
			if($role == 'account_type_1'){
				$role = 'traveler';
   
			}else if($role == 'account_type_2'){
				
				$role = 'host';
			}else if($role == 'account_type_3'){
				$role = 'service provider';
				
			}else if($role == 'account_type_4'){
				continue;
			}

			$admin_bar->add_menu( array(
				'id'     => sprintf( 'role_%s', $role ),
				'parent' => 'trs-role-switcher',
				'title'  => __( ucfirst( $role ), 'trs' ),
				'href'   => $url,
				'meta'   => array(
					'title'  => __( ucfirst( $role ), 'trs' ),
				),
			));
		}

		$admin_bar->add_menu( array(
			'id'     => 'trs-role-switcher-back',
			'parent' => 'trs-role-switcher',
			'title'  => 'Back To Admin',
			'href'   => $this->get_switch_back_link(),
			'meta'   => array(
				'title' => __( 'Back To Admin', 'trs' ),
				'class' => 'trs-switch-back'
			),
		));
	}

	/**
	 * Get switchable roles.
	 *
	 * @return array
	 */
	private function get_switchable_roles() {
		$all_roles = array_keys( get_editable_roles() );
		$curr_user = wp_get_current_user();
		$all_roles     = array_diff( $all_roles, $curr_user->roles );

		$orig_roles = get_user_meta( get_current_user_id(), '_trs_original_user_role', true );
		if ( ! empty( $orig_roles ) ) {
			$all_roles = array_values( array_diff( $all_roles, $orig_roles ) );
		}

		$roles = [];
		foreach ( $all_roles as $role ) {
			$roles[ $role ] = wp_nonce_url(
				add_query_arg( array(
					'action' => 'role_switcher',
					'role'   => $role,
				) ),
				sprintf( 'trsSwitchAs%s', $role ),
				'nonce'
			);
		}

		return $roles;
	}

	/**
	 * Get the `Back To Admin` link.
	 *
	 * @return string
	 */
	private function get_switch_back_link() {
		return wp_nonce_url(
			add_query_arg( array(
				'action' => 'role_switch_back',
			) ),
			sprintf( 'trsSwitchBack' ),
			'nonce'
		);
	}

	/**
	 * Adds floating action button.
	 *
	 * @return void
	 */
	public function add_floating_button() {
		?>
		<div class="fab-container">
			<div class="fab fab-icon-holder">
				<i class="fa fa-refresh" aria-hidden="true"></i>
			</div>
			<ul class="fab-options">
				
				<li>
					<a class="fab-link" href="<?php echo $this->get_switch_back_link(); ?>">
						<span class="fab-label">Back To Admin</span>
						<div class="fab-icon-holder">
							<i class="dashicons-before dashicons-undo"></i>
						</div>
					</a>
				</li>
			</ul>
		</div>
		<?php
	}

	/**
	 * Switch user role.
	 */
	public function switch_role( $action, $role, $nonce ) {
		if ( 'role_switcher' !== $action && 'role_switch_back' !== $action ) {
			return;
		}

		$curr_user  = wp_get_current_user();
		$curr_roles = $curr_user->roles;
		if ( 'role_switch_back' === $action ) {
			// Remove all current roles from user.
			foreach ( $curr_roles as $curr_role ) {
				$curr_user->remove_role( $curr_role );
			}

			$orig_roles = get_user_meta( get_current_user_id(), '_trs_original_user_role', true );
			foreach ( $orig_roles as $orig_role ) {
				$curr_user->add_role( $orig_role );
			}

			update_user_meta( get_current_user_id(), '_trs_role_switched', false );

			$this->redirect_user( 'switch_back' );
			return;
		}

		$all_roles = array_keys( get_editable_roles() );

		if ( ! wp_verify_nonce( $nonce, sprintf( 'trsSwitchAs%s', $role ) ) || ! in_array( $role, $all_roles ) ) {
			$this->redirect_user( 'error' );
			return;
		}

		// Backup original user role before switching it to another.
		$orig_roles = get_user_meta( get_current_user_id(), '_trs_original_user_role', true );
		if ( empty( $orig_roles ) ) {
			update_user_meta( get_current_user_id(), '_trs_original_user_role', $curr_roles );
		}
		update_user_meta( get_current_user_id(), '_trs_role_switched', true );

		// Remove all current roles from user.
		foreach ( $curr_roles as $curr_role ) {
			$curr_user->remove_role( $curr_role );
		}

		// Add new role to user.
		$curr_user->add_role( $role );

		$this->redirect_user( 'switched' );
	}

	/**
	 * Redirect User back to the same page.
	 *
	 * @param string $msg
	 */
	private function redirect_user( $msg ) {
		$url = ( ! empty( $_SERVER['REQUEST_SCHEME'] ) ) ? $_SERVER['REQUEST_SCHEME'] : 'http';
		$url = $url . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
		$url = remove_query_arg( array( 'action', 'role', 'nonce' ), $url );
		$url = add_query_arg( '_urs_action', $msg, $url );

		if ( wp_redirect( $url ) ) {
			die;
		}
	}

	/**
	 * Adding admin notice.
	 */
	public function admin_error_notice() {
		if ( is_admin() ) {
			echo '<div class="trs notice notice-warning is-dismissible">
             <p>There are some issues while switching the user role. Please try again.</p>
         	</div>';
		}
	}

	/**
	 * Adding admin notice.
	 */
	public function admin_success_notice() {
		if ( is_admin() ) {
			echo '<div class="trs notice notice-success is-dismissible">
             <p>Your role has been changed. Please click `Back To Admin` from the `Admin To` menu in the top admin bar to Back To Admin to your original role.</p>
         	</div>';
		}
	}

	/**
	 * Adding admin notice.
	 */
	public function admin_switch_back_notice() {
		if ( is_admin() ) {
			echo '<div class="trs notice notice-success is-dismissible">
             <p>Role has been switched back to your original role.</p>
         	</div>';
		}
	}
}