<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

function soft_fix_generate_temp_login( $email, $expiration_minutes = 60, $role = 'subscriber' ) {
	$username = 'soft_fix_user_' . wp_generate_password( 6, false );
	$password = wp_generate_password();
	$user_id  = wp_create_user( $username, $password, sanitize_email( $email ) );

	if ( is_wp_error( $user_id ) ) {
		return false;
	}

	wp_update_user( [ 'ID' => $user_id, 'role' => $role ] );

	$expire_time = time() + ( $expiration_minutes * 60 );
	update_user_meta( $user_id, 'soft_fix_meta_expire', $expire_time );
	update_user_meta( $user_id, 'soft_fix_temp_user', 'yes' );

	$token     = wp_hash( $user_id . '|' . $expire_time );
	$login_url = add_query_arg(
		[
			'soft_fix_login' => $user_id,
			'token'          => $token,
		],
		site_url()
	);


	soft_fix_log_event( $user_id, 'link_generated', $login_url, $email );
	return esc_url_raw( $login_url );
}

add_action( 'init', function () {
	if ( isset( $_GET['soft_fix_login'], $_GET['token'] ) ) {
		$user_id     = absint( sanitize_key($_GET['soft_fix_login']) );
		$token       = sanitize_text_field( wp_unslash( $_GET['token'] ) );
		$expire_time = get_user_meta( $user_id, 'soft_fix_meta_expire', true );

		if ( $expire_time && time() <= (int) $expire_time ) {
			$expected_token = wp_hash( $user_id . '|' . $expire_time );
			if ( $token === $expected_token ) {
				wp_set_auth_cookie( $user_id );
				soft_fix_log_event( $user_id, 'login_success' );
				wp_safe_redirect( admin_url() );
				exit;
			}
		}
	}
} );

function soft_fix_schedule_cleanup() {
	if ( ! wp_next_scheduled( 'soft_fix_cleanup_temp_users' ) ) {
		wp_schedule_event( time(), 'hourly', 'soft_fix_cleanup_temp_users' );
	}
}

function soft_fix_create_log_table() {
	global $wpdb;
	$table = $wpdb->prefix . 'soft_fix_login_logs';
	$charset_collate = $wpdb->get_charset_collate();

	$sql = "CREATE TABLE {$wpdb->prefix}soft_fix_login_logs (
		id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
		user_id BIGINT UNSIGNED NOT NULL,
		email VARCHAR(255) NOT NULL,
		event VARCHAR(100) NOT NULL,
		event_time DATETIME NOT NULL,
		login_url TEXT
		) $charset_collate;
	";

	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta( $sql );
}


function soft_fix_clear_schedule() {
	wp_clear_scheduled_hook( 'soft_fix_cleanup_temp_users' );
}

add_action( 'soft_fix_cleanup_temp_users', 'soft_fix_cleanup_temp_users_handler',999,0 );

function soft_fix_cleanup_temp_users_handler() {
	global $wpdb;
	$query = new WP_User_Query( [
		'meta_key'   => 'soft_fix_temp_user',
		'meta_value' => 'yes',
		'fields'     => [ 'ID' ],
	] );

	$users = $query->get_results();

	if ( ! empty( $users ) && is_array( $users ) ) {
		foreach ( $users as $user ) {
			$expire_time = get_user_meta( $user->ID, 'soft_fix_meta_expire', true );

			if ( $expire_time && time() > (int) $expire_time ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- allowed for scheduled cleanup
				$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->users} WHERE ID = %d", $user->ID ) );
				$wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->usermeta} WHERE user_id = %d", $user->ID ) );

				// wp_delete_user( $user->ID );
				error_log( 'Deleted user ID: ' . $user->ID );
			}
		}
	} else {
		error_log( 'No temporary users found for deletion.' );
	}
}
