<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_action( 'admin_menu', 'soft_fix_add_admin_menu' );

function soft_fix_add_admin_menu() {
	add_users_page(
		__( 'Temporary Login', 'temporary-login-by-softfix' ),
		__( 'Temporary Login', 'temporary-login-by-softfix' ),
		'manage_options',
		'soft-fix-temp-login',
		'soft_fix_admin_page'
	);
}

function soft_fix_admin_page() {
	global $wpdb;

	$generated_url = '';
	$table         = esc_sql( $wpdb->prefix . 'soft_fix_login_logs' ); // escaped table name

	if (
		isset( $_POST['soft_fix_generate'] ) &&
		check_admin_referer( 'soft_fix_generate_nonce_action', 'soft_fix_generate_nonce' )
	) {
		$email      = sanitize_email( wp_unslash( $_POST['soft_fix_email'] ?? '' ) );
		$expiration = absint( wp_unslash( $_POST['soft_fix_expire'] ?? 60 ) );
		$role       = sanitize_text_field( wp_unslash( $_POST['soft_fix_user_role'] ?? 'subscriber' ) );

		$all_roles = array_keys( wp_roles()->roles );
		if ( in_array( $role, $all_roles, true ) && is_email( $email ) ) {
			$generated_url = soft_fix_generate_temp_login( $email, $expiration, $role );
		}
	}

	if (
		isset( $_POST['soft_fix_disable_user_id'], $_POST['soft_fix_disable_user_nonce'] ) &&
		wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['soft_fix_disable_user_nonce'] ) ), 'soft_fix_disable_user_action' )
	) {
		$user_id = absint( $_POST['soft_fix_disable_user_id'] );

		if ( get_user_meta( $user_id, 'soft_fix_meta_expire', true ) ) {
			wp_delete_user( $user_id );
			soft_fix_log_event( $user_id, 'access_disabled' );

			echo '<div class="notice notice-success is-dismissible"><p>' .
			esc_html__( 'Access disabled and user removed.', 'temporary-login-by-softfix' ) .
			'</p></div>';
		}
	}

	$per_page     = 10;
	$paged        = isset( $_GET['paged'] ) ? max( 1, absint( $_GET['paged'] ) ) : 1;
	$offset       = ( $paged - 1 ) * $per_page;

	// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name sanitized via esc_sql.
	$total_items  = (int) $wpdb->get_var( "SELECT COUNT(*) FROM `{$table}`" );

	// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Safe custom query with prepared values.
	$logs = $wpdb->get_results(
		$wpdb->prepare(
			"SELECT * FROM `{$table}` ORDER BY event_time DESC LIMIT %d OFFSET %d",
			$per_page,
			$offset
		),
		ARRAY_A
	);

	$total_pages = ceil( $total_items / $per_page );
	?>
	<div class="softfix-temp-all-wrap">
		<h1><?php esc_html_e( 'Generate Temporary Login Link', 'temporary-login-by-softfix' ); ?></h1>

		<div class="content-wrap-softfix">
			<div class="form-wrap-softfix">
				<form method="post">
					<?php wp_nonce_field( 'soft_fix_generate_nonce_action', 'soft_fix_generate_nonce' ); ?>
					<p>
						<label for="soft_fix_email"><?php esc_html_e( 'Email', 'temporary-login-by-softfix' ); ?></label><br>
						<input type="email" id="soft_fix_email" name="soft_fix_email" required>
					</p>

					<p>
						<label for="soft_fix_user_role"><?php esc_html_e( 'Role', 'temporary-login-by-softfix' ); ?></label><br>
						<select name="soft_fix_user_role" id="soft_fix_user_role">
							<?php
							foreach ( wp_roles()->roles as $key => $role ) {
								echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $role['name'] ) . '</option>';
							}
							?>
						</select>
					</p>

					<p>
						<label for="soft_fix_expire"><?php esc_html_e( 'Expiration (minutes)', 'temporary-login-by-softfix' ); ?></label><br>
						<input type="number" id="soft_fix_expire" name="soft_fix_expire" value="60" min="1">
					</p>

					<p>
						<input type="submit" name="soft_fix_generate" class="button button-primary" value="<?php esc_attr_e( 'Generate Link', 'temporary-login-by-softfix' ); ?>">
					</p>
				</form>
			</div>
			<div class="message-wrap-softfix">
				<a class="link-softfix-support" href="http://softfix.in/contact-us?utm=softfix-org" target="_blank">
					<img src="<?php echo esc_url( SOFT_FIX_TEMPLOGIN_URL . 'assets/images/get-help.png' ); ?>" alt="Help Icon" width="150" height="150">
				</a>
			</div>
		</div>

		<?php if ( $generated_url ) : ?>
			<p><strong><?php esc_html_e( 'Generated Link:', 'temporary-login-by-softfix' ); ?></strong><br>
				<a href="<?php echo esc_url( $generated_url ); ?>" target="_blank"><?php echo esc_html( $generated_url ); ?></a></p>
			<?php endif; ?>
			<div class="clear" bis_skin_checked="1"></div>
			<div class="log-wrap-softfix table-responsive">
				<h2><?php esc_html_e( 'Login Logs', 'temporary-login-by-softfix' ); ?></h2>

				<table class="widefat">
					<thead>
						<tr>
							<th><?php esc_html_e( 'User ID', 'temporary-login-by-softfix' ); ?></th>
							<th><?php esc_html_e( 'Event', 'temporary-login-by-softfix' ); ?></th>
							<th><?php esc_html_e( 'Time', 'temporary-login-by-softfix' ); ?></th>
							<th><?php esc_html_e( 'Email', 'temporary-login-by-softfix' ); ?></th><th><?php esc_html_e( 'Login URL', 'temporary-login-by-softfix' ); ?></th>
							<th><?php esc_html_e( 'Actions', 'temporary-login-by-softfix' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $logs as $log ) : ?>
							<tr>
								<td><?php echo esc_html( $log['user_id'] ); ?></td>
								<td><?php echo esc_html( $log['event'] ); ?></td>
								<td><?php echo esc_html( $log['event_time'] ); ?></td>
								<td><?php echo esc_html( $log['email'] ); ?></td>
								<td>
									<?php if ( ! empty( $log['login_url'] ) ) : ?>
										<a href="<?php echo esc_url( $log['login_url'] ); ?>" target="_blank"><?php echo esc_html( $log['login_url'] ); ?></a>
									<?php endif; ?>
								</td>
								<td> <?php if ( ! empty( $log['login_url'] ) ) { ?>
									<form method="post">
										<?php wp_nonce_field( 'soft_fix_disable_user_action', 'soft_fix_disable_user_nonce' ); ?>
										<input type="hidden" name="soft_fix_disable_user_id" value="<?php echo esc_attr( $log['user_id'] ); ?>">
										<button type="submit" class="button button-small">
											<?php esc_html_e( 'Disable Access', 'temporary-login-by-softfix' ); ?>
										</button>
									</form>
								<?php } ?>
							</td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
			
			<?php if ( $total_pages > 1 ) : ?>
				<div class="tablenav">
					<div class="tablenav-pages">
						<ul class="pagination-links">
							<?php
							$pagination_links = paginate_links( [
								'base'      => esc_url( add_query_arg( 'paged', '%#%' ) ),
								'format'    => '',
								'prev_text' => esc_html__( '« Prev', 'temporary-login-by-softfix' ),
								'next_text' => esc_html__( 'Next »', 'temporary-login-by-softfix' ),
								'total'     => $total_pages,
								'current'   => $paged,
								'type'      => 'array',
							] );

							if ( ! empty( $pagination_links ) ) {
								foreach ( $pagination_links as $link ) {
									echo '<li class="page-link">' . wp_kses_post( $link ) . '</li>';
								}
							}
							?>
						</ul>
					</div>
				</div>

			<?php endif; ?>
		</div>

	</div>
	<?php
}
