<?php
/**
 * Plugin Name: TELO Affiliates Lite
 * Description: Simple affiliate management for WooCommerce: single link with parameter, affiliate dashboard via shortcode, cashout requests, public pages per affiliate, affiliate import/export, and more.
 * Version: 3.0.1
 * Author: Miguel Martinez
 * Author URI: https://qagencia.com/
 * Text Domain: telo-affiliates-lite
 * Requires at least: 6.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Domain Path: /languages
 **/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Detectar si la versión PRO está activa.
 */
if ( ! function_exists( 'telo_aff_pro_is_active' ) ) {
    function telo_aff_pro_is_active() {
        // Aseguramos plugin.php para is_plugin_active / get_plugins
        if ( ! function_exists( 'is_plugin_active' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        // Constante que puede definir el PRO
        if ( defined( 'TELO_AFFILIATES_IS_PRO' ) && TELO_AFFILIATES_IS_PRO ) {
            return true;
        }

        // Slugs posibles (1 o 2 "f")
        $candidates = array(
            'telo-affiliates-pro/telo-affiliates.php',
            'telo-affiliates-pro/telo-afiliates.php',
            'telo-afiliates-pro/telo-affiliates.php',
            'telo-afiliates-pro/telo-afiliates.php',
        );

        foreach ( $candidates as $slug ) {
            if ( is_plugin_active( $slug ) || is_plugin_active_for_network( $slug ) ) {
                return true;
            }
        }

        // Fallback: detectar por nombre del plugin activo
        if ( function_exists( 'get_plugins' ) ) {
            foreach ( get_plugins() as $file => $data ) {
                $name = isset( $data['Name'] ) ? $data['Name'] : '';
                if (
                    stripos( $name, 'TELO' ) !== false &&
                    stripos( $name, 'Affiliates' ) !== false &&
                    stripos( $name, 'Pro' ) !== false &&
                    ( is_plugin_active( $file ) || is_plugin_active_for_network( $file ) )
                ) {
                    return true;
                }
            }
        }

        return false;
    }
}

/**
 * Auto-desactivar Lite si la PRO está activa (solo admin).
 */
if ( is_admin() && telo_aff_pro_is_active() ) {
    if ( ! function_exists( 'deactivate_plugins' ) ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    deactivate_plugins( plugin_basename( __FILE__ ), false );

    add_action( 'admin_notices', function () {
        echo '<div class="notice notice-warning is-dismissible"><p>'
            . esc_html__( 'TELO Affiliates Lite se ha desactivado automáticamente porque se ha detectado la versión PRO activa.', 'telo-affiliates-lite' )
            . '</p></div>';
    } );

    return; // Evita registrar menús, hooks, etc.
}

/* =========================
 * 1) CONSTANTES (siempre primero)
 * ========================= */
if ( ! defined( 'TELO_AFFILIATES_IS_LITE' ) ) {
    define( 'TELO_AFFILIATES_IS_LITE', true );
}
if ( ! defined( 'TELO_AFFILIATES_VERSION' ) ) {
    define( 'TELO_AFFILIATES_VERSION', '3.0.1' );
}
if ( ! defined( 'TELO_AFFILIATES_FILE' ) ) {
    define( 'TELO_AFFILIATES_FILE', __FILE__ );
}
if ( ! defined( 'TELO_AFFILIATES_DIR' ) ) {
    define( 'TELO_AFFILIATES_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'TELO_AFFILIATES_URL' ) ) {
    define( 'TELO_AFFILIATES_URL', plugin_dir_url( __FILE__ ) );
}

// Fallbacks tracking usados también en admin
if ( ! defined( 'TELO_AFF_PARAM' ) ) {
    define( 'TELO_AFF_PARAM', 'pv' );
}
if ( ! defined( 'TELO_AFF_COOKIE' ) ) {
    define( 'TELO_AFF_COOKIE', 'telo_affiliate' );
}
if ( ! defined( 'TELO_AFF_COOKIE_DAYS' ) ) {
    define( 'TELO_AFF_COOKIE_DAYS', 30 );
}

/* =========================
 * 2) i18n (no hace falta cargar manualmente en .org)
 * ========================= */
// (Eliminado) load_plugin_textdomain(...) — Desaconsejado desde WP 4.6 para plugins alojados en .org

/* =========================
 * 3) Activación / Desactivación
 * ========================= */
register_activation_hook( __FILE__, function () {
    if ( ! class_exists( 'WooCommerce' ) ) {
        if ( ! function_exists( 'deactivate_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        deactivate_plugins( plugin_basename( __FILE__ ) );

        wp_die(
            __( 'TELO Affiliates requiere WooCommerce activo. Actívalo e inténtalo de nuevo.', 'telo-affiliates-lite' ),
            __( 'Requisito no cumplido', 'telo-affiliates-lite' ),
            array( 'back_link' => true )
        );
    }
} );

register_deactivation_hook( __FILE__, function () {
    // Limpiezas futuras (cron/transients) si procede
} );

/* =========================
 * 4) Licensing (desactivado en Lite)
 * ========================= */
if ( ! TELO_AFFILIATES_IS_LITE ) {
    $lic_file = TELO_AFFILIATES_DIR . 'includes/licensing.php';

    if ( file_exists( $lic_file ) ) {
        require_once $lic_file;
    } else {
        add_action( 'admin_notices', function () use ( $lic_file ) {
            echo '<div class="notice notice-error"><p>' .
                sprintf(
                    esc_html__( 'TELO Affiliates: falta el archivo de licencia: %s', 'telo-affiliates-lite' ),
                    esc_html( $lic_file )
                ) .
            '</p></div>';
        } );
        return;
    }
}

/* =========================
 * 5) Enlaces en la lista de plugins (Upgrade + Ajustes)
 * ========================= */
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function ( $links ) {
    // CTA a PRO
    $upgrade = '<a target="_blank" rel="noopener" href="https://licencias.qagencia.com/"><strong>' . esc_html__( 'Pásate a la versión PRO', 'telo-affiliates-lite' ) . '</strong></a>';
    array_unshift( $links, $upgrade );

    // Ajustes (pantalla principal del plugin)
    $settings_url = admin_url( 'admin.php?page=telo-affiliates-lite' );
    $settings     = '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Ajustes', 'telo-affiliates-lite' ) . '</a>';
    array_unshift( $links, $settings );

    return $links;
} );

/* =========================
 * 6) Gate de licencia (desactivado en Lite)
 * ========================= */
if ( ! TELO_AFFILIATES_IS_LITE && function_exists( 'telo_license_is_valid' ) && ! telo_license_is_valid() ) {
    // En PRO sin licencia válida, se corta la carga (la pantalla de licencia se gestiona en licensing.php)
    return;
}

/* =========================
 * 7) Compat Woo HPOS + aviso Woo faltante
 * ========================= */
add_action( 'before_woocommerce_init', function () {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );

add_action( 'admin_notices', function () {
    if ( ! class_exists( 'WooCommerce' ) ) {
        echo '<div class="notice notice-error"><p>' .
            esc_html__( 'TELO Affiliates requiere WooCommerce activo.', 'telo-affiliates-lite' ) .
        '</p></div>';
    }
} );

/**
 * Guard extra: si la PRO está activa, no seguimos cargando (por si acaso).
 */
if ( telo_aff_pro_is_active() ) {
    return;
}

/* =========================
 * 8) Includes del plugin
 * ========================= */
$includes = array(
    'includes/helpers.php',
    'includes/tracking.php',
    'includes/admin-list-table.php',
    'includes/admin.php',
    'includes/admin-pages.php',
    'includes/frontend.php',
    'includes/admin-requests.php',
    'includes/template-affiliate-page.php',
);

foreach ( $includes as $rel ) {
    $abs = TELO_AFFILIATES_DIR . $rel;
    if ( file_exists( $abs ) ) {
        require_once $abs;
    } else {
        add_action( 'admin_notices', function () use ( $rel ) {
            echo '<div class="notice notice-error"><p>' .
                sprintf(
                    esc_html__( 'TELO Affiliates: no se encontró el archivo requerido: %s', 'telo-affiliates-lite' ),
                    esc_html( $rel )
                ) .
            '</p></div>';
        } );
    }
}

/* =========================
 * 9) CSS admin (incluye pantallas bloqueadas)
 * ========================= */
add_action( 'admin_enqueue_scripts', function () {
    $pages = array(
        'telo-affiliates-lite',
        'telo-affiliates-pages',
        'telo-affiliates-import-export',
        'telo-affiliates-payments',
        'telo-affiliates-upgrade',
    );

    if ( isset( $_GET['page'] ) && in_array( sanitize_key( $_GET['page'] ), $pages, true ) ) {
        wp_enqueue_style(
            'telo-affiliates-admin',
            TELO_AFFILIATES_URL . 'assets/css/telo-affiliates.css',
            array(),
            TELO_AFFILIATES_VERSION
        );
    }
} );
