<?php
if (! defined('ABSPATH')) exit;

// 1) Constantes primero
if (! defined('TELO_AFF_PARAM'))          define('TELO_AFF_PARAM', 'pv');
if (! defined('TELO_AFF_COOKIE'))         define('TELO_AFF_COOKIE', 'telo_affiliate');
if (! defined('TELO_AFF_COOKIE_DAYS'))    define('TELO_AFF_COOKIE_DAYS', 30);

// Comprueba si un slug de afiliado existe en la opción guardada
function telo_affiliates_slug_exists($slug)
{
    $slug = sanitize_key($slug);
    $affiliates = get_option('telo_affiliates', array());
    return isset($affiliates[$slug]);
}

// Alias para compatibilidad: si llega ?partner= y no hay ?pv=, mapéalo a pv
if (
    TELO_AFF_PARAM !== 'partner'
    && empty($_GET[TELO_AFF_PARAM])
    && isset($_GET['partner'])
) {
    $partner_raw    = wp_unslash($_GET['partner']);
    $partner_sanit  = sanitize_key($partner_raw);

    if ($partner_sanit !== '') {
        $_GET[TELO_AFF_PARAM]     = $partner_sanit;
        $_REQUEST[TELO_AFF_PARAM] = $partner_sanit;
    }
}


// 2) Bypass de caché seguro cuando llega ?pv=...
if (isset($_GET[TELO_AFF_PARAM])) {
    if (! defined('DONOTCACHEPAGE')) define('DONOTCACHEPAGE', true);
    add_action('send_headers', function () {
        nocache_headers();
    }, 0);
}

// 3) Captura en init temprano
add_action('init', function () {
    if (empty($_GET[TELO_AFF_PARAM])) return;

    $partner = sanitize_key(wp_unslash($_GET[TELO_AFF_PARAM]));
    if ($partner === '') return;

    // Validación: solo si existe el slug
    if (! telo_affiliates_slug_exists($partner)) return;

    if (function_exists('WC') && WC()->session) {
        WC()->session->set(TELO_AFF_COOKIE, $partner);
    }

    if (function_exists('wc_setcookie')) {
        wc_setcookie(
            TELO_AFF_COOKIE,
            $partner,
            time() + (TELO_AFF_COOKIE_DAYS * DAY_IN_SECONDS),
            is_ssl(),
            true
        );
        $_COOKIE[TELO_AFF_COOKIE] = $partner;
    }

    // Cookie adicional en el apex para www↔no-www/subdominios
    $host = parse_url(home_url(), PHP_URL_HOST);
    if ($host) {
        $apex = '.' . preg_replace('/^www\./i', '', strtolower($host));
        @setcookie(
            TELO_AFF_COOKIE,
            $partner,
            time() + (TELO_AFF_COOKIE_DAYS * DAY_IN_SECONDS),
            '/',
            $apex,
            is_ssl(),
            true
        );
    }
}, 1);

// Rehidrata la sesión WC desde la cookie cuando Woo ya está inicializado
add_action('woocommerce_init', function () {
    if (function_exists('WC') && WC()->session && isset($_COOKIE[TELO_AFF_COOKIE])) {
        if (! WC()->session->get(TELO_AFF_COOKIE)) {
            WC()->session->set(TELO_AFF_COOKIE, sanitize_key(wp_unslash($_COOKIE[TELO_AFF_COOKIE])));
        }
    }
}, 1);

// Inserta hidden en checkout con el afiliado activo
add_action('woocommerce_after_order_notes', function () {
    $partner = telo_affiliates_current_partner();
    if ($partner) {
        echo '<input type="hidden" name="telo_affiliate" value="' . esc_attr($partner) . '">';
    }
});

// Helper unificado
function telo_affiliates_current_partner()
{
    $partner = '';

    // 1) Intentar desde sesión
    if (function_exists('WC') && WC()->session) {
        $partner = (string) WC()->session->get(TELO_AFF_COOKIE);
    }

    // 2) Fallback desde cookie
    if (! $partner && isset($_COOKIE[TELO_AFF_COOKIE])) {
        $partner = sanitize_key(wp_unslash($_COOKIE[TELO_AFF_COOKIE]));
    }

    // 3) Si el slug ya no existe, limpiar y devolver vacío
    if ($partner && ! telo_affiliates_slug_exists($partner)) {
        if (function_exists('WC') && WC()->session) {
            WC()->session->__unset(TELO_AFF_COOKIE);
        }
        if (function_exists('wc_setcookie')) {
            wc_setcookie(TELO_AFF_COOKIE, '', time() - DAY_IN_SECONDS, is_ssl(), true);
        }
        unset($_COOKIE[TELO_AFF_COOKIE]);

        $host = parse_url(home_url(), PHP_URL_HOST);
        if ($host) {
            $apex = '.' . preg_replace('/^www\./i', '', strtolower($host));
            @setcookie(TELO_AFF_COOKIE, '', time() - DAY_IN_SECONDS, '/', $apex, is_ssl(), true);
        }
        $partner = '';
    }

    return $partner;
}

/**
 * Guardar en el pedido
 */
add_action('woocommerce_checkout_create_order', function ($order) {
    $partner = '';
    if (! empty($_POST['telo_affiliate'])) {
        $partner = sanitize_key(wp_unslash($_POST['telo_affiliate']));
    }
    if (! $partner) {
        $partner = telo_affiliates_current_partner();
        if (! $partner && ! empty($_REQUEST[TELO_AFF_PARAM])) {
            $partner = sanitize_key(wp_unslash($_REQUEST[TELO_AFF_PARAM]));
        }
    }
    $order->update_meta_data('_telo_affiliate', $partner ? $partner : 'directo');
}, 10, 1);

add_action('woocommerce_new_order', function ($order_id) {
    $order = wc_get_order($order_id);
    if (! $order) return;

    $current = $order->get_meta('_telo_affiliate', true);
    if ($current && $current !== 'directo') return;

    $partner = telo_affiliates_current_partner();
    if (! $partner && ! empty($_REQUEST[TELO_AFF_PARAM])) {
        $partner = sanitize_key(wp_unslash($_REQUEST[TELO_AFF_PARAM]));
    }
    if ($partner) {
        $order->update_meta_data('_telo_affiliate', $partner);
        $order->save();
    }
}, 20);

/**
 * Decoración de URLs internas con ?pv=afiliado
 * Se puede desactivar por filtro: add_filter('telo_affiliates_enable_url_decoration', '__return_false');
 */
function telo_affiliates_decorate_url($url)
{
    // Permite desactivar globalmente
    if (false === apply_filters('telo_affiliates_enable_url_decoration', true)) return $url;

    // No tocar en admin ni en login
    if (is_admin()) return $url;
    $req_uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
    if ($req_uri && strpos($req_uri, 'wp-login.php') !== false) return $url;

    $partner = telo_affiliates_current_partner();
    if (! $partner || ! is_string($url) || $url === '') return $url;

    // Ya decorada
    if (strpos($url, TELO_AFF_PARAM . '=') !== false) return $url;

    // Ignorar anchors, vacíos, y esquemas no http(s)
    if ($url === '#' || preg_match('/^#.+$/', $url)) return $url;
    $parsed = wp_parse_url($url);
    $scheme = isset($parsed['scheme']) ? strtolower($parsed['scheme']) : '';
    if ($scheme && ! in_array($scheme, array('http', 'https'), true)) return $url;

    $path  = isset($parsed['path']) ? $parsed['path'] : '';
    $query = isset($parsed['query']) ? $parsed['query'] : '';

    // EXCLUSIONES: REST, AJAX, nonces, admin-ajax, wc-ajax
    if (strpos($url, '/wp-json/') !== false) return $url;
    if (strpos($query, 'rest_route=') !== false) return $url;
    if (strpos($query, 'wc-ajax=') !== false || strpos($path, '/wc-ajax/') !== false) return $url;
    if (strpos($path, 'admin-ajax.php') !== false) return $url;
    if (strpos($query, '_wpnonce=') !== false) return $url;

    // Opcional: no decorar ficheros
    if ($path && preg_match('/\.(?:pdf|zip|rar|7z|jpg|jpeg|png|gif|webp|svg|mp4|mp3|wav|docx?|xlsx?)$/i', $path)) {
        return $url;
    }

    // Solo mismo host (o relativo). Whitelist opcional más abajo.
    $current_host = isset($_SERVER['HTTP_HOST']) ? strtolower($_SERVER['HTTP_HOST']) : '';
    $target_host  = isset($parsed['host']) ? strtolower($parsed['host']) : '';
    $is_same_host = ($target_host === '' || $target_host === $current_host);

    // Whitelist opcional para cross-domain/subdominios:
    // $allowed = array('academy.tu-dominio.com', 'otraweb.com');
    // if ( $target_host && in_array($target_host, array_map('strtolower', $allowed), true) ) {
    //     $is_same_host = true;
    // }

    if ($is_same_host) {
        $url = add_query_arg(TELO_AFF_PARAM, $partner, $url);
    }

    return $url;
}

// Enlaces generados por WP/Woo y menús
add_filter('post_type_link', 'telo_affiliates_decorate_url');
add_filter('page_link', 'telo_affiliates_decorate_url');
add_filter('term_link', 'telo_affiliates_decorate_url');
add_filter('woocommerce_product_add_to_cart_url', 'telo_affiliates_decorate_url');

add_filter('nav_menu_link_attributes', function ($atts) {
    if (isset($atts['href'])) {
        $atts['href'] = telo_affiliates_decorate_url($atts['href']);
    }
    return $atts;
}, 10, 1);

// Decoración en contenido (al final para evitar duplicados), solo si está activo el toggle
add_filter('the_content', function ($content) {
    if (false === apply_filters('telo_affiliates_enable_url_decoration', true)) return $content;

    $partner = telo_affiliates_current_partner();
    if (! $partner) return $content;

    return preg_replace_callback('#href=("|\')(.*?)\1#i', function ($m) use ($partner) {
        $q = $m[1];
        $url = $m[2];
        if (strpos($url, TELO_AFF_PARAM . '=') !== false) return $m[0];

        $parsed = wp_parse_url($url);
        $current_host = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : '';
        $is_same_host =
            empty($parsed['host']) ||
            (! empty($parsed['host']) && $current_host && strtolower($parsed['host']) === strtolower($current_host));

        if ($is_same_host) {
            $url = add_query_arg(TELO_AFF_PARAM, $partner, $url);
        }
        return 'href=' . $q . esc_url($url) . $q;
    }, $content);
}, 20);

// Refuerzo JS: añade el parámetro de afiliado a los clics en enlaces del frontend
add_action('wp_footer', function () {
    if (is_admin() || wp_doing_ajax()) {
        return;
    }

    $current_partner = telo_affiliates_current_partner();
    if (! $current_partner) {
        // No hay afiliado activo: no añadimos JS extra
        return;
    }

    $q       = defined('TELO_AFF_PARAM') ? TELO_AFF_PARAM : 'pv';
    $partner = $q . '=' . $current_partner;

    wp_register_script(
        'telo-aff-tracking-inline',
        '',
        [],
        defined('TELO_AFFILIATES_VERSION') ? TELO_AFFILIATES_VERSION : false,
        true
    );
    wp_enqueue_script('telo-aff-tracking-inline');

    $partner_js = wp_json_encode($partner);
    $js         = "
    (function(){
      var partner = {$partner_js};
      var sameHost = location.host.toLowerCase();
      document.addEventListener('click', function(e){
        var a = e.target.closest('a[href]');
        if(!a) return;
        var href = a.getAttribute('href') || '';
        try {
          var u = new URL(href, document.baseURI);
          if (u.host.toLowerCase() === sameHost && !u.search.includes(partner)) {
            u.search += (u.search ? '&' : '?') + partner;
            a.setAttribute('href', u.toString());
          }
        } catch(err) {}
      });
    })();";
    wp_add_inline_script('telo-aff-tracking-inline', $js);
}, 99);
