<?php
if ( ! defined('ABSPATH') ) exit;

/**
 * ===========================================
 *  Identidad del producto (AISLADA)
 * ===========================================
 */
if ( ! defined('TELO_LICENSE_PLUGIN_SLUG') )
    define('TELO_LICENSE_PLUGIN_SLUG', 'telo-affiliates-lite');

if ( ! defined('TELO_LICENSE_PLUGIN_VERSION') )
    define('TELO_LICENSE_PLUGIN_VERSION', defined('TELO_AFFILIATES_VERSION') ? TELO_AFFILIATES_VERSION : 'dev');

if ( ! defined('TELO_LICENSE_TEXTDOMAIN') )
    define('TELO_LICENSE_TEXTDOMAIN', 'telo-affiliates-lite');

/**
 * ===========================================
 *  Constantes y opciones
 * ===========================================
 */
if ( ! defined('TELO_LICENSE_LAST_OK') )        define('TELO_LICENSE_LAST_OK',       'telo_license_last_ok_ts'); // último éxito real
if ( ! defined('TELO_LICENSE_OPTION_KEY') )     define('TELO_LICENSE_OPTION_KEY',    'telo_license_key');
if ( ! defined('TELO_LICENSE_OPTION_STATUS') )  define('TELO_LICENSE_OPTION_STATUS', 'telo_license_status'); // 'valid'|'invalid'
if ( ! defined('TELO_LICENSE_LAST_CHECK') )     define('TELO_LICENSE_LAST_CHECK',    'telo_license_last_check_ts');
if ( ! defined('TELO_LICENSE_REMOTE_STATUS') )  define('TELO_LICENSE_REMOTE_STATUS', 'telo_license_remote_status'); // 'active','suspended','revoked','expired','invalid','unknown'
if ( ! defined('TELO_LICENSE_EXPIRES_TS') )     define('TELO_LICENSE_EXPIRES_TS',    'telo_license_expires_ts');
if ( ! defined('TELO_LICENSE_LAST_ERROR') )     define('TELO_LICENSE_LAST_ERROR',    'telo_license_last_error');
if ( ! defined('TELO_LICENSE_CRON_HOOK') )      define('TELO_LICENSE_CRON_HOOK',     'telo_license_cron_check');

/**
 * ===========================================
 *  Helpers de zona horaria (mostrar en Madrid)
 *  — Guardamos SIEMPRE timestamps UTC —
 * ===========================================
 */
if ( ! function_exists('telo_license_tz') ) {
    function telo_license_tz() {
        static $tz = null;
        if ( $tz instanceof DateTimeZone ) return $tz;
        try {
            $tz = new DateTimeZone( apply_filters('telo_license_timezone', 'Europe/Madrid') );
        } catch ( Exception $e ) {
            $tz = new DateTimeZone('UTC');
        }
        return $tz;
    }
}

if ( ! function_exists('telo_license_format_dt') ) {
    /**
     * Formatea un timestamp UTC en la TZ configurada (Madrid por defecto).
     */
    function telo_license_format_dt( $ts, $with_time = true, $format = null ) {
        $ts = (int) $ts;
        if ( ! $ts ) return '—';

        if ( $format === null ) {
            $format = $with_time
                ? ( get_option('date_format') . ' ' . get_option('time_format') )
                : get_option('date_format');
        }

        if ( function_exists('wp_date') ) {
            return wp_date( $format, $ts, telo_license_tz() );
        }

        // Fallback WP antiguo:
        $dt = DateTime::createFromFormat('U', (string) $ts, new DateTimeZone('UTC'));
        if ( $dt ) { $dt->setTimezone( telo_license_tz() ); return $dt->format( $format ); }
        return date_i18n( $format, $ts );
    }
}

/**
 * ===========================================
 *  Parámetros (filtrables)
 * ===========================================
 */
function telo_license_check_every_admin()   { return (int) apply_filters('telo_license_check_every_admin',   30 * MINUTE_IN_SECONDS); }
function telo_license_check_every_valid()   { return (int) apply_filters('telo_license_check_every_valid',   30 * MINUTE_IN_SECONDS); }
function telo_license_check_every_invalid() { return (int) apply_filters('telo_license_check_every_invalid',  1 * HOUR_IN_SECONDS); }
function telo_license_grace_seconds()       { return (int) apply_filters('telo_license_grace_seconds',       72 * HOUR_IN_SECONDS); }
function telo_license_capability()          { return apply_filters('telo_license_capability', 'manage_options'); }
function telo_license_api_base()            { return apply_filters('telo_license_api_base', 'https://licencias.qagencia.com/wp-json/qa-license/v1'); }

/** Estado local rápido */
function telo_license_is_valid() {
    return get_option(TELO_LICENSE_OPTION_STATUS, 'invalid') === 'valid';
}

/**
 * ===========================================
 *  Menú de administración (pantalla SIEMPRE)
 * ===========================================
 */
add_action('admin_menu', function () {
    add_menu_page(
        __('Licencia TELO', 'telo-affiliates-lite'),
        __('Licencia TELO', 'telo-affiliates-lite'),
        telo_license_capability(),
        'telo-license',
        'telo_license_render_page',
        'dashicons-admin-network',
        56
    );

    if ( telo_license_is_valid() ) {
        add_submenu_page(
            'telo-affiliates-lite',
            __('Licencia', 'telo-affiliates-lite'),
            __('Licencia', 'telo-affiliates-lite'),
            telo_license_capability(),
            'telo-license',
            'telo_license_render_page',
            99
        );
        remove_menu_page('telo-license');
    }
}, 20);

/**
 * ===========================================
 *  Guardado / activación / desactivación
 * ===========================================
 */
add_action('admin_init', function () {
    if ( ! current_user_can( telo_license_capability() ) ) return;

    // Guardar y (re)activar SIEMPRE (aunque la clave no cambie)
    if ( isset($_POST['telo_save_license']) ) {
        check_admin_referer('telo_license_save', 'telo_license_nonce');

        $key_raw = isset($_POST['telo_license_key']) ? wp_unslash($_POST['telo_license_key']) : '';
        $key     = trim( sanitize_text_field( $key_raw ) );
        $prev    = trim( get_option(TELO_LICENSE_OPTION_KEY, '') );
        $changed = ($key !== $prev);

        // Guardamos la clave (aunque sea igual)
        update_option(TELO_LICENSE_OPTION_KEY, $key);

        // Si no hay clave → inválida inmediata
        if ( $key === '' ) {
            update_option(TELO_LICENSE_OPTION_STATUS, 'invalid');
            update_option(TELO_LICENSE_REMOTE_STATUS, 'invalid');
            wp_safe_redirect( admin_url('admin.php?page=telo-license&error=1') );
            exit;
        }

        // Reset mínimos si cambió la clave
        if ( $changed ) {
            update_option(TELO_LICENSE_LAST_OK,          0);
            update_option(TELO_LICENSE_LAST_CHECK,       0);
            update_option(TELO_LICENSE_EXPIRES_TS,       0);
            update_option(TELO_LICENSE_OPTION_STATUS,   'invalid');
            update_option(TELO_LICENSE_REMOTE_STATUS,   'unknown');
            update_option(TELO_LICENSE_LAST_ERROR,      '');
        }

        $ok = false;

        // 1) Intentar ACTIVAR explícitamente SIEMPRE
        $res = telo_license_remote_activate($key);

        if ( ! is_wp_error($res) && is_array($res) ) {
            $status     = isset($res['status']) ? strtolower( (string) $res['status'] ) : 'invalid';
            $expires_ts = (int) ( $res['expires_at'] ?? 0 );

            if ( $status === 'active' ) {
                $now = time();
                update_option(TELO_LICENSE_REMOTE_STATUS, 'active');
                update_option(TELO_LICENSE_EXPIRES_TS,    $expires_ts);
                update_option(TELO_LICENSE_OPTION_STATUS, 'valid');
                update_option(TELO_LICENSE_LAST_OK,       $now);
                update_option(TELO_LICENSE_LAST_CHECK,    $now);
                update_option(TELO_LICENSE_LAST_ERROR,    '');
                $ok = true;
            } else {
                // Guardamos estado remoto (expired / suspended / activation_limit / invalid, etc.)
                update_option(TELO_LICENSE_REMOTE_STATUS, $status);
                update_option(TELO_LICENSE_EXPIRES_TS,    $expires_ts);
                update_option(TELO_LICENSE_OPTION_STATUS, 'invalid');
            }
        } elseif ( is_wp_error($res) ) {
            // Rate limit → no marcamos error ni invalidamos; reintento pronto
            if ( $res->get_error_code() === 'rate_limited' ) {
                update_option(TELO_LICENSE_LAST_ERROR, '');
                update_option(TELO_LICENSE_LAST_CHECK, 0);
            } else {
                // Otro error de red → lo anotamos
                update_option(TELO_LICENSE_LAST_ERROR, $res->get_error_message() );
            }
        }

        // 2) Si no quedó activa, hacemos VALIDATE como respaldo (no re-activa por sí solo)
        if ( ! $ok ) {
            $ok = telo_license_refresh(true, 'admin');
        }

        wp_safe_redirect( admin_url('admin.php?page=telo-license' . ($ok ? '&updated=1' : '&error=1')) );
        exit;
    }

    // Forzar comprobación: primero intenta ACTIVAR; si no, VALIDATE
    if ( isset($_POST['telo_force_check']) && $_POST['telo_force_check'] === '1' ) {
        $key = trim( get_option(TELO_LICENSE_OPTION_KEY, '') );

        $ok = false;
        if ( $key !== '' ) {
            $res = telo_license_remote_activate($key);
            if ( ! is_wp_error($res) && is_array($res) ) {
                $status     = isset($res['status']) ? strtolower( (string) $res['status'] ) : 'invalid';
                $expires_ts = (int) ( $res['expires_at'] ?? 0 );

                if ( $status === 'active' ) {
                    $now = time();
                    update_option(TELO_LICENSE_REMOTE_STATUS, 'active');
                    update_option(TELO_LICENSE_EXPIRES_TS,    $expires_ts);
                    update_option(TELO_LICENSE_OPTION_STATUS, 'valid');
                    update_option(TELO_LICENSE_LAST_OK,       $now);
                    update_option(TELO_LICENSE_LAST_CHECK,    $now);
                    update_option(TELO_LICENSE_LAST_ERROR,    '');
                    $ok = true;
                } else {
                    update_option(TELO_LICENSE_REMOTE_STATUS, $status);
                    update_option(TELO_LICENSE_EXPIRES_TS,    $expires_ts);
                    update_option(TELO_LICENSE_OPTION_STATUS, 'invalid');
                }
            } elseif ( is_wp_error($res) && $res->get_error_code() === 'rate_limited' ) {
                update_option(TELO_LICENSE_LAST_ERROR, '');
                update_option(TELO_LICENSE_LAST_CHECK, 0);
            }
        }

        if ( ! $ok ) {
            $ok = telo_license_refresh(true, 'admin');
        }

        wp_safe_redirect( admin_url('admin.php?page=telo-license' . ($ok ? '&checked=1' : '&error=1')) );
        exit;
    }

    // Desactivar localmente (y avisar al servidor)
    if ( isset($_POST['telo_deactivate_license']) ) {
        check_admin_referer('telo_license_deactivate', 'telo_license_deactivate_nonce');
        $key = get_option(TELO_LICENSE_OPTION_KEY, '');
        if ( $key ) telo_license_remote_deactivate($key);
        update_option(TELO_LICENSE_OPTION_STATUS, 'invalid');
        wp_safe_redirect( admin_url('admin.php?page=telo-license&deactivated=1') );
        exit;
    }
});


/**
 * ===========================================
 *  Heartbeat + Cron
 * ===========================================
 */
add_action('init', function () {
    if ( ! wp_next_scheduled( TELO_LICENSE_CRON_HOOK ) ) {
        wp_schedule_event( time() + wp_rand(300, 1800), 'twicedaily', TELO_LICENSE_CRON_HOOK );
    }
});
add_action( TELO_LICENSE_CRON_HOOK, function () {
    telo_license_refresh(true, 'cron');
});

// Admin heartbeat
add_action('admin_init', function () {
    $prev_status = get_option(TELO_LICENSE_OPTION_STATUS, 'invalid');

    telo_license_maybe_refresh('admin');

    $now_status = get_option(TELO_LICENSE_OPTION_STATUS, 'invalid');
    if ( $prev_status === 'valid' && $now_status !== 'valid' ) {
        $page = isset($_GET['page']) ? sanitize_key($_GET['page']) : '';
        if ( $page && strpos($page, 'telo-affiliates-lite') === 0 ) {
            wp_safe_redirect( admin_url('admin.php?page=telo-license') );
            exit;
        }
    }
});

// Frontend heartbeat más laxo
add_action('init', function () {
    if ( is_admin() ) return;
    telo_license_maybe_refresh('frontend');
});

function telo_license_maybe_refresh( $context = 'frontend' ) {
    $last  = (int) get_option(TELO_LICENSE_LAST_CHECK, 0);
    $valid = telo_license_is_valid();

    $interval = $valid ? telo_license_check_every_valid() : telo_license_check_every_invalid();
    if ( $context === 'admin' ) {
        $interval = min( $interval, telo_license_check_every_admin() );
    }

    if ( $last === 0 || ( time() - $last ) >= $interval ) {
        telo_license_refresh(false, $context);
    }
}

/**
 * ===========================================
 *  Verificación con servidor (gracia solo red)
 * ===========================================
 */
function telo_license_refresh( $force = false, $context = 'general' ) {
    $key = get_option(TELO_LICENSE_OPTION_KEY, '');
    $now = time();

    if ( $key === '' ) {
        update_option(TELO_LICENSE_OPTION_STATUS, 'invalid');
        update_option(TELO_LICENSE_REMOTE_STATUS, 'invalid');
        update_option(TELO_LICENSE_LAST_CHECK, $now);
        return false;
    }

    // Throttle
    if ( ! $force ) {
        $last    = (int) get_option(TELO_LICENSE_LAST_CHECK, 0);
        $valid   = telo_license_is_valid();
        $min_gap = $valid ? telo_license_check_every_valid() : telo_license_check_every_invalid();
        if ( $context === 'admin' ) $min_gap = min( $min_gap, telo_license_check_every_admin() );
        if ( $last && ( $now - $last ) < $min_gap ) {
            return $valid;
        }
    }

    $resp = telo_license_remote_check( $key );

    if ( is_wp_error($resp) ) {
        $code = $resp->get_error_code();

        // 429: no mostrar error ni cambiar estado; reintento temprano
        if ( $code === 'rate_limited' ) {
            update_option(TELO_LICENSE_LAST_ERROR, '');
            update_option(TELO_LICENSE_LAST_CHECK, 0); // próximo intento rápido
            return telo_license_is_valid();
        }

        // Error de red real → gracia si estaba válida
        update_option(TELO_LICENSE_LAST_ERROR, $resp->get_error_message());

        if ( telo_license_is_valid() ) {
            $last_ok  = (int) get_option(TELO_LICENSE_LAST_OK, 0);
            $in_grace = $last_ok && ( ( $now - $last_ok ) <= telo_license_grace_seconds() );
            update_option(TELO_LICENSE_LAST_CHECK, $now);
            return $in_grace;
        }

        update_option(TELO_LICENSE_LAST_CHECK, $now);
        return false;
    }

    // Respuesta válida (200/403 con JSON)
    $status     = isset($resp['status']) ? strtolower( (string) $resp['status'] ) : 'invalid';
    $expires_ts = (int) ( $resp['expires_at'] ?? 0 );

    update_option(TELO_LICENSE_REMOTE_STATUS, $status);
    update_option(TELO_LICENSE_EXPIRES_TS,    $expires_ts);
    update_option(TELO_LICENSE_LAST_ERROR,    '');

    if ( $status === 'active' ) {
        update_option(TELO_LICENSE_OPTION_STATUS, 'valid' );
        update_option(TELO_LICENSE_LAST_OK,       $now );
        update_option(TELO_LICENSE_LAST_CHECK,    $now );
        return true;
    }

    // Cualquier otro estado remoto → inválido inmediato
    update_option(TELO_LICENSE_OPTION_STATUS, 'invalid' );
    update_option(TELO_LICENSE_LAST_CHECK,    $now );
    return false;
}

/**
 * ===========================================
 *  Utilidades “cliente”
 * ===========================================
 */
// Dominio sin esquema/path
function telo_license_normalize_site() {
    $host = parse_url( home_url(), PHP_URL_HOST );
    return $host ?: home_url();
}

// Hash estable por instalación **y por producto**
function telo_license_site_hash() {
    // opción por producto, no global
    $opt_key = 'telo_license_site_hash_' . TELO_LICENSE_PLUGIN_SLUG;
    $hash = get_option($opt_key, '');
    if ( ! $hash ) {
        // incluye el slug del producto en el hash
        $data = home_url() . '|' . wp_salt('auth') . '|' . TELO_LICENSE_PLUGIN_SLUG;
        $hash = substr( hash('sha256', $data), 0, 32 );
        update_option($opt_key, $hash, false);
    }
    return $hash;
}

/** Petición a /validate (devuelve 200|403; 429=rate limit) */
function telo_license_remote_check( $key ) {
    $endpoint = trailingslashit( telo_license_api_base() ) . 'validate';

    $payload = array(
        'key'            => (string) $key,
        'domain'         => telo_license_normalize_site(),
        'site_hash'      => telo_license_site_hash(),
        'plugin_slug'    => TELO_LICENSE_PLUGIN_SLUG,
        'plugin_version' => TELO_LICENSE_PLUGIN_VERSION,
    );

    $r = wp_remote_post( $endpoint, array(
        'timeout' => 7,
        'headers' => array('Accept'=>'application/json'),
        'body'    => $payload,
    ) );

    if ( is_wp_error($r) ) return $r;

    $code = (int) wp_remote_retrieve_response_code($r);
    $raw  = wp_remote_retrieve_body($r);

    // 429 primero
    if ( $code === 429 ) {
        return new WP_Error('rate_limited', 'El servidor de licencias está limitando la frecuencia (429).');
    }
    // Aceptamos 200 (válida) o 403 (no válida). Cualquier otro → error.
    if ( $code !== 200 && $code !== 403 ) {
        return new WP_Error('bad_response', 'HTTP ' . $code . ' al validar licencia');
    }

    $body = json_decode( $raw, true );
    if ( ! is_array($body) ) {
        return new WP_Error('bad_response', 'JSON inesperado del servidor de licencias');
    }

    // Normalizar estado y expiración
    $status = strtolower( (string) ( $body['status'] ?? 'invalid' ) );

    $expires_ts = 0;
    if ( isset($body['expires_at']) ) {
        $expires_ts = is_numeric($body['expires_at'])
            ? (int) $body['expires_at']
            : ( ( $t = strtotime( (string) $body['expires_at'] ) ) ? $t : 0 );
    }

    return array(
        'status'     => $status,     // active|expired|suspended|revoked|invalid
        'expires_at' => $expires_ts, // timestamp (0 si no aplica)
        'ok'         => ! empty($body['ok']),
    );
}

/** /activate (similar tratamiento de códigos) */
function telo_license_remote_activate( $key ) {
    $endpoint = trailingslashit( telo_license_api_base() ) . 'activate';

    $payload = array(
        'key'            => (string) $key,
        'domain'         => telo_license_normalize_site(),
        'site_hash'      => telo_license_site_hash(),
        'plugin_slug'    => TELO_LICENSE_PLUGIN_SLUG,
        'plugin_version' => TELO_LICENSE_PLUGIN_VERSION,
    );

    $r = wp_remote_post( $endpoint, array(
        'timeout' => 7,
        'headers' => array('Accept'=>'application/json'),
        'body'    => $payload,
    ) );

    if ( is_wp_error($r) ) return $r;

    $code = (int) wp_remote_retrieve_response_code($r);
    $raw  = wp_remote_retrieve_body($r);

    if ( $code === 429 ) {
        return new WP_Error('rate_limited', 'El servidor de licencias está limitando la frecuencia (429).');
    }
    if ( $code !== 200 && $code !== 403 ) {
        return new WP_Error('bad_response', 'Respuesta inesperada en /activate (HTTP ' . $code . ')');
    }

    $body = json_decode($raw, true);
    if ( ! is_array($body) ) {
        return new WP_Error('bad_response', 'JSON inesperado en /activate');
    }

    $status     = strtolower( (string) ( $body['status'] ?? 'invalid' ) );
    $expires_ts = 0;
    if ( isset($body['expires_at']) ) {
        $expires_ts = is_numeric($body['expires_at'])
            ? (int) $body['expires_at']
            : ( ( $t = strtotime( (string) $body['expires_at'] ) ) ? $t : 0 );
    }

    return array(
        'status'     => $status,
        'expires_at' => $expires_ts,
        'ok'         => ! empty($body['ok']),
    );
}

/** /deactivate (ahora incluye plugin_slug) */
function telo_license_remote_deactivate( $key ) {
    $endpoint = trailingslashit( telo_license_api_base() ) . 'deactivate';

    $payload = array(
        'key'            => (string) $key,
        'domain'         => telo_license_normalize_site(),
        'site_hash'      => telo_license_site_hash(),
        'plugin_slug'    => TELO_LICENSE_PLUGIN_SLUG,
        'plugin_version' => TELO_LICENSE_PLUGIN_VERSION,
    );

    return wp_remote_post( $endpoint, array(
        'timeout' => 7,
        'headers' => array('Accept'=>'application/json'),
        'body'    => $payload,
    ) );
}

/**
 * ===========================================
 *  Pantalla de licencia (mostrar en Madrid)
 * ===========================================
 */
function telo_license_render_page() {
    if ( ! current_user_can( telo_license_capability() ) ) {
        wp_die( esc_html__('No tienes permisos suficientes.', 'telo-affiliates-lite') );
    }

    $key      = get_option(TELO_LICENSE_OPTION_KEY, '');
    $status   = get_option(TELO_LICENSE_OPTION_STATUS, 'invalid');
    $remote   = get_option(TELO_LICENSE_REMOTE_STATUS, 'invalid');
    $expires  = (int) get_option(TELO_LICENSE_EXPIRES_TS, 0);
    $last     = (int) get_option(TELO_LICENSE_LAST_CHECK, 0);
    $last_err = get_option(TELO_LICENSE_LAST_ERROR, '');
    $is_ok    = ( $status === 'valid' );

    if ( isset($_GET['updated']) )     echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Licencia guardada.', 'telo-affiliates-lite') . '</p></div>';
    if ( isset($_GET['error']) )       echo '<div class="notice notice-error is-dismissible"><p>'   . esc_html__('La licencia no es válida o ha expirado.', 'telo-affiliates-lite') . '</p></div>';
    if ( isset($_GET['deactivated']) ) echo '<div class="notice notice-info is-dismissible"><p>'    . esc_html__('Licencia desactivada.', 'telo-affiliates-lite') . '</p></div>';
    if ( isset($_GET['checked']) )     echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Comprobación realizada.', 'telo-affiliates-lite') . '</p></div>';

    echo '<div class="wrap"><h1>' . esc_html__('Licencia — TELO Affiliates', 'telo-affiliates-lite') . '</h1>';
    echo '<div class="card" style="max-width:780px;padding:16px 20px;">';

    if ( $last_err ) {
        echo '<div class="notice notice-warning is-dismissible"><p>' .
             esc_html__('No se pudo contactar con el servidor de licencias. Se aplicará un periodo de gracia.', 'telo-affiliates-lite') .
             ' <em>(' . esc_html($last_err) . ')</em></p></div>';
    }

    echo '<form method="post" style="margin-top:12px;">';
    wp_nonce_field('telo_license_save', 'telo_license_nonce');

    echo '<table class="form-table" role="presentation"><tbody>';

    // Clave
    echo '<tr><th scope="row"><label for="telo_license_key">' . esc_html__('Clave de licencia', 'telo-affiliates-lite') . '</label></th><td>';
    echo '<input name="telo_license_key" id="telo_license_key" type="text" class="regular-text" value="' . esc_attr($key) . '" placeholder="XXXX-XXXX-XXXX-XXXX" style="width:100%" />';
    echo '</td></tr>';

    // Estado + expira + última verificación
    echo '<tr><th scope="row">' . esc_html__('Estado', 'telo-affiliates-lite') . '</th><td>';
    if ( $is_ok ) {
        echo '<span class="dashicons dashicons-yes-alt" style="color:#22a55b;"></span> <strong>' . esc_html__('Activada', 'telo-affiliates-lite') . '</strong>';
    } else {
        $label = ( $remote && $remote !== 'invalid' ) ? $remote : __('No activada', 'telo-affiliates-lite');
        echo '<span class="dashicons dashicons-warning" style="color:#d63638;"></span> <strong>' . esc_html( ucfirst($label) ) . '</strong>';
    }
    if ( $expires ) {
        echo ' &nbsp; <span style="color:#555;">(' . esc_html( telo_license_format_dt($expires) ) . ')</span>';
    }
    if ( $last ) {
        echo '<div style="color:#777;margin-top:4px;">' . esc_html__('Última verificación:', 'telo-affiliates-lite') . ' ' .
             esc_html( telo_license_format_dt($last) ) . '</div>';
    }
    echo '</td></tr>';

    echo '</tbody></table>';

    echo '<p class="submit">';
    echo '<button type="submit" name="telo_save_license" class="button button-primary">' . esc_html__('Guardar y activar', 'telo-affiliates-lite') . '</button> ';
    echo '<button type="submit" name="telo_force_check" value="1" class="button">' . esc_html__('Forzar comprobación', 'telo-affiliates-lite') . '</button>';
    echo '</p>';
    echo '</form>';

    if ( $is_ok ) {
        echo '<hr><form method="post">';
        wp_nonce_field('telo_license_deactivate', 'telo_license_deactivate_nonce');
        echo '<p><button type="submit" name="telo_deactivate_license" class="button">' . esc_html__('Desactivar licencia en este sitio', 'telo-affiliates-lite') . '</button></p>';
        echo '</form>';
    }

    echo '</div></div>';
}

/**
 * ===========================================
 *  Aviso global si no hay licencia válida
 * ===========================================
 */
add_action('admin_notices', function () {
    if ( ! current_user_can( telo_license_capability() ) ) return;
    if ( telo_license_is_valid() ) return;

    $screen = function_exists('get_current_screen') ? get_current_screen() : null;
    if ( $screen && $screen->id === 'toplevel_page_telo-license' ) return;

    echo '<div class="notice notice-error"><p>' .
        sprintf(
            esc_html__('TELO Affiliates requiere una licencia válida. Ve a %s para activarla.', 'telo-affiliates-lite'),
            '<a href="' . esc_url( admin_url('admin.php?page=telo-license') ) . '">' . esc_html__('Licencia', 'telo-affiliates-lite') . '</a>'
        ) .
        '</p></div>';
});
