<?php
if ( ! defined('ABSPATH') ) exit;

/**
 * Zona horaria por defecto para el plugin.
 */
if ( ! defined('TELO_AFF_TZ') ) {
    define('TELO_AFF_TZ', 'Europe/Madrid');
}

/**
 * ¿Lite?
 */
if ( ! function_exists('telo_aff_is_lite') ) {
    function telo_aff_is_lite() {
        return defined('TELO_AFFILIATES_IS_LITE') && TELO_AFFILIATES_IS_LITE;
    }
}

/**
 * Devuelve un objeto DateTimeZone fiable.
 */
if ( ! function_exists('telo_affiliates_timezone') ) {
    function telo_affiliates_timezone() {
        // 1) Constante del plugin
        if ( defined('TELO_AFF_TZ') && TELO_AFF_TZ ) {
            try { return new DateTimeZone(TELO_AFF_TZ); } catch (\Exception $e) {}
        }
        // 2) Ajuste del sitio
        if ( function_exists('wp_timezone') ) {
            try { return wp_timezone(); } catch (\Exception $e) {}
        }
        // 3) UTC
        return new DateTimeZone('UTC');
    }
}

/**
 * Formatea un timestamp según TZ del plugin.
 */
if ( ! function_exists('telo_affiliates_format_datetime') ) {
    function telo_affiliates_format_datetime($timestamp, $format = '') {
        $timestamp = (int) $timestamp;
        if ( ! $timestamp ) return '—';
        if ( ! $format ) {
            $format = get_option('date_format') . ' ' . get_option('time_format');
        }
        $tz = telo_affiliates_timezone();
        if ( function_exists('wp_date') ) {
            return wp_date($format, $timestamp, $tz);
        }
        try {
            $dt = new DateTime('@' . $timestamp);
            $dt->setTimezone($tz);
            return $dt->format($format);
        } catch (\Exception $e) {
            return date_i18n($format, $timestamp);
        }
    }
}

/**
 * Normaliza slug (quita NBSP/ZWSP/BOM y comillas tipográficas)
 */
if ( ! function_exists('telo_affiliates_normalize_slug') ) {
    function telo_affiliates_normalize_slug($s) {
        $s = is_string($s) ? $s : '';
        $s = preg_replace('/[\x{00A0}\x{200B}\x{200C}\x{200D}\x{FEFF}]/u', '', $s);
        $s = trim($s);
        $s = trim($s, "\"'“”‘’");
        return sanitize_key($s);
    }
}

/**
 * Busca un afiliado de forma tolerante y devuelve [slug_guardado, datos, slug_canonico]
 */
if ( ! function_exists('telo_affiliates_find_by_slug') ) {
    function telo_affiliates_find_by_slug($input_slug) {
        $affs = (array) get_option('telo_affiliates', []);
        $key  = telo_affiliates_normalize_slug($input_slug);
        if ( ! $key || empty($affs) ) return [null, null, null];

        // 1) Coincidencia directa
        if ( isset($affs[$key]) ) return [$key, $affs[$key], $key];

        // 2) Coincidencia tolerante
        foreach ($affs as $stored_key => $row) {
            if ( sanitize_key($stored_key) === $key ) {
                return [$stored_key, $row, sanitize_key($stored_key)];
            }
            if ( ! empty($row['slug']) && sanitize_key($row['slug']) === $key ) {
                return [$stored_key, $row, sanitize_key($row['slug'])];
            }
        }
        return [null, null, $key];
    }
}

/**
 * ¿WooCommerce activo?
 */
if ( ! function_exists('telo_aff_has_wc') ) {
    function telo_aff_has_wc() {
        return class_exists('WooCommerce') || defined('WC_VERSION');
    }
}

/**
 * Formateo de precio seguro si no está WooCommerce
 */
if ( ! function_exists('telo_aff_price') ) {
    function telo_aff_price($amount) {
        if ( function_exists('wc_price') ) return wc_price($amount);
        $currency = get_option('woocommerce_currency', 'EUR');
        return esc_html( number_format_i18n( (float) $amount, 2 ) ) . ' ' . esc_html($currency);
    }
}
