<?php
if (! defined('ABSPATH')) exit;

// Shortcode: [telo_affiliate_dashboard slug="juan"]
add_shortcode('telo_affiliate_dashboard', function ($atts) {
    if (! defined('DONOTCACHEPAGE')) define('DONOTCACHEPAGE', true);

    $atts = shortcode_atts(['slug' => ''], $atts, 'telo_affiliate_dashboard');

    $IS_LITE = function_exists('telo_aff_is_lite')
        ? telo_aff_is_lite()
        : (defined('TELO_AFFILIATES_IS_LITE') && TELO_AFFILIATES_IS_LITE);

    if (! telo_aff_has_wc()) {
        return '<div class="telo-aff-frontend telo-aff-no-wc">'
            . esc_html__('Este panel requiere WooCommerce activo para mostrar pedidos y comisiones.', 'telo-affiliates-lite')
            . '</div>';
    }

    $slug        = sanitize_key($atts['slug']);
    $affiliates  = get_option('telo_affiliates', array());

    if (! $slug || ! isset($affiliates[$slug])) {
        return '<p>' . esc_html__('Afiliado no encontrado.', 'telo-affiliates-lite') . '</p>';
    }

    // Encola CSS sólo si se usa el shortcode
    wp_enqueue_style(
        'telo-affiliates-frontend',
        TELO_AFFILIATES_URL . 'assets/css/telo-affiliates.css',
        [],
        defined('TELO_AFFILIATES_VERSION') ? TELO_AFFILIATES_VERSION : '1.0.0'
    );

    $data = $affiliates[$slug];
    $name = isset($data['name'])  ? $data['name']  : $slug;
    $rate = isset($data['rate'])  ? (float) $data['rate'] : 0.0;
    $mail = isset($data['email']) ? sanitize_email($data['email']) : '';
    $link = home_url(add_query_arg(array(TELO_AFF_PARAM => $slug), '/'));

    // --- Pedidos del afiliado (processing/completed) ---
    $orders_ids = function_exists('wc_get_orders') ? wc_get_orders(array(
        'type'       => 'shop_order',
        'limit'      => -1,
        'status'     => array('processing', 'completed'),
        'return'     => 'ids',
        'meta_key'   => '_telo_affiliate',
        'meta_value' => $slug,
    )) : array();

    $total_sales  = 0.0;
    $order_count  = 0;
    $orders       = array();
    foreach ((array) $orders_ids as $oid) {
        $o = wc_get_order($oid);
        if (! $o) continue;
        $orders[] = $o;
        $total_sales += (float) $o->get_total();
        $order_count++;
    }
    $commission_total = $total_sales * ($rate / 100);

    // Pendientes de pago (no marcados como _telo_aff_paid = yes)
    $pending_order_ids = array();
    $pending_base      = 0.0;

    foreach ($orders as $o) {
        if ($o->get_meta('_telo_aff_paid', true) !== 'yes') {
            $pending_order_ids[] = $o->get_id();
            $pending_base       += (float) $o->get_total();
        }
    }

    $pending_commission = $pending_base * ($rate / 100);

    // Mínimo para solicitar cobro (filtrable)
    $min_payout = (float) apply_filters('telo_affiliates_min_payout', 20.0);

    // Solicitudes previas (para bloquear si ya hay una pendiente)
    $requests = get_option('telo_affiliates_payout_requests', array());
    if (! is_array($requests)) $requests = array();

    $has_pending_request = false;
    foreach ($requests as $rq) {
        if (! empty($rq['slug']) && $rq['slug'] === $slug && ($rq['status'] ?? '') === 'pending') {
            $has_pending_request = true;
            break;
        }
    }

    // --- Flash messages ---
    $flash = array('type' => '', 'msg' => '');

    // --- Procesar "Solicitar cobro"
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['telo_aff_request'], $_POST['telo_aff_slug'])) {

        // En Lite NO procesamos nunca la solicitud
        if ($IS_LITE) {
            $flash = array(
                'type' => 'error',
                'msg'  => __('Esta acción está disponible en la versión PRO.', 'telo-affiliates-lite')
            );
        } else {
            $posted_slug = sanitize_key(wp_unslash($_POST['telo_aff_slug']));

            if ($posted_slug === $slug && check_admin_referer('telo_aff_request_' . $slug, 'telo_aff_nonce')) {

                if ($has_pending_request) {
                    $flash = array('type' => 'error', 'msg' => __('Ya existe una solicitud pendiente. Te avisaremos cuando se procese.', 'telo-affiliates-lite'));
                } elseif ($pending_commission < $min_payout) {
                    $flash = array('type' => 'error', 'msg' => sprintf(
                        __('No alcanzas el mínimo para solicitar cobro. Tienes %1$s y el mínimo es %2$s.', 'telo-affiliates-lite'),
                        function_exists('wc_price') ? wc_price($pending_commission) : number_format_i18n($pending_commission, 2),
                        function_exists('wc_price') ? wc_price($min_payout)       : number_format_i18n($min_payout, 2)
                    ));
                } else {
                    // Construir y guardar solicitud
                    $req = array(
                        'id'           => uniqid('payout_', true),
                        'slug'         => $slug,
                        'name'         => $name,
                        'email'        => $mail,
                        'rate'         => $rate,
                        'orders'       => array_map('absint', $pending_order_ids),
                        'base_total'   => round((float)$pending_base,  2),
                        'amount'       => round((float)$pending_commission, 2),
                        'timestamp'    => time(),
                        'status'       => 'pending',
                        'note'         => '',
                    );
                    $requests[] = $req;
                    update_option('telo_affiliates_payout_requests', $requests, false);

                    // Email a admin
                    $to      = get_option('admin_email');
                    $subject = sprintf(__('Nueva solicitud de cobro de afiliado: %s', 'telo-affiliates-lite'), $name);
                    $body    = sprintf(
                        '<p><strong>%1$s</strong></p>
                         <p>%2$s: %3$s<br>%4$s: %5$s<br>%6$s: %7$s%%</p>
                         <p>%8$s: %9$s<br>%10$s: %11$s</p>
                         <p>%12$s: %13$s</p>',
                        esc_html__('Hay una nueva solicitud de cobro pendiente.', 'telo-affiliates-lite'),
                        esc_html__('Afiliado', 'telo-affiliates-lite'),
                        esc_html($name) . ' (' . esc_html($slug) . ')',
                        esc_html__('Email', 'telo-affiliates-lite'),
                        esc_html($mail ?: '—'),
                        esc_html__('% comisión', 'telo-affiliates-lite'),
                        esc_html(number_format_i18n($rate, 2)),
                        esc_html__('Base pendiente', 'telo-affiliates-lite'),
                        function_exists('wc_price') ? wp_kses_post(wc_price($pending_base))       : number_format_i18n($pending_base, 2),
                        esc_html__('Comisión solicitada', 'telo-affiliates-lite'),
                        function_exists('wc_price') ? wp_kses_post(wc_price($pending_commission)) : number_format_i18n($pending_commission, 2),
                        esc_html__('Pedidos incluidos', 'telo-affiliates-lite'),
                        $pending_order_ids ? ('#' . esc_html(implode(', #', array_map('absint', $pending_order_ids)))) : '—'
                    );
                    $headers = array('Content-Type: text/html; charset=UTF-8');
                    wp_mail($to, $subject, $body, $headers);

                    $has_pending_request = true;
                    $flash = array('type' => 'success', 'msg' => __('¡Solicitud enviada! Te avisaremos cuando se procese el pago.', 'telo-affiliates-lite'));
                }
            } else {
                $flash = array('type' => 'error', 'msg' => __('Solicitud no válida. Recarga la página e inténtalo de nuevo.', 'telo-affiliates-lite'));
            }
        }
    }

    // Historial de pagos del afiliado
    $all_payouts = get_option('telo_affiliates_payouts', array());
    $payouts = array();
    if (is_array($all_payouts)) {
        foreach ($all_payouts as $p) {
            if (isset($p['slug']) && $p['slug'] === $slug) $payouts[] = $p;
        }
        usort($payouts, function ($a, $b) {
            return ($b['timestamp'] ?? 0) <=> ($a['timestamp'] ?? 0);
        });
    }

    // --- Interfaz ---
    ob_start();
?>
    <div class="telo-affiliate-dashboard telo-aff-frontend" style="padding:20px;">
        <div class="tituloPanelDetalle">
            <h2><?php echo esc_html__('Panel de Afiliado', 'telo-affiliates-lite'); ?>: <?php echo esc_html($name); ?></h2>
            <div class="poweredBy">
                <span class="poweredGrey">Powered by</span>
                <a href="https://qagencia.com" target="_blank" rel="noopener" style="text-decoration: none;">
                    <figure class="logos-card">
                        <?php $img_url = plugins_url('assets/images/grupo-qagencia.webp', TELO_AFFILIATES_FILE); ?>
                        <img src="<?php echo esc_url($img_url); ?>" alt="Logo qagencia" loading="lazy" decoding="async">
                    </figure>
                </a>
            </div>
        </div>

        <?php if ($flash['type']): ?>
            <div role="alert" style="margin:10px 0; padding:10px; border-radius:6px; <?php echo $flash['type'] === 'success' ? 'background:#e8f7ee;border:1px solid #b6e1c8;' : 'background:#fdecea;border:1px solid #f5c2c0;'; ?>">
                <?php echo esc_html($flash['msg']); ?>
            </div>
        <?php endif; ?>

        <?php
        $input_id = 'telo-aff-link-' . substr(md5($link . microtime()), 0, 8);
        $wrap_id  = 'telo-aff-wrap-' . substr(md5($input_id), 0, 6);
        ?>
        <div id="<?php echo esc_attr($wrap_id); ?>">
            <p><strong><?php esc_html_e('Tu enlace único:', 'telo-affiliates-lite'); ?></strong><br>
                <a href="<?php echo esc_url($link); ?>" target="_blank" rel="noopener"><?php echo esc_html($link); ?></a><br>
                <input type="text" id="<?php echo esc_attr($input_id); ?>" readonly value="<?php echo esc_attr($link); ?>" style="width:100%; margin-bottom:8px;">
                <button type="button" data-copy-target="<?php echo esc_attr($input_id); ?>" style="display:inline-block; padding:6px 10px; border:1px solid #ddd; background:#f6f7f7; cursor:pointer; border-radius:4px;">
                    <?php esc_html_e('Copiar enlace', 'telo-affiliates-lite'); ?>
                </button>
                <span class="copy-notice" aria-live="polite" style="margin-left:8px;"></span>
            </p>
        </div>


        <p><strong><?php esc_html_e('Comisión asignada:', 'telo-affiliates-lite'); ?></strong> <?php echo esc_html(number_format($rate, 2)); ?>%</p>

        <div class="telo-col-afiliados-panel">
            <div style="border:1px solid #eee; padding:12px;">
                <h3 style="margin:0 0 8px;"><?php esc_html_e('Resumen histórico', 'telo-affiliates-lite'); ?></h3>
                <p style="margin:0;"><strong><?php esc_html_e('Ventas', 'telo-affiliates-lite'); ?>:</strong> <?php echo esc_html($order_count); ?></p>
                <p style="margin:0;"><strong><?php esc_html_e('Importe total', 'telo-affiliates-lite'); ?>:</strong> <?php echo function_exists('wc_price') ? wp_kses_post(wc_price($total_sales)) : esc_html(number_format_i18n($total_sales, 2)); ?></p>
                <p style="margin:0;"><strong><?php esc_html_e('Comisión total', 'telo-affiliates-lite'); ?>:</strong> <?php echo function_exists('wc_price') ? wp_kses_post(wc_price($commission_total)) : esc_html(number_format_i18n($commission_total, 2)); ?></p>
            </div>

            <div style="border:1px solid #eee; padding:12px;">
                <h3 style="margin:0 0 8px;"><?php esc_html_e('Pendiente de pago', 'telo-affiliates-lite'); ?></h3>
                <p style="margin:0;"><strong><?php esc_html_e('Pedidos pendientes', 'telo-affiliates-lite'); ?>:</strong> <?php echo esc_html(count($pending_order_ids)); ?></p>
                <p style="margin:0;"><strong><?php esc_html_e('Base pendiente', 'telo-affiliates-lite'); ?>:</strong> <?php echo function_exists('wc_price') ? wp_kses_post(wc_price($pending_base)) : esc_html(number_format_i18n($pending_base, 2)); ?></p>
                <p style="margin:0;"><strong><?php esc_html_e('Comisión pendiente', 'telo-affiliates-lite'); ?>:</strong> <?php echo function_exists('wc_price') ? wp_kses_post(wc_price($pending_commission)) : esc_html(number_format_i18n($pending_commission, 2)); ?></p>
            </div>
        </div>

        <?php
        // Botón Solicitar Cobro — en Lite siempre deshabilitado + CTA
        $can_request_pro = (! $IS_LITE) && ($pending_commission >= $min_payout) && (! $has_pending_request);
        ?>
        <div style="margin-top:18px; padding:12px; border:1px dashed #ddd; border-radius:8px;">
            <form method="post">
                <?php wp_nonce_field('telo_aff_request_' . $slug, 'telo_aff_nonce'); ?>
                <input type="hidden" name="telo_aff_request" value="1" />
                <input type="hidden" name="telo_aff_slug" value="<?php echo esc_attr($slug); ?>" />
                <button type="submit"
                    <?php echo $can_request_pro ? '' : 'disabled'; ?>
                    style="padding:10px 14px; border-radius:6px; border:0; <?php echo $can_request_pro ? 'background:#1f8f4e;color:#fff;cursor:pointer;' : 'background:#e5e7eb;color:#6b7280;cursor:not-allowed;'; ?>">
                    <?php esc_html_e('Solicitar cobro', 'telo-affiliates-lite'); ?>
                </button>

                <div style="margin-top:6px; color:#6b7280;">
                    <?php
                    if ($IS_LITE) {
                        echo wp_kses_post(
                            sprintf(
                                __('Función disponible en <strong>PRO</strong>. %s', 'telo-affiliates-lite'),
                                '<a target="_blank" rel="noopener" href="https://licencias.qagencia.com/">' . esc_html__('Pásate a PRO', 'telo-affiliates-lite') . '</a>'
                            )
                        );
                    } else {
                        if ($has_pending_request) {
                            esc_html_e('Tienes una solicitud pendiente. Te avisaremos cuando se procese.', 'telo-affiliates-lite');
                        } elseif ($pending_commission < $min_payout) {
                            printf(
                                esc_html__('Necesitas al menos %2$s para solicitar el cobro. Actualmente tienes %1$s.', 'telo-affiliates-lite'),
                                function_exists('wc_price') ? wc_price($pending_commission) : number_format_i18n($pending_commission, 2),
                                function_exists('wc_price') ? wc_price($min_payout)       : number_format_i18n($min_payout, 2)
                            );
                        } else {
                            esc_html_e('Puedes solicitar el cobro de tus comisiones pendientes.', 'telo-affiliates-lite');
                        }
                    }
                    ?>
                </div>
            </form>

            <?php if ($IS_LITE) : ?>
                <div class="telo-aff-upgrade-wrap" style="margin-top:12px;">
                    <div class="telo-aff-upgrade-box" style="background:#fff;border:1px solid #e5e7eb;padding:16px;border-radius:6px;">
                        <strong><?php esc_html_e('Desbloquea pagos automáticos, solicitudes y más', 'telo-affiliates-lite'); ?></strong>
                        <p style="margin:.5em 0 0;">
                            <a class="button button-primary" target="_blank" rel="noopener" href="https://licencias.qagencia.com/">
                                <?php esc_html_e('Pásate a la versión PRO', 'telo-affiliates-lite'); ?>
                            </a>
                        </p>
                    </div>
                </div>
            <?php endif; ?>
        </div>

        <div class="telo-aff-section telo-aff-payouts">
            <h3 class="telo-aff-title"><?php esc_html_e('Historial de pagos', 'telo-affiliates-lite'); ?></h3>

            <?php if (empty($payouts)) : ?>
                <p class="telo-aff-empty"><?php esc_html_e('Aún no hay pagos registrados.', 'telo-affiliates-lite'); ?></p>
            <?php else :
                $col_fecha    = __('Fecha', 'telo-affiliates-lite');
                $col_pedidos  = __('Pedidos', 'telo-affiliates-lite');
                $col_base     = __('Base', 'telo-affiliates-lite');
                $col_percent  = __('%', 'telo-affiliates-lite');
                $col_comision = __('Comisión', 'telo-affiliates-lite');
                $col_nota     = __('Nota', 'telo-affiliates-lite');
            ?>
                <table class="telo-aff-table telo-aff-table--payouts">
                    <thead class="telo-aff-thead">
                        <tr>
                            <th scope="col" class="telo-aff-th telo-aff-th--left"><?php echo esc_html($col_fecha); ?></th>
                            <th scope="col" class="telo-aff-th telo-aff-th--left"><?php echo esc_html($col_pedidos); ?></th>
                            <th scope="col" class="telo-aff-th telo-aff-th--right"><?php echo esc_html($col_base); ?></th>
                            <th scope="col" class="telo-aff-th telo-aff-th--right"><?php echo esc_html($col_percent); ?></th>
                            <th scope="col" class="telo-aff-th telo-aff-th--right"><?php echo esc_html($col_comision); ?></th>
                            <th scope="col" class="telo-aff-th telo-aff-th--left"><?php echo esc_html($col_nota); ?></th>
                        </tr>
                    </thead>
                    <tbody class="telo-aff-tbody">
                        <?php foreach ($payouts as $p) :
                            $date     = ! empty($p['timestamp']) ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), (int)$p['timestamp']) : '—';
                            $orders_p = isset($p['orders']) && is_array($p['orders']) ? $p['orders'] : array();
                            $base     = isset($p['base_total']) ? (float)$p['base_total'] : 0.0;
                            $r        = isset($p['rate']) ? (float)$p['rate'] : 0.0;
                            $amt      = isset($p['amount']) ? (float)$p['amount'] : 0.0;
                            $note     = isset($p['note']) ? $p['note'] : '';
                        ?>
                            <tr class="telo-aff-tr">
                                <td class="telo-aff-td" data-label="<?php echo esc_attr($col_fecha); ?>">
                                    <?php echo esc_html($date); ?>
                                </td>
                                <td class="telo-aff-td" data-label="<?php echo esc_attr($col_pedidos); ?>">
                                    <?php echo $orders_p ? esc_html('#' . implode(', #', array_map('absint', $orders_p))) : '—'; ?>
                                </td>
                                <td class="telo-aff-td telo-aff-td--num" data-label="<?php echo esc_attr($col_base); ?>">
                                    <?php echo function_exists('wc_price') ? wp_kses_post(wc_price($base)) : esc_html(number_format_i18n($base, 2)); ?>
                                </td>
                                <td class="telo-aff-td telo-aff-td--num" data-label="<?php echo esc_attr($col_percent); ?>">
                                    <?php echo esc_html(number_format_i18n($r, 2)); ?>%
                                </td>
                                <td class="telo-aff-td telo-aff-td--num" data-label="<?php echo esc_attr($col_comision); ?>">
                                    <?php echo function_exists('wc_price') ? wp_kses_post(wc_price($amt)) : esc_html(number_format_i18n($amt, 2)); ?>
                                </td>
                                <td class="telo-aff-td" data-label="<?php echo esc_attr($col_nota); ?>">
                                    <?php echo esc_html($note); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
<?php
    // justo antes de cerrar el output del shortcode
    wp_register_script(
        'telo-aff-frontend-inline',
        '',
        [],
        defined('TELO_AFFILIATES_VERSION') ? TELO_AFFILIATES_VERSION : false,
        true
    );
    wp_enqueue_script('telo-aff-frontend-inline');

    // Textos traducibles para el JS
    $copied_txt = esc_js(__('¡Copiado!', 'telo-affiliates-lite'));
    $error_txt  = esc_js(__('No se pudo copiar', 'telo-affiliates-lite'));

    $js = "
(function(){
  var wrap = document.getElementById('" . esc_js($wrap_id) . "');
  if (!wrap) return;
  var btn = wrap.querySelector('button[data-copy-target]');
  if (!btn) return;
  var input = document.getElementById(btn.getAttribute('data-copy-target'));
  var notice = wrap.querySelector('.copy-notice');
  function showMsg(msg){
    if (notice){
      notice.textContent = msg;
      setTimeout(function(){ notice.textContent = ''; }, 2000);
    }
  }
  btn.addEventListener('click', function(){
    if (!input) return;
    var text = input.value;
    if (navigator.clipboard && window.isSecureContext) {
      navigator.clipboard.writeText(text).then(function(){ showMsg('{$copied_txt}'); })
        .catch(fallbackCopy);
    } else { fallbackCopy(); }
    function fallbackCopy(){
      var r = input.hasAttribute('readonly');
      if (r) input.removeAttribute('readonly');
      input.select();
      input.setSelectionRange(0, 99999);
      var ok = document.execCommand('copy');
      if (r) input.setAttribute('readonly','readonly');
      window.getSelection().removeAllRanges();
      showMsg(ok ? '{$copied_txt}' : '{$error_txt}');
    }
  });
})();";
    wp_add_inline_script('telo-aff-frontend-inline', $js);


    return ob_get_clean();
});
