<?php
if ( ! defined( 'ABSPATH' ) ) exit;

// =======================
// COLUMNA (LISTA CLÁSICA) + HPOS SOLO SI HAY WC
// =======================


if ( function_exists('telo_aff_pro_is_active') && telo_aff_pro_is_active() ) {
    return; // No registres menús/páginas del Lite si el PRO está activo
}

// en helpers.php y/o admin.php
if (!function_exists('telo_aff_is_lite')) {
    function telo_aff_is_lite() {
        return defined('TELO_AFFILIATES_IS_LITE') && TELO_AFFILIATES_IS_LITE;
    }
}


// Caja de upgrade mínima (por si no existe aún en otro include)
if ( ! function_exists('telo_aff_upgrade_box') ) {
    function telo_aff_upgrade_box( $title = '' ) { ?>
        <div class="telo-aff-upgrade-wrap" style="margin-top:16px;">
            <div class="telo-aff-upgrade-box" style="background:#fff;border:1px solid #ccd0d4;padding:20px;border-radius:6px;">
                <h2><?php echo esc_html( $title ?: __('Desbloquea esta función', 'telo-affiliates-lite') ); ?></h2>
                <p><?php esc_html_e('Esta característica está disponible en la versión PRO.', 'telo-affiliates-lite'); ?></p>
                <p><a class="button button-primary button-hero" target="_blank" rel="noopener" href="https://licencias.qagencia.com/"><?php esc_html_e('Pásate a la versión PRO', 'telo-affiliates-lite'); ?></a></p>
            </div>
        </div>
    <?php }
}

if ( function_exists('telo_aff_has_wc') ? telo_aff_has_wc() : class_exists('WooCommerce') ) {

    add_filter('manage_edit-shop_order_columns', function ($columns) {
        $columns['telo_affiliate'] = __('Afiliado', 'telo-affiliates-lite');
        return $columns;
    });

    add_action('manage_shop_order_posts_custom_column', function ($column, $post_id) {
        if ($column !== 'telo_affiliate') return;
        $order = function_exists('wc_get_order') ? wc_get_order($post_id) : null;
        if (!$order) { echo '<span style="color:#888;">—</span>'; return; }
        $partner = (string) $order->get_meta('_telo_affiliate', true);
        echo $partner && $partner !== 'directo'
            ? esc_html($partner)
            : '<span style="color:#888;">' . esc_html__('Directo', 'telo-affiliates-lite') . '</span>';
    }, 10, 2);

    add_filter('woocommerce_shop_order_list_table_columns', function ($columns) {
        $new = [];
        foreach ($columns as $key => $label) {
            $new[$key] = $label;
            if ($key === 'order_status') {
                $new['telo_affiliate'] = __('Afiliado', 'telo-affiliates-lite');
            }
        }
        if (!isset($new['telo_affiliate'])) {
            $new['telo_affiliate'] = __('Afiliado', 'telo-affiliates-lite');
        }
        return $new;
    });

    add_action('woocommerce_shop_order_list_table_custom_column', function ($column, $order) {
        if ($column !== 'telo_affiliate') return;
        if (!class_exists('WC_Order')) { echo '<span style="color:#888;">—</span>'; return; }
        if ($order instanceof WC_Order) {
            $partner = (string) $order->get_meta('_telo_affiliate', true);
            echo $partner && $partner !== 'directo'
                ? esc_html($partner)
                : '<span style="color:#888;">' . esc_html__('Directo', 'telo-affiliates-lite') . '</span>';
        } else {
            echo '<span style="color:#888;">—</span>';
        }
    }, 10, 2);
}


// ==============
// MENÚ AFILIADOS (con anti-duplicados)
// ==============

/**
 * Añade un submenu solo si no existe ya ese slug bajo el padre.
 */
function telo_aff_add_submenu_once($parent_slug, $page_title, $menu_title, $cap, $menu_slug, $callback) {
    global $submenu;
    if ( isset($submenu[$parent_slug]) ) {
        foreach ( (array)$submenu[$parent_slug] as $item ) {
            if ( isset($item[2]) && $item[2] === $menu_slug ) {
                return false; // ya existe
            }
        }
    }
    add_submenu_page($parent_slug, $page_title, $menu_title, $cap, $menu_slug, $callback);
    return true;
}

add_action('admin_menu', function () {
    global $admin_page_hooks;

    $parent_slug = 'telo-affiliates-lite';

    // Evita duplicar el menú toplevel si otro include ya lo añadió
    if ( ! isset($admin_page_hooks[ $parent_slug ]) ) {
        add_menu_page(
            __('Afiliados','telo-affiliates-lite'),
            __('Afiliados','telo-affiliates-lite'),
            'manage_options',
            $parent_slug,
            'telo_affiliates_dashboard',
            'dashicons-networking',
            56
        );
    }

    // Páginas (bloqueada en Lite)
    telo_aff_add_submenu_once(
        $parent_slug,
        __('Páginas de afiliado','telo-affiliates-lite'),
        __('Páginas','telo-affiliates-lite'),
        'manage_options',
        'telo-affiliates-pages',
        telo_aff_is_lite() ? 'telo_aff_render_locked_pages' : 'telo_aff_pages_render'
    );

    // Exportar/Importar (bloqueada en Lite)
    telo_aff_add_submenu_once(
        $parent_slug,
        __('Exportar/Importar','telo-affiliates-lite'),
        __('Exportar/Importar','telo-affiliates-lite'),
        'manage_options',
        'telo-affiliates-import-export',
        telo_aff_is_lite() ? 'telo_aff_render_locked_import_export' : 'telo_aff_render_import_export'
    );

    // Cobros (bloqueada en Lite)
    telo_aff_add_submenu_once(
        $parent_slug,
        __('Cobros','telo-affiliates-lite'),
        __('Cobros','telo-affiliates-lite'),
        'manage_options',
        'telo-affiliates-payments',
        telo_aff_is_lite() ? 'telo_aff_render_locked_payments' : 'telo_aff_render_payments'
    );

    // Upgrade
    telo_aff_add_submenu_once(
        $parent_slug,
        __('Pásate a la versión PRO','telo-affiliates-lite'),
        __('Pásate a la versión PRO','telo-affiliates-lite'),
        'manage_options',
        'telo-affiliates-upgrade',
        'telo_aff_render_upgrade_landing'
    );
}, 20);

// Oculta posibles submenús duplicados añadidos por CPT cuando es Lite
add_action('admin_menu', function () {
    if ( ! telo_aff_is_lite() ) return;
    remove_submenu_page('telo-affiliates-lite', 'edit.php?post_type=telo_aff_page');
    remove_submenu_page('telo-affiliates-lite', 'edit.php?post_type=telo_aff_payout');
}, 999);

if (!function_exists('telo_aff_has_wc')) {
    function telo_aff_has_wc() {
        return class_exists('WooCommerce') || defined('WC_VERSION');
    }
}


if (!function_exists('telo_aff_upgrade_box')) {
    function telo_aff_upgrade_box($title = '') {
        ?>
        <div class="telo-aff-upgrade-wrap" style="margin-top:16px;">
            <div class="telo-aff-upgrade-box" style="background:#fff;border:1px solid #ccd0d4;padding:20px;border-radius:6px;">
                <h2><?php echo esc_html($title ?: __('Desbloquea esta función', 'telo-affiliates-lite')); ?></h2>
                <p><?php esc_html_e('Esta característica está disponible en la versión PRO.', 'telo-affiliates-lite'); ?></p>
                <p><a class="button button-primary button-hero" target="_blank" rel="noopener" href="https://licencias.qagencia.com/">
                    <?php esc_html_e('Pásate a la versión PRO', 'telo-affiliates-lite'); ?>
                </a></p>
            </div>
        </div>
        <?php
    }
}

if (!function_exists('telo_aff_render_upgrade_landing')) {
    function telo_aff_render_upgrade_landing() {
        echo '<div class="wrap telo-aff-locked"><h1>' . esc_html__('Pásate a la versión PRO', 'telo-affiliates-lite') . '</h1>';
        echo '<div class="telo-aff-blurred" style="min-height:220px;border:1px dashed #ccd0d4;background:#f6f7f7;filter:blur(3px);"></div>';
        if (function_exists('telo_aff_upgrade_box')) {
            telo_aff_upgrade_box(__('Todas las funciones PRO', 'telo-affiliates-lite'));
        }
        echo '</div>';
    }
}


// =======================
// SCREEN OPTIONS (per_page)
// =======================
add_filter('set-screen-option', function ($status, $option, $value) {
    if ($option === 'telo_affiliates_per_page') {
        return (int) $value;
    }
    return $status;
}, 10, 3);

add_action('load-toplevel_page_telo-affiliates-lite', function () {
    add_screen_option('per_page', [
        'label'   => __('Afiliados por página', 'telo-affiliates-lite'),
        'default' => 10,
        'option'  => 'telo_affiliates_per_page',
    ]);
});


// =======================
// HELPERS (pedidos/importe)
// =======================
function telo_affiliates_get_orders_by_partner($slug, $statuses = array('processing', 'completed'))
{
    if (! function_exists('wc_get_orders')) return array();

    $ids = wc_get_orders(array(
        'type'       => 'shop_order',
        'status'     => $statuses,
        'limit'      => -1,
        'return'     => 'ids',
        'meta_key'   => '_telo_affiliate',
        'meta_value' => $slug,
    ));

    $orders = array();
    foreach ((array) $ids as $oid) {
        $o = wc_get_order($oid);
        if ($o) $orders[] = $o;
    }
    return $orders;
}

function telo_affiliates_get_unpaid_orders_by_partner($slug, $statuses = array('processing', 'completed'))
{
    $orders = telo_affiliates_get_orders_by_partner($slug, $statuses);
    $unpaid = array();
    foreach ($orders as $order) {
        $paid_flag = $order->get_meta('_telo_aff_paid', true);
        if ($paid_flag !== 'yes') {
            $unpaid[] = $order;
        }
    }
    return $unpaid;
}

function telo_affiliates_calculate_commission($orders, $rate_percent)
{
    $total_sales = 0.0;
    foreach ($orders as $order) {
        $total_sales += (float) $order->get_total();
    }
    $commission = $total_sales * ((float)$rate_percent / 100);
    return array($total_sales, $commission);
}


// ====== EXPORT / IMPORT HANDLERS ======
// En Lite: bloqueamos los endpoints (no permiten acción real).
if ( telo_aff_is_lite() ) {

    add_action('admin_post_telo_affiliates_export', function () {
        wp_die( esc_html__('Esta función está disponible en la versión PRO.', 'telo-affiliates-lite') );
    });

    add_action('admin_post_telo_affiliates_import', function () {
        wp_die( esc_html__('Esta función está disponible en la versión PRO.', 'telo-affiliates-lite') );
    });

} else {

    add_action('admin_post_telo_affiliates_export', 'telo_affiliates_handle_export');
    add_action('admin_post_telo_affiliates_import', 'telo_affiliates_handle_import');
}

/**
 * Exporta afiliados a JSON (solo slug, name, rate) — solo PRO
 */
function telo_affiliates_handle_export()
{
    if ( telo_aff_is_lite() ) {
        wp_die( esc_html__('Esta función está disponible en la versión PRO.', 'telo-affiliates-lite') );
    }

    if (! current_user_can('manage_options')) {
        wp_die(esc_html__('No tienes permisos suficientes.', 'telo-affiliates-lite'));
    }
    check_admin_referer('telo_affiliates_export');

    $aff = get_option('telo_affiliates', array());
    $out = array();
    foreach ((array) $aff as $slug => $data) {
        $out[] = array(
            'slug' => (string) $slug,
            'name' => isset($data['name']) ? (string) $data['name'] : (string) $slug,
            'rate' => isset($data['rate']) ? (float) $data['rate'] : 0.0,
        );
    }

    $json = wp_json_encode($out, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    if ($json === false) {
        wp_die(esc_html__('No se pudo generar el JSON.', 'telo-affiliates-lite'));
    }

    $filename = 'telo-affiliates-' . date('Ymd-His') . '.json';
    nocache_headers();
    header('Content-Type: application/json; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Content-Length: ' . strlen($json));
    echo $json;
    exit;
}

/**
 * Importa afiliados desde JSON — solo PRO
 */
function telo_affiliates_handle_import()
{
    if ( telo_aff_is_lite() ) {
        wp_die( esc_html__('Esta función está disponible en la versión PRO.', 'telo-affiliates-lite') );
    }

    if (! current_user_can('manage_options')) {
        wp_die(esc_html__('No tienes permisos suficientes.', 'telo-affiliates-lite'));
    }
    check_admin_referer('telo_affiliates_import', 'telo_affiliates_import_nonce');

    if (empty($_FILES['telo_affiliates_json']['tmp_name'])) {
        telo_affiliates_import_redirect_error(__('No se recibió ningún archivo.', 'telo-affiliates-lite'));
    }

    $tmp = $_FILES['telo_affiliates_json']['tmp_name'];
    $raw = file_get_contents($tmp);
    if ($raw === false || $raw === '') {
        telo_affiliates_import_redirect_error(__('Archivo vacío o ilegible.', 'telo-affiliates-lite'));
    }

    $data = json_decode($raw, true);
    if (! is_array($data)) {
        telo_affiliates_import_redirect_error(__('JSON inválido.', 'telo-affiliates-lite'));
    }

    // Normaliza: aceptar array de objetos [{slug,name,rate}] o mapa {slug:{name,rate}}
    $entries = array();

    if (isset($data[0]) && is_array($data[0])) {
        foreach ($data as $row) {
            if (!is_array($row)) continue;
            $entries[] = $row;
        }
    } else {
        foreach ($data as $k => $row) {
            if (!is_array($row)) continue;
            $row['slug'] = isset($row['slug']) ? $row['slug'] : $k;
            $entries[] = $row;
        }
    }

    $aff = get_option('telo_affiliates', array());
    if (!is_array($aff)) $aff = array();

    $imported = 0;
    $updated  = 0;
    $skipped  = 0;

    foreach ($entries as $row) {
        $slug = isset($row['slug']) ? sanitize_key($row['slug']) : '';
        if ($slug === '') {
            $skipped++;
            continue;
        }

        $name = isset($row['name']) ? sanitize_text_field($row['name']) : $slug;
        $rate = isset($row['rate']) ? (float) $row['rate'] : 0.0;
        if ($rate < 0) $rate = 0.0;

        if (isset($aff[$slug])) {
            $aff[$slug]['name'] = $name ?: $slug;
            $aff[$slug]['rate'] = $rate;
            $updated++;
        } else {
            $aff[$slug] = array(
                'name' => $name ?: $slug,
                'rate' => $rate,
            );
            $imported++;
        }
    }

    update_option('telo_affiliates', $aff);

    $args = array(
        'page'     => 'telo-affiliates-lite',
        'ta_imp'   => $imported,
        'ta_upd'   => $updated,
        'ta_skp'   => $skipped,
    );
    wp_safe_redirect(add_query_arg($args, admin_url('admin.php')));
    exit;
}

function telo_affiliates_import_redirect_error($msg)
{
    $args = array(
        'page'   => 'telo-affiliates-lite',
        'ta_err' => rawurlencode(sanitize_text_field($msg)),
    );
    wp_safe_redirect(add_query_arg($args, admin_url('admin.php')));
    exit;
}


// ==============
// DASHBOARD ADMIN
// ==============
function telo_affiliates_dashboard()
{
    if (! current_user_can('manage_options')) {
        wp_die(esc_html__('No tienes permisos suficientes.', 'telo-affiliates-lite'));
    }

    // ---------------------------
    // Alta de afiliado (LIBRE)
    // ---------------------------
    if (isset($_POST['telo_new_affiliate'])) {
        check_admin_referer('telo_affiliates_save', 'telo_affiliates_nonce');

        $name = isset($_POST['affiliate_name']) ? sanitize_text_field(wp_unslash($_POST['affiliate_name'])) : '';
        $slug = isset($_POST['affiliate_slug']) ? sanitize_key(wp_unslash($_POST['affiliate_slug'])) : '';
        $rate = isset($_POST['affiliate_rate']) ? (float) wp_unslash($_POST['affiliate_rate']) : 0;

        if ($name && $slug && $rate >= 0) {
            $affiliates = get_option('telo_affiliates', array());
            if (isset($affiliates[$slug])) {
                echo '<div class="notice notice-error"><p>' . esc_html__('Ese slug ya existe. Usa otro.', 'telo-affiliates-lite') . '</p></div>';
            } else {
                $affiliates[$slug] = array('name' => $name, 'rate' => $rate);
                update_option('telo_affiliates', $affiliates);
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Afiliado guardado.', 'telo-affiliates-lite') . '</p></div>';
            }
        } else {
            echo '<div class="notice notice-error"><p>' . esc_html__('Completa todos los campos correctamente.', 'telo-affiliates-lite') . '</p></div>';
        }
    }

    // ---------------------------
    // Borrar afiliado (LIBRE)
    // ---------------------------
    if (isset($_POST['telo_delete_affiliate'])) {
        check_admin_referer('telo_affiliates_delete', 'telo_affiliates_delete_nonce');

        $slug_to_delete = sanitize_key(wp_unslash($_POST['telo_delete_affiliate']));
        $affiliates = get_option('telo_affiliates', array());

        if (isset($affiliates[$slug_to_delete])) {
            unset($affiliates[$slug_to_delete]);
            update_option('telo_affiliates', $affiliates);

            echo '<div class="notice notice-success is-dismissible"><p>' .
                sprintf(esc_html__('Afiliado %s eliminado.', 'telo-affiliates-lite'), esc_html($slug_to_delete)) .
                '</p></div>';
        } else {
            echo '<div class="notice notice-warning is-dismissible"><p>' .
                esc_html__('El afiliado ya no existe o fue eliminado.', 'telo-affiliates-lite') .
                '</p></div>';
        }
    }

    // ---------------------------
    // Editar afiliado (LIBRE)
    // ---------------------------
    if (isset($_POST['telo_update_affiliate'])) {
        check_admin_referer('telo_affiliates_update', 'telo_affiliates_update_nonce');

        $old_slug = isset($_POST['telo_old_slug']) ? sanitize_key(wp_unslash($_POST['telo_old_slug'])) : '';
        $new_name = isset($_POST['telo_edit_name']) ? sanitize_text_field(wp_unslash($_POST['telo_edit_name'])) : '';
        $new_slug = isset($_POST['telo_edit_slug']) ? sanitize_key(wp_unslash($_POST['telo_edit_slug'])) : '';
        $new_rate = isset($_POST['telo_edit_rate']) ? (float) wp_unslash($_POST['telo_edit_rate']) : 0;

        $affiliates = get_option('telo_affiliates', array());

        if (! $old_slug || ! isset($affiliates[$old_slug])) {
            echo '<div class="notice notice-error"><p>' . esc_html__('Afiliado a editar no encontrado.', 'telo-affiliates-lite') . '</p></div>';
        } elseif (! $new_slug || $new_rate < 0) {
            echo '<div class="notice notice-error"><p>' . esc_html__('Datos inválidos. Revisa slug y porcentaje.', 'telo-affiliates-lite') . '</p></div>';
        } elseif ($new_slug !== $old_slug && isset($affiliates[$new_slug])) {
            echo '<div class="notice notice-error"><p>' . esc_html__('El nuevo slug ya existe. Elige otro.', 'telo-affiliates-lite') . '</p></div>';
        } else {
            unset($affiliates[$old_slug]);
            $affiliates[$new_slug] = array(
                'name' => $new_name ? $new_name : $new_slug,
                'rate' => max(0, $new_rate),
            );
            update_option('telo_affiliates', $affiliates);

            // Reasignación de pedidos/histórico si cambia slug
            if ($new_slug !== $old_slug) {
                if (function_exists('wc_get_order_statuses') && function_exists('wc_get_orders')) {
                    $all_statuses = array_map(
                        function ($s) { return preg_replace('/^wc-/', '', $s); },
                        array_keys(wc_get_order_statuses())
                    );
                    $orders_to_update = wc_get_orders(array(
                        'limit'      => -1,
                        'status'     => $all_statuses,
                        'return'     => 'objects',
                        'meta_query' => array(
                            array(
                                'key'   => '_telo_affiliate',
                                'value' => $old_slug,
                            )
                        ),
                    ));
                    foreach ($orders_to_update as $o) {
                        $o->update_meta_data('_telo_affiliate', $new_slug);
                        $o->save();
                    }
                }
                $payouts = get_option('telo_affiliates_payouts', array());
                $changed = false;
                if (is_array($payouts)) {
                    foreach ($payouts as &$p) {
                        if (! empty($p['slug']) && $p['slug'] === $old_slug) {
                            $p['slug'] = $new_slug;
                            $changed = true;
                        }
                    }
                    if ($changed) update_option('telo_affiliates_payouts', $payouts);
                }
            }

            echo '<div class="notice notice-success is-dismissible"><p>' .
                esc_html__('Afiliado actualizado correctamente.', 'telo-affiliates-lite') .
                '</p></div>';
        }
    }

    // ---------------------------
    // Marcar comisiones como pagadas (SOLO PRO)
    // ---------------------------
    if ( isset($_POST['telo_pay_affiliate']) ) {
        if ( telo_aff_is_lite() ) {
            echo '<div class="notice notice-info is-dismissible"><p>' .
                esc_html__('La gestión de cobros está disponible en la versión PRO.', 'telo-affiliates-lite') .
                '</p></div>';
        } else {
            // === LÓGICA PRO ORIGINAL ===
            check_admin_referer('telo_affiliates_pay', 'telo_affiliates_pay_nonce');

            $slug_pay = sanitize_key(wp_unslash($_POST['telo_pay_affiliate']));
            $note     = isset($_POST['telo_pay_note']) ? sanitize_textarea_field(wp_unslash($_POST['telo_pay_note'])) : '';

            $affiliates_all = get_option('telo_affiliates', array());
            if (! isset($affiliates_all[$slug_pay])) {
                echo '<div class="notice notice-error"><p>' . esc_html__('Afiliado no encontrado.', 'telo-affiliates-lite') . '</p></div>';
            } else {
                $rate = (float) $affiliates_all[$slug_pay]['rate'];
                $unpaid_orders = telo_affiliates_get_unpaid_orders_by_partner($slug_pay, array('processing', 'completed'));

                if (empty($unpaid_orders)) {
                    echo '<div class="notice notice-warning is-dismissible"><p>' . esc_html__('No hay comisiones pendientes para este afiliado.', 'telo-affiliates-lite') . '</p></div>';
                } else {
                    list($base_total, $amount) = telo_affiliates_calculate_commission($unpaid_orders, $rate);

                    $payout_id = uniqid('telo_payout_', true);
                    $order_ids = array();

                    foreach ($unpaid_orders as $o) {
                        $o->update_meta_data('_telo_aff_paid', 'yes');
                        $o->update_meta_data('_telo_aff_payout_id', $payout_id);
                        $o->update_meta_data('_telo_aff_paid_ts', time());
                        $o->save();
                        $order_ids[] = $o->get_id();
                    }

                    $payouts = get_option('telo_affiliates_payouts', array());
                    $payouts[] = array(
                        'id'         => $payout_id,
                        'slug'       => $slug_pay,
                        'rate'       => $rate,
                        'orders'     => $order_ids,
                        'base_total' => $base_total,
                        'amount'     => $amount,
                        'note'       => $note,
                        'timestamp'  => time(),
                    );
                    update_option('telo_affiliates_payouts', $payouts);

                    do_action('telo_affiliates_payout_recorded', $payout_id, $slug_pay, $order_ids, $amount);

                    if (function_exists('rocket_clean_domain')) { rocket_clean_domain(); }
                    if (class_exists('\LiteSpeed_Cache_API')) { \LiteSpeed_Cache_API::purge_all(); }
                    if (function_exists('w3tc_flush_all')) { w3tc_flush_all(); }
                    if (function_exists('wpfc_clear_all_cache')) { wpfc_clear_all_cache(); }
                    if (function_exists('wp_cache_clear_cache')) { wp_cache_clear_cache(); }
                    if (function_exists('wp_cache_flush')) { wp_cache_flush(); }

                    echo '<div class="notice notice-success is-dismissible"><p>' .
                        sprintf(
                            esc_html__('Marcadas como pagadas las comisiones de %1$s por %2$s.', 'telo-affiliates-lite'),
                            esc_html($slug_pay),
                            function_exists('wc_price') ? wp_kses_post(wc_price($amount)) : esc_html(number_format_i18n($amount, 2))
                        ) .
                        '</p></div>';
                }
            }
        }
    }

    // ======================
    // Render de la pantalla
    // ======================
    echo '<div class="wrap telo-aff-admin"><h1>' . esc_html__('Gestión de Afiliados', 'telo-affiliates-lite') . '</h1>';

    // Alta
    echo '<div class="telo-aff-new-affiliate-header-full">';
    echo '<div class="telo-aff-new-affiliate">';
    echo '<h2>' . esc_html__('Añadir Afiliado', 'telo-affiliates-lite') . '</h2>';
    echo '<form method="post" class="telo-affiliate-form-add-affiliate">';
    wp_nonce_field('telo_affiliates_save', 'telo_affiliates_nonce');
    echo '<p><input type="text" name="affiliate_name" placeholder="' . esc_attr__('Nombre', 'telo-affiliates-lite') . '" required></p>';
    echo '<p><input type="text" name="affiliate_slug" placeholder="' . esc_attr__('Slug único (ej: juan)', 'telo-affiliates-lite') . '" required></p>';
    echo '<p><input type="number" step="0.01" min="0" name="affiliate_rate" placeholder="' . esc_attr__('Comisión %', 'telo-affiliates-lite') . '" required></p>';
    echo '<p><input type="submit" name="telo_new_affiliate" class="button button-primary" value="' . esc_attr__('Guardar', 'telo-affiliates-lite') . '"></p>';
    echo '</form>';
    echo '</div>';

    // Tabla + Import/Export UI
    if (! class_exists('WP_List_Table')) {
        require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
    }

    if (! class_exists('TELO_Affiliates_List_Table')) {
        require_once __DIR__ . '/admin-list-table.php';
    }


    echo '<div class="telo-aff-import-export-header">';
    echo '<h2 style="margin-bottom: 0;">' . esc_html__('Importar / Exportar Afiliados', 'telo-affiliates-lite') . '</h2>';
    echo '<p style="font-size: .9em;">' . esc_html__('Aquí puedes importar y exportar los afiliados registrados en tu tienda para llevarlos a otra web.', 'telo-affiliates-lite') . '</p>';

    echo '<div class="telo-aff-import-export" style="display:flex; gap:12px; align-items:center; margin:12px 0 6px;">';

    if ( telo_aff_is_lite() ) {
        // Bloqueado en Lite: UI con blur + CTA
        echo '<div class="telo-aff-locked" style="width:100%">';
        echo '<div class="telo-aff-blurred" style="min-height:120px; border:1px dashed #ccd0d4; background:#f6f7f7; filter:blur(3px);"></div>';
        telo_aff_upgrade_box( __('Importar / Exportar (solo PRO)', 'telo-affiliates-lite') );
        echo '</div>';
    } else {
        // UI real SOLO PRO
        // Exportar JSON
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
        wp_nonce_field('telo_affiliates_export');
        echo '<input type="hidden" name="action" value="telo_affiliates_export">';
        echo '<button type="submit" class="button button-secondary">' . esc_html__('Exportar JSON', 'telo-affiliates-lite') . '</button>';
        echo '</form>';

        // Importar JSON
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" enctype="multipart/form-data">';
        wp_nonce_field('telo_affiliates_import', 'telo_affiliates_import_nonce');
        echo '<input type="hidden" name="action" value="telo_affiliates_import">';
        echo '<input type="file" name="telo_affiliates_json" accept=".json,application/json" required> ';
        echo '<button type="submit" class="button button-primary">' . esc_html__('Importar JSON', 'telo-affiliates-lite') . '</button>';
        echo '</form>';
    }

    echo '</div>';
    echo '</div>';
    echo '</div>'; // telo-aff-new-affiliate-header-full

    // Avisos post-importación
    if (isset($_GET['ta_err'])) {
        $ta_err = sanitize_text_field(wp_unslash($_GET['ta_err']));

        if ($ta_err !== '') {
            echo '<div class="notice notice-error is-dismissible"><p>' .
                esc_html($ta_err) .
                '</p></div>';
        }
    } elseif (isset($_GET['ta_imp']) || isset($_GET['ta_upd']) || isset($_GET['ta_skp'])) {
        $imp = isset($_GET['ta_imp']) ? absint($_GET['ta_imp']) : 0;
        $upd = isset($_GET['ta_upd']) ? absint($_GET['ta_upd']) : 0;
        $skp = isset($_GET['ta_skp']) ? absint($_GET['ta_skp']) : 0;

        echo '<div class="notice notice-success is-dismissible"><p>' .
            sprintf(
                esc_html__('Importados: %1$d · Actualizados: %2$d · Omitidos: %3$d', 'telo-affiliates-lite'),
                $imp,
                $upd,
                $skp
            ) .
            '</p></div>';
    }

    if (class_exists('TELO_Affiliates_List_Table')) {
        $list_table = new TELO_Affiliates_List_Table();
        $list_table->prepare_items();

        echo '<form method="get" style="margin-bottom:8px;">';
        echo '<input type="hidden" name="page" value="telo-affiliates-lite" />';
        $list_table->search_box(__('Buscar afiliado', 'telo-affiliates-lite'), 'telo-aff');
        echo '</form>';

        $list_table->display();
    } else {
        echo '<div class="notice notice-error"><p>' .
            esc_html__('No se ha encontrado la clase TELO_Affiliates_List_Table. Verifica includes/admin-list-table.php.', 'telo-affiliates-lite') .
            '</p></div>';
    }

    // ---------------------------
    // Borrar afiliado (GET/POST) – LIBRE
    // ---------------------------
    if (isset($_REQUEST['telo_delete_affiliate'])) {
        check_admin_referer('telo_affiliates_delete', 'telo_affiliates_delete_nonce');

        $slug_to_delete = sanitize_key(wp_unslash($_REQUEST['telo_delete_affiliate']));
        $affiliates     = get_option('telo_affiliates', array());

        if (isset($affiliates[$slug_to_delete])) {
            unset($affiliates[$slug_to_delete]);
            update_option('telo_affiliates', $affiliates);

            echo '<div class="notice notice-success is-dismissible"><p>' .
                sprintf(esc_html__('Afiliado %s eliminado.', 'telo-affiliates-lite'), esc_html($slug_to_delete)) .
                '</p></div>';
        } else {
            echo '<div class="notice notice-warning is-dismissible"><p>' .
                esc_html__('El afiliado ya no existe o fue eliminado.', 'telo-affiliates-lite') .
                '</p></div>';
        }
    }

    // Historial de pagos (solo lectura; si nunca hubo pagos, quedará vacío)
    $payouts = get_option('telo_affiliates_payouts', array());
    usort($payouts, function ($a, $b) {
        return ($b['timestamp'] ?? 0) <=> ($a['timestamp'] ?? 0);
    });

    echo '<h2 style="margin-top:28px;">' . esc_html__('Historial de pagos', 'telo-affiliates-lite') . '</h2>';

    if (empty($payouts)) {
        echo '<p>' . esc_html__('Aún no hay registros de pagos.', 'telo-affiliates-lite') . '</p>';
    } else {
        echo '<table class="widefat fixed striped"><thead><tr>
                <th>' . esc_html__('Fecha', 'telo-affiliates-lite') . '</th>
                <th>' . esc_html__('Afiliado', 'telo-affiliates-lite') . '</th>
                <th>' . esc_html__('Pedidos incluidos', 'telo-affiliates-lite') . '</th>
                <th>' . esc_html__('Base', 'telo-affiliates-lite') . '</th>
                <th>' . esc_html__('%', 'telo-affiliates-lite') . '</th>
                <th>' . esc_html__('Comisión pagada', 'telo-affiliates-lite') . '</th>
                <th>' . esc_html__('Nota', 'telo-affiliates-lite') . '</th>
              </tr></thead><tbody>';

        foreach ($payouts as $p) {
            $date = !empty($p['timestamp'])
                ? ( function_exists('telo_affiliates_format_datetime') ? telo_affiliates_format_datetime((int) $p['timestamp']) : date_i18n( get_option('date_format').' '.get_option('time_format'), (int)$p['timestamp'] ) )
                : '—';

            $slug    = isset($p['slug']) ? $p['slug'] : '';
            $orders  = isset($p['orders']) && is_array($p['orders']) ? $p['orders'] : array();
            $base    = isset($p['base_total']) ? (float)$p['base_total'] : 0.0;
            $rate    = isset($p['rate']) ? (float)$p['rate'] : 0.0;
            $amount  = isset($p['amount']) ? (float)$p['amount'] : 0.0;
            $note    = isset($p['note']) ? $p['note'] : '';

            $orders_links = array();
            foreach ($orders as $oid) {
                $orders_links[] = '<a href="' . esc_url(admin_url('post.php?post=' . absint($oid) . '&action=edit')) . '" target="_blank" rel="noopener">#' . absint($oid) . '</a>';
            }

            echo '<tr>
                    <td>' . esc_html($date) . '</td>
                    <td>' . esc_html($slug) . '</td>
                    <td>' . ($orders_links ? implode(', ', $orders_links) : '—') . '</td>
                    <td>' . ( function_exists('wc_price') ? wp_kses_post(wc_price($base)) : esc_html(number_format_i18n($base, 2)) ) . '</td>
                    <td>' . esc_html(number_format_i18n($rate, 2)) . '%</td>
                    <td>' . ( function_exists('wc_price') ? wp_kses_post(wc_price($amount)) : esc_html(number_format_i18n($amount, 2)) ) . '</td>
                    <td>' . esc_html($note) . '</td>
                  </tr>';
        }

        echo '</tbody></table>';
    }

    echo '<p>' . sprintf(
        esc_html__('Usa el shortcode %s para mostrar el panel del afiliado en una página.', 'telo-affiliates-lite'),
        '<code>[telo_affiliate_dashboard slug="tu_slug"]</code>'
    ) . '</p>';

    echo '</div>'; // .wrap
}

// ======================
// METABOX EN EL PEDIDO
// ======================
add_action('add_meta_boxes', function () {
    add_meta_box(
        'telo_affiliate_box',
        __('Afiliado', 'telo-affiliates-lite'),
        function ($post) {
            $order = function_exists('wc_get_order') ? wc_get_order($post->ID) : null;
            if (!$order) {
                echo '<p>—</p>';
                return;
            }
            $partner = (string) $order->get_meta('_telo_affiliate', true);
            echo '<p><strong>' . esc_html__('Afiliado asociado:', 'telo-affiliates-lite') . '</strong> ' .
                ($partner ? esc_html($partner) : '<span style="color:#888;">' . esc_html__('Directo', 'telo-affiliates-lite') . '</span>') .
                '</p>';
        },
        'shop_order',
        'side',
        'high'
    );
});

// ======================
// JS: copiar enlace (solo en nuestra página)
// ======================
// justo antes de cerrar el output del shortcode
add_action('admin_enqueue_scripts', function($hook){
    // Cubre todas las pantallas del plugin: telo-affiliates, telo-affiliates-pages, telo-affiliates-requests, etc.
    $on_plugin_screen = ( isset($_GET['page']) && strpos(sanitize_key($_GET['page']), 'telo-affiliates-lite') === 0 );
    if ( ! $on_plugin_screen ) {
        return;
    }

    // Ruta segura al JS
    $base_url = defined('TELO_AFFILIATES_URL') ? TELO_AFFILIATES_URL : plugin_dir_url( __FILE__ );
    $version  = defined('TELO_AFFILIATES_VERSION') ? TELO_AFFILIATES_VERSION : false;

    wp_enqueue_script(
        'telo-aff-admin-copy',
        $base_url . 'assets/js/admin-copy.js',
        array(),
        $version,
        true // en footer
    );

    // Texto “Copiado” i18n
    wp_localize_script( 'telo-aff-admin-copy', 'teloAffI18n', array(
        'copied' => __( 'Copiado', 'telo-affiliates-lite' ),
        'fallbackError' => __( 'No se pudo copiar', 'telo-affiliates-lite' ),
    ) );
});
