<?php
if (! defined('ABSPATH')) exit;


if (function_exists('telo_aff_pro_is_active') && telo_aff_pro_is_active()) {
    return;
}

/** =========================================================
 * Cargar WP_List_Table si no está (para la tabla de páginas)
 * ========================================================= */

if (! class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/** =========================================================
 * Helpers básicos
 * ========================================================= */
if (! function_exists('telo_aff_is_lite')) {
    function telo_aff_is_lite()
    {
        return defined('TELO_AFFILIATES_IS_LITE') && TELO_AFFILIATES_IS_LITE;
    }
}

// Caja de upgrade mínima por si no existe
if (! function_exists('telo_aff_upgrade_box')) {
    function telo_aff_upgrade_box($title = '')
    { ?>
        <div class="telo-aff-upgrade-wrap" style="margin-top:16px;">
            <div class="telo-aff-upgrade-box" style="background:#fff;border:1px solid #ccd0d4;padding:20px;border-radius:6px;">
                <h2><?php echo esc_html($title ?: __('Desbloquea esta función', 'telo-affiliates-lite')); ?></h2>
                <p><?php esc_html_e('Esta característica está disponible en la versión PRO.', 'telo-affiliates-lite'); ?></p>
                <p><a class="button button-primary button-hero" target="_blank" rel="noopener" href="https://licencias.qagencia.com/"><?php esc_html_e('Pásate a la versión PRO', 'telo-affiliates-lite'); ?></a></p>
            </div>
        </div>
<?php }
}

/** =========================================================
 *  Helpers de Páginas de Afiliado (idénticos a tus originales)
 * ========================================================= */
function telo_aff_pages_get_parent_id()
{
    $parent = get_page_by_path('panel-afiliado', OBJECT, 'page');
    if ($parent instanceof WP_Post) return (int) $parent->ID;

    $parent_id = wp_insert_post(array(
        'post_type'   => 'page',
        'post_status' => 'publish',
        'post_parent' => 0,
        'post_title'  => __('Panel de Afiliado', 'telo-affiliates-lite'),
        'post_name'   => 'panel-afiliado',
        'post_content' => '',
    ));
    return (int) $parent_id;
}

function telo_aff_pages_find_by_slug($slug)
{
    $slug = sanitize_key($slug);

    $posts = get_posts(array(
        'post_type'   => 'page',
        'post_status' => array('publish', 'draft', 'pending', 'private', 'trash'),
        'meta_key'    => '_telo_aff_slug',
        'meta_value'  => $slug,
        'numberposts' => 1,
        'orderby'     => 'ID',
        'order'       => 'DESC',
        'suppress_filters' => true,
    ));
    if (!empty($posts)) return $posts[0];

    $by_path = get_page_by_path('panel-afiliado/' . $slug, OBJECT, 'page');
    if ($by_path instanceof WP_Post) return $by_path;

    return false;
}

function telo_aff_pages_create($slug, $name)
{
    $slug = sanitize_key($slug);
    $name = $name ? sanitize_text_field($name) : $slug;

    $parent_id = telo_aff_pages_get_parent_id();
    $existing  = telo_aff_pages_find_by_slug($slug);

    if ($existing instanceof WP_Post) {
        $post_id = (int) $existing->ID;

        if ($existing->post_status === 'trash') {
            wp_untrash_post($post_id);
        }

        wp_update_post(array(
            'ID'          => $post_id,
            'post_parent' => $parent_id,
            'post_title'  => sprintf(__('Panel de Afiliado: %s', 'telo-affiliates-lite'), $name),
            'post_name'   => ($existing->post_name !== $slug) ? $slug : $existing->post_name,
        ));

        update_post_meta($post_id, '_wp_page_template', 'telo-affiliates/affiliate-page.php');
        telo_aff_mark_noindex($post_id);
        update_post_meta($post_id, '_telo_aff_slug', $slug);
        update_post_meta($post_id, '_telo_aff_is_panel', 'yes');

        $content = get_post_field('post_content', $post_id);
        if (strpos($content, '[telo_affiliate_dashboard') === false) {
            $content = trim($content) . "\n\n" . '[telo_affiliate_dashboard slug="' . $slug . '"]';
            wp_update_post(array('ID' => $post_id, 'post_content' => $content));
        }

        return $post_id;
    }

    $postarr = array(
        'post_type'    => 'page',
        'post_status'  => 'publish',
        'post_parent'  => $parent_id,
        'post_title'   => sprintf(__('Panel de Afiliado: %s', 'telo-affiliates-lite'), $name),
        'post_name'    => $slug,
        'post_content' => '[telo_affiliate_dashboard slug="' . $slug . '"]',
        'meta_input'   => array(
            '_wp_page_template' => 'telo-affiliates/affiliate-page.php',
            '_telo_aff_slug'    => $slug,
            '_telo_aff_is_panel' => 'yes',
        ),
    );

    $post_id = wp_insert_post($postarr, true);

    return (!is_wp_error($post_id) && $post_id) ? (int) $post_id : 0;
}

function telo_aff_pages_trash($post_id)
{
    $post = get_post($post_id);
    if (! $post || $post->post_type !== 'page') return false;

    $parent = $post->post_parent ? get_post($post->post_parent) : null;
    if ($parent && $parent->post_name !== 'panel-afiliado') return false;

    wp_trash_post($post_id);
    return true;
}

function telo_aff_pages_restore($post_id)
{
    $post = get_post($post_id);
    if (! $post || $post->post_type !== 'page') return false;

    $ok = false;
    if ($post->post_parent) {
        $parent = get_post($post->post_parent);
        $ok = ($parent && $parent->post_name === 'panel-afiliado');
    }
    if (! $ok) {
        $slug_meta = get_post_meta($post_id, '_telo_aff_slug', true);
        $ok = ! empty($slug_meta);
    }
    if (! $ok) return false;

    wp_untrash_post($post_id);
    return true;
}

function telo_aff_pages_delete_perm($post_id)
{
    $post = get_post($post_id);
    if (! $post || $post->post_type !== 'page') return false;

    $ok = false;
    if ($post->post_parent) {
        $parent = get_post($post->post_parent);
        $ok = ($parent && $parent->post_name === 'panel-afiliado');
    }
    if (! $ok) {
        $slug_meta = get_post_meta($post_id, '_telo_aff_slug', true);
        $ok = ! empty($slug_meta);
    }
    if (! $ok) return false;

    wp_delete_post($post_id, true);
    return true;
}

function telo_aff_mark_noindex($post_id)
{
    if (! $post_id) return;

    update_post_meta($post_id, '_telo_aff_is_panel', 'yes');

    if (defined('WPSEO_VERSION')) {
        update_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', '1');
        update_post_meta($post_id, '_yoast_wpseo_meta-robots-nofollow', '1');
    }

    if (defined('RANK_MATH_VERSION')) {
        $rm_robots = get_post_meta($post_id, 'rank_math_robots', true);
        if (!is_array($rm_robots)) $rm_robots = [];
        $rm_robots = array_unique(array_merge($rm_robots, ['noindex', 'nofollow']));
        update_post_meta($post_id, 'rank_math_robots', $rm_robots);
    }
}

function telo_aff_pages_get_trashed_rows()
{
    $trashed = get_posts(array(
        'post_type'   => 'page',
        'post_status' => 'trash',
        'numberposts' => -1,
        'meta_key'    => '_telo_aff_slug',
        'orderby'     => 'post_title',
        'order'       => 'ASC',
        'suppress_filters' => true,
    ));

    $by_parent = get_posts(array(
        'post_type'   => 'page',
        'post_status' => 'trash',
        'numberposts' => -1,
        'post_parent' => telo_aff_pages_get_parent_id(),
        'suppress_filters' => true,
    ));

    $map = array();
    foreach ($trashed as $p) {
        $map[$p->ID] = $p;
    }
    foreach ($by_parent as $p) {
        $map[$p->ID] = $p;
    }

    return array_values($map);
}

/** =========================================================
 * List Table (como en tu versión)
 * ========================================================= */
class TELO_Affiliate_Pages_Table extends WP_List_Table
{

    protected $view_mode = 'active'; // 'active' | 'trash'
    protected $affiliates = array();
    protected $trash_rows = array();

    public function __construct($view = 'active')
    {
        parent::__construct([
            'singular' => 'affiliate_page',
            'plural'   => 'affiliate_pages',
            'ajax'     => false,
        ]);
        $this->view_mode = ($view === 'trash') ? 'trash' : 'active';
        $this->affiliates = (array) get_option('telo_affiliates', array());
        $this->trash_rows = telo_aff_pages_get_trashed_rows();
    }

    public function get_views()
    {
        $total_aff = count($this->affiliates);
        $trash_cnt = count($this->trash_rows);

        $base = admin_url('admin.php?page=telo-affiliates-pages');

        $views = array();

        $views['active'] = sprintf(
            '<a href="%s"%s>%s <span class="count">(%d)</span></a>',
            esc_url($base),
            $this->view_mode === 'active' ? ' class="current"' : '',
            esc_html__('Activas', 'telo-affiliates-lite'),
            (int) $total_aff
        );

        $views['trash'] = sprintf(
            '<a href="%s"%s>%s <span class="count">(%d)</span></a>',
            esc_url(add_query_arg('view', 'trash', $base)),
            $this->view_mode === 'trash' ? ' class="current"' : '',
            esc_html__('Papelera', 'telo-affiliates-lite'),
            (int) $trash_cnt
        );

        return $views;
    }

    public function get_columns()
    {
        return [
            'name'    => __('Nombre', 'telo-affiliates-lite'),
            'slug'    => __('Slug', 'telo-affiliates-lite'),
            'url'     => __('URL de la página', 'telo-affiliates-lite'),
            'actions' => __('Acciones', 'telo-affiliates-lite'),
        ];
    }

    protected function get_sortable_columns()
    {
        return [
            'name' => ['name', true],
            'slug' => ['slug', false],
        ];
    }

    public function prepare_items()
    {
        $columns  = $this->get_columns();
        $hidden   = [];
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = [$columns, $hidden, $sortable];

        if ($this->view_mode === 'trash') {
            $rows = $this->build_trash_rows();
        } else {
            $rows = $this->build_active_rows();
        }

        if (! empty($_REQUEST['s'])) {
            $s = sanitize_text_field(wp_unslash($_REQUEST['s']));
            $rows = array_values(array_filter($rows, function ($r) use ($s) {
                $name = isset($r['name']) ? $r['name'] : '';
                $slug = isset($r['slug']) ? $r['slug'] : '';
                return (stripos($name, $s) !== false) || (stripos($slug, $s) !== false);
            }));
        }

        $orderby = isset($_REQUEST['orderby']) ? sanitize_key($_REQUEST['orderby']) : 'name';
        $order   = (isset($_REQUEST['order']) && 'desc' === strtolower($_REQUEST['order'])) ? 'desc' : 'asc';

        if (in_array($orderby, ['name', 'slug'], true)) {
            usort($rows, function ($a, $b) use ($orderby, $order) {
                $cmp = strcasecmp((string)$a[$orderby], (string)$b[$orderby]);
                return $order === 'asc' ? $cmp : -$cmp;
            });
        }

        $per_page     = $this->get_items_per_page('telo_aff_pages_per_page', 10);
        $current_page = $this->get_pagenum();
        $total_items  = count($rows);

        $rows_page = array_slice($rows, max(0, ($current_page - 1) * $per_page), $per_page);

        $this->items = $rows_page;

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => max(1, (int)ceil($total_items / max(1, $per_page))),
        ]);
    }

    protected function build_active_rows()
    {
        $rows = [];
        foreach ($this->affiliates as $slug => $data) {
            $name = isset($data['name']) ? (string)$data['name'] : (string)$slug;
            $page = telo_aff_pages_find_by_slug($slug);

            $url   = '';
            $pid   = 0;
            $trash = false;

            if ($page instanceof WP_Post) {
                $pid = (int) $page->ID;
                if ($page->post_status === 'trash') {
                    $trash = true;
                } else {
                    $url = get_permalink($pid);
                }
            }

            $rows[] = [
                'name'   => $name,
                'slug'   => $slug,
                'url'    => $url,
                'pageid' => $pid,
                'trash'  => $trash,
                'view'   => 'active',
            ];
        }
        return $rows;
    }

    protected function build_trash_rows()
    {
        $rows = [];
        $aff = $this->affiliates;

        foreach ($this->trash_rows as $p) {
            $slug_meta = get_post_meta($p->ID, '_telo_aff_slug', true);
            $slug = $slug_meta ? sanitize_key($slug_meta) : sanitize_key($p->post_name);
            $name = isset($aff[$slug]['name']) ? $aff[$slug]['name'] : $slug;

            $rows[] = [
                'name'   => (string) $name,
                'slug'   => (string) $slug,
                'url'    => '',
                'pageid' => (int) $p->ID,
                'trash'  => true,
                'view'   => 'trash',
            ];
        }
        return $rows;
    }

    protected function column_default($item, $column_name)
    {
        switch ($column_name) {
            case 'name':
                return '<strong>' . esc_html($item['name']) . '</strong>';

            case 'slug':
                return '<code>' . esc_html($item['slug']) . '</code>';

            case 'url':
                if ($item['view'] === 'trash') {
                    return '<em>' . esc_html__('En papelera', 'telo-affiliates-lite') . '</em>';
                }
                if (! empty($item['url'])) {
                    return sprintf(
                        '<div style="display:flex;gap:8px;align-items:center;">
                            <input type="text" readonly value="%1$s" style="width:100%%;">
                            <button type="button" class="button button-secondary telo_copy_btn" data-copy="%1$s">%2$s</button>
                            <a class="button" target="_blank" rel="noopener" href="%1$s">%3$s</a>
                         </div>',
                        esc_url($item['url']),
                        esc_html__('Copiar', 'telo-affiliates-lite'),
                        esc_html__('Ver', 'telo-affiliates-lite')
                    );
                }
                return '<em>' . esc_html__('No hay página creada', 'telo-affiliates-lite') . '</em>';

            case 'actions':
                ob_start();

                if ($item['view'] === 'active' && empty($item['url'])) {
                    echo '<form method="post" style="display:inline-block;margin-right:8px;">';
                    wp_nonce_field('telo_aff_pages_create_' . $item['slug']);
                    echo '<input type="hidden" name="telo_aff_slug" value="' . esc_attr($item['slug']) . '">';
                    echo '<button type="submit" name="telo_create_page" class="button button-primary">' .
                        esc_html__('Crear página', 'telo-affiliates-lite') . '</button>';
                    echo '</form>';

                    if ($item['trash']) {
                        $trash_url = esc_url(add_query_arg(['page' => 'telo-affiliates-pages', 'view' => 'trash'], admin_url('admin.php')));
                        echo '<a class="button" href="' . $trash_url . '">' . esc_html__('Ver en Papelera', 'telo-affiliates-lite') . '</a>';
                    }
                }

                if ($item['view'] === 'active' && ! empty($item['pageid']) && ! $item['trash']) {
                    echo '<form method="post" style="display:inline-block;margin-left:8px;" onsubmit="return confirm(\'' .
                        esc_js(__('¿Seguro que quieres enviar esta página a la papelera?', 'telo-affiliates-lite')) .
                        '\');">';
                    wp_nonce_field('telo_aff_pages_delete_' . $item['slug']);
                    echo '<input type="hidden" name="telo_aff_slug" value="' . esc_attr($item['slug']) . '">';
                    echo '<input type="hidden" name="telo_aff_page_id" value="' . absint($item['pageid']) . '">';
                    echo '<button type="submit" name="telo_delete_page" class="button button-secondary" style="color:#b32d2e;">' .
                        esc_html__('Eliminar página', 'telo-affiliates-lite') . '</button>';
                    echo '</form>';
                }

                if ($item['view'] === 'trash') {
                    echo '<form method="post" style="display:inline-block;margin-right:8px;">';
                    wp_nonce_field('telo_aff_pages_restore_' . $item['pageid']);
                    echo '<input type="hidden" name="telo_aff_page_id" value="' . absint($item['pageid']) . '">';
                    echo '<button type="submit" name="telo_restore_page" class="button button-primary">' .
                        esc_html__('Restaurar', 'telo-affiliates-lite') . '</button>';
                    echo '</form>';

                    echo '<form method="post" style="display:inline-block;" onsubmit="return confirm(\'' .
                        esc_js(__('¿Seguro que quieres borrar permanentemente esta página?', 'telo-affiliates-lite')) .
                        '\');">';
                    wp_nonce_field('telo_aff_pages_delete_perm_' . $item['pageid']);
                    echo '<input type="hidden" name="telo_aff_page_id" value="' . absint($item['pageid']) . '">';
                    echo '<button type="submit" name="telo_delete_page_perm" class="button" style="color:#b32d2e;">' .
                        esc_html__('Borrar permanentemente', 'telo-affiliates-lite') . '</button>';
                    echo '</form>';
                }

                return ob_get_clean();
        }
        return '';
    }
}

/** =========================================================
 * Render Páginas — bloqueado en Lite
 * ========================================================= */
function telo_aff_render_locked_pages()
{
    echo '<div class="wrap telo-aff-locked"><h1>' . esc_html__('Páginas de afiliados', 'telo-affiliates-lite') . '</h1>';
    echo '<div class="telo-aff-blurred" style="min-height:260px;border:1px dashed #ccd0d4;background:#f6f7f7;filter:blur(3px);"></div>';
    telo_aff_upgrade_box(__('Crea paneles de afiliado con 1 clic', 'telo-affiliates-lite'));
    echo '</div>';
}

function telo_aff_pages_render()
{
    if (telo_aff_is_lite()) {
        telo_aff_render_locked_pages();
        return;
    }

    if (! current_user_can('manage_options')) {
        wp_die(esc_html__('No tienes permisos suficientes.', 'telo-affiliates-lite'));
    }

    $view = 'active';
    if (isset($_GET['view'])) {
        $view_param = sanitize_key(wp_unslash($_GET['view']));
        if ($view_param === 'trash') {
            $view = 'trash';
        }
    }


    // --- ACCIONES POST (Solo PRO) ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['telo_create_page'], $_POST['telo_aff_slug'])) {
            $slug = sanitize_key(wp_unslash($_POST['telo_aff_slug']));
            check_admin_referer('telo_aff_pages_create_' . $slug);

            $aff = (array) get_option('telo_affiliates', array());
            if (isset($aff[$slug])) {
                $name = isset($aff[$slug]['name']) ? $aff[$slug]['name'] : $slug;
                $pid  = telo_aff_pages_create($slug, $name);
                echo $pid
                    ? '<div class="notice notice-success is-dismissible"><p>' . sprintf(esc_html__('Página creada/restaurada para %s.', 'telo-affiliates-lite'), esc_html($slug)) . '</p></div>'
                    : '<div class="notice notice-error is-dismissible"><p>' . esc_html__('No se pudo crear la página.', 'telo-affiliates-lite') . '</p></div>';
            } else {
                echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Afiliado no encontrado.', 'telo-affiliates-lite') . '</p></div>';
            }
        }

        if (isset($_POST['telo_delete_page'], $_POST['telo_aff_page_id'], $_POST['telo_aff_slug'])) {
            $slug    = sanitize_key(wp_unslash($_POST['telo_aff_slug']));
            $page_id = absint($_POST['telo_aff_page_id']);
            check_admin_referer('telo_aff_pages_delete_' . $slug);

            echo ($page_id && telo_aff_pages_trash($page_id))
                ? '<div class="notice notice-success is-dismissible"><p>' . sprintf(esc_html__('Página movida a la papelera para %s.', 'telo-affiliates-lite'), esc_html($slug)) . '</p></div>'
                : '<div class="notice notice-error is-dismissible"><p>' . esc_html__('No se pudo eliminar la página.', 'telo-affiliates-lite') . '</p></div>';
        }

        if (isset($_POST['telo_restore_page'], $_POST['telo_aff_page_id'])) {
            $page_id = absint($_POST['telo_aff_page_id']);
            check_admin_referer('telo_aff_pages_restore_' . $page_id);

            echo ($page_id && telo_aff_pages_restore($page_id))
                ? '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Página restaurada.', 'telo-affiliates-lite') . '</p></div>'
                : '<div class="notice notice-error is-dismissible"><p>' . esc_html__('No se pudo restaurar la página.', 'telo-affiliates-lite') . '</p></div>';
        }

        if (isset($_POST['telo_delete_page_perm'], $_POST['telo_aff_page_id'])) {
            $page_id = absint($_POST['telo_aff_page_id']);
            check_admin_referer('telo_aff_pages_delete_perm_' . $page_id);

            echo ($page_id && telo_aff_pages_delete_perm($page_id))
                ? '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Página borrada permanentemente.', 'telo-affiliates-lite') . '</p></div>'
                : '<div class="notice notice-error is-dismissible"><p>' . esc_html__('No se pudo borrar permanentemente la página.', 'telo-affiliates-lite') . '</p></div>';
        }

        if (isset($_POST['telo_empty_trash'])) {
            check_admin_referer('telo_aff_pages_empty_trash');

            $trashed = telo_aff_pages_get_trashed_rows();
            $count = 0;
            foreach ($trashed as $p) {
                if (telo_aff_pages_delete_perm($p->ID)) $count++;
            }
            echo '<div class="notice notice-success is-dismissible"><p>' .
                sprintf(esc_html__('Papelera vaciada: %d páginas borradas.', 'telo-affiliates-lite'), (int)$count) .
                '</p></div>';

            $view = 'trash';
        }
    }

    // --- RENDER ---
    echo '<div class="wrap telo-aff-admin">';
    echo '<h1>' . esc_html__('Páginas de afiliados', 'telo-affiliates-lite') . '</h1>';

    $table = new TELO_Affiliate_Pages_Table($view);
    echo '<div class="wp-filter">';
    echo $table->views();
    echo '</div>';

    if ($view === 'trash' && count(telo_aff_pages_get_trashed_rows()) > 0) {
        echo '<form method="post" style="margin:10px 0;">';
        wp_nonce_field('telo_aff_pages_empty_trash');
        echo '<button type="submit" name="telo_empty_trash" class="button button-secondary" ' .
            'onclick="return confirm(\'' . esc_js(__('¿Vaciar papelera? Esta acción no se puede deshacer.', 'telo-affiliates-lite')) . '\');">' .
            esc_html__('Vaciar papelera', 'telo-affiliates-lite') . '</button>';
        echo '</form>';
    }

    $table->prepare_items();

    echo '<form method="get" style="margin-bottom:8px;">';
    echo '<input type="hidden" name="page" value="telo-affiliates-pages" />';
    if ($view === 'trash') echo '<input type="hidden" name="view" value="trash" />';
    $table->search_box(__('Buscar afiliado', 'telo-affiliates-lite'), 'telo-aff-pages');
    echo '</form>';

    $table->display();

    if ($view === 'active') {
        echo '<p style="margin-top:12px;">' .
            esc_html__('Cada página se crea en', 'telo-affiliates-lite') . ' <code>/panel-afiliado/{slug}</code>. ' .
            esc_html__('El contenido usa el shortcode del panel del afiliado y se envuelve en', 'telo-affiliates-lite') .
            ' <code>.telo-aff-page</code> ' . esc_html__('para estilo.', 'telo-affiliates-lite') .
            '</p>';

        echo '<p>' . sprintf(
            esc_html__('Usa el shortcode %s para mostrar el panel del afiliado en una página.', 'telo-affiliates-lite'),
            '<code>[telo_affiliate_dashboard slug="tu_slug"]</code>'
        ) . '</p>';
    }

    echo '</div>';
}

/** =========================================================
 * Pantallas bloqueadas de Import/Export y Cobros (Lite)
 * ========================================================= */
function telo_aff_render_locked_import_export()
{
    echo '<div class="wrap telo-aff-locked"><h1>' . esc_html__('Exportar / Importar', 'telo-affiliates-lite') . '</h1>';
    echo '<div class="telo-aff-blurred" style="min-height:220px;border:1px dashed #ccd0d4;background:#f6f7f7;filter:blur(3px);"></div>';
    telo_aff_upgrade_box(__('Importa y exporta tu red en segundos', 'telo-affiliates-lite'));
    echo '</div>';
}
function telo_aff_render_locked_payments()
{
    echo '<div class="wrap telo-aff-locked"><h1>' . esc_html__('Cobros', 'telo-affiliates-lite') . '</h1>';
    echo '<div class="telo-aff-blurred" style="min-height:220px;border:1px dashed #ccd0d4;background:#f6f7f7;filter:blur(3px);"></div>';
    telo_aff_upgrade_box(__('Gestiona pagos y estados con un clic', 'telo-affiliates-lite'));
    echo '</div>';
}

/** =========================================================
 * (Stubs opcionales) Render reales para PRO si no están en otro include
 * ========================================================= */
if (! function_exists('telo_aff_render_import_export')) {
    function telo_aff_render_import_export()
    {
        echo '<div class="wrap"><h1>' . esc_html__('Exportar / Importar', 'telo-affiliates-lite') . '</h1>';
        echo '<p>' . esc_html__('(Contenido PRO; asegúrate de cargar el include correspondiente en la versión PRO)', 'telo-affiliates-lite') . '</p>';
        echo '</div>';
    }
}
if (! function_exists('telo_aff_render_payments')) {
    function telo_aff_render_payments()
    {
        echo '<div class="wrap"><h1>' . esc_html__('Cobros', 'telo-affiliates-lite') . '</h1>';
        echo '<p>' . esc_html__('(Contenido PRO; asegúrate de cargar el include correspondiente en la versión PRO)', 'telo-affiliates-lite') . '</p>';
        echo '</div>';
    }
}

/** =========================================================
 * JS “Copiar” (solo en esta pantalla)
 * ========================================================= */
add_action('admin_enqueue_scripts', function ($hook) {
    $page = isset($_GET['page'])
        ? sanitize_key(wp_unslash($_GET['page']))
        : '';

    if ($page !== 'telo-affiliates-pages') {
        return;
    }

    // handle vacío, cargado en footer
    wp_register_script(
        'telo-aff-admin-inline',
        '',
        [],
        defined('TELO_AFFILIATES_VERSION') ? TELO_AFFILIATES_VERSION : false,
        true
    );
    wp_enqueue_script('telo-aff-admin-inline');

    $copied = esc_js(__('Copiado', 'telo-affiliates-lite'));
    $js = "
    (function(){
      document.addEventListener('click', function(e){
        var btn = e.target.closest('.telo_copy_btn');
        if(!btn) return;
        var text = btn.getAttribute('data-copy') || '';
        if(!text) return;
        navigator.clipboard.writeText(text).then(function(){
          var original = btn.textContent;
          btn.textContent = '{$copied}';
          btn.disabled = true;
          setTimeout(function(){
            btn.textContent = original;
            btn.disabled = false;
          }, 1500);
        });
      });
    })();";
    wp_add_inline_script('telo-aff-admin-inline', $js);
});


/** =========================================================
 * Robots: noindex en paneles / robots.txt
 * ========================================================= */
add_filter('wp_robots', function ($robots) {
    if (!is_singular('page')) return $robots;

    $post_id = get_queried_object_id();
    if (!$post_id) return $robots;

    // Primero, por meta
    $is_aff_panel = get_post_meta($post_id, '_telo_aff_is_panel', true);

    // Si no hay meta, detecta por parent existente (sin crearlo)
    if (!$is_aff_panel) {
        $parent = get_page_by_path('panel-afiliado', OBJECT, 'page'); // <- NO crea
        if ($parent instanceof WP_Post && (int) wp_get_post_parent_id($post_id) === (int) $parent->ID) {
            $is_aff_panel = 'yes';
        }
    }

    if ($is_aff_panel) {
        $robots['noindex'] = true;
        $robots['nofollow'] = true;
        $robots['max-snippet'] = '0';
        $robots['max-image-preview'] = 'none';
        $robots['max-video-preview'] = '0';
    }
    return $robots;
});
