<?php
// includes/admin-list-table.php
if (! defined('ABSPATH')) exit;

if (! class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class TELO_Affiliates_List_Table extends WP_List_Table
{

    public function __construct()
    {
        parent::__construct([
            'singular' => 'affiliate',
            'plural'   => 'affiliates',
            'ajax'     => false,
        ]);
    }

    public function get_columns()
    {
        return [
            'name'              => __('Nombre', 'telo-affiliates-lite'),
            'slug'              => __('Slug', 'telo-affiliates-lite'),
            'link'              => __('Enlace', 'telo-affiliates-lite'),
            'rate'              => __('% Comisión', 'telo-affiliates-lite'),
            'sales'             => __('Ventas', 'telo-affiliates-lite'),
            'commission_total'  => __('Comisión total', 'telo-affiliates-lite'),
            'pending'           => __('Pendiente de pago', 'telo-affiliates-lite'),
            'actions'           => __('Acciones', 'telo-affiliates-lite'),
        ];
    }

    protected function get_sortable_columns()
    {
        return [
            'name'              => ['name', true],
            'slug'              => ['slug', false],
            'rate'              => ['rate', false],
            'sales'             => ['sales', false],
            'commission_total'  => ['commission_total', false],
            'pending'           => ['pending', false],
        ];
    }

    public function get_no_items()
    {
        esc_html_e('No hay afiliados todavía. Añade el primero con el formulario de arriba.', 'telo-affiliates-lite');
    }

    public function prepare_items()
    {
        $columns  = $this->get_columns();
        $hidden   = [];
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = [$columns, $hidden, $sortable];

        $affiliates = (array) get_option('telo_affiliates', []);
        $rows = [];
        $seen = [];

        foreach ($affiliates as $slug => $data) {
            $slug = sanitize_key($slug);
            if ($slug === '') continue;

            // De-duplicado por slug
            if (isset($seen[$slug])) continue;
            $seen[$slug] = true;

            $name = isset($data['name']) ? (string)$data['name'] : (string)$slug;
            $rate = isset($data['rate']) ? (float)$data['rate'] : 0.0;
            $link = home_url(add_query_arg([defined('TELO_AFF_PARAM') ? TELO_AFF_PARAM : 'pv' => $slug], '/'));

            $rows[] = [
                'slug' => $slug,
                'name' => $name,
                'rate' => $rate,
                'link' => $link,
            ];
        }

        // Buscar por nombre/slug
        if (! empty($_REQUEST['s'])) {
            $s = sanitize_text_field(wp_unslash($_REQUEST['s']));
            $rows = array_values(array_filter($rows, function ($r) use ($s) {
                return (stripos($r['name'], $s) !== false) || (stripos($r['slug'], $s) !== false);
            }));
        }

        // Calcular métricas ANTES de ordenar/paginar
        foreach ($rows as &$row) {
            $orders = function_exists('telo_affiliates_get_orders_by_partner')
                ? telo_affiliates_get_orders_by_partner($row['slug'], ['processing', 'completed'])
                : [];
            $order_count = count($orders);

            if (function_exists('telo_affiliates_calculate_commission')) {
                list($sales_total, $commission_total) = telo_affiliates_calculate_commission($orders, $row['rate']);
            } else {
                $sales_total = 0.0;
                $commission_total = 0.0;
            }

            $unpaid_orders = function_exists('telo_affiliates_get_unpaid_orders_by_partner')
                ? telo_affiliates_get_unpaid_orders_by_partner($row['slug'], ['processing', 'completed'])
                : [];
            $pending_count = count($unpaid_orders);

            if (function_exists('telo_affiliates_calculate_commission')) {
                list($pending_base, $pending_commission) = telo_affiliates_calculate_commission($unpaid_orders, $row['rate']);
            } else {
                $pending_commission = 0.0;
            }

            $row['sales_count']         = (int) $order_count;
            $row['sales_total']         = (float) $sales_total;
            $row['commission_total']    = (float) $commission_total;
            $row['pending_commission']  = (float) $pending_commission;
            $row['pending_count']       = (int) $pending_count;
        }
        unset($row);

        // Orden
        $orderby = isset($_REQUEST['orderby']) ? sanitize_key($_REQUEST['orderby']) : 'name';
        $order   = (isset($_REQUEST['order']) && 'desc' === strtolower($_REQUEST['order'])) ? 'desc' : 'asc';

        if (in_array($orderby, ['name', 'slug', 'rate', 'sales', 'commission_total', 'pending'], true)) {
            usort($rows, function ($a, $b) use ($orderby, $order) {
                switch ($orderby) {
                    case 'rate':
                        $cmp = $a['rate'] <=> $b['rate'];
                        break;
                    case 'sales':
                        $cmp = $a['sales_count'] <=> $b['sales_count'];
                        break;
                    case 'commission_total':
                        $cmp = $a['commission_total'] <=> $b['commission_total'];
                        break;
                    case 'pending':
                        $cmp = $a['pending_commission'] <=> $b['pending_commission'];
                        break;
                    case 'name':
                    case 'slug':
                    default:
                        $cmp = strcasecmp((string)$a[$orderby], (string)$b[$orderby]);
                        break;
                }
                return $order === 'asc' ? $cmp : -$cmp;
            });
        }

        // Paginación (usar misma option que definiste en admin.php)
        $per_page     = $this->get_items_per_page('telo_affiliates_per_page', 10);
        $current_page = max(1, $this->get_pagenum());
        $total_items  = count($rows);

        $offset   = ($current_page - 1) * $per_page;
        $rows_page = array_slice($rows, $offset, $per_page);

        $this->items = $rows_page;

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => max(1, (int)ceil($total_items / max(1, $per_page))),
        ]);
    }

    protected function column_default($item, $column_name)
    {
        switch ($column_name) {
            case 'name':
                return '<strong>' . esc_html($item['name']) . '</strong>';

            case 'slug':
                return '<code>' . esc_html($item['slug']) . '</code>';

            case 'link':
                return sprintf(
                    '<a href="%1$s" target="_blank" rel="noopener">%1$s</a><br>
                     <div style="display:flex;gap:8px;align-items:center;margin-top:4px;">
                        <input type="text" readonly value="%1$s" style="width:100%%;">
                        <button type="button" class="button button-secondary telo_copy_btn" data-copy="%1$s">%2$s</button>
                     </div>',
                    esc_url($item['link']),
                    esc_html__('Copiar', 'telo-affiliates-lite')
                );

            case 'rate':
                return esc_html(number_format_i18n((float)$item['rate'], 2)) . '%';

            case 'sales':
                $sales_total = function_exists('wc_price')
                    ? wp_kses_post(wc_price($item['sales_total']))
                    : esc_html(number_format_i18n($item['sales_total'], 2));

                return esc_html($item['sales_count']) . ' ' . esc_html__('ventas', 'telo-affiliates-lite')
                    . '<br>' . $sales_total;


            case 'commission_total':
                return function_exists('wc_price')
                    ? wp_kses_post(wc_price($item['commission_total']))
                    : esc_html(number_format_i18n($item['commission_total'], 2));


            case 'pending':
                $html = function_exists('wc_price')
                    ? wp_kses_post(wc_price($item['pending_commission']))
                    : esc_html(number_format_i18n($item['pending_commission'], 2));

                if ($item['pending_commission'] > 0) {
                    $html .= ' <small>(' . esc_html($item['pending_count']) . ' ' . esc_html__('pedidos', 'telo-affiliates-lite') . ')</small>';
                }
                return $html;


            case 'actions':
                $is_lite = function_exists('telo_aff_is_lite') ? telo_aff_is_lite() : (defined('TELO_AFFILIATES_IS_LITE') && TELO_AFFILIATES_IS_LITE);

                ob_start();
?>
                <div>
                    <!-- Borrar -->
                    <form method="post" onsubmit="return confirm('<?php echo esc_js(__('¿Seguro que quieres borrar este afiliado? Esto no modifica pedidos ya creados.', 'telo-affiliates-lite')); ?>');" style="display:inline-block;margin:0 8px 8px 0;">
                        <?php wp_nonce_field('telo_affiliates_delete', 'telo_affiliates_delete_nonce'); ?>
                        <input type="hidden" name="telo_delete_affiliate" value="<?php echo esc_attr($item['slug']); ?>">
                        <button type="submit" class="button-link-delete" style="color:#b32d2e;"><?php esc_html_e('Borrar', 'telo-affiliates-lite'); ?></button>
                    </form>

                    <!-- Editar -->
                    <details style="display:block;margin:6px 0;">
                        <summary style="cursor:pointer;"><?php esc_html_e('Editar', 'telo-affiliates-lite'); ?></summary>
                        <form method="post" style="margin-top:8px;">
                            <?php wp_nonce_field('telo_affiliates_update', 'telo_affiliates_update_nonce'); ?>
                            <input type="hidden" name="telo_old_slug" value="<?php echo esc_attr($item['slug']); ?>">

                            <p style="margin:6px 0;">
                                <label><?php esc_html_e('Nombre', 'telo-affiliates-lite'); ?></label><br>
                                <input type="text" name="telo_edit_name" value="<?php echo esc_attr($item['name']); ?>" style="width:100%;">
                            </p>

                            <p style="margin:6px 0;">
                                <label><?php esc_html_e('Slug', 'telo-affiliates-lite'); ?></label><br>
                                <input type="text" name="telo_edit_slug" value="<?php echo esc_attr($item['slug']); ?>" style="width:100%;">
                            </p>

                            <p style="margin:6px 0;">
                                <label><?php esc_html_e('% Comisión', 'telo-affiliates-lite'); ?></label><br>
                                <input type="number" step="0.01" min="0" name="telo_edit_rate" value="<?php echo esc_attr(number_format((float)$item['rate'], 2, '.', '')); ?>" style="width:120px;">
                            </p>

                            <p style="margin:8px 0 0;">
                                <button type="submit" name="telo_update_affiliate" class="button"><?php esc_html_e('Guardar cambios', 'telo-affiliates-lite'); ?></button>
                            </p>
                        </form>
                    </details>

                    <!-- Pagar -->
                    <?php if ((float)$item['pending_commission'] > 0) : ?>
                        <div style="margin-top:6px; padding:8px; border:1px solid #ddd; background:#fafafa;">
                            <div><strong><?php esc_html_e('Pendiente:', 'telo-affiliates-lite'); ?></strong>
                                <?php echo function_exists('wc_price') ? wc_price($item['pending_commission']) : esc_html(number_format_i18n($item['pending_commission'], 2)); ?>
                                (<?php echo esc_html($item['pending_count']); ?> <?php esc_html_e('pedidos', 'telo-affiliates-lite'); ?>)
                            </div>

                            <?php if ($is_lite) : ?>
                                <div class="telo-aff-upgrade-wrap" style="margin-top:6px;">
                                    <div class="telo-aff-upgrade-box" style="background:#fff;border:1px solid #e5e7eb;padding:10px;border-radius:6px;">
                                        <em><?php esc_html_e('Para marcar comisiones pagadas y gestionar cobros necesitas la versión PRO.', 'telo-affiliates-lite'); ?></em>
                                        <p style="margin:.5em 0 0;">
                                            <a class="button button-primary" target="_blank" rel="noopener" href="https://licencias.qagencia.com/">
                                                <?php esc_html_e('Pásate a PRO', 'telo-affiliates-lite'); ?>
                                            </a>
                                        </p>
                                    </div>
                                </div>
                            <?php else : ?>
                                <form method="post" onsubmit="return confirm('<?php echo esc_js(__('¿Confirmas marcar como pagadas estas comisiones?', 'telo-affiliates-lite')); ?>');" style="margin-top:6px;">
                                    <?php wp_nonce_field('telo_affiliates_pay', 'telo_affiliates_pay_nonce'); ?>
                                    <input type="hidden" name="telo_pay_affiliate" value="<?php echo esc_attr($item['slug']); ?>">
                                    <textarea name="telo_pay_note" rows="2" style="width:100%;" placeholder="<?php echo esc_attr__('Nota (opcional): referencia transferencia, etc.', 'telo-affiliates-lite'); ?>"></textarea>
                                    <p style="margin-top:6px;"><button type="submit" class="button button-primary"><?php esc_html_e('Marcar pagado', 'telo-affiliates-lite'); ?></button></p>
                                </form>
                            <?php endif; ?>
                        </div>
                    <?php else: ?>
                        <em style="color:#007cba;"><?php esc_html_e('Al día', 'telo-affiliates-lite'); ?></em>
                    <?php endif; ?>
                </div>
<?php
                return ob_get_clean();
        }
        return '';
    }
}
