<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class taxonomy_note_Settings {

    const OPTION_ENABLED_MAP = 'taxonomy_note_enabled_map';

    /**
     * Constructor.
     */
    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
    }

    /**
     * Add settings page under Settings.
     */
    public function add_settings_page() {
        add_options_page(
            __( 'Taxonomy Note', 'taxonomy-note' ),
            __( 'Taxonomy Note', 'taxonomy-note' ),
            'manage_options',
            'taxonomy-note',
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Register settings.
     */
    public function register_settings() {
        register_setting(
            'taxonomy_note_settings_group',
            self::OPTION_ENABLED_MAP,
            array(
                'type'              => 'array',
                'sanitize_callback' => array( $this, 'sanitize_enabled_map' ),
                'default'           => array(),
            )
        );

        add_settings_section(
            'taxonomy_note_settings_section_main',
            __( 'Configuration', 'taxonomy-note' ),
            array( $this, 'render_main_section_intro' ),
            'taxonomy-note'
        );

        add_settings_field(
            'taxonomy_note_settings_field_matrix',
            __( 'Enable note panel for post types & taxonomies', 'taxonomy-note' ),
            array( $this, 'render_matrix_field' ),
            'taxonomy-note',
            'taxonomy_note_settings_section_main'
        );
    }

    /**
     * Sanitize enabled map.
     *
     * Expected format:
     * array(
     *   'post_type' => array(
     *      'taxonomy' => 1,
     *   ),
     * )
     *
     * @param mixed $value Raw value.
     * @return array
     */
    public function sanitize_enabled_map( $value ) {
        $clean = array();

        if ( ! is_array( $value ) ) {
            return $clean;
        }

        foreach ( $value as $post_type => $tax_list ) {
            $post_type = sanitize_key( $post_type );
            if ( empty( $post_type ) || ! is_array( $tax_list ) ) {
                continue;
            }

            foreach ( $tax_list as $taxonomy => $flag ) {
                $taxonomy = sanitize_key( $taxonomy );
                if ( empty( $taxonomy ) ) {
                    continue;
                }
                if ( '1' === (string) $flag || 1 === $flag ) {
                    $clean[ $post_type ][ $taxonomy ] = 1;
                }
            }
        }

        return $clean;
    }

    /**
     * Render settings section intro.
     */
    public function render_main_section_intro() {
        echo '<p>' . esc_html__( 'Select for which post types and taxonomies you want to show the "term note" panel.', 'taxonomy-note' ) . '</p>';
        echo '<p>' . esc_html__( 'When enabled, the panel will list all terms assigned to that taxonomy for the current post and provide a text input for a note.', 'taxonomy-note' ) . '</p>';
    }

    /**
     * Render matrix of checkboxes: post types vs taxonomies.
     */
    public function render_matrix_field() {
        $enabled_map = get_option( self::OPTION_ENABLED_MAP, array() );

        // Get public post types.
        $post_types = get_post_types(
            array(
                'public' => true,
            ),
            'objects'
        );

        if ( empty( $post_types ) ) {
            echo '<p>' . esc_html__( 'No public post types found.', 'taxonomy-note' ) . '</p>';
            return;
        }

        echo '<table class="widefat striped">';
        echo '<thead>';
        echo '<tr>';
        echo '<th>' . esc_html__( 'Post type', 'taxonomy-note' ) . '</th>';
        echo '<th>' . esc_html__( 'Taxonomies', 'taxonomy-note' ) . '</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ( $post_types as $pt ) {
            $taxonomies = get_object_taxonomies( $pt->name, 'objects' );
            echo '<tr>';
            echo '<td><strong>' . esc_html( $pt->labels->name ) . '</strong><br/><code>' . esc_html( $pt->name ) . '</code></td>';
            echo '<td>';

            if ( empty( $taxonomies ) ) {
                echo '<em>' . esc_html__( 'No taxonomies.', 'taxonomy-note' ) . '</em>';
            } else {
                foreach ( $taxonomies as $tax ) {
                    $checked = isset( $enabled_map[ $pt->name ][ $tax->name ] );
                    $field_name = sprintf(
                        '%s[%s][%s]',
                        esc_attr( self::OPTION_ENABLED_MAP ),
                        esc_attr( $pt->name ),
                        esc_attr( $tax->name )
                    );
                    echo '<label style="display:inline-block;margin-right:12px;margin-bottom:4px;">';
                    echo '<input type="checkbox" name="' . esc_attr($field_name) . '" value="1" ' . checked( $checked, true, false ) . ' />';
                    echo ' ' . esc_html( $tax->labels->name ) . ' <code>' . esc_html( $tax->name ) . '</code>';
                    echo '</label>';
                }
            }

            echo '</td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
    }

    /**
     * Render settings page.
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Taxonomy Note Settings', 'taxonomy-note' ); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'taxonomy_note_settings_group' );
                do_settings_sections( 'taxonomy-note' );
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Get enabled map.
     *
     * @return array
     */
    public function get_enabled_map() {
        $map = get_option( self::OPTION_ENABLED_MAP, array() );
        return is_array( $map ) ? $map : array();
    }
}
