<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class taxonomy_note_Metabox {

	/**
	 * @var taxonomy_note_Settings
	 */
	protected $settings;

	/**
	 * Constructor.
	 *
	 * @param taxonomy_note_Settings $settings Settings instance.
	 */
	public function __construct( taxonomy_note_Settings $settings ) {
		$this->settings = $settings;

		add_action( 'add_meta_boxes', array( $this, 'register_meta_boxes' ) );
		add_action( 'save_post', array( $this, 'save_post' ), 10, 2 );
	}

	/**
	 * Register meta boxes for enabled post type / taxonomy combinations.
	 *
	 * @param string $post_type Post type.
	 */
	public function register_meta_boxes( $post_type ) {
		$enabled_map = $this->settings->get_enabled_map();

		if ( empty( $enabled_map[ $post_type ] ) || ! is_array( $enabled_map[ $post_type ] ) ) {
			return;
		}

		foreach ( $enabled_map[ $post_type ] as $taxonomy => $flag ) {
			if ( ! taxonomy_exists( $taxonomy ) ) {
				continue;
			}

			$tax_obj = get_taxonomy( $taxonomy );
			$title   = sprintf(
				/* translators: 1: taxonomy label */
				__( '%s notes', 'taxonomy-note' ),
				$tax_obj && isset( $tax_obj->labels->name ) ? $tax_obj->labels->name : $taxonomy
			);

			add_meta_box(
				'taxonomy_note_' . $taxonomy . '_notes',
				$title,
				array( $this, 'render_meta_box' ),
				$post_type,
				'normal',
				'default',
				array(
					'taxonomy' => $taxonomy,
				)
			);
		}
	}

	/**
	 * Render meta box for a given taxonomy.
	 *
	 * @param WP_Post $post Post object.
	 * @param array   $box  Metabox args.
	 */
	public function render_meta_box( $post, $box ) {
		$taxonomy = isset( $box['args']['taxonomy'] ) ? $box['args']['taxonomy'] : '';
		if ( empty( $taxonomy ) || ! taxonomy_exists( $taxonomy ) ) {
			echo '<p>' . esc_html__( 'Invalid taxonomy.', 'taxonomy-note' ) . '</p>';
			return;
		}

		wp_nonce_field( 'taxonomy_note_save_notes_' . $taxonomy, 'taxonomy_note_nonce_' . $taxonomy );

		$terms = get_the_terms( $post->ID, $taxonomy );

		// Fetch stored notes: array(term_id => note).
		$meta_key = '_taxonomy_note_notes_' . $taxonomy;
		$stored   = get_post_meta( $post->ID, $meta_key, true );
		if ( ! is_array( $stored ) ) {
			$stored = array();
		}

		if ( empty( $terms ) || is_wp_error( $terms ) ) {
			echo '<p>' . esc_html__( 'No terms assigned yet. Assign some terms in the taxonomy box to see them here.', 'taxonomy-note' ) . '</p>';
			return;
		}

		echo '<p>' . esc_html__( 'For each assigned term, you can set a free text note. If you remove a term from the taxonomy, it will disappear from this panel on the next reload.', 'taxonomy-note' ) . '</p>';
		echo '<table class="widefat striped">';
		echo '<thead><tr>';
		echo '<th>' . esc_html__( 'Term', 'taxonomy-note' ) . '</th>';
		echo '<th>' . esc_html__( 'Note', 'taxonomy-note' ) . '</th>';
		echo '</tr></thead>';
		echo '<tbody>';

		foreach ( $terms as $term ) {
			$note = isset( $stored[ $term->term_id ] ) ? $stored[ $term->term_id ] : '';
			echo '<tr>';
			echo '<td><strong>' . esc_html( $term->name ) . '</strong><br/><code>' . esc_html( $term->slug ) . ' (#' . intval( $term->term_id ) . ')</code></td>';
			echo '<td>';
			printf(
				'<input type="text" class="regular-text" name="taxonomy_note_notes[%1$s][%2$d]" value="%3$s" />',
				esc_attr( $taxonomy ),
				intval( $term->term_id ),
				esc_attr( $note )
			);
			echo '</td>';
			echo '</tr>';
		}

		echo '</tbody>';
		echo '</table>';
	}

	/**
	 * Save notes when post is saved.
	 *
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post    Post object.
	 */
	public function save_post( $post_id, $post ) {

		// Only for non-autosave, non-revision.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		if ( 'revision' === $post->post_type ) {
			return;
		}

		// Permessi base.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		$enabled_map = $this->settings->get_enabled_map();

		if ( empty( $enabled_map[ $post->post_type ] ) || ! is_array( $enabled_map[ $post->post_type ] ) ) {
			return;
		}

		foreach ( $enabled_map[ $post->post_type ] as $taxonomy => $flag ) {

			if ( ! $flag ) {
				continue;
			}

			if ( ! taxonomy_exists( $taxonomy ) ) {
				continue;
			}

			// Check nonce per tassonomia.
			$nonce_field = 'taxonomy_note_nonce_' . $taxonomy;

			if ( ! isset( $_POST[ $nonce_field ] ) ) {
				continue;
			}

			// 🔐 Sanifichiamo SUBITO l'input dal superglobal.
			$nonce_value = isset( $_POST[ $nonce_field ] )
				? sanitize_text_field( wp_unslash( $_POST[ $nonce_field ] ) )
				: '';

			if ( ! wp_verify_nonce( $nonce_value, 'taxonomy_note_save_notes_' . $taxonomy ) ) {
				continue;
			}

			$meta_key = '_taxonomy_note_notes_' . $taxonomy;

			// Get current terms for this taxonomy to avoid storing notes for terms that are no longer assigned.
			$terms    = get_the_terms( $post_id, $taxonomy );
			$term_ids = array();

			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
				foreach ( $terms as $term ) {
					$term_ids[] = (int) $term->term_id;
				}
			}

			$notes_for_tax = array();

			// Recupero sicuro delle note SOLO per questa tassonomia.
			if (
				isset( $_POST['taxonomy_note_notes'] )
				&& isset( $_POST['taxonomy_note_notes'][ $taxonomy ] )
				&& is_array( $_POST['taxonomy_note_notes'][ $taxonomy ] )
			) {
				// 🔐 Anche qui: unslash + sanitize nella stessa espressione.
				$raw_notes_for_tax = array_map(
					'sanitize_text_field',
					wp_unslash( $_POST['taxonomy_note_notes'][ $taxonomy ] )
				);

				foreach ( $raw_notes_for_tax as $term_id => $note ) {
					$term_id = (int) $term_id;

					// Only keep for terms actually assigned to this post.
					if ( ! in_array( $term_id, $term_ids, true ) ) {
						continue;
					}

					// $note è già sanificato da array_map().
					if ( '' !== $note ) {
						$notes_for_tax[ $term_id ] = $note;
					}
				}
			}

			if ( empty( $notes_for_tax ) ) {
				delete_post_meta( $post_id, $meta_key );
			} else {
				update_post_meta( $post_id, $meta_key, $notes_for_tax );
			}
		}
	}
}
