<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://taxonomy.builders/
 * @since      1.0.0
 *
 * @package    Taxonomy_Builders
 * @subpackage Taxonomy_Builders/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Taxonomy_Builders
 * @subpackage Taxonomy_Builders/public
 * @author     Fulfillment Global Corporation <wordpress@fulfillment.global>
 */
class Taxonomy_Builders_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param string $plugin_name The name of the plugin.
	 * @param string $version The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;

		// Override template for WordPress default/custom term archive page , if hook setting IS NOT SET for category/taxonomy archive page.
		add_filter( 'template_include', array( $this, 'txb_override_taxonomy_template' ), 9999, 1 );

		// Append/Prepend page content , with existing post content , on single post page , if hook setting IS NOT SET for single post.
		add_filter( 'the_content', array( $this, 'txb_add_page_content_function' ) );

		// Retrive hook setting fields , when setting IS ADDED for single post.
		add_filter( 'wp', array( $this, 'txb_hook_setting_for_single_page' ) );

		// AJAX callback function for pagination , on term archive page.
		add_action( 'wp_ajax_nopriv_txb_taxonomy_ajax_pagination', array( $this, 'txb_taxonomy_ajax_pagination' ) );
		add_action( 'wp_ajax_txb_taxonomy_ajax_pagination', array( $this, 'txb_taxonomy_ajax_pagination' ) );
	}


	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function txb_public_enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Taxonomy_Builders_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Taxonomy_Builders_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		// Taxonomy builders CSS : public.
		wp_enqueue_style( 'taxonomy-builders-public', plugin_dir_url( __FILE__ ) . 'css/taxonomy-builders-public.css', array(), $this->version );
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function txb_public_enqueue_scripts() {
		global $wp_query;

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Taxonomy_Builders_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Taxonomy_Builders_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		// Taxonomy builders JS : public.
		wp_enqueue_script( 'taxonomy-builders-public', TAXONOMY_BUILDERS_PLUGIN_PATH . 'public/js/taxonomy-builders-public.js', array( 'jquery' ), $this->version, true );

		// Localize data for AJAX pagination.
		$current_term_id       = get_queried_object_id();
		$current_term_taxonomy = get_term( $current_term_id );

		$post_type                  = '';
		$current_term_taxonomy_data = '';

		if ( is_wp_error( $current_term_taxonomy ) ) {
			$error_string = $current_term_taxonomy->get_error_message();
		} else {
			if ( isset( $current_term_taxonomy ) ) {
				$post_type                  = get_taxonomy( $current_term_taxonomy->taxonomy )->object_type[0];
				$current_term_taxonomy_data = $current_term_taxonomy->taxonomy;
			}
		}

		wp_localize_script(
			'taxonomy-builders-public',
			'taxonomybuilders',
			array(
				'ajaxurl'               => admin_url( 'admin-ajax.php' ),
				'current_term_id'       => $current_term_id,
				'current_term_taxonomy' => $current_term_taxonomy_data,
				'post_type'             => $post_type,
				'query_vars'            => wp_json_encode( $wp_query->query ),
			)
		);
	}


	/**
	 * Override template for WordPress default/custom term archive page , if hook setting is not set for category/taxonomy archive page.
	 *
	 * @param  string $template    Template path.
	 * @return string
	 */
	public function txb_override_taxonomy_template( $template ) {

		if ( is_archive() ) {

			$current_term_id = get_queried_object_id();
			$current_term_object   = get_term( $current_term_id );
			$current_term_taxonomy = $current_term_object->taxonomy;
			$txb_all_data = get_term_meta( $current_term_id, 'txb_all_data', true );
			
			if ( ! empty( $txb_all_data ) ) {
				$txb_options = get_option( 'txb_options' );
				$txb_hook_settings_archive = $txb_options['txb_hook_settings_archive'];

				// if ( ! empty( $txb_hook_settings_archive ) ) {
				// 	$txb_hook_settings_unserialize_archive = unserialize( $txb_hook_settings_archive );
				// }

				foreach ( $txb_hook_settings_archive as $key_archive => $value_archive ) {
					if ( $current_term_taxonomy === $value_archive['txb_taxonomy_slug_archive'] ) {
						if ( ! empty( $value_archive['txb_hook_type_archive'] )
							&& ! empty( $value_archive['txb_hook_name_archive'] )
						) {
							if ( 'txb_action_archive' === $value_archive['txb_hook_type_archive'] ) {
								add_action( $value_archive['txb_hook_name_archive'], array( $this, 'txb_display_page_content_archive' ) );
							} else {
								add_filter( $value_archive['txb_hook_name_archive'], array( $this, 'txb_display_page_content_archive' ) );
							}
						} else {
							$template = plugin_dir_path( __FILE__ ) . 'templates/txb-taxonomy.php';
						}
					}
				}
			}
		}

		return $template;
	}


	/**
	 * Dynamic action/filter hook callback function , for category/taxonomy archive page.
	 *
	 * @return void
	 */
	public function txb_display_page_content_archive() {
		$current_term_id = get_queried_object_id();
		$txb_all_data    = get_term_meta( $current_term_id, 'txb_all_data', true );

		if ( ! empty( $txb_all_data ) ) {
			$txb_all_data_array = get_post( $txb_all_data );

			$content = apply_filters( 'the_content', $txb_all_data_array->post_content );
			echo wp_kses_post( $content );
		}
	}

	/**
	 * Append/Prepend page content , with existing post content , on single post page , if hook setting is not set for single post.
	 *
	 * @param  string $content    content.
	 * @return string
	 */
	public function txb_add_page_content_function( $content ) {
		
		if ( is_single() ) {

			global $post,$wpdb;
			
			$txb_append_data_term_hook         = array();
			$txb_append_data_term_hook_details = array();

			// Retrive hook setting fields data and separate out the array for which hook name is NOT ADDED for single post.
			$txb_options = get_option( 'txb_options' );
			$txb_hook_settings_single = $txb_options['txb_hook_settings_single'];
			
			if ( ! empty( $txb_hook_settings_single ) ) {
				// $txb_hook_settings_unserialize_single = unserialize( $txb_hook_settings_single );

				foreach ( $txb_hook_settings_single as $key_single => $value_single ) {
					if ( empty( $value_single['txb_hook_type_single'] )
						|| empty( $value_single['txb_hook_name_single'] )
					) {
						$txb_append_data_term_hook[] = $value_single['txb_taxonomy_slug_single'];
						$txb_append_data_term_hook_details[ $value_single['txb_taxonomy_slug_single'] ] = $txb_hook_settings_single;
					}
				}
			}
			
			$txb_append_data          = array();
			$txb_description_position = array();
			$txb_priority             = array();

			
			$taxonomies = get_taxonomies( '','names' );
			$get_all_term_id = wp_get_post_terms( $post->ID, $taxonomies, array( "fields" => "ids" ) );

			if ( ! empty( $get_all_term_id ) && is_array( $get_all_term_id ) ) {
				foreach ( $get_all_term_id as $get_all_term_id_key => $get_all_term_id_value ) {
					$txb_append_data_check = get_term_meta( $get_all_term_id_value, 'txb_append_data', true );

					if ( ! empty( $txb_append_data_check ) ) {
						$txb_append_data_term_taxonomy       = get_term( $get_all_term_id_value );
						$txb_append_data_term_taxonomy_value = $txb_append_data_term_taxonomy->taxonomy;

						// Separate out array to Append/Prepend page content , for which hook is not added for single post.
						if ( in_array( $txb_append_data_term_taxonomy_value, $txb_append_data_term_hook ) ) {
							$txb_append_data[]          = get_term_meta( $get_all_term_id_value, 'txb_append_data', true );
							$txb_description_position[] = get_term_meta( $get_all_term_id_value, 'txb_description_position', true );
							$txb_priority[]             = get_term_meta( $get_all_term_id_value, 'txb_priority', true );
						}
					}
				}

				// Re-structure array , based on data we get.
				$sort_result = array_map(
					function ( $page, $position, $priority ) {
						return compact( 'page', 'position', 'priority' );
					},
					$txb_append_data,
					$txb_description_position,
					$txb_priority
				);

				// Sort array based on priority.
				usort(
					$sort_result,
					function ( $a, $b ) {
						return $a['priority'] <=> $b['priority'];
					}
				);

				if ( is_array( $sort_result ) && ! empty( $sort_result ) ) {
					$modify_array = array();
					foreach ( $sort_result as $sort_result_key => $sort_result_value ) {
						$modify_array[ $sort_result_value['page'] ][] = $sort_result_value;
					}

					foreach ( $modify_array as $key => $value ) {
						$filter_array[ $key ]['both_yes']    = array_keys( array_column( $value, 'position' ), 'txb-both-yes' );
						$filter_array[ $key ]['prepend_yes'] = array_keys( array_column( $value, 'position' ), 'txb-prepend-yes' );
						$filter_array[ $key ]['append_yes']  = array_keys( array_column( $value, 'position' ), 'txb-append-yes' );
					}

					foreach ( $filter_array as $filter_array_key => $filter_array_value ) {
						if ( ! empty( $filter_array_value['both_yes'] ) ) {
							$a             = $filter_array_value['both_yes'][0];
							$final_array[] = $modify_array[ $filter_array_key ][ $a ];
						} else {
							if ( ! empty( $filter_array_value['prepend_yes'] ) ) {
								$b             = $filter_array_value['prepend_yes'][0];
								$final_array[] = $modify_array[ $filter_array_key ][ $b ];
							}
							if ( ! empty( $filter_array_value['append_yes'] ) ) {
								$c             = $filter_array_value['append_yes'][0];
								$final_array[] = $modify_array[ $filter_array_key ][ $c ];
							}
						}
					}

					$content_append  = '';
					$content_prepend = '';
					$content_both    = '';

					// Get Append/Prepend content and set in different variables.
					foreach ( $final_array as $sort_result_key => $sort_result_value ) {
						if ( 'txb-prepend-yes' === $sort_result_value['position'] ) {
							remove_filter( 'se152488_the_content', array( $this, 'txb_add_page_content' ) );
							$content_prepend .= apply_filters( 'se152488_the_content', get_post_field( 'post_content', $sort_result_value['page'] ) );
						}

						if ( 'txb-both-yes' === $sort_result_value['position'] ) {
							remove_filter( 'se152488_the_content', array( $this, 'txb_add_page_content' ) );
							$content_both .= apply_filters( 'se152488_the_content', get_post_field( 'post_content', $sort_result_value['page'] ) );
						}

						if ( 'txb-append-yes' === $sort_result_value['position'] ) {
							remove_filter( 'se152488_the_content', array( $this, 'txb_add_page_content' ) );
							$content_append .= apply_filters( 'se152488_the_content', get_post_field( 'post_content', $sort_result_value['page'] ) );
						}
					}

					// Concat Append/Prepend content variable into one.
					$content = $content_prepend . $content_both . $content . $content_both . $content_append;
				}
			}
		}

		return preg_replace('/<!--[\s\S]*?-->/', '', $content);
	}

	/**
	 * Retrive hook setting fields , when setting IS ADDED for single post.
	 *
	 * @since    1.0.0
	 */
	public function txb_hook_setting_for_single_page() {
		
		if ( is_single() ) {

			global $post,$wpdb;
			
			$txb_append_data_term_default      = array();
			$txb_append_data_term_hook         = array();
			$txb_append_data_term_hook_details = array();

			// Retrive hook setting fields data and separate out the array for which hook name IS ADDED for single post.
			$txb_options = get_option( 'txb_options' );
			$txb_hook_settings_single = $txb_options['txb_hook_settings_single'];

			if ( ! empty( $txb_hook_settings_single ) && is_array( $txb_hook_settings_single ) ) {
				foreach ( $txb_hook_settings_single as $txb_hook_settings_unserialize_key_single => $txb_hook_settings_unserialize_value_single ) {
					if ( ! empty( $txb_hook_settings_unserialize_value_single['txb_hook_type_single'] )
						&& ! empty( $txb_hook_settings_unserialize_value_single['txb_hook_name_single'] )
					) {
						$txb_append_data_term_hook[] = $txb_hook_settings_unserialize_value_single['txb_taxonomy_slug_single'];
						$txb_append_data_term_hook_details[ $txb_hook_settings_unserialize_value_single['txb_taxonomy_slug_single'] ] = $txb_hook_settings_unserialize_value_single;
					}
				}
			}

			$taxonomies = get_taxonomies( '','names' );
			$get_all_term_id = wp_get_post_terms( $post->ID, $taxonomies, array( "fields" => "ids" ) );

			if ( ! empty( $get_all_term_id ) && is_array( $get_all_term_id ) ) {
				foreach ( $get_all_term_id as $get_all_term_id_key => $get_all_term_id_value ) {
					$txb_append_data_check = get_term_meta( $get_all_term_id_value, 'txb_append_data', true );

					if ( ! empty( $txb_append_data_check ) ) {
						$txb_append_data_term_taxonomy       = get_term( $get_all_term_id_value );
						$txb_append_data_term_taxonomy_value = $txb_append_data_term_taxonomy->taxonomy;

						// Separate the array : Hook setting is added for current single post term.
						if ( in_array( $txb_append_data_term_taxonomy_value, $txb_append_data_term_hook ) ) {
							$txb_append_data_term_default[ $txb_append_data_term_taxonomy_value ] = array(

								'term_id_single'       => $get_all_term_id_value,
								'term_taxonomy_single' => $txb_append_data_term_taxonomy_value,
							);
						}
					}
				}
			}

			// Filter the array and call dynamic action/filter hook callback function.
			foreach ( $txb_append_data_term_hook_details as $txb_append_data_term_hook_details_key => $txb_append_data_term_hook_details_value ) {
				foreach ( $txb_append_data_term_default as $txb_append_data_term_default_key => $txb_append_data_term_default_value ) {
					if ( $txb_append_data_term_hook_details_key === $txb_append_data_term_default_key ) {
						if ( 'txb_action_single' === $txb_append_data_term_hook_details_value['txb_hook_type_single'] ) {
							add_action( $txb_append_data_term_hook_details_value['txb_hook_name_single'], array( $this, 'txb_display_page_content_single' ) );
						} else {
							add_filter( $txb_append_data_term_hook_details_value['txb_hook_name_single'], array( $this, 'txb_display_page_content_single' ) );
						}
					}
				}
			}
		}
	}

	/**
	 * Dynamic action/filter hook callback function , for category/taxonomy archive page..
	 *
	 * @since    1.0.0
	 */
	public function txb_display_page_content_single() {
		global $post,$wpdb;

		$txb_all_data_array_old = '';

		$txb_append_data_term_hook         = array();
		$txb_append_data_term_hook_details = array();

		// Retrive hook setting fields data and separate out the array for which hook name IS ADDED for single post.
		$txb_options = get_option( 'txb_options' );
		$txb_hook_settings_single = $txb_options['txb_hook_settings_single'];

		if ( ! empty( $txb_hook_settings_single ) && is_array( $txb_hook_settings_single ) ) {
			
			foreach ( $txb_hook_settings_single as $txb_hook_settings_unserialize_key_single => $txb_hook_settings_unserialize_value_single ) {
				if ( ! empty( $txb_hook_settings_unserialize_value_single['txb_hook_type_single'] )
					&& ! empty( $txb_hook_settings_unserialize_value_single['txb_hook_name_single'] )
				) {
					$txb_append_data_term_hook[] = $txb_hook_settings_unserialize_value_single['txb_taxonomy_slug_single'];
					$txb_append_data_term_hook_details[ $txb_hook_settings_unserialize_value_single['txb_taxonomy_slug_single'] ] = $txb_hook_settings_unserialize_value_single;
				}
			}
		}

		$taxonomies = get_taxonomies( '','names' );
		$get_all_term_id = wp_get_post_terms( $post->ID, $taxonomies, array( "fields" => "ids" ) );

		if ( ! empty( $get_all_term_id ) && is_array( $get_all_term_id ) ) {
			foreach ( $get_all_term_id as $get_all_term_id_key => $get_all_term_id_value ) {
				$txb_append_data_check = get_term_meta( $get_all_term_id_value, 'txb_append_data', true );

				if ( ! empty( $txb_append_data_check ) ) {
					$txb_append_data_term_taxonomy       = get_term( $get_all_term_id_value );
					$txb_append_data_term_taxonomy_value = $txb_append_data_term_taxonomy->taxonomy;

					// Filter out the array and show content at specific hook.
					if ( in_array( $txb_append_data_term_taxonomy_value, $txb_append_data_term_hook ) ) {
						$current_term_id = $get_all_term_id_value;
						$txb_all_data    = get_term_meta( $current_term_id, 'txb_append_data', true );

						if ( ! empty( $txb_all_data ) ) {
							$txb_all_data_array = get_post( $txb_all_data );

							if ( $txb_all_data_array->ID !== $txb_all_data_array_old ) {
								$content = apply_filters( 'the_content', $txb_all_data_array->post_content );
								echo wp_kses_post( $content );
								$txb_all_data_array_old = $txb_all_data_array->ID;
							}
						}
					}
				}
			}
		}
	}

	/**
	 * AJAX callback function for pagination , on term archive page.
	 */
	public function txb_taxonomy_ajax_pagination() {
		
		// Get page number.
		$query_vars['paged'] = sanitize_text_field( $_POST['page'] );

		// Get POST variables.
		$current_term_id       = sanitize_text_field( $_POST['current_term_id'] );
		$current_term_taxonomy = sanitize_text_field( $_POST['current_term_taxonomy'] );
		$post_type             = sanitize_text_field( $_POST['post_type'] );

		$post_list = new WP_Query(
			array(
				'post_type'   => $post_type,
				'paged'       => $query_vars['paged'],
				'post_status' => 'publish',
				'tax_query'   => array(
					array(
						'taxonomy' => $current_term_taxonomy,
						'field'    => 'term_id',
						'terms'    => $current_term_id,
					),
				),
			)
		);

		$GLOBALS['wp_query'] = $post_list;

		if ( $post_list->have_posts() ) {

			// Show same page content for all pagination page OR First page of pagination.
			$txb_all_data = get_term_meta( $current_term_id, 'txb_all_data', true );
			if ( ! empty( $txb_all_data ) ) {
				$txb_show_page_content = get_term_meta( $current_term_id, 'txb_show_page_content', true );

				if ( 'show-page-content-yes' === $txb_show_page_content || 'admin-ajax.php' === $query_vars['paged'] ) {

					$txb_all_page_content = get_term_meta( $current_term_id, 'txb_all_page_content', true );
					if ( $query_vars['paged'] > 1 && ! empty( $txb_all_page_content ) ) {
						$txb_all_page_content_array = get_post( $txb_all_page_content );
					} else {
						$txb_all_page_content_array = get_post( $txb_all_data );
					} 
					?>

					<div class="txb-taxonomy-page-details-section">
						<div class="txb-taxonomy-page-title-and-image">
							<h4></h4>
							<?php
							$txb_taxonomy_page_image = wp_get_attachment_image_src( get_post_thumbnail_id( $txb_all_page_content_array->ID ), 'large' );
							$txb_taxonomy_page_image = ! empty( $txb_taxonomy_page_image ) ? $txb_taxonomy_page_image[0] : '';
							
							echo(
								has_post_thumbnail( $txb_all_page_content_array->ID )
								? '<img class="txb-taxonomy-page-image" src="' . esc_url( $txb_taxonomy_page_image ) . '">'
								: ''
								// : '<img class="txb-taxonomy-page-image" src="' . plugin_dir_url( __DIR__ ) . 'public/images/no-image.jpg" alt="">'
							);
							?>
						</div>
						<div class="txb-taxonomy-page-content">
							<?php
							$content = apply_filters( 'the_content', get_post_field( 'post_content', $txb_all_page_content_array->ID ) );
							echo wp_kses_post( $content );
							?>
						</div>
					</div>
					<?php
				}

				// Show post grid on term archive page.
				?>
				<div class="card txb-taxonomy-post-card">
					<?php
					while ( $post_list->have_posts() ) {
						$post_list->the_post();
						$txb_taxonomy_page_image = wp_get_attachment_image_src( get_post_thumbnail_id( $post_list->ID ), 'large' );
						$txb_taxonomy_page_image = ! empty( $txb_taxonomy_page_image ) ? $txb_taxonomy_page_image[0] : '';

						echo '<div class="txb-taxonomy-post-image-block">' .
						'<a href="' . esc_url( get_post_permalink( get_the_ID() ) ) . '" class="txb-taxonomy-link">' .
								(
									has_post_thumbnail( get_the_ID() )
									? '<img class="txb-taxonomy-post-image" src="' . esc_url( $txb_taxonomy_page_image ) . '">'
									: ''
								) .
							'<h6 class="txb-taxonomy-post-title">' . esc_attr( get_the_title() ) . '</h6>' .
							'<p class="txb-taxonomy-post-content">' . wp_kses_post( get_the_excerpt() ) . '</p>' .
						'</a>' .
						'</div>';
					}
					?>
				</div>
				<?php

				// Show the numeric pagination.
				$GLOBALS['wp_query']->max_num_pages = $post_list->max_num_pages;

				the_posts_pagination(
					array(
						'screen_reader_text' => ' ',
						'before_page_number' => '<span class="meta-nav screen-reader-text">' . __( 'Page', 'taxonomy-builders' ) . ' </span>',
					)
				);

				wp_reset_query();
				wp_reset_postdata();
			} else {
				// echo '<div class="txb-no-data-found">' . esc_attr__( 'No Data Found', 'taxonomy-builders' ) . '</div>';
			}
		} else {
			echo '<div class="txb-no-data-found">' . esc_attr__( 'No Data Found', 'taxonomy-builders' ) . '</div>';
		}

		wp_die();
	}
}
