<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://taxonomy.builders
 * @since      1.0.0
 *
 * @package    Taxonomy_Builders
 * @subpackage Taxonomy_Builders/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Taxonomy_Builders
 * @subpackage Taxonomy_Builders/admin
 * @author     Author <abc@xyz.com>
 */
class Taxonomy_Builders_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
		
		// Submenu page to add action/filter hook for category/taxonomy archive page and single post.
		add_action( 'admin_menu', array( $this, 'txb_hook_setting' ) );

		// Action to add and save custom column field in taxonomy
		add_action( 'admin_init', array( $this, 'txb_admin_process' ), 100 );

		// Show the feedback popup.
		add_action( 'admin_head', array( $this, 'txb_show_feedback_popup' ) );
		
		// Save next popup show timings.
		add_action( 'wp_ajax_nopriv_txb_save_next_popup_time', array( $this, 'txb_save_next_popup_time' ) );
		add_action( 'wp_ajax_txb_save_next_popup_time', array( $this, 'txb_save_next_popup_time' ) );

		// Remove popup timing , and never show popup again.
		add_action( 'wp_ajax_nopriv_txb_not_show_popup', array( $this, 'txb_not_show_popup' ) );
		add_action( 'wp_ajax_txb_not_show_popup', array( $this, 'txb_not_show_popup' ) );
	}

	/**
	 * Register the stylesheets for the admin-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function txb_admin_enqueue_styles( $hook ) {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Taxonomy_Builders_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Taxonomy_Builders_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		global $pagenow;

		// Taxonomy builders CSS : admin.
		if ( 'term.php' === $pagenow || $hook == 'settings_page_txb-hook-setting' ) {
			wp_register_style( 'taxonomy-builders-admin', TAXONOMY_BUILDERS_PLUGIN_PATH . 'admin/css/taxonomy-builders-admin.css', array(), TAXONOMY_BUILDERS_VERSION );
			wp_enqueue_style( 'taxonomy-builders-admin' );
		}

		if ( 'options-general.php' === $pagenow ) {
			wp_enqueue_style( 'taxonomy-builders-hook-setting', TAXONOMY_BUILDERS_PLUGIN_PATH . 'admin/css/taxonomy-builders-hook-setting.css', array(), TAXONOMY_BUILDERS_VERSION );
			wp_enqueue_style( 'taxonomy-builders-jqueryui', TAXONOMY_BUILDERS_PLUGIN_PATH . 'admin/css/jquery-ui.css', array(), TAXONOMY_BUILDERS_VERSION );
		}
	}


	/**
	 * Register the JavaScript for the admin-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function txb_admin_enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Taxonomy_Builders_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Taxonomy_Builders_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		global $pagenow;

		// Taxonomy builders JS : admin.
		if ( 'term.php' === $pagenow ) {
			wp_enqueue_script( 'taxonomy-builders-admin', TAXONOMY_BUILDERS_PLUGIN_PATH . 'admin/js/taxonomy-builders-admin.js', array( 'jquery' ), $this->version, true );

			$txb_plugin_activation_timestamp = get_option( 'txb_plugin_activation_timestamp', true );
			$txb_first_popup_display         = get_option( 'txb_first_popup_display', true );

			$txb_next_popup_timestamp = get_option( 'txb_next_popup_timestamp', true );
			$txb_next_popup_display   = get_option( 'txb_next_popup_display', true );

			// Localize data for feedback popup.
			wp_localize_script(
				'taxonomy-builders-admin',
				'taxonomybuildersadmin',
				array(
					'ajaxurl'                         => admin_url( 'admin-ajax.php' ),
					'txb_plugin_activation_timestamp' => esc_attr( $txb_plugin_activation_timestamp ),
					'txb_first_popup_display'         => esc_attr( $txb_first_popup_display ),
					'txb_next_popup_timestamp'        => esc_attr( $txb_next_popup_timestamp ),
					'txb_next_popup_display'          => esc_attr( $txb_next_popup_display ),
				)
			);
		}

		if ( 'options-general.php' === $pagenow ) {
			wp_enqueue_script( 'taxonomy-builders-hook-setting', TAXONOMY_BUILDERS_PLUGIN_PATH . 'admin/js/taxonomy-builders-hook-setting.js', array( 'jquery' ), $this->version, true );
			wp_enqueue_script( 'jquery-ui-accordion' );
		}
	}

	/**
	 * Submenu page to add action/filter hook for category/taxonomy archive page and single post.
	 *
	 * @since    1.0.0
	 */
	public function txb_hook_setting() {
		add_submenu_page(
			'options-general.php',
			__( 'Taxonomy.Builders Hook Settings', 'taxonomy-builders' ),
			__( 'Taxonomy.Builders Hook Settings', 'taxonomy-builders' ),
			'manage_options',
			'txb-hook-setting',
			array( $this, 'txb_hook_setting_callback' )
		);
	}

	/**
	 * Created setting fields for action/filter hook for category/taxonomy archive page and single post.
	 *
	 * @since    1.0.0
	 */
	public function txb_hook_setting_callback() {
		include_once( TAXONOMY_BUILDERS_DIR . '/admin/settings/txb-settings.php' );
	}

	/**
	 * Function to add and save custom column field in taxonomy
	 *
	 * @since    1.0.0
	 */
	function txb_admin_process(){

		global $pagenow;

		// Taking some variables
		$cat_args = array(
					'public' 	=> true,
					'show_ui'	=> true,
				);
		$get_taxonomy_slug = get_taxonomies( $cat_args );
		
		if ( ! empty( $get_taxonomy_slug ) && is_array( $get_taxonomy_slug ) ) {
			foreach ( $get_taxonomy_slug as $get_taxonomy_slug_key => $get_taxonomy_slug_value ) {

				// Save taxonomy fields
				add_action( "{$get_taxonomy_slug_value}_edit_form_fields", array( $this, 'txb_edit_taxonomy_field' ), 10, 2 );
				add_action( 'edited_'.$get_taxonomy_slug_value, array( $this, 'txb_save_taxonomy_meta' ) );
			}
		}

		register_setting( 'txb_plugin_options', 'txb_options', array($this, 'txb_validate_options') );

	}

	function txb_validate_options( $input ) {

		$txb_options = get_option( 'txb_options' );

		if( isset( $input['txb_hook_settings_archive'] ) && isset( $input['txb_hook_settings_archive'] ) ) {
			$new_input = txb_clean( $input );
			return $new_input;
		}

		return $input;
	}

	/**
	 * Add form field on edit-taxonomy page
	 * 
	 * @since 1.0.0
	 */
	function txb_edit_taxonomy_field( $term, $taxonomy ) {
		include_once( TAXONOMY_BUILDERS_DIR . '/admin/form-field/edit-form.php' );
	}

	/**
	 * Function to add term field on edit screen
	 * 
	 * @since 1.0.0
	 */
	function txb_save_taxonomy_meta( $term_id ) {

		$data['txb_all_data'] 				= isset( $_POST['txb_all_data'] ) 				? sanitize_text_field( $_POST['txb_all_data'] ) 			: '';
		$data['txb_hide_listing'] 			= isset( $_POST['txb_hide_listing'] ) 			? sanitize_text_field( $_POST['txb_hide_listing'] ) 		: '';
		$data['txb_hide_paging'] 			= isset( $_POST['txb_hide_paging'] ) 			? sanitize_text_field( $_POST['txb_hide_paging'] ) 			: '';
		$data['txb_show_page_content'] 		= isset( $_POST['txb_show_page_content'] ) 		? sanitize_text_field( $_POST['txb_show_page_content'] ) 	: '';
		$data['txb_all_page_content'] 		= isset( $_POST['txb_all_page_content'] ) 		? sanitize_text_field( $_POST['txb_all_page_content'] ) 	: '';
		$data['txb_append_data'] 			= isset( $_POST['txb_append_data'] ) 			? sanitize_text_field( $_POST['txb_append_data'] ) 			: '';
		$data['txb_description_position'] 	= isset( $_POST['txb_description_position'] ) 	? sanitize_text_field( $_POST['txb_description_position'] ) : '';
		$data['txb_priority'] 				= isset( $_POST['txb_priority'] ) 				? sanitize_text_field( $_POST['txb_priority'] ) 			: '';
		
		if( ! empty( $data ) ) {
			foreach ($data as $txb_key => $txb_value) {
				update_term_meta( $term_id, $txb_key, $txb_value );
			}
		}
	}

	/**
	 * Show the feedback popup.
	 *
	 * @since    1.0.0
	 */
	public function txb_show_feedback_popup() {
		?>
		<div style="display: none;" id="txb-modal" class="txb-modal-popup txb-feedback-popup">
			<div class="txb-popup-content">
				<?php
				echo esc_html__( 'Please let us know how we can improve by giving your feedback here!', 'taxonomy-builders' ) . '   ' . '<a target="_blank" href="' . esc_url( 'https://taxonomy.builders' ) . '">' . esc_html__( 'Feedback', 'taxonomy-builders' ) . '</a>';
				?>
			</div>
			<div class="txb-popup-button">
				<?php
				echo '<a onclick="tb_remove();" class="button-primary txb-cancel-btn" href="#">' . esc_html__( 'Cancel', 'taxonomy-builders' ) . '</a>';
				echo '<a onclick="tb_remove();" class="button-primary txb-not-show-btn" href="#">' . esc_html__( "Don't show again", 'taxonomy-builders' ) . '</a>';
				?>
			</div>
		</div>
		<a title="Give your feedback" style="display: none;" href="#TB_inline?&width=250&height=120&inlineId=txb-modal" id="txb-open-popup" class="thickbox txb-open-popup"><?php esc_html_e( 'Give Feedback', 'taxonomy-builders' ); ?></a>

		<?php
	}


	/**
	 * Save the next popup show timings.
	 *
	 * @since    1.0.0
	 */
	public function txb_save_next_popup_time() {
		if ( ! empty( $_POST['txb_next_date'] ) ) {
			update_option( 'txb_next_popup_timestamp', sanitize_text_field( $_POST['txb_next_date'] ) );
			update_option( 'txb_next_popup_display', 'txb_next_popup_no' );
			update_option( 'txb_first_popup_display', 'txb_first_popup_yes' );
			echo '1';
		}
		wp_die();
	}


	/**
	 * Remove the popup timing , and never show popup again.
	 *
	 * @since    1.0.0
	 */
	public function txb_not_show_popup() {
		if ( isset( $_POST['txb_never_show_popup'] ) && 'txb_never' === $_POST['txb_never_show_popup'] ) {
			update_option( 'txb_next_popup_timestamp', '' );
			update_option( 'txb_next_popup_display', '' );
			update_option( 'txb_plugin_activation_timestamp', '' );
			update_option( 'txb_first_popup_display', '' );
			echo '1';
		}
		wp_die();
	}
}
