/**
 * TalkGenAI Article Generation - Unified Job Integration
 * Single form handles both app-based and standalone articles
 */

(function($) {
    'use strict';

    // Unified progress messages (distributed over ~300 seconds / 5 minutes)
    const articleProgressMessages = [
        { text: "🤖 Connecting to AI server...", progress: 5 },
        { text: "⏱️ This can take up to 5 minutes — please wait...", progress: 8 },
        { text: "📝 Analyzing your topic & title...", progress: 12 },
        { text: "🧠 Understanding context & instructions...", progress: 16 },
        { text: "🔗 Gathering internal links for your article...", progress: 20 },
        { text: "📚 Researching & outlining article structure...", progress: 25 },
        { text: "🗂️ Building detailed content outline...", progress: 30 },
        { text: "✍️ Writing article content...", progress: 35 },
        { text: "📖 Crafting paragraphs & sections...", progress: 42 },
        { text: "📝 Expanding key points & arguments...", progress: 48 },
        { text: "🔗 Embedding internal & external links...", progress: 54 },
        { text: "❓ Generating FAQ section...", progress: 60 },
        { text: "🎨 Formatting & styling text...", progress: 66 },
        { text: "📊 Adding meta description & SEO...", progress: 72 },
        { text: "🔍 Checking grammar & coherence...", progress: 78 },
        { text: "🛠️ Polishing content & structure...", progress: 83 },
        { text: "📦 Preparing HTML output...", progress: 88 },
        { text: "✨ Finalizing your article...", progress: 93 },
        { text: "🔒 Running final quality checks...", progress: 97 },
        { text: "🚀 Almost there...", progress: 100 },
        // Extended waiting guidance (keeps user informed beyond estimate)
        { text: "⏳ Almost done — please keep this tab open...", progress: 102 },
        { text: "📡 Do not close or refresh — final content is coming...", progress: 104 },
        { text: "🧭 Still working — longer articles take a bit more time...", progress: 106 },
        { text: "🔒 Final checks in progress...", progress: 108 },
        { text: "🚀 Wrapping up — thanks for your patience...", progress: 110 }
    ];

    // Timer for simulated smooth progress
    let articleProgressInterval = null;
    let articleTimerInterval = null;
    let articleProgressMessageIndex = 0;
    let activeProgressMessages = null;
    let articleStartTime = null;

    // Estimated total duration in seconds (matches the 300s / 5 minute message distribution)
    const ESTIMATED_DURATION_S = 300;

    /**
     * Format seconds into M:SS string
     */
    function formatTime(totalSeconds) {
        const m = Math.floor(totalSeconds / 60);
        const s = Math.floor(totalSeconds % 60);
        return m + ':' + (s < 10 ? '0' : '') + s;
    }

    /**
     * Start simulated smooth progress (ignores backend sparse updates)
     * @param {Array} messages - Progress messages array to use
     */
    function startArticleProgress(messages) {
        stopArticleProgress(); // Clear any existing timer

        activeProgressMessages = messages || articleProgressMessages;
        articleProgressMessageIndex = 0;
        articleStartTime = Date.now();

        // Distribute messages over ~160 seconds
        const totalMs = ESTIMATED_DURATION_S * 1000;
        const steps = Math.max(1, activeProgressMessages.length - 1);
        const stepMs = Math.floor(totalMs / steps);

        // Show first message immediately
        TalkGenAI_JobManager.showProgress(
            activeProgressMessages[0].text,
            activeProgressMessages[0].progress
        );
        TalkGenAI_JobManager.updateProgressTimer(
            formatTime(0) + ' / ~' + formatTime(ESTIMATED_DURATION_S)
        );

        // Advance through messages
        articleProgressInterval = setInterval(() => {
            if (articleProgressMessageIndex < activeProgressMessages.length - 1) {
                articleProgressMessageIndex++;
                const msg = activeProgressMessages[articleProgressMessageIndex];
                TalkGenAI_JobManager.showProgress(msg.text, msg.progress);
            }
        }, stepMs);

        // Update elapsed timer every second
        articleTimerInterval = setInterval(() => {
            if (!articleStartTime) return;
            const elapsed = Math.floor((Date.now() - articleStartTime) / 1000);
            TalkGenAI_JobManager.updateProgressTimer(
                formatTime(elapsed) + ' / ~' + formatTime(ESTIMATED_DURATION_S)
            );
        }, 1000);
    }

    /**
     * Stop simulated progress and show completion
     */
    function stopArticleProgress() {
        if (articleProgressInterval) {
            clearInterval(articleProgressInterval);
            articleProgressInterval = null;
        }
        if (articleTimerInterval) {
            clearInterval(articleTimerInterval);
            articleTimerInterval = null;
        }
        activeProgressMessages = null;
        articleStartTime = null;
    }

    /**
     * Parse manual internal URLs from textarea (one per line)
     * Supports LTR (URL|Anchor) and RTL (Anchor|URL) auto-detection
     */
    function parseInternalUrls(rawText) {
        return rawText
            .split('\n')
            .map(line => {
                line = line.trim();
                if (!line) {
                    return null;
                }

                // Split by pipe
                const parts = line.split('|').map(p => p.trim());
                if (parts.length === 0) {
                    return null;
                }

                // Auto-detect format based on which part contains http://https://
                let url = '';
                let anchor = '';

                if (parts.length === 1) {
                    // No pipe: just URL
                    if (parts[0].startsWith('http://') || parts[0].startsWith('https://')) {
                        url = parts[0];
                    } else {
                        return null; // Invalid
                    }
                } else {
                    // Has pipe: detect order
                    const firstIsUrl = parts[0].startsWith('http://') || parts[0].startsWith('https://');
                    const secondIsUrl = parts[1].startsWith('http://') || parts[1].startsWith('https://');

                    if (firstIsUrl && !secondIsUrl) {
                        // LTR format: URL|Anchor
                        url = parts[0];
                        anchor = parts.slice(1).join('|');
                    } else if (!firstIsUrl && secondIsUrl) {
                        // RTL format: Anchor|URL
                        anchor = parts[0];
                        url = parts.slice(1).join('|');
                    } else if (firstIsUrl && secondIsUrl) {
                        // Both are URLs? Take first as URL, second as anchor (unlikely but handle)
                        url = parts[0];
                        anchor = parts[1];
                    } else {
                        // Neither is URL - invalid
                        return null;
                    }
                }

                if (!url || !(url.startsWith('http://') || url.startsWith('https://'))) {
                    return null;
                }

                return { url: url, anchor: anchor };
            })
            .filter(item => item !== null);
    }

    /**
     * Article Generator with Job System (Unified)
     */
    window.TalkGenAI_ArticleJob = {
        /**
         * Safely parse JSON that might arrive as an escaped string
         */
        _safeParse: function(raw) {
            if (typeof raw !== 'string') return raw;
            try { return JSON.parse(raw); } catch (e1) {}
            try {
                let txt = raw.trim();
                if ((txt.startsWith('"') && txt.endsWith('"')) || (txt.startsWith("'") && txt.endsWith("'"))) {
                    txt = txt.slice(1, -1);
                }
                txt = txt.replace(/\\"/g, '"').replace(/\\\\/g, '\\');
                return JSON.parse(txt);
            } catch (e2) {
                return null;
            }
        },

        /**
         * Generate article using unified form (handles both app-based and standalone)
         */
        generateUnifiedArticle: async function() {
            try {
                // Guard against double-submit
                const $btn = $('#generate-article-btn');
                if ($btn.prop('disabled') || $btn.data('tgaiBusy')) {
                    return;
                }
                $btn.data('tgaiBusy', true);

                // Determine mode from radio toggle
                const articleSource = $('input[name="article_source"]:checked').val() || 'app-based';
                const isAppBased = (articleSource === 'app-based');

                // Collect shared form fields
                const articleTitle = $('#article_title').val().trim();
                const topic = $('#article_topic').val().trim() || articleTitle;

                const articleLength = $('#article_length').val() || 'medium';
                const internalUrlsRaw = $('#internal_urls').val().trim();
                const autoInternalLinks = $('#auto_internal_links').is(':checked');
                const includeExternalLink = $('#include_external_link').is(':checked');
                const includeFaq = $('#include_faq').is(':checked');
                const createImage = !$('#create_image').prop('disabled') && $('#create_image').is(':checked');

                // Parse manual internal URLs
                const internalUrls = parseInternalUrls(internalUrlsRaw);

                if (isAppBased) {
                    // === APP-BASED MODE ===
                    const appId = $('#target_app').val();
                    if (!appId) {
                        alert('Please select an app to generate an article for!');
                        $btn.data('tgaiBusy', false);
                        return;
                    }

                    // Validate title/topic (should be auto-populated, but user may have cleared)
                    if (!articleTitle) {
                        alert('Please enter an article title!');
                        $btn.data('tgaiBusy', false);
                        return;
                    }

                    // Load app data for app_spec and app_html
                    let appSpec = null;
                    let appHtml = '';

                    try {
                        const raw = await $.ajax({
                            url: ajaxurl,
                            type: 'POST',
                            dataType: 'text',
                            data: {
                                action: 'talkgenai_load_app',
                                nonce: (typeof talkgenai_nonce !== 'undefined') ? talkgenai_nonce : (talkgenai_ajax && talkgenai_ajax.nonce),
                                app_id: appId,
                                id: appId
                            }
                        });
                        const resp = this._safeParse(raw) || {};
                        const data = resp && (resp.data || resp.app || resp);

                        appHtml = (data && data.html_content) ? data.html_content : '';

                        // Extract app_spec from the loaded data for AI customization
                        if (data && data.json_spec) {
                            appSpec = data.json_spec;
                            if (typeof appSpec === 'string') {
                                try {
                                    appSpec = JSON.parse(appSpec);
                                } catch (e) {
                                    console.warn('Failed to parse app_spec:', e);
                                    appSpec = null;
                                }
                            }
                        }
                    } catch (e) {
                        console.error('Error loading app data:', e);
                    }

                    const appPageUrl = $('#app_url').val() || '';

                    // Build input data for app-based article job
                    const inputData = {
                        app_title: articleTitle,
                        app_description: topic,
                        article_length: articleLength,
                        app_spec: appSpec,
                        app_url: appPageUrl,
                        app_html: appHtml,
                        // New enhanced fields
                        article_title: articleTitle,
                        topic: topic,
                        internal_urls: internalUrls,
                        include_faq: includeFaq,
                        include_external_link: includeExternalLink,
                        auto_internal_links: autoInternalLinks,
                        ...(createImage ? { create_image: true } : {}),
                    };

                    try { console.log('TalkGenAI_ArticleJob: createJob app-based payload', inputData); } catch (e) {}

                    // Disable button and start progress
                    $btn.prop('disabled', true);
                    startArticleProgress(articleProgressMessages);

                    // Create job as 'article' type (backend stays the same)
                    await TalkGenAI_JobManager.createJob('article', inputData, {
                        onProgress: (progress) => {
                            console.log('Backend progress update:', progress.percent + '%', '(using simulated progress instead)');
                        },
                        onSuccess: (result) => {
                            this._handleSuccess(result, $btn);
                        },
                        onError: (error, errorData) => {
                            this._handleError(error, errorData, $btn);
                        }
                    });

                } else {
                    // === STANDALONE MODE ===
                    if (!articleTitle) {
                        alert('Please enter an article title!');
                        $btn.data('tgaiBusy', false);
                        return;
                    }
                    const writingStyleId = ($('#tgai_writing_style_id').val() || '').trim();
                    const inputData = {
                        article_title: articleTitle,
                        topic: topic,
                        internal_urls: internalUrls,
                        include_faq: includeFaq,
                        article_length: articleLength,
                        auto_internal_links: autoInternalLinks,
                        include_external_link: includeExternalLink,
                        is_standalone: true,
                        ...(createImage ? { create_image: true } : {}),
                        ...(writingStyleId ? { writing_style_id: writingStyleId } : {}),
                    };

                    try { console.log('TalkGenAI_ArticleJob: createJob standalone payload', inputData); } catch (e) {}

                    // Disable button and start progress
                    $btn.prop('disabled', true);
                    startArticleProgress(articleProgressMessages);

                    // Create job as 'standalone_article' type
                    await TalkGenAI_JobManager.createJob('standalone_article', inputData, {
                        onProgress: (progress) => {
                            console.log('Backend progress update:', progress.percent + '%', '(using simulated progress instead)');
                        },
                        onSuccess: (result) => {
                            this._handleSuccess(result, $btn);
                        },
                        onError: (error, errorData) => {
                            this._handleError(error, errorData, $btn);
                        }
                    });
                }

            } catch (error) {
                console.error('Article generation error:', error);

                stopArticleProgress();
                TalkGenAI_JobManager.hideProgress();

                this.showNotification('Error: ' + error.message, 'error');
                $('#generate-article-btn').prop('disabled', false).data('tgaiBusy', false);
            }
        },

        /**
         * Handle successful article generation (shared by both modes)
         */
        _handleSuccess: function(result, $btn) {
            try { console.log('Article job success result:', result); } catch(e) {}

            stopArticleProgress();
            TalkGenAI_JobManager.hideProgress();

            this.displayArticle(result);
            this.loadArticleHistory();

            $btn.prop('disabled', false).data('tgaiBusy', false);

            // Handle generated image if present
            const generatedImage = result.generated_image || (result.json_spec && result.json_spec.generated_image);
            if (generatedImage && generatedImage.data) {
                this._handleGeneratedImage(generatedImage, result);
            }

            // Success message + FAQ badge
            const hasSchemas = result.html && result.html.includes('application/ld+json');
            const hasFaqSchema = (result.faq_schema || (result.json_spec && result.json_spec.faq_schema)) && typeof (result.faq_schema || result.json_spec.faq_schema) === 'object';
            if (hasFaqSchema) {
                $('#faq-schema-badge').show();
                this.showNotification('Article generated with validated FAQ schema!', 'success');
            } else {
                $('#faq-schema-badge').hide();
                this.showNotification('Article generated successfully!', 'success');
            }
        },

        /**
         * Handle a generated image: swap placeholder src with Blob URL and show upload button
         */
        _handleGeneratedImage: function(imageData, result) {
            try {
                // Decode base64 → Uint8Array → Blob → object URL for instant preview
                const binary = atob(imageData.data);
                const bytes = new Uint8Array(binary.length);
                for (let i = 0; i < binary.length; i++) {
                    bytes[i] = binary.charCodeAt(i);
                }
                const blob = new Blob([bytes], { type: imageData.mime_type || 'image/webp' });
                const blobUrl = URL.createObjectURL(blob);

                // Update the placeholder <img> src in the article preview
                const $placeholder = $('#article-content figure.tgai-article-image-placeholder img, #article-result-area figure.tgai-article-image-placeholder img');
                if ($placeholder.length) {
                    $placeholder.attr('src', blobUrl);
                }

                // Store job ID for upload
                this._lastJobId = result.job_id || (result.json_spec && result.json_spec.job_id) || null;

                // Disable Create Draft until the auto-upload completes
                const $draftBtn = $('#create-draft-btn');
                this._draftBtnOriginalHtml = $draftBtn.html();
                $draftBtn.prop('disabled', true)
                    .html('<span class="dashicons dashicons-upload" style="vertical-align:middle;margin-right:3px;"></span>'
                        + '<span class="talkgenai-draft-btn-label">Uploading image…</span>');

                // Auto-upload to WordPress in the background
                this._autoUploadImage();

            } catch (e) {
                console.warn('[TalkGenAI] Failed to process generated image:', e);
            }
        },

        /**
         * Inject/show the "Upload Image to WordPress" button near the article result actions
         */
        _showImageUploadButton: function() {
            // Remove any existing button first
            $('#tgai-upload-image-btn-wrap').remove();

            const $wrap = $('<div id="tgai-upload-image-btn-wrap" style="margin-top:10px;"></div>');
            const $btn = $('<button type="button" class="button" id="tgai-upload-image-btn">'
                + '<span class="dashicons dashicons-format-image" style="vertical-align:middle;margin-right:4px;"></span>'
                + 'Upload Image to WordPress'
                + '</button>');
            $wrap.append($btn);

            // Insert after the create-draft-group or at the end of the result area header actions
            if ($('#create-draft-group').length) {
                $('#create-draft-group').after($wrap);
            } else if ($('#article-result-area').length) {
                $('#article-result-area').prepend($wrap);
            }

            const self = this;
            $btn.off('click').on('click', function() {
                self._uploadImageToWordPress($(this));
            });
        },

        /**
         * Automatically upload the generated image to WP Media Library.
         * Called immediately after article generation completes.
         * Disables Create Draft until done; shows retry on failure.
         */
        _autoUploadImage: function() {
            if (!this._lastJobId) {
                // No job ID — re-enable draft button and bail
                $('#create-draft-btn').prop('disabled', false).html(this._draftBtnOriginalHtml || 'Create Post Draft');
                return;
            }

            // Status indicator below the draft button group
            $('#tgai-image-upload-status').remove();
            const $status = $('<div id="tgai-image-upload-status" style="margin-top:8px;font-size:13px;color:#555;">'
                + '<span class="dashicons dashicons-update spin" style="vertical-align:middle;margin-right:4px;font-size:16px;"></span>'
                + 'Uploading image to WordPress…'
                + '</div>');
            if ($('#faq-schema-badge').length) {
                $('#faq-schema-badge').after($status);
            } else if ($('#create-draft-group').length) {
                $('#create-draft-group').after($status);
            }

            const self = this;
            const $draftBtn = $('#create-draft-btn');
            const originalHtml = this._draftBtnOriginalHtml;

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'talkgenai_upload_article_image',
                    nonce: (typeof talkgenai_nonce !== 'undefined') ? talkgenai_nonce : (talkgenai_ajax && talkgenai_ajax.nonce),
                    job_id: self._lastJobId,
                },
                success: function(response) {
                    if (response.success && response.data) {
                        const data = response.data;

                        // Update DOM preview with real WP URL
                        if (data.url) {
                            $('#article-content figure.tgai-article-image-placeholder img, #article-result-area figure.tgai-article-image-placeholder img')
                                .attr('src', data.url);

                            // Strip the placeholder figure — PHP will inject a clean
                            // centered <img> before the first <h2> using attachment_id.
                            if (self._lastArticleHtml) {
                                const $tmp = $('<div>').html(self._lastArticleHtml);
                                const $fig = $tmp.find('figure[data-tgai-image-placeholder="1"]');
                                if ($fig.length) {
                                    $fig.remove();
                                    self._lastArticleHtml = $tmp.html();
                                }
                            }
                        }

                        // Store for safety patch fallback in createDraft
                        self._lastAttachmentUrl = data.url || null;
                        self._lastAttachmentId = data.attachment_id || null;
                        self._lastAttachmentAlt = data.alt || null;

                        // Re-enable Create Draft
                        $draftBtn.prop('disabled', false).html(originalHtml);

                        // Update status + Media Library link
                        let statusHtml = '<span style="color:#00a32a;">✅ Image uploaded to WordPress</span>';
                        if (data.attachment_id) {
                            const adminBase = ajaxurl.replace(/admin-ajax\.php$/, '');
                            const editUrl = adminBase + 'post.php?post=' + data.attachment_id + '&action=edit';
                            statusHtml += ' — <a href="' + editUrl + '" target="_blank">View in Media Library →</a>';
                        }
                        $status.html(statusHtml);

                    } else {
                        // Upload failed — re-enable draft (without image) + offer retry
                        $draftBtn.prop('disabled', false).html(originalHtml);
                        const msg = (response.data && response.data.message) || 'Upload failed.';
                        self._showUploadRetry($status, msg);
                    }
                },
                error: function(xhr, status, error) {
                    $draftBtn.prop('disabled', false).html(originalHtml);
                    self._showUploadRetry($status, 'Network error: ' + error);
                }
            });
        },

        _showUploadRetry: function($status, msg) {
            const self = this;
            $status.html('<span style="color:#d63638;">⚠️ ' + msg + '</span> ');
            const $retry = $('<button type="button" class="button button-small" style="margin-left:6px;">Retry Upload</button>');
            $status.append($retry);
            $retry.on('click', function() {
                // Re-disable draft button and retry
                const $draftBtn = $('#create-draft-btn');
                $draftBtn.prop('disabled', true)
                    .html('<span class="dashicons dashicons-upload" style="vertical-align:middle;margin-right:3px;"></span>'
                        + '<span class="talkgenai-draft-btn-label">Uploading image…</span>');
                $status.html('<span class="dashicons dashicons-update spin" style="vertical-align:middle;margin-right:4px;font-size:16px;"></span>Uploading image to WordPress…');
                self._autoUploadImage();
            });
        },

        /**
         * Upload the generated image to the WordPress media library via AJAX (manual fallback)
         */
        _uploadImageToWordPress: function($btn) {
            if (!this._lastJobId) {
                this.showNotification('Cannot upload: job ID not found. Try re-generating the article.', 'error');
                return;
            }

            $btn.prop('disabled', true).text('Uploading…');

            const self = this;
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'talkgenai_upload_article_image',
                    nonce: (typeof talkgenai_nonce !== 'undefined') ? talkgenai_nonce : (talkgenai_ajax && talkgenai_ajax.nonce),
                    job_id: this._lastJobId,
                },
                success: function(response) {
                    if (response.success && response.data) {
                        const data = response.data;
                        $btn.text('✅ Uploaded!');
                        self.showNotification('Image uploaded to Media Library!', 'success');

                        // Replace blob URL with the real WP media URL in the preview
                        if (data.url) {
                            // 1. Update the live preview DOM
                            const $placeholder = $('#article-content figure.tgai-article-image-placeholder img, #article-result-area figure.tgai-article-image-placeholder img');
                            $placeholder.attr('src', data.url);

                            // 2. Strip the placeholder figure — PHP will inject a clean
                            //    centered <img> before the first <h2> using attachment_id.
                            if (self._lastArticleHtml) {
                                const $tmp = $('<div>').html(self._lastArticleHtml);
                                const $fig = $tmp.find('figure[data-tgai-image-placeholder="1"]');
                                if ($fig.length) {
                                    $fig.remove();
                                    self._lastArticleHtml = $tmp.html();
                                }
                            }
                        }

                        // Show a link to edit the attachment
                        if (data.attachment_id) {
                            // Derive admin base URL from ajaxurl (/wp-admin/admin-ajax.php → /wp-admin/)
                            const adminBase = (typeof ajaxurl !== 'undefined') ? ajaxurl.replace(/admin-ajax\.php$/, '') : '/wp-admin/';
                            const editUrl = adminBase + 'post.php?post=' + data.attachment_id + '&action=edit';
                            const $link = $('<a class="button" target="_blank" style="margin-left:8px;">View in Media Library →</a>').attr('href', editUrl);
                            $btn.after($link);
                        }
                    } else {
                        $btn.prop('disabled', false).text('Upload Image to WordPress');
                        const msg = (response.data && response.data.message) || 'Upload failed.';
                        self.showNotification('Error: ' + msg, 'error');
                    }
                },
                error: function(xhr, status, error) {
                    $btn.prop('disabled', false).text('Upload Image to WordPress');
                    self.showNotification('Network error: ' + error, 'error');
                }
            });
        },

        /**
         * Handle article generation error (shared by both modes)
         */
        _handleError: function(error, errorData, $btn) {
            stopArticleProgress();
            TalkGenAI_JobManager.hideProgress();

            if (errorData && errorData.error === 'ai_provider_error' && errorData.ai_message) {
                const safeMessage = String(errorData.ai_message || 'AI provider error. Please try again.');
                this.showNotification(safeMessage, 'error');
            } else if (errorData && errorData.ai_message) {
                this.showNotification('Error: ' + error, 'error');

                const isHtml = errorData.is_html || (errorData.ai_message && errorData.ai_message.trim().startsWith('<'));
                if (isHtml) {
                    if ($('#article-content').length) {
                        $('#article-content').html(errorData.ai_message);
                        $('#article-result-area').show().css('display','block');
                    } else if ($('#article-result-area').length) {
                        $('#article-result-area').html(errorData.ai_message).show().css('display','block');
                    } else {
                        const $container = $('<div id="article-result-area" style="margin-top:16px;"></div>');
                        $container.html(errorData.ai_message);
                        $('.wrap h1').after($container);
                    }
                }
            } else {
                this.showNotification('Error: ' + error, 'error');
            }

            $btn.prop('disabled', false).data('tgaiBusy', false);
        },

        /**
         * Display generated article
         */
        displayArticle: function(result) {
            try { console.log('displayArticle payload:', result); } catch(e) {}
            // Normalize fields - check 'html' first (current format from API)
            const html = result.html || result.article_html || result.article || '';
            // Meta description can be at top level or inside json_spec
            const metaDescription = result.meta_description || result.metaDescription ||
                                   (result.json_spec && result.json_spec.meta_description) || '';
            const wordCount = result.word_count || result.wordCount ||
                             (result.json_spec && result.json_spec.word_count) || 0;
            const provider = result.provider || result.ai_provider ||
                            (result.json_spec && result.json_spec.provider) || '';
            const generationTime = typeof result.generation_time !== 'undefined' ? result.generation_time :
                                  (result.json_spec && result.json_spec.generation_time) || result.generationTime;

            console.log('[displayArticle] Extracted HTML length:', html.length);
            console.log('[displayArticle] Meta description:', metaDescription?.substring(0, 50));
            console.log('[displayArticle] Word count:', wordCount);

            // Simple sanitizer (mirror of admin.js sanitizeHtml)
            function sanitizeHtml(unsafeHtml) {
                try {
                    const template = document.createElement('template');
                    template.innerHTML = String(unsafeHtml || '');
                    const disallowedTags = new Set(['SCRIPT', 'STYLE', 'IFRAME', 'OBJECT', 'EMBED', 'LINK']);
                    const walker = document.createTreeWalker(template.content, NodeFilter.SHOW_ELEMENT, null);
                    const toRemove = [];
                    while (walker.nextNode()) {
                        const el = walker.currentNode;
                        if (disallowedTags.has(el.tagName)) {
                            toRemove.push(el);
                            continue;
                        }
                        [...el.attributes].forEach(attr => {
                            const name = attr.name.toLowerCase();
                            const value = (attr.value || '').trim();
                            if (name.startsWith('on')) {
                                el.removeAttribute(attr.name);
                            } else if ((name === 'href' || name === 'src') && /^javascript:/i.test(value)) {
                                el.removeAttribute(attr.name);
                            }
                        });
                    }
                    toRemove.forEach(n => n.remove());
                    return template.innerHTML;
                } catch (_) { return String(unsafeHtml || ''); }
            }

            const safeHtml = sanitizeHtml(html);

            // Visual HTML panel (preserve tabs and buttons)
            let ensuredContainer = false;
            if ($('#article-content').length) {
                $('#article-content').html(safeHtml);
                $('#article-result-area').show().css('display','block');
                // Ensure Visual tab is selected
                $('.talkgenai-tab-button').removeClass('active');
                $('.talkgenai-tab-panel').removeClass('active');
                $('.talkgenai-tab-button[data-tab="visual"]').addClass('active');
                $('#visual-tab').addClass('active');
                ensuredContainer = true;
            } else if ($('#article-result-area').length) {
                const $vc = $('<div class="article-visual-content"></div>').html(safeHtml);
                $('#article-result-area').append($vc).show().css('display','block');
                $('.talkgenai-tab-button').removeClass('active');
                $('.talkgenai-tab-panel').removeClass('active');
                $('.talkgenai-tab-button[data-tab="visual"]').addClass('active');
                $('#visual-tab').addClass('active');
                ensuredContainer = true;
            } else if ($('#article-output').length) {
                const $above = $('#article-output');
                const $container = $('<div id="article-result-area" style="margin-top:16px;"></div>');
                $container.html(safeHtml);
                $above.before($container);
                ensuredContainer = true;
            } else if ($('.wrap h1').length) {
                const $container = $('<div id="article-result-area" style="margin-top:16px;"></div>');
                $container.html(safeHtml);
                $('.wrap h1').after($container);
                ensuredContainer = true;
            }

            // Code view panel - include FAQ schema as JSON-LD if available
            if ($('#article-code').length) {
                var faqSchema = result.faq_schema || (result.json_spec && result.json_spec.faq_schema) || null;
                var codeHtml = html;
                if (faqSchema && typeof faqSchema === 'object') {
                    codeHtml += '\n\n<!-- FAQ Schema (JSON-LD) -->\n<script type="application/ld+json">\n' + JSON.stringify(faqSchema, null, 2) + '\n</script>';
                }
                $('#article-code').text(codeHtml);
            }

            // Ensure Visual tab is active and visible
            try {
                const $tabs = $('.talkgenai-tab-button');
                if ($tabs.length) {
                    $tabs.removeClass('active');
                    $('.talkgenai-tab-panel').removeClass('active');
                    $('.talkgenai-tab-button[data-tab="visual"]').addClass('active');
                    $('#visual-tab').addClass('active');
                }
            } catch (e) { /* ignore */ }

            // Try to push content into a WordPress editor if present
            try {
                if (window.tinymce && tinymce.editors && tinymce.editors.length) {
                    tinymce.editors.forEach(function(ed){
                        if (ed && typeof ed.setContent === 'function') {
                            ed.setContent(html);
                        }
                    });
                }
            } catch (e) { /* ignore */ }

            // Fallback: populate common textareas if they exist
            const textareas = [
                'textarea#article_output',
                'textarea[name="article_output"]',
                'textarea#talkgenai_article',
                'textarea[name="talkgenai_article"]'
            ];
            textareas.forEach(function(sel){
                if ($(sel).length) {
                    $(sel).val(html);
                }
            });

            // Meta description targets
            if ($('#meta-description-text').length) {
                $('#meta-description-text').val(metaDescription);
                const len = (metaDescription || '').length;
                if ($('#meta-description-length').length) {
                    $('#meta-description-length').text(len);
                }
                $('.talkgenai-meta-description-section').show();
            }
            if ($('#meta-description').length) {
                $('#meta-description').text(metaDescription);
            }

            if ($('#word-count').length) {
                $('#word-count').text(wordCount);
            }

            if ($('#ai-provider').length) {
                $('#ai-provider').text(provider);
            }

            if ($('#generation-time').length && typeof generationTime !== 'undefined') {
                try {
                    const t = Number(generationTime);
                    $('#generation-time').text(isNaN(t) ? generationTime : (t.toFixed ? t.toFixed(1) + 's' : String(t)));
                } catch (e) {
                    $('#generation-time').text(String(generationTime));
                }
            }

            // Scroll to article/result area if present, else fallback
            if ($('#article-result-area').length) {
                $('html, body').animate({ scrollTop: $('#article-result-area').offset().top - 100 }, 500);
            } else if ($('#article-output').length) {
                $('html, body').animate({ scrollTop: $('#article-output').offset().top - 100 }, 500);
            }

            // Store article data for Create Draft feature
            this._lastArticleHtml = html;
            this._uploadedImageData = null; // Reset on new article — set again after upload
            this._lastAttachmentUrl = null; // Reset; set after successful image upload
            this._lastAttachmentId = null;
            this._lastAttachmentAlt = null;
            this._lastMetaDescription = metaDescription;
            // FAQ schema is returned inside json_spec from the backend
            this._lastFaqSchema = result.faq_schema || (result.json_spec && result.json_spec.faq_schema) || null;
            // Focus keyword for Yoast/RankMath and image alt/title
            this._lastFocusKeyword = result.focus_keyword || (result.json_spec && result.json_spec.focus_keyword) || '';

            // Show the Create Draft button group and re-enable it for the new article
            $('#create-draft-group').show();
            $('#create-draft-group .talkgenai-draft-link').remove();
            // Reset toggle to Post (default)
            $('#draft-post-type').val('post');
            $('.talkgenai-toggle-btn').removeClass('active').filter('[data-value="post"]').addClass('active');
            $('#create-draft-btn')
                .prop('disabled', false)
                .html('<span class="dashicons dashicons-welcome-write-blog" style="vertical-align:middle;margin-right:3px;"></span><span class="talkgenai-draft-btn-label">Create Post Draft</span>');

            if (!html) {
                this.showNotification('Job completed but article content was empty. Check server logs.', 'warning');
            }

            // Wire copy buttons (bottom + top)
            $('#copy-visual-btn, #copy-visual-btn-top').off('click').on('click', function() {
                // Copy as rich HTML so pasting into editors preserves formatting
                if (navigator.clipboard && navigator.clipboard.write) {
                    var htmlBlob = new Blob([html], { type: 'text/html' });
                    var textBlob = new Blob([$('#article-content').text()], { type: 'text/plain' });
                    navigator.clipboard.write([new ClipboardItem({
                        'text/html': htmlBlob,
                        'text/plain': textBlob
                    })]).then(function() {
                        if (window.TalkGenAI_ArticleJob) {
                            window.TalkGenAI_ArticleJob.showNotification('Visual content copied!', 'success');
                        }
                    }).catch(function() {
                        // Fallback to plain text
                        navigator.clipboard.writeText($('#article-content').text()).then(function() {
                            if (window.TalkGenAI_ArticleJob) {
                                window.TalkGenAI_ArticleJob.showNotification('Visual content copied (text format)!', 'success');
                            }
                        });
                    });
                } else {
                    navigator.clipboard.writeText($('#article-content').text()).then(function() {
                        if (window.TalkGenAI_ArticleJob) {
                            window.TalkGenAI_ArticleJob.showNotification('Visual content copied!', 'success');
                        }
                    });
                }
            });
            $('#copy-code-btn, #copy-code-btn-top').off('click').on('click', function() {
                navigator.clipboard.writeText(html).then(() => {
                    if (window.TalkGenAI_ArticleJob) {
                        window.TalkGenAI_ArticleJob.showNotification('Code copied!', 'success');
                    }
                });
            });
            $('#copy-meta-description-btn, #copy-meta-btn-top').off('click').on('click', function() {
                navigator.clipboard.writeText(metaDescription).then(() => {
                    if (window.TalkGenAI_ArticleJob) {
                        window.TalkGenAI_ArticleJob.showNotification('Meta Description copied!', 'success');
                    }
                });
            });

            // Show/hide top meta copy button based on whether meta description exists
            if (metaDescription) {
                $('#copy-meta-btn-top').show();
            }
        },

        /**
         * Load article history
         */
        loadArticleHistory: async function() {
            try {
                console.log('[loadArticleHistory] Fetching article history...');
                const results = await TalkGenAI_JobManager.getHistory('article', 50);
                console.log('[loadArticleHistory] Results count:', results ? results.length : 0);

                // Populate dropdown
                TalkGenAI_JobManager.populateHistoryDropdown(
                    'article-history-select',
                    results,
                    (result) => {
                        const date = new Date(result.created_at).toLocaleDateString();
                        const title = (result.app_title || (result.result_data && result.result_data.app_title) || 'Untitled').toString();
                        const wordCountRaw = (result.word_count != null) ? result.word_count : (result.result_data ? result.result_data.word_count : 0);
                        const wordCount = Number.isFinite(Number(wordCountRaw)) ? Number(wordCountRaw) : 0;
                        return `${title} (${wordCount} words - ${date})`;
                    }
                );

                // Update count
                $('#article-history-count').text(results.length);

            } catch (error) {
                console.error('[loadArticleHistory] Failed to load article history:', error);
            }
        },

        /**
         * Load article from history
         */
        loadFromHistory: async function(resultId) {
            try {
                TalkGenAI_JobManager.showProgress('Loading article...', 50);
                const result = await TalkGenAI_JobManager.loadResult(resultId);
                TalkGenAI_JobManager.hideProgress();

                // Restore article title to the form field so Create Draft works
                const restoredTitle = result.app_title ||
                    (result.result_data && result.result_data.app_title) ||
                    (result.result_data && result.result_data.title) || '';
                if (restoredTitle && !$('#article_title').val().trim()) {
                    $('#article_title').val(restoredTitle);
                }

                this.displayArticle(result.result_data);
                this.showNotification('Article loaded successfully', 'success');
            } catch (error) {
                TalkGenAI_JobManager.hideProgress();
                this.showNotification('Error loading article: ' + error.message, 'error');
            }
        },

        /**
         * Delete article from history
         */
        deleteFromHistory: async function(resultId) {
            if (!confirm('Are you sure you want to delete this article?')) {
                return;
            }

            try {
                await TalkGenAI_JobManager.deleteResult(resultId);
                this.loadArticleHistory();
                $('#article-history-select').val('');
                this.showNotification('Article deleted successfully', 'success');
            } catch (error) {
                this.showNotification('Error deleting article: ' + error.message, 'error');
            }
        },

        /**
         * Create a WordPress draft post/page from the last generated article
         */
        createDraft: function() {
            const title = $('#article_title').val().trim();
            const postType = $('#draft-post-type').val() || 'post';
            let fullHtml = this._lastArticleHtml || '';
            const metaDescription = this._lastMetaDescription || '';

            if (!title) {
                this.showNotification('Please enter an article title before creating a draft.', 'warning');
                return;
            }
            if (!fullHtml) {
                this.showNotification('No article content available. Generate an article first.', 'warning');
                return;
            }

            // _lastArticleHtml was already patched in-place by the upload callback.
            // Safety fallback: strip any leftover placeholder figure — PHP injects the image.
            if (fullHtml.includes('data-tgai-image-placeholder="1"')) {
                fullHtml = fullHtml.replace(
                    /<figure[^>]*tgai-article-image-placeholder[^>]*>[\s\S]*?<\/figure>/,
                    ''
                );
            }

            // Strip any leftover <script> tags from content (legacy safety)
            const content = fullHtml.replace(/<script\s+type=["']application\/ld\+json["'][^>]*>[\s\S]*?<\/script>/gi, '').trim();

            // FAQ schema is stored as a separate object, not embedded in HTML
            const faqSchema = this._lastFaqSchema ? JSON.stringify(this._lastFaqSchema) : '';

            const $btn = $('#create-draft-btn');
            $btn.prop('disabled', true);
            const originalText = $btn.html();
            $btn.html('<span class="dashicons dashicons-update spin" style="vertical-align:middle;margin-right:3px;"></span> Creating...');

            const self = this;

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'talkgenai_create_draft',
                    nonce: (typeof talkgenai_nonce !== 'undefined') ? talkgenai_nonce : (talkgenai_ajax && talkgenai_ajax.nonce),
                    post_type: postType,
                    title: title,
                    content: content,
                    meta_description: metaDescription,
                    focus_keyword: this._lastFocusKeyword || '',
                    faq_schema: faqSchema,
                    attachment_id: this._lastAttachmentId || 0
                },
                success: function(response) {
                    if (response.success && response.data) {
                        const editLink = response.data.edit_link;

                        // Keep button disabled to prevent duplicate drafts
                        $btn.html('<span class="dashicons dashicons-yes" style="vertical-align:middle;margin-right:3px;"></span> Draft Created');

                        // Remove any previous draft link, then show a prominent Edit button
                        $('#create-draft-group .talkgenai-draft-link').remove();
                        if (editLink) {
                            var typeLabel = postType === 'page' ? 'Page' : 'Post';
                            $('<a class="button talkgenai-edit-draft-btn talkgenai-draft-link" href="' + editLink + '" target="_blank"><span class="dashicons dashicons-edit"></span> Edit ' + typeLabel + ' \u2192</a>')
                                .insertAfter($btn.closest('.talkgenai-create-draft-group'));
                        }
                    } else {
                        // On error, re-enable so user can retry
                        $btn.prop('disabled', false).html(originalText);
                        const errMsg = (response.data && response.data.message) || 'Failed to create draft.';
                        self.showNotification(errMsg, 'error');
                    }
                },
                error: function(xhr, status, error) {
                    // On network error, re-enable so user can retry
                    $btn.prop('disabled', false).html(originalText);
                    self.showNotification('Network error: ' + error, 'error');
                }
            });
        },

        /**
         * Show notification
         */
        showNotification: function(message, type = 'info') {
            const $notice = $('<div class="notice is-dismissible"></div>')
                .addClass('notice-' + type);
            const $p = $('<p></p>').text(message);
            $notice.append($p);

            $('.wrap h1').after($notice);

            setTimeout(() => {
                $notice.fadeOut(() => $notice.remove());
            }, 5000);
        },

        /**
         * Initialize event listeners
         */
        init: function() {
            try { console.log('TalkGenAI_ArticleJob: init bindings (unified)'); } catch(e) {}

            // Unbind any legacy handlers from old two-tab structure
            $(document).off('submit', '#talkgenai-articles-form');
            $(document).off('click', '#generate-article-btn');
            $(document).off('submit', '#talkgenai-standalone-articles-form');
            $(document).off('click', '#generate-standalone-article-btn');
            $('#talkgenai-articles-form').off('submit');
            $('#talkgenai-standalone-articles-form').off('submit');
            $('#generate-article-btn').off('click');
            $('#generate-standalone-article-btn').off('click');

            // Bind global event for article display
            try {
                $(document).off('talkgenai:article-generated').on('talkgenai:article-generated', (e, payload) => {
                    this.displayArticle(payload);
                    try {
                        if ($('#article-result-area').length) {
                            $('#article-result-area').show().css('display','block');
                            $('html, body').animate({ scrollTop: $('#article-result-area').offset().top - 100 }, 400);
                        }
                    } catch(_) {}
                });
            } catch(_) {}

            // Unified form handlers
            $(document)
                .off('click.talkgenai', '#generate-article-btn')
                .on('click.talkgenai', '#generate-article-btn', (e) => {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    this.generateUnifiedArticle();
                });

            $(document)
                .off('submit.talkgenai', '#talkgenai-unified-article-form')
                .on('submit.talkgenai', '#talkgenai-unified-article-form', (e) => {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    this.generateUnifiedArticle();
                });

            // Load from history
            $('#load-article-from-history-btn').on('click', () => {
                const resultId = $('#article-history-select').val();
                if (resultId) {
                    this.loadFromHistory(resultId);
                } else {
                    alert('Please select an article from history');
                }
            });

            // Delete from history
            $('#delete-article-from-history-btn').on('click', () => {
                const resultId = $('#article-history-select').val();
                if (resultId) {
                    this.deleteFromHistory(resultId);
                } else {
                    alert('Please select an article to delete');
                }
            });

            // Create Draft button
            $(document).on('click', '#create-draft-btn', (e) => {
                e.preventDefault();
                this.createDraft();
            });

            // Post/Page type toggle
            $(document).on('click', '.talkgenai-toggle-btn', function() {
                var $btn = $(this);
                $btn.closest('.talkgenai-post-type-toggle').find('.talkgenai-toggle-btn').removeClass('active');
                $btn.addClass('active');
                var value = $btn.data('value');
                $('#draft-post-type').val(value);
                var typeLabel = value === 'page' ? 'Page' : 'Post';
                $('#create-draft-btn .talkgenai-draft-btn-label').text('Create ' + typeLabel + ' Draft');
            });

            // More options dropdown toggle
            $(document).on('click', '#more-options-btn', function(e) {
                e.stopPropagation();
                var $dropdown = $('#more-options-dropdown');
                var isOpen = $dropdown.hasClass('open');
                $dropdown.toggleClass('open', !isOpen);
                $(this).attr('aria-expanded', String(!isOpen));
            });

            // Close dropdown when clicking outside
            $(document).on('click.tgai-more-options', function(e) {
                if (!$(e.target).closest('.talkgenai-more-options-wrapper').length) {
                    $('#more-options-dropdown').removeClass('open');
                    $('#more-options-btn').attr('aria-expanded', 'false');
                }
            });

            // Close dropdown after any item is clicked
            $(document).on('click', '.talkgenai-dropdown-item', function() {
                $('#more-options-dropdown').removeClass('open');
                $('#more-options-btn').attr('aria-expanded', 'false');
            });

            // Load history on page load
            this.loadArticleHistory();
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        if ($('#generate-article-btn').length > 0) {
            TalkGenAI_ArticleJob.init();
        }
    });

})(jQuery);
