<?php
/**
 * Plugin Name: Tahc AI Hybrid Chat
 * Plugin URI: https://tahc.ai/wordpress
 * Description: Add AI-powered hybrid chat to your WordPress site. Connects to the Tahc service (tahc.ai) for AI chat functionality. This plugin is free and open source.
 * Version: 1.0.4
 * Author: Tahc
 * Author URI: https://tahc.ai
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: tahc-ai-hybrid-chat
 * Requires at least: 5.0
 * Requires PHP: 7.2
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('TAHC_CHAT_VERSION', '1.0.4');
define('TAHC_CHAT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('TAHC_CHAT_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Main Tahc Chat Widget Class
 */
class Tahc_Chat_Widget {

    /**
     * Constructor
     */
    public function __construct() {
        // Add settings page to admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Register settings
        add_action('admin_init', array($this, 'register_settings'));

        // Enqueue admin styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));

        // Enqueue widget script
        add_action('wp_enqueue_scripts', array($this, 'enqueue_chat_widget'));

        // Add data attribute to script tag
        add_filter('script_loader_tag', array($this, 'add_widget_data_attribute'), 10, 3);

        // Add settings link on plugins page
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));
    }

    /**
     * Enqueue admin styles on plugin settings page
     */
    public function enqueue_admin_styles($hook) {
        // Only load on our settings page
        if ('settings_page_tahc-chat-settings' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'tahc-admin-style',
            TAHC_CHAT_PLUGIN_URL . 'admin-style.css',
            array(),
            TAHC_CHAT_VERSION
        );
    }

    /**
     * Add admin menu page
     */
    public function add_admin_menu() {
        add_options_page(
            'Tahc Chat Settings',
            'Tahc Chat',
            'manage_options',
            'tahc-chat-settings',
            array($this, 'settings_page')
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('tahc_chat_settings', 'tahc_workspace_id', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));

        register_setting('tahc_chat_settings', 'tahc_chat_enabled', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true
        ));
    }

    /**
     * Settings page HTML
     */
    public function settings_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }

        // Get workspace ID
        $workspace_id = get_option('tahc_workspace_id', '');
        $chat_enabled = get_option('tahc_chat_enabled', true);

        // WordPress handles settings-updated via the Settings API
        settings_errors('tahc_chat_settings');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div class="tahc-welcome-banner">
                <h2><?php esc_html_e('Welcome to Tahc Hybrid Chat!', 'tahc-ai-hybrid-chat'); ?></h2>
                <p>
                    <?php esc_html_e('Combine the speed of AI with the empathy of real humans to provide exceptional customer support.', 'tahc-ai-hybrid-chat'); ?>
                </p>
                <p class="tahc-subtitle">
                    <strong><?php esc_html_e('Free to try!', 'tahc-ai-hybrid-chat'); ?></strong> <?php esc_html_e('The Tahc service offers a 14-day free trial - no credit card required.', 'tahc-ai-hybrid-chat'); ?>
                </p>
            </div>

            <div class="tahc-card">
                <h2><?php esc_html_e('Quick Setup', 'tahc-ai-hybrid-chat'); ?></h2>

                <?php if (empty($workspace_id)): ?>
                    <div class="tahc-alert tahc-alert-warning">
                        <strong><?php esc_html_e('First Time Setup:', 'tahc-ai-hybrid-chat'); ?></strong>
                        <ol>
                            <li><?php printf(
                                /* translators: %s: signup link */
                                esc_html__("Don't have an account? %s", 'tahc-ai-hybrid-chat'),
                                '<a href="https://dash.tahc.ai/signup" target="_blank">' . esc_html__('Sign up for free at tahc.ai', 'tahc-ai-hybrid-chat') . '</a>'
                            ); ?></li>
                            <li><?php esc_html_e('After signup, copy your Workspace ID from your dashboard', 'tahc-ai-hybrid-chat'); ?></li>
                            <li><?php esc_html_e('Paste it below and click "Save Changes"', 'tahc-ai-hybrid-chat'); ?></li>
                        </ol>
                    </div>
                <?php else: ?>
                    <div class="tahc-alert tahc-alert-success">
                        <strong><?php esc_html_e('Chat Widget Active!', 'tahc-ai-hybrid-chat'); ?></strong>
                        <p><?php esc_html_e('Your Tahc chat widget is installed and ready. Visit your site to see it in action!', 'tahc-ai-hybrid-chat'); ?></p>
                    </div>
                <?php endif; ?>

                <form method="post" action="options.php">
                    <?php
                    settings_fields('tahc_chat_settings');
                    do_settings_sections('tahc_chat_settings');
                    ?>

                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="tahc_workspace_id"><?php esc_html_e('Workspace ID', 'tahc-ai-hybrid-chat'); ?></label>
                            </th>
                            <td>
                                <input
                                    type="text"
                                    id="tahc_workspace_id"
                                    name="tahc_workspace_id"
                                    value="<?php echo esc_attr($workspace_id); ?>"
                                    class="regular-text"
                                    placeholder="your-company"
                                />
                                <p class="description">
                                    <?php esc_html_e('Your unique workspace identifier from your Tahc dashboard.', 'tahc-ai-hybrid-chat'); ?>
                                    <a href="https://dash.tahc.ai/dashboard" target="_blank"><?php esc_html_e('Find it here', 'tahc-ai-hybrid-chat'); ?> &rarr;</a>
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="tahc_chat_enabled"><?php esc_html_e('Enable Chat Widget', 'tahc-ai-hybrid-chat'); ?></label>
                            </th>
                            <td>
                                <label>
                                    <input
                                        type="checkbox"
                                        id="tahc_chat_enabled"
                                        name="tahc_chat_enabled"
                                        value="1"
                                        <?php checked($chat_enabled, true); ?>
                                    />
                                    <?php esc_html_e('Show chat widget on your website', 'tahc-ai-hybrid-chat'); ?>
                                </label>
                            </td>
                        </tr>
                    </table>

                    <?php submit_button(esc_html__('Save Changes', 'tahc-ai-hybrid-chat')); ?>
                </form>
            </div>

            <div class="tahc-card">
                <h2><?php esc_html_e('What is Hybrid Chat?', 'tahc-ai-hybrid-chat'); ?></h2>
                <p class="tahc-description">
                    <strong><?php esc_html_e('Hybrid Chat', 'tahc-ai-hybrid-chat'); ?></strong> <?php esc_html_e('blends the speed of AI with the empathy of real humans. Your customers get instant answers from AI, with seamless handoff to human agents when needed.', 'tahc-ai-hybrid-chat'); ?>
                </p>

                <h3><?php esc_html_e('Features:', 'tahc-ai-hybrid-chat'); ?></h3>
                <ul class="tahc-features-list">
                    <li><strong><?php esc_html_e('AI-Powered Responses', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Instant answers trained on your content', 'tahc-ai-hybrid-chat'); ?></li>
                    <li><strong><?php esc_html_e('Human Handoff', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Seamlessly escalate to real people', 'tahc-ai-hybrid-chat'); ?></li>
                    <li><strong><?php esc_html_e('Orchestrations', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Custom AI behavior based on page, visitor, or intent', 'tahc-ai-hybrid-chat'); ?></li>
                    <li><strong><?php esc_html_e('Multi-Language', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Auto-detects and responds in 10+ languages', 'tahc-ai-hybrid-chat'); ?></li>
                    <li><strong><?php esc_html_e('Fully Customizable', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Match your brand perfectly', 'tahc-ai-hybrid-chat'); ?></li>
                    <li><strong><?php esc_html_e('Mobile Responsive', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Works beautifully on all devices', 'tahc-ai-hybrid-chat'); ?></li>
                    <li><strong><?php esc_html_e('Office Hours', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Set your availability schedule', 'tahc-ai-hybrid-chat'); ?></li>
                    <li><strong><?php esc_html_e('Analytics', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Track conversations and performance', 'tahc-ai-hybrid-chat'); ?></li>
                    <li><strong><?php esc_html_e('iOS App', 'tahc-ai-hybrid-chat'); ?></strong> - <?php esc_html_e('Respond to customers from anywhere', 'tahc-ai-hybrid-chat'); ?></li>
                </ul>
            </div>

            <div class="tahc-card">
                <h2><?php esc_html_e('Pricing', 'tahc-ai-hybrid-chat'); ?></h2>
                <p class="tahc-description">
                    <strong><?php esc_html_e('The Tahc service offers a 14-day free trial', 'tahc-ai-hybrid-chat'); ?></strong> <?php esc_html_e('on all plans. No credit card required.', 'tahc-ai-hybrid-chat'); ?>
                </p>
                <div class="tahc-pricing-grid">
                    <div class="tahc-pricing-plan">
                        <h4><?php esc_html_e('Starter', 'tahc-ai-hybrid-chat'); ?></h4>
                        <p class="tahc-price">$19<span>/<?php esc_html_e('mo', 'tahc-ai-hybrid-chat'); ?></span></p>
                        <p class="tahc-plan-details"><?php esc_html_e('100 pages, 1 domain', 'tahc-ai-hybrid-chat'); ?></p>
                    </div>
                    <div class="tahc-pricing-plan tahc-popular">
                        <h4><?php esc_html_e('Pro', 'tahc-ai-hybrid-chat'); ?> <span class="tahc-badge"><?php esc_html_e('POPULAR', 'tahc-ai-hybrid-chat'); ?></span></h4>
                        <p class="tahc-price">$39<span>/<?php esc_html_e('mo', 'tahc-ai-hybrid-chat'); ?></span></p>
                        <p class="tahc-plan-details"><?php esc_html_e('500 pages, 3 domains', 'tahc-ai-hybrid-chat'); ?></p>
                    </div>
                    <div class="tahc-pricing-plan">
                        <h4><?php esc_html_e('Team', 'tahc-ai-hybrid-chat'); ?></h4>
                        <p class="tahc-price">$79<span>/<?php esc_html_e('mo', 'tahc-ai-hybrid-chat'); ?></span></p>
                        <p class="tahc-plan-details"><?php esc_html_e('Unlimited, white-label', 'tahc-ai-hybrid-chat'); ?></p>
                    </div>
                </div>
                <p class="tahc-pricing-note">
                    <?php esc_html_e('All plans include unlimited conversations, AI responses, human handoff, and analytics.', 'tahc-ai-hybrid-chat'); ?>
                </p>
            </div>

            <div class="tahc-help-section">
                <h3><?php esc_html_e('Need Help?', 'tahc-ai-hybrid-chat'); ?></h3>
                <p>
                    <?php printf(
                        /* translators: %1$s: documentation link, %2$s: support link */
                        esc_html__('Visit our %1$s or %2$s', 'tahc-ai-hybrid-chat'),
                        '<a href="https://tahc.ai/docs" target="_blank">' . esc_html__('documentation', 'tahc-ai-hybrid-chat') . '</a>',
                        '<a href="https://dash.tahc.ai/support" target="_blank">' . esc_html__('contact support', 'tahc-ai-hybrid-chat') . '</a>'
                    ); ?>
                </p>
                <p class="tahc-cta">
                    <a href="https://dash.tahc.ai/dashboard" target="_blank" class="button button-primary button-hero">
                        <?php esc_html_e('Open Tahc Dashboard', 'tahc-ai-hybrid-chat'); ?> &rarr;
                    </a>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Enqueue chat widget script
     */
    public function enqueue_chat_widget() {
        // Check if enabled
        $chat_enabled = get_option('tahc_chat_enabled', true);
        if (!$chat_enabled) {
            return;
        }

        // Get workspace ID
        $workspace_id = get_option('tahc_workspace_id', '');

        // Don't add if no workspace ID
        if (empty($workspace_id)) {
            return;
        }

        // Register and enqueue the widget script
        wp_enqueue_script(
            'tahc-chat-widget',
            'https://app.tahc.ai/widget.js',
            array(),
            TAHC_CHAT_VERSION,
            array('in_footer' => true)
        );
    }

    /**
     * Add data-workspace-id attribute to widget script tag
     */
    public function add_widget_data_attribute($tag, $handle, $src) {
        if ('tahc-chat-widget' !== $handle) {
            return $tag;
        }

        $workspace_id = get_option('tahc_workspace_id', '');
        if (empty($workspace_id)) {
            return $tag;
        }

        // Add the data-workspace-id attribute
        $tag = str_replace(
            ' src=',
            ' data-workspace-id="' . esc_attr($workspace_id) . '" src=',
            $tag
        );

        return $tag;
    }

    /**
     * Add settings link to plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="options-general.php?page=tahc-chat-settings">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

// Initialize the plugin
new Tahc_Chat_Widget();
