<?php
/**
 * @package    SyncNexusWC
 * @author     Claudiu Maftei <admin@bizzmags.ro>
 */
namespace ClaudiuMaftei\SyncNexusWC;
use \stdClass;
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class Nexus{
    
    public $config;
    public $config_front;

    public function __construct(){
        $this->loadConfig();
    }

    public function loadConfig()
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $this->config=new stdClass;
        $this->config_front=new stdClass;
        $sql=$wpdb->prepare("select * from ".$wpdb->prefix."syncnexuswc_config where %d",1);
        $results=$wpdb->get_results($sql);
        if($results){
            foreach($results as $r){
                if(!isset($this->config->{$r->config_name}))
                    $this->config->{$r->config_name}=$r->config_value;
                if(!isset($this->config_front->{$r->config_name}) && $r->show_front==1)
                    $this->config_front->{$r->config_name}=$r->config_value;
            }
        }
    }

    public function is_woocommerce_activated(){
        if ( class_exists( 'woocommerce' ) ) { return true; } else { return false; }
    }

    public function is_wpo_wcpdf_activated(){
        if ( class_exists( 'WPO_WCPDF' ) ) { return true; } else { return false; }
    }

    public function is_bizzmagsmarketplace_activated(){
        if ( defined( 'BIZZMAGSMARKETPLACE_VERSION' ) ) { return true; } else { return false; }
    }
    public function get_wpo_wcpdf_invoice_prefix()
    {
        $settings=get_option("wpo_wcpdf_documents_settings_invoice");
        if(is_array($settings) && isset($settings['number_format']) && is_array($settings['number_format']) && isset($settings['number_format']['prefix']))
            return $settings['number_format']['prefix'];
        return '';
    }
    public function checkIfProBizzmags()
    {
        if ( defined( 'BIZZMAGSMARKETPLACE_VERSION' ) && version_compare(BIZZMAGSMARKETPLACE_VERSION,'10.0.0','>'))
            return true;
        return false;
    }
    public function getLastWPOinvoiceNumber()
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($wpdb->get_var( $wpdb->prepare("show tables like %s", $wpdb->prefix.'wcpdf_invoice_number'))!="")
        {
            $sql=$wpdb->prepare("select max(id) from ".$wpdb->prefix."wcpdf_invoice_number where %d",1);
            return $wpdb->get_var($sql);
        }
        return "-";
    }
    public function saveNexusApiCredentials()
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        check_admin_referer( 'syncnexuswc_save_api_credentials' );
        $nexus_ip=isset($_POST['nexus_ip'])?sanitize_text_field(wp_unslash($_POST['nexus_ip'])):"";
        $nexus_port=isset($_POST['nexus_port'])?sanitize_text_field(wp_unslash($_POST['nexus_port'])):"";
        $nexus_api_key=isset($_POST['nexus_api_key'])?sanitize_text_field(wp_unslash($_POST['nexus_api_key'])):"";
        $nexus_protocol=isset($_POST['nexus_protocol'])?sanitize_text_field(wp_unslash($_POST['nexus_protocol'])):"";
        if (!filter_var($nexus_ip, FILTER_VALIDATE_IP))
        {
            $nexus_ip="";
             $sql=$wpdb->prepare("update ".$wpdb->prefix."syncnexuswc_config set config_value=%s where config_name='nexus_ip'",array($nexus_ip));
            $wpdb->query($sql);
            $this->loadConfig();
        }
        if($nexus_port!="" && $nexus_api_key!="" && $nexus_ip!="")
        {
            $sql=$wpdb->prepare("update ".$wpdb->prefix."syncnexuswc_config set config_value=%s where config_name='nexus_ip'",array($nexus_ip));
            $wpdb->query($sql);
            $sql=$wpdb->prepare("update ".$wpdb->prefix."syncnexuswc_config set config_value=%s where config_name='nexus_port'",array($nexus_port));
            $wpdb->query($sql);
            $sql=$wpdb->prepare("update ".$wpdb->prefix."syncnexuswc_config set config_value=%s where config_name='nexus_api_key'",array($nexus_api_key));
            $wpdb->query($sql);
            $sql=$wpdb->prepare("update ".$wpdb->prefix."syncnexuswc_config set config_value=%s where config_name='nexus_protocol'",array($nexus_protocol));
            $wpdb->query($sql);
            $this->loadConfig();

            $check_api_response=$this->doNexusRequest("api/v1/read","facturi_clienti",array("anluna"=>gmdate("Ym")));
            if(is_array($check_api_response) && isset($check_api_response['status']) && $check_api_response['status']=='error')
                return array('status'=>'error','msg'=>array($check_api_response['msg']));
            if($check_api_response=="")
                return array('status'=>'error','msg'=>__("The API Endpoint does not seam right","sync-nexus-wc"));
            $have_errors=$this->getResultError($check_api_response);
            if($have_errors!="")
                return array('status'=>'error','msg'=>$have_errors);
            else
            {
                $this->saveLog(__("Saved API configuration","sync-nexus-wc"));
                return array('status'=>'updated','msg'=>array(__("Nexus ERP API credentials saved with success","sync-nexus-wc")));
            }
        }
        else
            return array('status'=>'error','msg'=>__("Nexus ERP API IP, Port and Key are required","sync-nexus-wc"));
    }

    public function getResultError($result)
    {
        if (is_object($result) && isset($result->isError) && $result->isError == true && isset($result->message) && is_array($result->message)) {
            foreach ($result->message as $msg) {
                $this->saveLog($msg);
            }

            return $result->message;
        } elseif (is_object($result) && isset($result->message) && is_string($result->message) && $result->isError == true) {
            if ($result->message != '') {
                $this->saveLog($result->message);
            }

            return $result->message;
        }

        return '';
    }

    public function returnResultError($message="")
    {
        return ["msg"=>$message,"status"=>"error"];
    }

    public function updateConfigValue($name="", $value="")
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $name=sanitize_text_field($name);
        if($name=='emag_rank_log')
            $value=sanitize_textarea_field($value);
        else
            $value=sanitize_text_field($value);
        if($name!="")
        {
            $sql=$wpdb->prepare("update ".$wpdb->prefix."syncnexuswc_config set config_value=%s where config_name=%s",array($value,$name));
            $wpdb->query($sql);
            $this->config->$name=$value;
        }
    }

    public function getConfigValue($name="")
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $name=sanitize_text_field($name);
        if($name!="")
        {
            $sql=$wpdb->prepare("select config_value from ".$wpdb->prefix."syncnexuswc_config where config_name=%s",array($name));
            $result=$wpdb->get_row($sql);
            if(isset($result->config_value))
                return $result->config_value;
        }
    }
    
    public function getNexusServerStatus($just_status=false)
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $url=$this->config->nexus_protocol."://".$this->config->nexus_ip.":".$this->config->nexus_port;
        $response = wp_remote_get( $url, array("timeout"=>5) );
        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            return $error_message;
        }
        else
        {
            if($just_status)
                return wp_remote_retrieve_response_code( $response );
            $body = wp_remote_retrieve_body( $response );
            return $body;
        }
    }
    public function doNexusRequest($resource="",$action="", $body_arr=array(), $request_type="POST", $params="")
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $request_type=sanitize_text_field($request_type);
        $resource=sanitize_text_field($resource);
        $action=sanitize_text_field($action);
        if($this->config->nexus_ip=="" || $this->config->nexus_port=="" || $this->config->nexus_api_key=="")
            return $this->returnResultError(__("Please add your Nexus ERP server details first","sync-nexus-wc"));
        $url="";
        $url=$this->config->nexus_protocol."://".$this->config->nexus_ip.":".$this->config->nexus_port."/".$resource."/".$action;
        
        $token=base64_encode($this->config->nexus_api_key.":");
        $body=wp_json_encode($body_arr);
        
        if($request_type=="POST")
        {
            $response = wp_remote_post($url, array(
                'headers'     => array('Authorization' => 'Basic ' . $token, 'Content-Type' => 'application/json'),
                'body'        => $body,
                'method'      => 'POST',
                'data_format' => 'body',
                'timeout'     => 15,
            ));
        }
        else if($request_type=="PATCH")
        {
            $response = wp_remote_post($url, array(
                'headers'     => array('Authorization' => 'Basic ' . $token, 'Content-Type' => 'application/json'),
                'body'        => $body,
                'method'      => 'PATCH',
                'timeout'     => 15,
            ));
        }
        else
        {
            $arguments=array('method' => $request_type,'body' => $params, 'headers'=>array('Authorization' => 'Basic ' . $token), 'timeout' => 15);
            $response = wp_remote_post( $url, $arguments );
        }
        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            return $this->returnResultError($error_message);
        }
        else
        {
            $body = wp_remote_retrieve_body( $response );
            $result=json_decode($body);
            return $result;
        }
    }

    public function saveLog($log="")
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if($log!="")
        {
            $user_id=get_current_user_id();
            $log=sanitize_textarea_field($log);
            $sql=$wpdb->prepare("insert into ".$wpdb->prefix."syncnexuswc_logs set user_id=%d, log=%s, mdate=%s",array($user_id,$log,microtime(true)));
            $wpdb->query($sql);

            $sql=$wpdb->prepare("delete from ".$wpdb->prefix."syncnexuswc_logs where mdate<%s",strtotime(gmdate("Y-m-d")." -60 days"));
            $wpdb->query($sql);
        }
    }

    public function addActionSchedulerTask($task="", $args=array(), $group="")
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($task!="")
        {
            if ( ! class_exists( 'ActionScheduler_Versions', false ) )
                require_once(WC()->plugin_path()."/packages/action-scheduler/action-scheduler.php");
            if(!as_has_scheduled_action($task,$args,$group))
            {
                $action_added=true;
                $action_id=as_enqueue_async_action( $task, $args, $group, true, 0);
                if(!$action_id)
                    $action_added=false;
                return $action_added;
            }
        }
        return false;
    }
    
    public function substrKeepWordsOnly($string, $max_characters)
    {
        if (strlen($string) <= $max_characters)
            return $string;
        $trimmed_string = substr($string, 0, $max_characters);
        $last_space_position = strrpos($trimmed_string, ' ');
        if ($last_space_position !== false)
            $trimmed_string = substr($trimmed_string, 0, $last_space_position);
        return $trimmed_string;
    }

    
    public function saveNexusConfiguration()
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        check_admin_referer( 'syncnexuswc_save_configuration' );

        $send_wc_invoices=isset($_POST['send_wc_invoices'])?sanitize_text_field(wp_unslash($_POST['send_wc_invoices'])):"";
        $send_emag_invoices=isset($_POST['send_emag_invoices'])?sanitize_text_field(wp_unslash($_POST['send_emag_invoices'])):"";
        $nexus_send_invoice_status=isset($_POST['nexus_send_invoice_status'])?sanitize_text_field(wp_unslash($_POST['nexus_send_invoice_status'])):"";
        $nexus_invoice_source=isset($_POST['nexus_invoice_source'])?sanitize_text_field(wp_unslash($_POST['nexus_invoice_source'])):"";
        $nexus_order_meta_cui=isset($_POST['nexus_order_meta_cui'])?sanitize_text_field(wp_unslash($_POST['nexus_order_meta_cui'])):"";
        $nexus_order_meta_cui_child=isset($_POST['nexus_order_meta_cui_child'])?sanitize_text_field(wp_unslash($_POST['nexus_order_meta_cui_child'])):"";
        $nexus_order_meta_reg_com=isset($_POST['nexus_order_meta_reg_com'])?sanitize_text_field(wp_unslash($_POST['nexus_order_meta_reg_com'])):"";
        $nexus_order_meta_reg_com_child=isset($_POST['nexus_order_meta_reg_com_child'])?sanitize_text_field(wp_unslash($_POST['nexus_order_meta_reg_com_child'])):"";
        $nexus_order_meta_pers_jur=isset($_POST['nexus_order_meta_pers_jur'])?sanitize_text_field(wp_unslash($_POST['nexus_order_meta_pers_jur'])):"";
        $nexus_order_meta_pers_jur_child=isset($_POST['nexus_order_meta_pers_jur_child'])?sanitize_text_field(wp_unslash($_POST['nexus_order_meta_pers_jur_child'])):"";
        $nexus_order_meta_pers_jur_value=isset($_POST['nexus_order_meta_pers_jur_value'])?sanitize_text_field(wp_unslash($_POST['nexus_order_meta_pers_jur_value'])):"";
        $nexus_prod_batch_import_nr=isset($_POST['nexus_prod_batch_import_nr'])?sanitize_text_field(wp_unslash($_POST['nexus_prod_batch_import_nr'])):"";
        $nexus_prod_import_interval=isset($_POST['nexus_prod_import_interval'])?sanitize_text_field(wp_unslash($_POST['nexus_prod_import_interval'])):"";
        $nexus_invoice_cont_produs=isset($_POST['nexus_invoice_cont_produs'])?sanitize_text_field(wp_unslash($_POST['nexus_invoice_cont_produs'])):"";
        $nexus_invoice_tip_produs=isset($_POST['nexus_invoice_tip_produs'])?sanitize_text_field(wp_unslash($_POST['nexus_invoice_tip_produs'])):"";
        $nexus_invoice_cont_produs_discount=isset($_POST['nexus_invoice_cont_produs_discount'])?sanitize_text_field(wp_unslash($_POST['nexus_invoice_cont_produs_discount'])):"";
        $nexus_invoice_tip_produs_discount=isset($_POST['nexus_invoice_tip_produs_discount'])?sanitize_text_field(wp_unslash($_POST['nexus_invoice_tip_produs_discount'])):"";
        $nexus_invoice_cont_produs_livrare=isset($_POST['nexus_invoice_cont_produs_livrare'])?sanitize_text_field(wp_unslash($_POST['nexus_invoice_cont_produs_livrare'])):"";
        $nexus_invoice_tip_produs_livrare=isset($_POST['nexus_invoice_tip_produs_livrare'])?sanitize_text_field(wp_unslash($_POST['nexus_invoice_tip_produs_livrare'])):"";
        $nexus_send_invoices_auto_validate=isset($_POST['nexus_send_invoices_auto_validate'])?sanitize_text_field(wp_unslash($_POST['nexus_send_invoices_auto_validate'])):"";

        $nexus_send_invoices_live=isset($_POST['nexus_send_invoices_live'])?sanitize_text_field(wp_unslash($_POST['nexus_send_invoices_live'])):"";

        if($nexus_prod_import_interval<30)
            $nexus_prod_import_interval=30;
        if($nexus_prod_batch_import_nr<1)
            $nexus_prod_batch_import_nr=1;

        $this->updateConfigValue("send_wc_invoices",$send_wc_invoices);
        $this->updateConfigValue("send_emag_invoices",$send_emag_invoices);
        $this->updateConfigValue("nexus_send_invoice_status",$nexus_send_invoice_status);
        $this->updateConfigValue("nexus_invoice_source",$nexus_invoice_source);
        $this->updateConfigValue("nexus_order_meta_cui",$nexus_order_meta_cui);
        $this->updateConfigValue("nexus_order_meta_cui_child",$nexus_order_meta_cui_child);
        $this->updateConfigValue("nexus_order_meta_reg_com",$nexus_order_meta_reg_com);
        $this->updateConfigValue("nexus_order_meta_reg_com_child",$nexus_order_meta_reg_com_child);
        $this->updateConfigValue("nexus_order_meta_pers_jur",$nexus_order_meta_pers_jur);
        $this->updateConfigValue("nexus_order_meta_pers_jur_child",$nexus_order_meta_pers_jur_child);
        $this->updateConfigValue("nexus_prod_batch_import_nr",(int)$nexus_prod_batch_import_nr);
        $this->updateConfigValue("nexus_prod_import_interval",(int)$nexus_prod_import_interval);
        $this->updateConfigValue("nexus_invoice_cont_produs",$nexus_invoice_cont_produs);
        $this->updateConfigValue("nexus_invoice_tip_produs",$nexus_invoice_tip_produs);
        $this->updateConfigValue("nexus_invoice_cont_produs_discount",$nexus_invoice_cont_produs_discount);
        $this->updateConfigValue("nexus_invoice_tip_produs_discount",$nexus_invoice_tip_produs_discount);
        $this->updateConfigValue("nexus_invoice_cont_produs_livrare",$nexus_invoice_cont_produs_livrare);
        $this->updateConfigValue("nexus_invoice_tip_produs_livrare",$nexus_invoice_tip_produs_livrare);
        $this->updateConfigValue("nexus_send_invoices_auto_validate",$nexus_send_invoices_auto_validate);

        $this->updateConfigValue("nexus_send_invoices_live",$nexus_send_invoices_live);

        $this->saveLog(__("Saved Settings","sync-nexus-wc"));
        return array('status'=>'updated','msg'=>__("Settings updated","sync-nexus-wc"));
    }
    public function getProductMetas()
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $sql=$wpdb->prepare("
        select DISTINCT pm.meta_key 
        FROM ".$wpdb->prefix."postmeta pm
        INNER JOIN ".$wpdb->prefix."posts p ON pm.post_id = p.ID
        WHERE p.post_type = 'product'
        AND pm.meta_key LIKE %s
        AND pm.meta_key NOT IN ('_edit_lock', '_edit_last')
        ",'_%');
        return $wpdb->get_results($sql);
    }
    public function getProductVariationName($prod_id, $variation_id) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $product = wc_get_product($prod_id);
        if (!$product || 'variable' !== $product->get_type()) {
            return null;
        }
        $parent_title = $product->get_name();
        $variation = wc_get_product($variation_id);
        if (!$variation || $variation->get_parent_id() != $prod_id) {
            return null;
        }
        $attributes = $variation->get_variation_attributes();
        $variation_title = $parent_title;
        foreach ($attributes as $attribute => $value) {
            $taxonomy = str_replace('attribute_', '', $attribute);
            $term = get_term_by('slug', $value, $taxonomy);
            if ($term && !is_wp_error($term)) {
                $variation_title .= ' - ' . $term->name;
            }
        }
        return $variation_title;
    }
    public function getWcProductDefaultCategory($product_id) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $product = wc_get_product($product_id);
        if(!$product)
            return 0;
        $categories = $product->get_category_ids();
        if (empty($categories)) {
            return 0;
        }
        $last_child_category = null;
        $max_depth = -1;
        foreach ($categories as $category_id) {
            $category = get_term($category_id, 'product_cat');
            $depth = 0;
            while ($category->parent != 0) {
                $depth++;
                $category = get_term($category->parent, 'product_cat');
            }
            if ($depth > $max_depth) {
                $max_depth = $depth;
                $last_child_category = get_term($category_id, 'product_cat');
            }
        }
        if(isset($last_child_category->term_id))
            return $last_child_category->term_id;
        return $last_child_category;
    }    
    public function getCategoryname($category_id) {
        $term = get_term($category_id, 'product_cat');
        if (is_wp_error($term) || !$term) {
            return '';
        }
        return $term->name;
    }
    public function getLogsPages($logs_per_page=100)
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $logs_per_page=(int)$logs_per_page;
        $sql=$wpdb->prepare("select count(*) from ".$wpdb->prefix."syncnexuswc_logs where %d",1);
        $total=$wpdb->get_var($sql);
        if($total>0)
        {
            $arr=[];
            $pages=$total/$logs_per_page;
            $pages=ceil($pages);
            for($i=1;$i<=$pages;$i++)
            {
                $arr[]=$i;
            }
            return $arr;
        }
        return [1];
    }

    public function getLogs($logs_page=1,$logs_per_page=100)
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $logs_page=(int)$logs_page;
        $logs_per_page=(int)$logs_per_page;
        if($logs_page==1)
            $start=0;
        else
            $start=($logs_page-1)*$logs_per_page;
        $sql=$wpdb->prepare("select mdate, log from ".$wpdb->prefix."syncnexuswc_logs where 1 order by mdate desc limit %d, %d",array((int)$start,(int)$logs_per_page));
        return $wpdb->get_results($sql);
    }
    
    public function getWcProductAttributes($product_id) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $product = wc_get_product($product_id);
        if (!$product) {
            return null;
        }
        $attributes = $product->get_attributes();
        $attributes_data = [];
        foreach ($attributes as $name => $attribute) {
            if ($attribute->is_taxonomy()) {
                $terms = wc_get_product_terms($product_id, $name, ['fields' => 'names']);
                $value = implode(', ', $terms);
            } else {
                $value = implode(', ', $attribute->get_options());
            }

            $attributes_data[] = [
                'name' => $name,
                'value' => $value,
            ];
        }
        return $attributes_data;
    }

    public function getWcVariationAttributes($variation_id) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $variation = wc_get_product($variation_id);
        if (!$variation || 'variation' !== $variation->get_type()) {
            return null;
        }
        $attributes = $variation->get_variation_attributes();
        $attributes_data = [];
        foreach ($attributes as $attribute_name => $attribute_value) {
            $taxonomy = str_replace('attribute_', '', $attribute_name);
            $term = get_term_by('slug', $attribute_value, $taxonomy);
            if ($term && !is_wp_error($term)) {
                $name = $taxonomy;
                $value = $term->name;
            } else {
                $name = $attribute_name;
                $value = $attribute_value;
            }
            $attributes_data[] = [
                'name' => str_ireplace("attribute_","",$name),
                'value' => $value,
            ];
        }
        return $attributes_data;
    }

    public function getWcVariationTaxonomyAttributes($variation_id) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $variation = wc_get_product($variation_id);
        if (!$variation || 'variation' !== $variation->get_type()) {
            return null;
        }
        $attributes = $variation->get_variation_attributes();
        $attributes_data = [];
        foreach ($attributes as $attribute_name => $attribute_value) {
            $taxonomy = str_replace('attribute_', '', $attribute_name);
            $term = get_term_by('slug', $attribute_value, $taxonomy);
            if ($term && !is_wp_error($term)) {
                $name = $taxonomy;
                $value = $term->name;
                $attributes_data[] = [
                    'name' => str_ireplace("attribute_","",$name),
                    'value' => $value,
                ];
            }
        }
        return $attributes_data;
    }
    
    public function unique_multidimensional_array($array) {
        $tempArray = array();
        $keyArray = array();
        
        foreach ($array as $val) {
            if (!in_array(serialize($val), $keyArray)) {
                $keyArray[] = serialize($val);
                $tempArray[] = $val;
            }
        }
        return $tempArray;
    }
    public function clearAllLogs()
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        check_admin_referer( 'syncnexuswc_clear_all_logs' );
        $sql=$wpdb->prepare("truncate table ".$wpdb->prefix."syncnexuswc_logs where %d",1);
        $sql=str_replace("where 1","",$sql);
        if(!$wpdb->query($sql))
            return array('status'=>'error','msg'=>array(__("Error in clearing all logs","sync-nexus-wc")));
        else
        {
            $sql=$wpdb->prepare("alter table ".$wpdb->prefix."syncnexuswc_logs AUTO_INCREMENT=%d",1);
            $wpdb->query($sql);
            return array('status'=>'updated','msg'=>array(__("Cleared all logs with success","sync-nexus-wc")));
        }
    }
    public function getOrderMetaKeys()
    {
        global $wpdb;

        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $keys = [];

        // 1) Legacy (posts + postmeta)
        $legacy_sql = "
            SELECT DISTINCT pm.meta_key
            FROM {$wpdb->postmeta} pm
            INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
            WHERE p.post_type IN ('shop_order','shop_order_placehold')
        ";

        // 2) HPOS (custom order tables) — only if the table exists
        $hpos_table = $wpdb->prefix . 'wc_orders_meta';
        $has_hpos_table = ( $wpdb->get_var( $wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $hpos_table
        ) ) === $hpos_table );

        if ( $has_hpos_table ) {
            $hpos_sql = "SELECT DISTINCT meta_key FROM {$hpos_table}";
            $sql = "{$legacy_sql} UNION {$hpos_sql}";
        } else {
            $sql = $legacy_sql;
        }

        // UNION already de-dupes, but keep it tidy
        $keys = $wpdb->get_col( $sql );

        return $keys;
    }
    public function translateAndLimitString($str="",$max=128)
    {
        setlocale(LC_ALL, 'en_US.utf8');
        $transliterator = \Transliterator::create('Any-Latin; Latin-ASCII');
        if($str!="")
        {
            $str=trim($str);
            $str=$transliterator->transliterate($str);
            if (strlen($str) > $max)
            {
                $str = substr($str, 0, $max);
            }
        }
        return $str;
    }
    public function checkDependencies()
    {
        $dependencies=true;
        if(!$this->is_woocommerce_activated())
        {
            $this->saveLog(__("WooCommerce is not installed or activated, please install or activate it in order to continue","sync-nexus-wc"));
            $dependencies=false;
        }
        if(!$this->is_wpo_wcpdf_activated())
        {
            //$this->saveLog(__("PDF Invoices & Packing Slips for WooCommerce is not installed or activated, please install or activate it in order to continue","sync-nexus-wc"));
            //$dependencies=false;
        }
        if(!$this->is_bizzmagsmarketplace_activated())
        {
            //$this->saveLog(__("BizzmagsMarketplace is not installed or activated, please install or activate it in order to continue","sync-nexus-wc"));
            //$dependencies=false;
        }
        return $dependencies;
    }
    public function getSectorNumberFromString($string)
    {
        $adr_sec = ['sector', 'sectorul', 'sect'];
        $pattern = '/^(' . implode('|', $adr_sec) . ')(\.?:)?\s*(\d+)/i';
        if (preg_match($pattern, $string, $matches))
            return (int)$matches[3];
        return 0;
    }
    public function getLocationDetailsForNexus($order)
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $return=new stdClass;
        $return->nume="";
        $return->id_locatie=0;
        $return->cod_judet="";
        $return->siruta=0;
        $return->id_zona=0;
        $return->zona="";

        if($order)
        {
            $state_code = $order->get_billing_state();
            $billing_country = $order->get_billing_country();
            $states = WC()->countries->get_states($billing_country);
            $state_name = isset($states[$state_code]) ? $states[$state_code] : $state_code;

            if($state_code=="B" || strtolower(trim($this->translateAndLimitString($state_code,1000)))=="bucuresti")//Bucuresti
            {
                $sql=$wpdb->prepare("select cod_judet from ".$wpdb->prefix."syncnexuswc_siruta where nume_fd=%s order by id asc limit 1",'MUNICIPIUL BUCURESTI');
                $return->cod_judet = $wpdb->get_var($sql);
            }
            else if($state_name!="")
            {
                $state_name=$this->translateAndLimitString($state_name,1000);
                $state_name=trim(strtoupper($state_name));
                $sql=$wpdb->prepare("select cod_judet from ".$wpdb->prefix."syncnexuswc_siruta where judet_fd=%s order by id asc limit 1",$state_name);
                $return->cod_judet = $wpdb->get_var($sql);
            }

            $city=$order->get_billing_city();
            $city=trim($city);

            if($return->cod_judet=="B")//Bucuresti
            {
                $sector=0;
                if(is_numeric($city) and in_array($city,[1,2,3,4,5,6]))//sector scris direct ca numar
                {
                    $sector=$city;
                    $city="BUCURESTI SECTORUL ".$city;
                }
                else//sector scris cu prefix "Sector"
                {
                    $sect_nr=$this->getSectorNumberFromString($city);
                    if($sect_nr>0)
                    {
                        $city="BUCURESTI SECTORUL ".$sect_nr;
                        $sector=$sect_nr;
                    }
                }
                if($sector==0)//sector lipsa, poate il gasim in adresa
                {
                    $full_address=$order->get_billing_address_1();
                    if(trim($order->get_billing_address_2())!="")
                        $full_address=$order->get_billing_address_1()." ".$order->get_billing_address_2();
                    $parsed=$this->parseAddressForNexus($full_address);
                    if(isset($parsed['sector']) && (int)$parsed['sector']>0)
                    {
                        $city="BUCURESTI SECTORUL ".(int)$parsed['sector'];
                        $sector=(int)$parsed['sector'];
                    }
                }
                if($sector==0)//nu avem sector, ramanem la municipiul bucuresti
                    $city="MUNICIPIUL BUCURESTI";
            }

            if($city!="")
            {
                $city=$this->translateAndLimitString($city,1000);
                $city=strtoupper($city);
                $state_code=$return->cod_judet;
                if($state_code!="")
                {
                    $sql=$wpdb->prepare("select id, siruta, id_zona, nume, zona from ".$wpdb->prefix."syncnexuswc_siruta where nume_fd=%s and cod_judet=%s order by id asc limit 1",array($city,$state_code));
                    $result=$wpdb->get_row($sql);
                    if(isset($result->id))
                    {
                        $return->id_locatie=$result->id;
                        $return->nume=$result->nume;
                        $return->siruta=$result->siruta;
                        $return->id_zona=$result->id_zona;
                        $return->zona=$result->zona;
                    }
                }
            }
        }
        return $return;
    }
    
    public function getEmagLocationDetailsForNexus($order)
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $return=new stdClass;
        $return->nume="";
        $return->id_locatie=0;
        $return->cod_judet="";
        $return->siruta=0;
        $return->id_zona=0;
        $return->zona="";

        if($order)
        {
            $billing_country = $order->customer->billing_country;
            $full_address=$order->customer->billing_street;
            $state_name=$order->customer->billing_suburb;
            $state_name=trim(strtoupper($state_name));
            $state_code="";
            $states = WC()->countries->get_states($billing_country);
            if (!empty($states)) {
                foreach ($states as $wc_state_code => $state_name_in_list)
                {
                    if (strtolower(trim($this->translateAndLimitString($state_name_in_list,1000)))==strtolower($state_name))
                    {
                        $state_code=$wc_state_code;
                        break;
                    }
                }
            }
            if($state_code=="")
                $state_code=$order->customer->billing_suburb;

            if($state_code=="B" || strtolower(trim($this->translateAndLimitString($state_code,1000)))=="bucuresti")//Bucuresti
            {
                $sql=$wpdb->prepare("select cod_judet from ".$wpdb->prefix."syncnexuswc_siruta where nume_fd=%s order by id asc limit 1",'MUNICIPIUL BUCURESTI');
                $return->cod_judet = $wpdb->get_var($sql);
            }
            else if($state_name!="")
            {
                $state_name=$this->translateAndLimitString($state_name,1000);
                $state_name=trim(strtoupper($state_name));
                $sql=$wpdb->prepare("select cod_judet from ".$wpdb->prefix."syncnexuswc_siruta where judet_fd=%s order by id asc limit 1",$state_name);
                $return->cod_judet = $wpdb->get_var($sql);
            }

            $city=$order->customer->billing_city;
            $city=trim($city);

            if($return->cod_judet=="B")//Bucuresti
            {
                $sector=0;
                if(is_numeric($city) and in_array($city,[1,2,3,4,5,6]))//sector scris direct ca numar
                {
                    $sector=$city;
                    $city="BUCURESTI SECTORUL ".$city;
                }
                else//sector scris cu prefix "Sector"
                {
                    $sect_nr=$this->getSectorNumberFromString($city);
                    if($sect_nr>0)
                    {
                        $city="BUCURESTI SECTORUL ".$sect_nr;
                        $sector=$sect_nr;
                    }
                }
                if($sector==0)//sector lipsa, poate il gasim in adresa
                {
                    $parsed=$this->parseAddressForNexus($full_address);
                    if(isset($parsed['sector']) && (int)$parsed['sector']>0)
                    {
                        $city="BUCURESTI SECTORUL ".(int)$parsed['sector'];
                        $sector=(int)$parsed['sector'];
                    }
                }
                if($sector==0)//nu avem sector, ramanem la municipiul bucuresti
                    $city="MUNICIPIUL BUCURESTI";
            }

            if($city!="")
            {
                $city=$this->translateAndLimitString($city,1000);
                $city=strtoupper($city);
                $state_code=$return->cod_judet;
                if($state_code!="")
                {
                    $sql=$wpdb->prepare("select id, siruta, id_zona, nume, zona from ".$wpdb->prefix."syncnexuswc_siruta where nume_fd=%s and cod_judet=%s order by id asc limit 1",array($city,$state_code));
                    $result=$wpdb->get_row($sql);
                    if(isset($result->id))
                    {
                        $return->id_locatie=$result->id;
                        $return->nume=$result->nume;
                        $return->siruta=$result->siruta;
                        $return->id_zona=$result->id_zona;
                        $return->zona=$result->zona;
                    }
                }
            }
        }
        return $return;
    }

    public function parseAddressForNexus($address)
    {
        $adr_str_start = ['aleea', 'bulevard', 'bulevardul', 'bd'];
        $pattern = '/^(' . implode('|', $adr_str_start) . ')(\.?:)?/i';
        if (preg_match($pattern, $address))
            $address="Str ".$address;

        $adr_str_start = ['Strada', 'Str'];
        $pattern = '/^(' . implode('|', $adr_str_start) . ')(\.?:)?/i';
        if (!preg_match($pattern, $address))
            $address="Str ".$address;

        $adr_str = ['strada', 'str'];
        $adr_nr = ['numar', 'numarul', 'nr'];
        $adr_bl = ['bloc', 'blocul', 'bl'];
        $adr_sca = ['scara', 'sc'];
        $adr_ap = ['apartament', 'apartamentul', 'ap'];
        $adr_sec = ['sector', 'sectorul', 'sect'];
        $adr_et = ['etaj', 'etajul', 'et'];

        // Normalize the address string
        $address = preg_replace('/\s+/', ' ', $address); // Replace multiple spaces with a single space
        $address = trim($address);

        // Define patterns for parsing
        $patterns = [
            'street' => '(' . implode('|', $adr_str) . ')[\s\.]+(.*?)(?= nr| bl| sc| et| ap| sect|,|$)',
            'number' => '(' . implode('|', $adr_nr) . ')[\s\.]+(.*?)\b(?=,| bl| sc| et| ap| sect|$)',
            'block' => '(' . implode('|', $adr_bl) . ')[\s\.]+(.*?)\b(?=,| sc| et| ap| sect|$)',
            'staircase' => '(' . implode('|', $adr_sca) . ')[\s\.]+(.*?)\b(?=,| et| ap| sect|$)',
            'floor' => '(' . implode('|', $adr_et) . ')[\s\.]+(\d+)\b', // Capture only the numeric floor value
            'apartment' => '(' . implode('|', $adr_ap) . ')[\s\.]+(.*?)\b(?=,| sect|$)',
            'sector' => '(' . implode('|', $adr_sec) . ')[\s\.]+(.*?)\b(?=,|$)'
        ];

        // Initialize result array
        $result = [
            'street' => null,
            'number' => null,
            'block' => null,
            'staircase' => null,
            'floor' => null,
            'apartment' => null,
            'sector' => null
        ];

        // Match each pattern against the address
        foreach ($patterns as $key => $pattern) {
            if (preg_match('/' . $pattern . '/i', $address, $matches)) {
                $result[$key] = trim($matches[2] ?? '');
            }
        }
        
        return $result;
    }

    public function displayNexusAddressToBeSent($order)
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($order)
        {
            $full_address=$order->get_billing_address_1();
            if(trim($order->get_billing_address_2())!="")
                $full_address=$order->get_billing_address_1()." ".$order->get_billing_address_2();
            $adr_det=$this->parseAddressForNexus($full_address);
            $location=$this->getLocationDetailsForNexus($order);

            $data_css='
            .nexusserp_borderpadme{
                border: 1px solid #ccc;
                padding:4px;
            }
            .nexusserp_colormered{
                color: red;
            }
            .nexusserp_colormeorange{
                color:orange;
            }
            .nexusserp_nexus_order_address{
                overflow-x:auto;
            }
            .nexusserp_error_msg{
                color:red;
                font-weight:bold;
                font-size:larger;
            }
            ';
            wp_register_style( 'syncnexuswc_wc_order_page_inline_css', '' );
            wp_enqueue_style( 'syncnexuswc_wc_order_page_inline_css' );
            wp_add_inline_style("syncnexuswc_wc_order_page_inline_css",$data_css);
            $location_err='nexusserp_colormered';
            if(is_numeric($location->siruta) && (int)$location->siruta>0)
                $location_err="";

            $adr_score=0;
            if($adr_det['street']!="")
                $adr_score++;
            if($adr_det['number']!="")
                $adr_score++;
            if($adr_det['staircase']!="")
                $adr_score++;
            if($adr_det['floor']!="")
                $adr_score++;
            if($adr_det['apartment']!="")
                $adr_score++;

            if($adr_score<=2 && $location_err=="")
                $location_err='nexusserp_colormeorange';

            $emag_order_id = $order->get_meta('_emag_order_id');
            if(!$emag_order_id)
                $emag_order_id = get_post_meta($order->get_id(),"_emag_order_id",true);
            ?>
            <div class="nexusserp_nexus_order_address">
            <?php
            if($emag_order_id)
            {
                ?>
                <h3>eMAG ID: <?php echo esc_html($emag_order_id);?></h3>
                <?php
            }
            ?>
            <h3 style="text-wrap:inherit;"><?php echo esc_html("Address to be sent to Nexus Server","sync-nexus-wc");?></h3>
            <?php
            if($location_err=="nexusserp_colormered")
            {
                ?>
                <div class="nexusserp_error_msg">
                    <?php
                    echo esc_html_e("The address to be sent to Nexus is wrong, please edit the billing address and save the order to fix the problem before setting the order as Completed","sync-nexus-wc");
                    ?>.
                </div>
                <?php
            }
            ?>
            <table class="<?php echo esc_attr($location_err);?>">
                <tr>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Locality","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("ID","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("State","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Siruta","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Area","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"></td>
                </tr>
                <tr>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($location->nume);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($location->id_locatie);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($location->cod_judet);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($location->siruta);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($location->zona);?></td>
                    <td class="nexusserp_borderpadme"></td>
                </tr>
                <tr>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Street","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Nr","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Block","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Staircase","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Floor","sync-nexus-wc");?></strong></td>
                    <td class="nexusserp_borderpadme"><strong><?php echo esc_html("Apartment","sync-nexus-wc");?></strong></td>
                </tr>
                <tr>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($adr_det['street']);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($adr_det['number']);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($adr_det['block']);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($adr_det['staircase']);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($adr_det['floor']);?></td>
                    <td class="nexusserp_borderpadme"><?php echo esc_html($adr_det['apartment']);?></td>
                </tr>
            </table>
            </div>
            <?php
        }
    }

    public function sendInvoiceToNexusServer($status, $order)
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $trigger_status=$this->config->nexus_send_invoice_status;
        if($order && $status!="")
        {
            $status_check=str_ireplace("wc-", "", $trigger_status);
            if($status_check==$status)
            {
                $this->saveLog(__("Preparing invoice to send to Nexus server","sync-nexus-wc"));
                
                $invoice_sent=(int)get_post_meta($order->id,"_invoice_sent_to_nexus",true);
                if ($invoice_sent==$order->id)
                {
                    $this->saveLog(__("SKIP: Invoice already sent to SyncNexusWC server","sync-nexus-wc"));
                    $syncnexuswc_invoice_error_messages=get_transient('claudius_syncnexuswc_invoice_error_messages_' . $order->id);
                    if(!$syncnexuswc_invoice_error_messages || !is_array($syncnexuswc_invoice_error_messages))
                        $syncnexuswc_invoice_error_messages=[];
                    $syncnexuswc_invoice_error_messages[]=__("SKIP: Invoice already sent to SyncNexusWC server","sync-nexus-wc");
                    set_transient('claudius_syncnexuswc_invoice_error_messages_' . $order->id, $syncnexuswc_invoice_error_messages, 30);
                    return true;
                }
                $emag_order=false;
                $emag_order_id=(int)get_post_meta($order->id,"_emag_order_id",true);

                if ($emag_order_id>0)
                    $emag_order=true;

                if($emag_order)
                {
                    $this->saveLog(__("Order is from eMAG","sync-nexus-wc"));
                    if($this->config->send_wc_invoices!="yes")
                    {
                        $this->saveLog(__("SKIP: Send eMAG Invoices to Nexus ERP is set to No","sync-nexus-wc"));
                        return true;
                    }
                }
                else
                {
                    $this->saveLog(__("Order is from WC","sync-nexus-wc"));
                    if($this->config->send_wc_invoices!="yes")
                    {
                        $this->saveLog(__("SKIP: Send WC Invoices to Nexus ERP is set to No","sync-nexus-wc"));
                        return true;
                    }
                }

                if(!$this->checkDependencies())
                {
                    $this->saveLog(__("Cannot continue with invoice sending to nexus server because of dependencies","sync-nexus-wc"));
                    return false;
                }
                
                if($emag_order)
                    $this->sendEmagInvoiceToNexus($order);
                else
                    $this->sendWCInvoiceToNexus($order);
                
            }
        }
    }

    public function sendWCInvoiceToNexus($order)
    {
        if(!$order)
            return false;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        
        $id_importex=$this->config->nexus_invoice_source;

        $invoice_number=$order->id;

        if(function_exists('wcpdf_get_invoice'))
        {
            $invoice = wcpdf_get_invoice($order->id);
            if($invoice)
                $invoice_number = $invoice->get_number( $invoice->get_type(), null, '', true );
            else
            {
                $this->saveLog(__("SKIP: cannot load invoice for order","sync-nexus-wc").": ".$order->id);
                return false;
            }
        }
        $fname=$order->get_billing_first_name();
        $lname=$order->get_billing_last_name();

        $den_delegat="";
        if($fname!="")
            $den_delegat=$fname;
        if($lname!="")
            $den_delegat.=" ".$lname;

        $cui="";
        if($this->config->nexus_order_meta_cui!="")
        {
            $meta_data=$order->get_meta_data();
            foreach ($meta_data as $meta) {
                if ($meta->get_data()['key'] === $this->config->nexus_order_meta_cui) {
                    $value = $meta->get_data()['value'];
                    if($this->config->nexus_order_meta_cui_child!="" && isset($value[$this->config->nexus_order_meta_cui_child]))
                        $cui=$value[$this->config->nexus_order_meta_cui_child];
                    else if(is_string($value))
                        $cui=$value;
                    else
                        $cui="";
                }
            }
        }

        //if($cui=="")
        //    $cui="0000000000000";

        $cui=str_ireplace("ro","",$cui);

        $reg_com="";
        if($this->config->nexus_order_meta_reg_com!="")
        {
            $meta_data=$order->get_meta_data();
            foreach ($meta_data as $meta) {
                if ($meta->get_data()['key'] === $this->config->nexus_order_meta_reg_com) {
                    $value = $meta->get_data()['value'];
                    if($this->config->nexus_order_meta_reg_com_child!="" && isset($value[$this->config->nexus_order_meta_reg_com_child]))
                        $reg_com=$value[$this->config->nexus_order_meta_reg_com_child];
                    else if(is_string($value))
                        $reg_com=$value;
                    else
                        $reg_com="";
                }
            }
        }

        $pers_fiz="1";
        if($this->config->nexus_order_meta_pers_jur!="")
        {
            $meta_data=$order->get_meta_data();
            foreach ($meta_data as $meta) {
                if ($meta->get_data()['key'] === $this->config->nexus_order_meta_pers_jur) {
                    $value = $meta->get_data()['value'];
                    if($this->config->nexus_order_meta_pers_jur_child!="" && isset($value[$this->config->nexus_order_meta_pers_jur_child]) && $this->config->nexus_order_meta_pers_jur_value!="" && $value[$this->config->nexus_order_meta_pers_jur_child]==$this->config->nexus_order_meta_pers_jur_value)
                        $pers_fiz="0";
                    else if(is_string($value) && $this->config->nexus_order_meta_pers_jur_value!="" && $value==$this->config->nexus_order_meta_pers_jur_value)
                        $pers_fiz="0";
                    else
                        $pers_fiz="1";
                }
            }
        }

        $full_address=$order->get_billing_address_1();
        if(trim($order->get_billing_address_2())!="")
            $full_address=$order->get_billing_address_1()." ".$order->get_billing_address_2();

        $location=$this->getLocationDetailsForNexus($order);
        $adr_det=$this->parseAddressForNexus($full_address);

        $den_partener=$order->get_billing_company();
        if($den_partener=="")
            $den_partener=$den_delegat;
        
        $customer_note = $order->get_customer_note();
        $platitor_tva=0;
        if($pers_fiz==0)
            $platitor_tva=1;
        $id_partner="";
        $id_address="";
        $partner=[
           "parametri" => [
                 "id_importex" => $this->translateAndLimitString($id_importex,25),
                 "manage_existing" => 1, 
                 "single_tran" => 1
              ], 
           "linii" => [
                [
                   "id_intern" => "", 
                   "id_partener" => "", 
                   "id_localitate" => (int)$location->id_locatie,
                   "cif_cnp" => $this->translateAndLimitString($cui,20),
                   "denumire" => $this->translateAndLimitString($den_partener,100), 
                   "pers_fizica" => (int)$pers_fiz,
                   "platitor_tva" => $platitor_tva, //not known
                   "registru_comert" => $this->translateAndLimitString($reg_com,15), 
                   "banca" => "", 
                   "contul" => "", 
                   "adresa" => $this->translateAndLimitString($full_address,250),
                   "email" => $this->translateAndLimitString($order->get_billing_email(),50),
                   "website" => "", 
                   "fax" => "", 
                   "telefon" => $this->translateAndLimitString($order->get_billing_phone(),60), 
                   "telefon_serv" => "", 
                   "manager" => "", 
                   "cod_tara" => $order->get_billing_country(),
                   "den_localitate" => "", 
                   "den_regiune" => "", 
                   "id_clasificare" => "", 
                   "den_clasificare" => "", 
                   "id_clasificare2" => "", 
                   "den_clasificare2" => "", 
                   "id_clasificare3" => "", 
                   "den_clasificare3" => "", 
                   "id_agent" => "", 
                   "den_agent" => "", 
                   "termen_incasare" => "30", 
                   "termen_plata" => "30", 
                   "moneda" => "RON", 
                   "observatii" => "", 
                   "limita_credit" => "0.00", 
                   "restanta_max" => "0", 
                   "cod_card" => "", 
                   "id_disc" => "", 
                   "den_disc" => "", 
                   "id_zona_comerciala" => "", 
                   "den_zona_comerciala" => "", 
                   "client_ret" => "0" 
                ] 
             ] 
        ];

        $this->saveLog(__("Creating or updating partner on Nexus server","sync-nexus-wc"));

        if($this->config->nexus_send_invoices_live=="no")
            $this->saveLog(print_r($partner,true));
        else
        {
            //check if we can get an ID
            $check_api_response=$this->doNexusRequest("api/v3/read","parteneri",array("denumire"=>$this->translateAndLimitString($den_partener,100)));
            $have_errors=$this->getResultError($check_api_response);
            if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
            {
                $partner["linii"][0]["id_intern"]=$check_api_response->result[0]->id_intern;
                $id_partner=$check_api_response->result[0]->id_intern;
            }
            $check_api_response=$this->doNexusRequest("api/v1/import","parteneri",$partner);
            $have_errors=$this->getResultError($check_api_response);
            if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK')
                $this->saveLog(__("Created or updated partner on Nexus server with success","sync-nexus-wc")." ".$this->translateAndLimitString($den_partener,100));
            else
            {
                $syncnexuswc_invoice_error_messages=get_transient('claudius_syncnexuswc_invoice_error_messages_' . $order->id);
                if(!$syncnexuswc_invoice_error_messages || !is_array($syncnexuswc_invoice_error_messages))
                    $syncnexuswc_invoice_error_messages=[];
                $syncnexuswc_invoice_error_messages[]=$have_errors;
                set_transient('claudius_syncnexuswc_invoice_error_messages_' . $order->id, $syncnexuswc_invoice_error_messages, 30);
            }
        }

        $address = [
            "parametri" => [
                 "manage_existing" => 1, 
                 "id_importex" => $this->translateAndLimitString($id_importex,25)
              ], 
            "linii" => [
                [
                       "id_intern" => "", 
                       "id_extern" => (int)$order->get_customer_id(), 
                       "cod_judet" => $this->translateAndLimitString($location->cod_judet,2),
                       "id_localitate" => (int)$location->id_locatie,
                       "cod_siruta" => $this->translateAndLimitString($location->siruta,6),
                       "observatii" => $full_address,
                       "denumire" => $this->translateAndLimitString($den_delegat,50),
                       "id_partener" => "", 
                       "cod_partener" => "", 
                       "den_partener" => $this->translateAndLimitString($den_partener,100), 
                       "cod_tara" => $order->get_billing_country(), 
                       "den_localitate" => $this->translateAndLimitString($location->nume,100), 
                       "den_regiune" => $this->translateAndLimitString($location->zona,100), 
                       "strada" => $this->translateAndLimitString($adr_det['street'],50), 
                       "numar" => $this->translateAndLimitString($adr_det['number'],6),
                       "bloc" => $this->translateAndLimitString($adr_det['block'],50),
                       "scara" => $this->translateAndLimitString($adr_det['staircase'],50),
                       "etaj" => $this->translateAndLimitString($adr_det['floor'],50),
                       "apartament" => $this->translateAndLimitString($adr_det['apartment'],50),
                       "cod_postal" =>  $this->translateAndLimitString($order->get_billing_postcode(), 7),
                       "telefon" => $this->translateAndLimitString($order->get_billing_phone(),25), 
                       "zona" => $this->translateAndLimitString($location->zona,25), 
                       "email" => $order->get_billing_email(),
                       "gln" => "0", 
                       "este_sediu" => "1", 
                       "id_clasificare" => "", 
                       "id_clasificare2" => "", 
                       "id_clasificare3" => "" 
                ] 
            ]
        ];
        
        $this->saveLog(__("Creating or updating address on Nexus server","sync-nexus-wc"));

        if($this->config->nexus_send_invoices_live=="no")
            $this->saveLog(print_r($address,true));
        else
        {
            //check if we can get an ID
            $check_api_response=$this->doNexusRequest("api/v3/read","adrese_parteneri",array("denumire"=>$this->translateAndLimitString($den_delegat,50),"den_partener"=>$this->translateAndLimitString($den_partener,100)));
            $have_errors=$this->getResultError($check_api_response);
            if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
            {
                $address["linii"][0]["id_intern"]=$check_api_response->result[0]->id_intern;
                $id_address=$check_api_response->result[0]->id_intern;
            }
            $check_api_response=$this->doNexusRequest("api/v1/import","adrese",$address);
            $have_errors=$this->getResultError($check_api_response);
            if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK')
                $this->saveLog(__("Created or updated address on Nexus server with success","sync-nexus-wc")." ".$this->translateAndLimitString($full_address,999));
            else
            {
                $syncnexuswc_invoice_error_messages=get_transient('claudius_syncnexuswc_invoice_error_messages_' . $order->id);
                if(!$syncnexuswc_invoice_error_messages || !is_array($syncnexuswc_invoice_error_messages))
                    $syncnexuswc_invoice_error_messages=[];
                $syncnexuswc_invoice_error_messages[]=$have_errors;
                set_transient('claudius_syncnexuswc_invoice_error_messages_' . $order->id, $syncnexuswc_invoice_error_messages, 30);
            }
        }

        $den_client=$order->get_billing_company();
        if($den_client=="")
            $den_client=$den_delegat;

        $order_items = $order->get_items();
        $invoice_items=[];
        foreach ($order_items as $item_id => $item)
        {
            $product_name = $item->get_name();
            $product_id = $item->get_product_id();
            $quantity = $item->get_quantity();
            $subtotal = $item->get_subtotal();
            $total = $item->get_total();
            $total_with_tax = $item->get_total() + $item->get_total_tax();
            $total_without_tax = $item->get_total();
            $product = $item->get_product();
            $sku = $product->get_sku();
            $taxes = $item->get_taxes();
            $tax_details = $taxes['total'];
            $tax_rate = !empty($tax_details) ? key($tax_details) : 0;
            $tax_percent = \WC_Tax::get_rate_percent($tax_rate);
            $tax_percent = str_replace('%', '', $tax_percent);
            $id_produs="";
            //check if we can get an ID
            $check_api_response=$this->doNexusRequest("api/v3/read","produse",array("denumire"=>$this->translateAndLimitString($product_name,110)));
            $have_errors=$this->getResultError($check_api_response);
            if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
                $id_produs=$check_api_response->result[0]->id_intern;
            $total_without_tax=$total_without_tax/$quantity;
            $total_with_tax=$total_with_tax/$quantity;
            $invoice_items[]=[
                "id_produs" => $this->translateAndLimitString($id_produs,25),
                "cod_produs" => $this->translateAndLimitString($sku, 20),
                "den_produs" => $this->translateAndLimitString($product_name, 173),
                //"tip_produs" => "N",
                "cont_produs" => $this->config->nexus_invoice_cont_produs,
                //"serie_produs" => "",
                "cantitate" => $quantity,
                "pret_vanzare" => number_format(round($total_without_tax,2),2,".",""),
                "pret_vanzare_tva" => number_format(round($total_with_tax,2),2,".",""),
                "cota_tva_ies" => $tax_percent,//////////////////////
                "denumire_sup" => "", 
                "id_centru_profit" => "", 
                "den_centru_profit" => "", 
                "numar_auto" => "", 
                "id_aviz" => "", 
                "serie_aviz" => "", 
                "numar_aviz" => "", 
                "id_comanda" => "", 
                "serie_comanda" => "", 
                "numar_comanda" => "", 
                "corectie_tva" => 0, 
                "discount" => 0, 
                "discount_proc" => 0 
            ];
        }
        $shipping_items = $order->get_shipping_methods();

        foreach ($shipping_items as $shipping_item)
        {
            $shipping_name = $shipping_item->get_method_title();
            $shipping_method_code = $shipping_item->get_method_id();
            $shipping_cost = $shipping_item->get_total();
            $shipping_tax = $shipping_item->get_total_tax();
            $shipping_cost_with_vat = $shipping_cost + $shipping_tax;
            $shipping_cost_with_vat = number_format(round($shipping_cost_with_vat,2),2,".","");
            $effective_tax_rate = ($shipping_cost > 0) ? ($shipping_tax / $shipping_cost) * 100 : 0;
            $effective_tax_rate = ceil($effective_tax_rate);
            $id_produs="";
            //check if we can get an ID
            $check_api_response=$this->doNexusRequest("api/v3/read","produse",array("denumire"=>$this->translateAndLimitString($shipping_name,110)));
            $have_errors=$this->getResultError($check_api_response);
            if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
                $id_produs=$check_api_response->result[0]->id_intern;
            else
            {
                $carrier=new stdClass;
                $carrier->title=$shipping_name;
                $carrier->sku=$shipping_method_code;
                $carrier->tax=$effective_tax_rate;
                $carrier->description=$this->translateAndLimitString($shipping_name,10000);
                $carrier->price_incl_tax=$shipping_cost_with_vat;
                $id_produs=$this->createNexusCarrierProduct($carrier);
            }
            $invoice_items[]=[
                "id_produs" => $this->translateAndLimitString($id_produs,25),
                "cod_produs" => $this->translateAndLimitString($shipping_method_code, 20),
                "den_produs" => $this->translateAndLimitString($shipping_name, 173), 
                "tip_produs" => $this->config->nexus_invoice_tip_produs,
                "cont_produs" => $this->config->nexus_invoice_cont_produs,
                //"serie_produs" => "",
                "cantitate" => 1.0000,
                "pret_vanzare" => number_format(round($shipping_cost,2),2,".",""),
                "pret_vanzare_tva" => number_format(round($shipping_cost_with_vat,2),2,".",""),
                "cota_tva_ies" => $effective_tax_rate,
                "denumire_sup" => "", 
                "id_centru_profit" => "", 
                "den_centru_profit" => "", 
                "numar_auto" => "", 
                "id_aviz" => "", 
                "serie_aviz" => "", 
                "numar_aviz" => "", 
                "id_comanda" => "", 
                "serie_comanda" => "", 
                "numar_comanda" => "", 
                "corectie_tva" => 0, 
                "discount" => 0, 
                "discount_proc" => 0 
            ];
        }

        $inv_series=$id_importex;
        $inv_number=$order->id;
        if($invoice_number!="")
        {
            if (preg_match('/^([A-Z]+)-0*(\d+)$/', $invoice_number, $matches)) {
                $inv_series = $matches[1];
                $inv_number = $matches[2];
            }
            if($inv_series=="")
                $inv_series=$id_importex;
            if($inv_number=="")
                $inv_number=$order->id;
        }
        $validare=false;
        if($this->config->nexus_send_invoices_auto_validate=='yes')
            $validare=true;
        $test_suffix="";
        $data=[
            "parametri" => [
                "id_importex" => $this->translateAndLimitString($id_importex,25),
                "single_tran" => 1,
                "gen_pvmpma" => 0,
                "inchidere_cmd" => 0
            ],
            "antete" => [
                "id_document" => $this->translateAndLimitString($inv_series."-".$inv_number.$test_suffix,25),
                "tip_document" => "Factura",//implicit Factura
                "serie_document" => $this->translateAndLimitString($inv_series.$test_suffix,10),
                "numar_document" => $inv_number.$test_suffix,//$order->id,//$this->translateAndLimitString($invoice_number,20),
                "data_document" => $order->get_date_created()->date('Y-m-d'),
                "data_scadenta" => gmdate("Y-m-d",strtotime($order->get_date_created()->date('Y-m-d')." +1 month")),
                "cod_factura" => "",
                "moneda" => "RON",
                "curs" => 1.0000,
                "id_gestiune" => "",////
                "den_gestiune" => $this->translateAndLimitString($id_importex,30),////
                "cont_debit" => "4111.  .  .  .   ",
                "id_client" => $id_partner,
                "cif_client" => $this->translateAndLimitString($cui,20),
                "den_client" => $this->translateAndLimitString($den_client,100),
                //"id_delegat" => "",///
                //"den_delegat" => $this->translateAndLimitString($den_delegat,30),
                //"id_agent" => "",//
                //"den_agent" => $this->translateAndLimitString($id_importex,100),
                //"id_facturist" => "",
                //"den_facturist" => $this->translateAndLimitString($id_importex,100),
                "id_adresa" => $id_address,
                "den_adresa" => $this->translateAndLimitString($den_delegat,50),//it is 50 because at address is 50 and here is 100....
                //"id_transport" => "",
                //"den_transport" => $this->translateAndLimitString($id_importex,100),
                "observatii" => $this->translateAndLimitString($customer_note,10000),
                "observatii_print" => "",
                "livrat_afara_ro" => false,
                "taxare_inversa" => false,
                "incasare_bon_fiscal" => false,
                //"data_anaf" => gmdate("Y-m-d"),/////////////
                //"id_solicitare" => "",///
                "validare" => $validare,
                "anulare" => false,
                "linii" => $invoice_items
            ]
        ];

        $this->saveLog(__("Sending invoices to Nexus server","sync-nexus-wc")."[1]");//WC

        if($this->config->nexus_send_invoices_live=="no")
        {
            $this->saveLog(print_r($data,true));
        }
        else
        {
            $check_api_response=$this->doNexusRequest("api/v1/import","facturi_clienti",$data);
            $have_errors=$this->getResultError($check_api_response);
            if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK')
            {
                update_post_meta($order->id,"_invoice_sent_to_nexus",$order->id);
                $this->saveLog(__("Created invoice on Nexus server with success","sync-nexus-wc")." ".$this->translateAndLimitString($invoice_number,20));
                return true;
            }
            else
            {
                $syncnexuswc_invoice_error_messages=get_transient('claudius_syncnexuswc_invoice_error_messages_' . $order->id);
                if(!$syncnexuswc_invoice_error_messages || !is_array($syncnexuswc_invoice_error_messages))
                    $syncnexuswc_invoice_error_messages=[];
                $syncnexuswc_invoice_error_messages[]=$have_errors;
                set_transient('claudius_syncnexuswc_invoice_error_messages_' . $order->id, $syncnexuswc_invoice_error_messages, 30);
            }
        }
    }
    public function createNexusCarrierProduct($product, $class_prefix="ship-")
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($product)
        {
            $id_importex=$this->config->nexus_invoice_source;
            $category_name=$class_prefix.$id_importex;
            $data=[
               "parametri" => [
                     "id_importex" => $this->translateAndLimitString($id_importex,25),
                     "manage_existing" => 1
                  ], 
               "linii" => [
                        [
                           "id_clasa" => $category_name, 
                           "id_intern" => "",
                           "tip" => "N", 
                           "denumire" => $this->translateAndLimitString($category_name,52), 
                           "id_grupa" => "", 
                           "den_grupa" => "", 
                           "cod_prior" => "", 
                           "errorlist" => "" 
                        ] 
                     ] 
            ]; 

            if($this->config->nexus_send_invoices_live=="no")
            {
                $this->saveLog(print_r($data,true));
            }
            else
            {
                $id_clasa="";
                //check if we can get an ID
                $check_api_response=$this->doNexusRequest("api/v3/read","clase_produse",array("id_extern"=>$this->translateAndLimitString($category_name,25),"denumire"=>$this->translateAndLimitString($category_name,100)));
                $have_errors=$this->getResultError($check_api_response);
                if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
                {
                    $data["linii"][0]["id_intern"]=$check_api_response->result[0]->id_intern;
                    $id_clasa=$check_api_response->result[0]->id_intern;
                }
                $check_api_response=$this->doNexusRequest("api/v1/import","clase_produse",$data);
                $have_errors=$this->getResultError($check_api_response);
                if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK')
                    $this->saveLog(__("Created or updated product class on Nexus server with success","sync-nexus-wc")." ".$this->translateAndLimitString($category_name,100));
                else
                {

                }
            }

            $weight_in_kg=0.00;
            $product_tax=str_replace("%","",$product->tax);
            $data = [
            "parametri" => [
                 "id_importex" => $this->translateAndLimitString($id_importex,25), 
                 "manage_existing" => 1, 
                 "single_tran" => 1 
              ], 
            "linii" => [
                    "id_produs" => $product->sku,//$product->get_id(), 
                    "id_intern" => "",
                    "tip" => "N", 
                    "cod_extern" => $this->translateAndLimitString($product->sku,20), 
                    "denumire" => $this->translateAndLimitString($product->title,110), 
                    "descriere" => "", 
                    "id_clasa" => "", ///////
                    "den_clasa" => $this->translateAndLimitString($category_name,30), 
                    "cota_tva_ies" => $this->translateAndLimitString($product_tax,2), 
                    "cota_tva_int" => $this->translateAndLimitString($product_tax,2), 
                    "um" => "BUC", 
                    "um2" => "", 
                    "conv_um2" => "0.00000", 
                    "den_brand" => "", 
                    "volum" => "0.000", 
                    "greutate" => $this->translateAndLimitString($weight_in_kg,20), 
                    "ambalare" => "0.00", 
                    "garantie" => "0", 
                    "densitate" => "0.0000", 
                    "latime" => "0.000", 
                    "valoric" => "0", 
                    "locație" => "", 
                    "cod_echiv_furnizor" => "", 
                    "observatii" => "", 
                    "cod_tara_origine" => "", 
                    "cod_nc8" => "0", 
                    "spec_tip" => "TXT", 
                    "spec_descriere" => $this->translateAndLimitString($product->description,10000), 
                    "spec_instalare" => "", 
                    "spec_utilizare" => "", 
                    "id_furn" => "", 
                    "cif_furn" => "", 
                    "den_furn" => "", 
                    "pret_achizitie" => $this->translateAndLimitString($product->price_incl_tax,100), 
                    "stas" => "", 
                    "dim" => "", 
                    "culoare" => "", 
                    "val_pct_fidelitate" => "0.0000", 
                    "cod_selectie" => "", 
                    "errorlist" => "", 
                    "status" => "", 
                    "mod_vanzare_easy_retail" => 1,
                    "validare" => 1 
                 ] 
            ];
            if($this->config->nexus_send_invoices_live=="no")
            {
                $this->saveLog(print_r($data,true));
            }
            else
            {
                $id_produs="";
                //check if we can get an ID
                $check_api_response=$this->doNexusRequest("api/v3/read","produse",array("denumire"=>$this->translateAndLimitString($product->title,110)));
                $have_errors=$this->getResultError($check_api_response);
                if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
                {
                    $data["linii"][0]["id_intern"]=$check_api_response->result[0]->id_intern;
                    $id_produs=$check_api_response->result[0]->id_intern;
                    return $id_produs;
                }
                $check_api_response=$this->doNexusRequest("api/v1/import","produse",$data);
                $have_errors=$this->getResultError($check_api_response);
                if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK')
                {
                    $this->saveLog(__("Created or updated product on Nexus server with success","sync-nexus-wc")." ".$this->translateAndLimitString($product->title,110));
                    //check if we can get an ID
                    $check_api_response=$this->doNexusRequest("api/v3/read","produse",array("denumire"=>$this->translateAndLimitString($product->title,110)));
                    $have_errors=$this->getResultError($check_api_response);
                    if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
                        return $check_api_response->result[0]->id_intern;
                }
                else
                {

                }
            }
        }
        return "";
    }
    public function sendEmagInvoiceToNexus($order)
    {
        return false;
    }
    public function getWcProductsCount()
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $args = array(
            'post_type'   => 'product',
            //'post_status' => 'publish',
            'post_status'    => array('publish', 'draft', 'private'), // Excludes 'trash', 'pending'
            'posts_per_page' => -1,
            'fields'      => 'ids',
        );

        $query = new \WP_Query($args);
        return $query->found_posts;
    }
    public function getWcProductIDs($start = 0, $limit = 10)
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $args = array(
            'post_type'      => 'product',
            //'post_status'    => 'publish',
            'post_status'    => array('publish', 'draft', 'private'), // Excludes 'trash', 'pending'
            'posts_per_page' => $limit,
            'offset'         => $start,
            'fields'         => 'ids',
        );

        $query = new \WP_Query($args);
        return $query->posts; // Return product IDs as an array
    }
    public function importNexusProductsHook()
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        check_admin_referer( 'syncnexuswc_import_nexus_products' );
        $nexus_prod_import_cancel=isset($_POST['nexus_prod_import_cancel'])?(int)$_POST['nexus_prod_import_cancel']:0;

        if($nexus_prod_import_cancel==1)
        {
            $this->stopBatchImport();
            $this->saveLog(__("Cancelled the Nexus products import process","sync-nexus-wc"));
            $this->config->nexus_prod_import_started=0;
            return array('status'=>'updated','msg'=>__("Cancelled the Nexus products import process","sync-nexus-wc"));
        }
        else if(is_file(WC()->plugin_path()."/packages/action-scheduler/action-scheduler.php"))
        {
            $this->startBatchImport();//needs to be before scheduling
            $action_added=$this->addActionSchedulerTask("claudius_syncnexuswc_import_products_hook",array(),"sync-nexus-wc");
            if($action_added)
            {
                $this->saveLog(__("Triggered the Nexus products import process","sync-nexus-wc"));
                $this->config->nexus_prod_import_started=1;
                return array('status'=>'updated','msg'=>__("Set the import Nexus products background process, the import will start shortly","sync-nexus-wc"));
            }
            else
            {
                $this->saveLog(__("Error in triggering the Nexus products import process","sync-nexus-wc"));
                return $this->returnResultError(__("Error in setting the import Nexus products background process, please try again later","sync-nexus-wc"));
            }
        }
    }
    public function startBatchImport()
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $this->updateConfigValue('nexus_prod_import_started', 1);
        $this->updateConfigValue('nexus_prod_imported_cnt', 0);
        $this->updateConfigValue('nexus_prod_import_start', 0);
        as_unschedule_all_actions( "claudius_syncnexuswc_import_products_hook", array(), "sync-nexus-wc" );
        as_unschedule_all_actions( "claudius_syncnexuswc_import_products_batch_reschedule_hook", array(), "sync-nexus-wc" );
    }
    public function stopBatchImport()
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $this->updateConfigValue('nexus_prod_import_started', 0);
        $this->updateConfigValue('nexus_prod_imported_cnt', 0);
        $this->updateConfigValue('nexus_prod_import_start', 0);
    }
    public function importNexusProducts()
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($this->getConfigValue("nexus_prod_import_started")==1)
        {
            if($this->getNexusServerStatus(true)!="200")
            {
                $this->saveLog(__("Nexus server API service not online, stopping the process","sync-nexus-wc"));
                $this->stopBatchImport();
                return;
            }
            $products_to_import=$this->getConfigValue('nexus_prod_batch_import_nr');
            if((int)$products_to_import==0)
                $products_to_import=10;
            if((int)$products_to_import>100)
                $products_to_import=100;

            if($this->getConfigValue("nexus_prod_import_started")==1)
            {
                $prod_count=$this->getWcProductsCount();
                $nexus_prod_import_start=$this->getConfigValue('nexus_prod_import_start');
                $nexus_prod_import_limit=$products_to_import;
                $total_imported_porducts=$nexus_prod_import_start;
                $product_ids=$this->getWcProductIDs($nexus_prod_import_start,$nexus_prod_import_limit);
                if(is_countable($product_ids) && count($product_ids)>0)
                {
                    foreach($product_ids as $prod_id)
                    {
                        $product = wc_get_product($prod_id);
                        if($product)
                        {
                            $this->sendProductsToNexus($product);
                            $total_imported_porducts++;
                            $this->updateConfigValue("nexus_prod_imported_cnt",$total_imported_porducts);
                            $this->updateConfigValue("nexus_prod_import_start",$total_imported_porducts);
                        }
                        else
                            $this->saveLog(__("Could not load product for ID","sync-nexus-wc").": ".$prod_id);
                    }

                    if($total_imported_porducts==$prod_count)
                    {
                        $this->saveLog(__("Finished the import products to Nexus server","sync-nexus-wc"));
                        $this->stopBatchImport();
                        return true;
                    }

                    $total_pages = ceil($prod_count / $nexus_prod_import_limit);
                    $current_page = ceil((int)$this->getConfigValue("nexus_prod_imported_cnt") / $nexus_prod_import_limit);
                    $this->saveLog(__("Imported batch of products into Nexus","sync-nexus-wc")." to_import=".$prod_count." ".$current_page." / ".$total_pages);
                    $action_added=$this->addActionSchedulerTask("claudius_syncnexuswc_import_products_batch_reschedule_hook",array(),"sync-nexus-wc");
                    if(!$action_added)
                    {
                        $this->saveLog(__("Error in setting up the import products Nexus batch process","sync-nexus-wc"));
                        $this->stopBatchImport();
                    }
                }
                else
                {
                    $this->stopBatchImport();
                    $this->saveLog(__("No products found to send to Nexus","sync-nexus-wc"));
                }
            }
            else
            {
                $this->stopBatchImport();
                $this->saveLog(__("Import not started or cancelled","sync-nexus-wc"));
            }
        }
        else
        {
            $this->stopBatchImport();
            $this->saveLog(__("Import not started or cancelled","sync-nexus-wc"));
        }
    }
    public function importNexusProductsBatchReschedule()
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($this->config->nexus_prod_import_started==1 && is_file(WC()->plugin_path()."/packages/action-scheduler/action-scheduler.php"))
        {
            $action_added=$this->addActionSchedulerTask("claudius_syncnexuswc_import_products_hook",array(),"sync-nexus-wc");
            if(!$action_added)
            {
                $this->saveLog(__("Error in setting up the import products Nexus batch process","sync-nexus-wc"));
                $this->stopBatchImport();
            }
        }
    }
    public function getImportProdStatusAjax()
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($this->getConfigValue("nexus_prod_import_started")==0)
        {
            echo "1";
            return;
        }
        $imported=$this->getConfigValue("nexus_prod_imported_cnt");
        $total_product_to_push=$this->getWcProductsCount();
        $import_percent=0;
        if((int)$total_product_to_push>0)
            $import_percent=intval(($imported*100)/$total_product_to_push);
        if($total_product_to_push==0)
        {
            $this->stopBatchImport();
            $this->saveLog(__("Cancelled the Emag products import process, nothing to import","sync-nexus-wc"));
            $this->config->nexus_prod_import_started=0;
        }
        ?>
        <div class="progress-bar">
            <span class="progress-bar-fill" style="width: <?php echo esc_attr($import_percent);?>%;"></span>
        </div>
        <?php
        echo esc_html($imported)."/".esc_html($total_product_to_push);
    }
    public function getCronStatus($hook="")
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($hook!="")
        {
            $hook=sanitize_text_field($hook);
            $pending=$this->getActionSchedulerTaskStatusPending($hook);
            $running=$this->getActionSchedulerTaskStatusRunning($hook);
            if(count($pending)>0)
                return "pending";
            if(count($running)>0)
                return "running";
        }
        return false;
    }
    public function getActionSchedulerTaskStatusPending($hook="")
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($hook!="")
        {
            $hook=sanitize_text_field($hook);
            if ( ! class_exists( 'ActionScheduler_Versions', false ) )
                require_once(WC()->plugin_path()."/packages/action-scheduler/action-scheduler.php");
            return as_get_scheduled_actions(
                array(
                    "hook"=>$hook,
                    "status"=>\ActionScheduler_Store::STATUS_PENDING
                )
            );
        }
        return array();
    }
    public function getActionSchedulerTaskStatusRunning($hook="")
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($hook!="")
        {
            $hook=sanitize_text_field($hook);
            if ( ! class_exists( 'ActionScheduler_Versions', false ) )
                require_once(WC()->plugin_path()."/packages/action-scheduler/action-scheduler.php");
            return as_get_scheduled_actions(
                array(
                    "hook"=>$hook,
                    "status"=>\ActionScheduler_Store::STATUS_RUNNING
                )
            );
        }
        return array();
    }
    public function getCronNextRun($hook="")
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($hook!="")
        {
            $pending="";
            $running="";
            if ( ! class_exists( 'ActionScheduler_Versions', false ) )
                require_once(WC()->plugin_path()."/packages/action-scheduler/action-scheduler.php");
            $sql=$wpdb->prepare("select scheduled_date_local, status from ".$wpdb->prefix."actionscheduler_actions where hook=%s and (status=%s or status=%s)",array(sanitize_text_field($hook),sanitize_text_field(\ActionScheduler_Store::STATUS_RUNNING),sanitize_text_field(\ActionScheduler_Store::STATUS_PENDING)));
            $results=$wpdb->get_results($sql);
            if(is_array($results))
            {
                foreach($results as $result)
                {
                    if($result->status==\ActionScheduler_Store::STATUS_PENDING)
                        $pending=$result->scheduled_date_local;
                    if($result->status==\ActionScheduler_Store::STATUS_RUNNING)
                        $running=$result->scheduled_date_local;
                }
            }
            if($pending!="")
                return $pending;
            if($running!="")
                return $running;
        }
        return "";
    }
    public function getCronLastRun($hook="")
    {
        global $wpdb;
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($hook!="")
        {
            $completed="";
            if ( ! class_exists( 'ActionScheduler_Versions', false ) )
                require_once(WC()->plugin_path()."/packages/action-scheduler/action-scheduler.php");
            $sql=$wpdb->prepare("select last_attempt_local, status from ".$wpdb->prefix."actionscheduler_actions where hook=%s and status=%s order by action_id desc limit 1",array(sanitize_text_field($hook),sanitize_text_field(\ActionScheduler_Store::STATUS_COMPLETE)));
            $results=$wpdb->get_results($sql);
            if(is_array($results))
            {
                foreach($results as $result)
                {
                    if($result->status==\ActionScheduler_Store::STATUS_COMPLETE)
                        $completed=$result->last_attempt_local;
                }
            }
            if($completed!="")
                return $completed;
        }
        return "";
    }
    public function sendProductsToNexus($product)
    {
        if ($product->is_type('variable'))
        {
            $variation_ids = $product->get_children();
            foreach($variation_ids as $variation_id)
                $this->sendProductToNexus($product,$variation_id);
        }
        else
            $this->sendProductToNexus($product);
    }
    public function sendProductToNexus($product, $variation_id=0)
    {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if($product)
        {
            $product_parent = $product;

            if($variation_id>0)
            {
                $product = new \WC_Product_Variation($variation_id);
                if(!$product)
                    return false;
            }
            
            $product_id = $product_parent->get_id();

            $title = $product->get_title();
            if($variation_id>0)
                $title=$this->getProductVariationName($product_id,$variation_id);

            $id_importex=$this->config->nexus_invoice_source;
            $sku = $product->get_sku();
            $description = $product->get_description();
            if(trim($sku)=="")
            {
                $this->saveLog(__("SKIP create product, missing SKU for","sync-nexus-wc").": ".$product->get_id()." ".$variation_id);
                return;
            }

            $tax_class = $product->get_tax_class();
            $tax_class = $tax_class ? $tax_class : 'standard';
            $tax_rates = \WC_Tax::get_rates($tax_class);
            $product_tax=0;
            if ( ! empty($tax_rates) ) {
                foreach ($tax_rates as $rate) {
                    $product_tax = $rate['rate'];
                    break;
                }
            }
            $weight = $product->get_weight();
            $weight_in_kg=(float)$weight;
            if($weight_in_kg<=0)
                $weight_in_kg=0;
            $weight_unit = get_option('woocommerce_weight_unit');
            if($weight>0)
            {
                switch ($weight_unit) {
                    case 'g':  // grams
                        $weight_in_kg = $weight / 1000;
                        break;
                    case 'lbs':  // pounds
                        $weight_in_kg = $weight * 0.453592;
                        break;
                    case 'oz':  // ounces
                        $weight_in_kg = $weight * 0.0283495;
                        break;
                    case 'kg':  // already in kg
                    default:
                        $weight_in_kg = $weight;
                        break;
                }
            }
            $category_id_last=$this->getWcProductDefaultCategory($product->get_id());
            $category_name=$this->getCategoryname($category_id_last);
            if($category_name=="")
            {
                $category_id_last=$this->getWcProductDefaultCategory($product_parent->get_id());
                $category_name=$this->getCategoryname($category_id_last);
                if($category_name=="")
                    $category_name="Uncategorized";
            }

            $data=[
               "parametri" => [
                     "id_importex" => $this->translateAndLimitString($id_importex,25),
                     "manage_existing" => 1
                  ], 
               "linii" => [
                        [
                           "id_clasa" => "", 
                           "id_intern" => "",
                           "tip" => "N", 
                           "denumire" => $this->translateAndLimitString($category_name,52), 
                           "id_grupa" => "", 
                           "den_grupa" => "", 
                           "cod_prior" => "", 
                           "errorlist" => "" 
                        ] 
                     ] 
            ]; 

            if($this->config->nexus_send_invoices_live=="no")
            {
                $this->saveLog(print_r($data,true));
            }
            else
            {
                $id_clasa="";
                //check if we can get an ID
                $check_api_response=$this->doNexusRequest("api/v3/read","clase_produse",array("denumire"=>$this->translateAndLimitString($category_name,100)));
                $have_errors=$this->getResultError($check_api_response);
                if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
                {
                    $data["linii"][0]["id_intern"]=$check_api_response->result[0]->id_intern;
                    $id_clasa=$check_api_response->result[0]->id_intern;
                }
                $check_api_response=$this->doNexusRequest("api/v1/import","clase_produse",$data);
                $have_errors=$this->getResultError($check_api_response);
                if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK')
                    $this->saveLog(__("Created or updated product class on Nexus server with success","sync-nexus-wc")." ".$this->translateAndLimitString($category_name,100));
                else
                {

                }
            }

            $weight_in_kg=number_format(round($weight_in_kg,2),2,".","");
            $price_incl_tax = wc_get_price_including_tax($product);//$product->get_price_including_tax();

            $data = [
            "parametri" => [
                 "id_importex" => $this->translateAndLimitString($id_importex,25), 
                 "manage_existing" => 1, 
                 "single_tran" => 1 
              ], 
            "linii" => [
                    "id_produs" => $product->get_id(), 
                    "id_intern" => "",
                    "tip" => "N", 
                    "cod_extern" => $this->translateAndLimitString($sku,20), 
                    "denumire" => $this->translateAndLimitString($title,110), 
                    "descriere" => "", 
                    "id_clasa" => "", ///////
                    "den_clasa" => $this->translateAndLimitString($category_name,30), 
                    "cota_tva_ies" => $this->translateAndLimitString($product_tax,2), 
                    "cota_tva_int" => $this->translateAndLimitString($product_tax,2), 
                    "um" => "BUC", 
                    "um2" => "", 
                    "conv_um2" => "0.00000", 
                    "den_brand" => "", 
                    "volum" => "0.000", 
                    "greutate" => $this->translateAndLimitString($weight_in_kg,20), 
                    "ambalare" => "0.00", 
                    "garantie" => "0", 
                    "densitate" => "0.0000", 
                    "latime" => "0.000", 
                    "valoric" => "0", 
                    "locație" => "", 
                    "cod_echiv_furnizor" => "", 
                    "observatii" => "", 
                    "cod_tara_origine" => "", 
                    "cod_nc8" => "0", 
                    "spec_tip" => "TXT", 
                    "spec_descriere" => $this->translateAndLimitString($description,10000), 
                    "spec_instalare" => "", 
                    "spec_utilizare" => "", 
                    "id_furn" => "", 
                    "cif_furn" => "", 
                    "den_furn" => "", 
                    "pret_achizitie" => $this->translateAndLimitString($price_incl_tax,100), 
                    "stas" => "", 
                    "dim" => "", 
                    "culoare" => "", 
                    "val_pct_fidelitate" => "0.0000", 
                    "cod_selectie" => "", 
                    "errorlist" => "", 
                    "status" => "", 
                    "mod_vanzare_easy_retail" => 1,
                    "validare" => 1 
                 ] 
            ];
            if($this->config->nexus_send_invoices_live=="no")
            {
                $this->saveLog(print_r($data,true));
            }
            else
            {
                $id_produs="";
                //check if we can get an ID
                $check_api_response=$this->doNexusRequest("api/v3/read","produse",array("denumire"=>$this->translateAndLimitString($title,110)));
                $have_errors=$this->getResultError($check_api_response);
                if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK' && isset($check_api_response->result) && is_array($check_api_response->result) && count($check_api_response->result)>0)
                {
                    $data["linii"][0]["id_intern"]=$check_api_response->result[0]->id_intern;
                    $id_produs=$check_api_response->result[0]->id_intern;
                }
                $check_api_response=$this->doNexusRequest("api/v1/import","produse",$data);
                $have_errors=$this->getResultError($check_api_response);
                if($have_errors=='' && is_object($check_api_response) && isset($check_api_response->message) && $check_api_response->message=='OK')
                    $this->saveLog(__("Created or updated product on Nexus server with success","sync-nexus-wc")." ".$this->translateAndLimitString($title,110));
                else
                {

                }
            }
        }
    }
}