<?php
/*
* Plugin Name: SWPanel Migrator
* Description: Migration tool for WordPress sites in SWPanel.
* Version: 1.0.0
* Author: SWPanel
* License: GPLv2 or later
* License URI: https://www.gnu.org/licenses/gpl-2.0.html
* Text Domain: swpanel-migrator
* Domain Path: /languages
* Requires at least: 6.8
* Requires PHP: 7.4
*/

defined('ABSPATH') || exit;

if ( file_exists( __DIR__ . '/vendor/autoload.php' ) ) {
    require __DIR__ . '/vendor/autoload.php';
}

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

class SWPM_ApiException extends \Exception {}
class SWPM_AuthException extends \Exception {}
class SWPM_ParamsException extends \Exception {}
class SWPM_Exception extends \Exception {}

add_action('admin_menu', function () {
    add_menu_page(
        __('SWPanel Migrator', 'swpanel-migrator'),
        __('SWPanel Migrator', 'swpanel-migrator'),
        'manage_options',
        'swpanel_migrator',
        'swpanel_migrator_admin_page',
        plugin_dir_url(__FILE__).'assets/images/logo.svg',
        0
    );
});

function swpanel_migrator_admin_page(){
    ?>
    <div id="swpanel-header">
        <img src="<?php echo esc_url(plugin_dir_url(__FILE__) . "assets/images/logo_top.svg");?>" alt="SWPanel Logo">
        <h1><?php echo esc_html__('SWPanel Migrator', 'swpanel-migrator'); ?></h1>
    </div>

    <form id="export-form" method="POST" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <?php wp_nonce_field( 'swpanel_migrator_export', 'swpanel_migrator_nonce' ); ?>
        <div id="export-block">
            <p><?php echo esc_html__('Export your WordPress site to SWPanel using your migration token.', 'swpanel-migrator'); ?></p>
            <input type="hidden" name="action" value="swpanel_migrator_export">
            <input type="text" id="token" name="token" value="" placeholder="<?php esc_html_e( 'Token', 'swpanel-migrator' ); ?>" />
            <input type="text" id="secret_key" name="secret_key" value="" placeholder="<?php esc_html_e( 'Secret Key', 'swpanel-migrator' ); ?>" />
        </div>
        
        <div class="alert-message">
            <div class="icon-message">
                <img src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'assets/images/box_alerta.svg'); ?>">
            </div>
            <div class="content-message">
                <h1><?php esc_html_e('Clear the cache', 'swpanel-migrator'); ?></h1>
                <p><?php esc_html_e('If your site uses a WordPress caching plugin (e.g., WP Rocket, W3 Total Cache, WP Fastest Cache), please clear or purge its cache before exporting to avoid potential migration issues.', 'swpanel-migrator'); ?></p>
            </div>
        </div>

         <div class="alert-message">
            <div class="icon-message">
                <img src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'assets/images/box_alerta.svg'); ?>">
            </div>
            <div class="content-message">
                <h1><?php esc_html_e('Maintenance Mode Activation', 'swpanel-migrator'); ?></h1>
                <p><?php esc_html_e('During the migration process, your website will automatically enter maintenance mode to ensure a safe and uninterrupted transfer. Normal functionality will be restored once the migration is complete.', 'swpanel-migrator'); ?></p>
            </div>
        </div>

        <button type="submit" id="button-block" class="green-button">
            <?php esc_html_e('Start Export', 'swpanel-migrator'); ?>
        </button>
    </form>

    <div id="migration-status">
        <div id="migration-full-block">
            <div id="chart-full-block">
                <div id="chart-block">
                    <div id="chart-container"></div>
                    <div id="chart-text"></div>
                </div>
                <div id="progress-list"></div>
            </div>
            <div id="migration-block">
                <div id="migration-status-block">
                    <div id="status-text"></div>
                    <div id="progress-bar">
                        <div id="progress-fill"></div>
                    </div>
                    <div id="progress-details"></div>
                </div>
            </div>
        </div>
        <button id="back" onclick="window.location.href='<?php echo esc_url(admin_url('admin.php?page=swpanel_migrator')); ?>'">
            <?php esc_html_e('Back', 'swpanel-migrator'); ?>
        </button>
    </div>
    
    <div class="error-block" id="error-retry">
        <div class="error-message">
            <div class="icon-message">
                <img src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'assets/images/box_error.svg');?>">
            </div>
            <div class="content-message">
                <h1><?php echo esc_html__('Error', 'swpanel-migrator'); ?></h1>
                <p id='error-migration'><?php esc_html_e('No migration in progress.', 'swpanel-migrator'); ?></p>
            </div>
        </div>
        <button id="back" onclick="window.location.href='<?php echo esc_url(admin_url('admin.php?page=swpanel_migrator')); ?>'">
            <?php esc_html_e('Back', 'swpanel-migrator'); ?>
        </button>
        <button id="retry-migration">
            <?php esc_html_e('Retry Migration', 'swpanel-migrator'); ?>
        </button>
    </div>

    <?php
}

add_action('admin_enqueue_scripts', function() {
    if (isset($_GET['page']) && $_GET['page'] === 'swpanel_migrator') {
        wp_enqueue_style(
            'google-fonts',
            'https://fonts.googleapis.com/css2?family=Roboto:wght@300;400&display=swap',
            array(),
            '1.0',
        );
        wp_enqueue_style(
            'swpanel_migrator_style',
            plugin_dir_url(__FILE__) . 'css/index.css',
            array(),
            '1.0'
        );
        wp_enqueue_style(
            'swpanel_migrator_style_awesome',
            'https://static.swpanel.com/fonts/font-awesome-2021/css/all.min.css',
            array(),
            '1.0'
        );
        wp_enqueue_script(
            'swpanel_migrator_creating_charts_core',
            'https://static.swpanel.com/charts/FusionCharts-2021/Powercharts/js/fusioncharts.js',
            array(),
            '1.0',
            true
        );
        wp_enqueue_script(
            'swpanel_migrator_creating_charts',
            'https://static.swpanel.com/charts/FusionCharts-2021/Powercharts/js/fusioncharts.charts.js',
            array('swpanel_migrator_creating_charts_core'),
            '1.0',
            true
        );
        wp_enqueue_script(
            'swpanel_migrator',
            plugin_dir_url(__FILE__) . 'js/swpanel-migrator.js',
            array('swpanel_migrator_creating_charts'),
            '1.0',
            true
        );
        wp_set_script_translations('swpanel_migrator', 'swpanel-migrator', plugin_dir_path(__FILE__) . 'languages');
        wp_localize_script( 'swpanel_migrator', 'swpanel_migrator_data', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'run_task' => get_option('swpanel_migrator_run_task', 0)
        ));
    }
});

add_action('admin_post_swpanel_migrator_export', function () {
    if (!current_user_can('manage_options'))
    {
        wp_die(esc_html__('You do not have permission to perform this action.', 'swpanel-migrator'));
    }

    $nonce = isset($_POST['swpanel_migrator_nonce']) ? sanitize_text_field( wp_unslash($_POST['swpanel_migrator_nonce']) ) : '';
    if ( ! $nonce || ! wp_verify_nonce( $nonce, 'swpanel_migrator_export' ) ) {
        wp_die( 'Invalid nonce' );
    }

    $token = isset($_POST['token']) ? sanitize_text_field($_POST['token']) : '';
    $secret_key = isset($_POST['secret_key']) ? sanitize_text_field($_POST['secret_key']) : '';

    update_option('swpanel_migrator_token', $token);
    update_option('swpanel_migrator_secret_key', $secret_key);

    delete_option('swpanel_migrator_status');

    if (!get_option('swpanel_migrator_run_task')) {
        update_option('swpanel_migrator_run_task', 1);
        swpanel_migrator_update_migration_status(['step_index' => 0]);
        as_schedule_single_action(time(), 'swpanel_migrator_long_task');
    }
    wp_redirect(admin_url('admin.php?page=swpanel_migrator&migration=started'));
    exit;
});

add_action('wp_ajax_swpanel_migrator_create_task', function () {
    if (!get_option('swpanel_migrator_run_task')) {
        update_option('swpanel_migrator_run_task', 1);
        swpanel_migrator_update_migration_status([
            'status'=> 'waiting',
            'step_index' => 0
        ]);
        as_schedule_single_action(time(), 'swpanel_migrator_long_task', array(), '', true);
    }
    wp_send_json_success();
});

add_action('swpanel_migrator_long_task', function () {
    try {
        update_option('swpanel_migrator_maintenance', 1);

        swpanel_migrator_update_migration_status(['status' => 'running']);
        swpanel_migrator_decode_token();

        swpanel_migrator_update_migration_status([
            'step_index' => 1,
            'status' => 'running',
            'message' => __('Setting up destination site', 'swpanel-migrator') . '...'
        ]);

        swpanel_migrator_init_destination();

        swpanel_migrator_update_migration_status([
            'step_index' => 2,
            'status' => 'running',
            'message' => __('Creating SQL dump', 'swpanel-migrator') . '...'
        ]);

        swpanel_migrator_create_sql_dump();

        swpanel_migrator_update_migration_status([
            'step_index' => 3,
            'message' => __('Uploading files', 'swpanel-migrator') . '...'
        ]);

        swpanel_migrator_send_chunks();

        swpanel_migrator_update_migration_status([
            'step_index' => 4,
            'message' => __('Executing SQL destination', 'swpanel-migrator') . '...'
        ]);

        swpanel_migrator_execute_dest_sql();

        swpanel_migrator_delete_sql_dump();

        swpanel_migrator_update_migration_status([
            'step_index' => 5,
            'message' => __('Updating WordPress destination', 'swpanel-migrator') . '...'
        ]);
        swpanel_migrator_update_wordpress();

        update_option('swpanel_migrator_maintenance', 0);

        delete_option('swpanel_migrator_run_task');
        swpanel_migrator_update_migration_status([
            'step_index' => 6,
            'status' => 'finished',
            'message' => __('Migration completed successfully', 'swpanel-migrator')
        ]);
    } catch (\Exception $e) {
        swpanel_migrator_delete_sql_dump();
        delete_option('swpanel_migrator_run_task');
        update_option('swpanel_migrator_maintenance', 0);
        swpanel_migrator_update_migration_status([
            'status'  => 'error',
            'message' => $e->getMessage()
        ]);
    }
});

function swpanel_migrator_init_destination() {
    try {
        swpanel_migrator_wp_remote_api_request('POST', [], '/init');
    } catch (SWPM_ApiException $e) {
        throw new \Exception(esc_html__('Failed to set up destination site. Please try again.', 'swpanel-migrator'));
    } catch (SWPM_AuthException $e) {
        throw new \Exception(esc_html__("The token you provided is invalid or expired. Please request a new one and try again.", 'swpanel-migrator'));
    } catch (SWPM_ParamsException $e) {
        throw new \Exception(esc_html__("Required parameters are missing.", 'swpanel-migrator'));
    } catch (SWPM_Exception $e) {
        throw new \Exception(esc_html__('Could not connect to the migration service. Please check your connection or try again later.', 'swpanel-migrator'));
    } catch (Exception $e) {
        throw new \Exception(esc_html__('Failed to install wordpress on the destination server. Please try again.', 'swpanel-migrator'));
    }
}

function swpanel_migrator_update_wordpress() {
    try {
        global $wpdb;
        $body = [
            'title' => get_option('blogname', ''),
            'description' => get_option('blogdescription', ''),
            'language' => get_locale(),
            'table_prefix' => $wpdb->prefix,
            'abspath' => ABSPATH,
            'domain'  => wp_parse_url(get_site_url())['host']
        ];
        swpanel_migrator_wp_remote_api_request('PATCH', $body);
    } catch (SWPM_ApiException $e) {
        throw new \Exception(esc_html__('Failed to install wordpress on the destination server. Please try again.', 'swpanel-migrator'));
    } catch (SWPM_AuthException $e) {
        throw new \Exception(esc_html__("The token you provided is invalid or expired. Please request a new one and try again.", 'swpanel-migrator'));
    } catch (SWPM_ParamsException $e) {
        throw new \Exception(esc_html__("Required parameters are missing.", 'swpanel-migrator'));
    } catch (SWPM_Exception $e) {
        throw new \Exception(esc_html__('Could not connect to the migration service. Please check your connection or try again later.', 'swpanel-migrator'));
    } catch (Exception $e) {
        throw new \Exception(esc_html__('Failed to install wordpress on the destination server. Please try again.', 'swpanel-migrator'));
    }
}
function swpanel_migrator_decode_token() {
    try {
        $migrator_token = get_option('swpanel_migrator_token', '');
        $secret_key = get_option('swpanel_migrator_secret_key', '');
        $decoded = JWT::decode($migrator_token, new Key($secret_key, 'HS256'));
        update_option('swpanel_migrator_token_url', $decoded->token);
        update_option('swpanel_migrator_url', $decoded->url);
    } catch (Exception $e) {
        throw new \Exception(esc_html__('The migration token is invalid. Please request a new one.', 'swpanel-migrator'));
    }
}
function swpanel_migrator_send_chunks (){
    try {
        $chunk_size = 1024 * 1024;
        $batch        = [];
        $batch_size   = 0;
        $total_files  = 0;

        global $wp_filesystem;
        if (empty($wp_filesystem))
        {
            require_once ABSPATH.'/wp-admin/includes/file.php';
            WP_Filesystem();
        }

        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator(dirname(WP_CONTENT_DIR), RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($files as $file) {
            if ($file->isFile()) {
                $total_files++;
            }
        }

        swpanel_migrator_update_migration_status([
            /* translators: %1$d is the total number of files to send */
            'message' => sprintf(__('Uploading files… (0 of %1$d)', 'swpanel-migrator'), $total_files)
        ]);

        $processed = 0;
        foreach ($files as $file) {
            if (!$file->isFile()) continue;

            $processed++;
            $file_path = $file->getPathname();
            $relative    = str_replace(ABSPATH, '', $file_path);
            $file_size   = $file->getSize();

            $content = $wp_filesystem->get_contents($file_path);
            if ($content === false) {
                continue;
            }

            $units = ['B', 'KB', 'MB', 'GB', 'TB'];
            $bytes = max($file_size, 0);
            $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
            $pow = min($pow, count($units) - 1);
            $bytes /= (1 << (10 * $pow)); 
            $file_size_units = round($bytes, 2) . ' ' . $units[$pow];

            $chunk_number = 0;
            $offset = 0;
            $total_chunks = ceil($file_size / $chunk_size);
            while ($offset < $file_size) {
                $details = 
                swpanel_migrator_update_migration_status([
                    /* translators: %1$d is the total number of files to send */
                    'message' => sprintf(__('Uploading files… (%1$d of %2$d)', 'swpanel-migrator'), $processed, $total_files),
                    'details' => $total_chunks > 1
                    /* translators: %1$s is the path of the file, %2$d is the actual chunk sending, %3$d is the total chunks of the file, %4$s is the file size and its unit */
                        ? sprintf(__('%1$s (%2$d of %3$d) - %4$s', 'swpanel-migrator'), $relative, $chunk_number + 1, $total_chunks, $file_size_units)
                    /* translators: %1$s is the path of the file, %2$s is the file size and its unit */
                        : sprintf(__('%1$s - %2$s', 'swpanel-migrator'), $relative, $file_size_units),
                    'percent' => ($processed / $total_files) * 100
                ]);
                $slice = substr($content, $offset, $chunk_size);
                $entry = [
                    'path'         => $relative,
                    'chunk'        => $chunk_number,
                    'data'         => base64_encode($slice),
                ];
                $entry_size = strlen($entry['data']);

                if ($batch_size + $entry_size > $chunk_size && !empty($batch)) {
                    swpanel_migrator_send_chunk_to_api($batch);
                    $batch       = [];
                    $batch_size  = 0;
                }
                $batch[]    = $entry;
                $batch_size += $entry_size;

                $offset += $chunk_size;
                $chunk_number++;
            }
        }

        if (!empty($batch)) {
            swpanel_migrator_send_chunk_to_api($batch);
        }
    } catch (SWPM_ApiException $e) {
        throw new \Exception(esc_html__('The migration failed after multiple attempts. Please try again.', 'swpanel-migrator'));
    } catch (SWPM_AuthException $e) {
        throw new \Exception(esc_html__("The token you provided is invalid or expired. Please request a new one and try again.", 'swpanel-migrator'));
    } catch (SWPM_ParamsException $e) {
        throw new \Exception(esc_html__("Required parameters are missing.", 'swpanel-migrator'));
    } catch (SWPM_Exception $e) {
        throw new \Exception(esc_html__('Could not connect to the migration service. Please check your connection or try again later.', 'swpanel-migrator'));
    } catch (Exception $e) {
        throw new \Exception(esc_html__('Could not send files to the migration service. Please try again.', 'swpanel-migrator'));
    }
}
function swpanel_migrator_delete_sql_dump() {
    global $wp_filesystem;
    if ( empty( $wp_filesystem ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }
    $upload = wp_upload_dir();
    $uploads_dir = trailingslashit( $upload['basedir'] ) . 'swpanel-migrator/';
    $dump_file = $uploads_dir . get_option('swpanel_migrator_sql_uuid', '');
    if ( $wp_filesystem->exists( $dump_file ) ) {
        $wp_filesystem->delete( $dump_file );
    }
}
function swpanel_migrator_create_sql_dump() {
    try {
        global $wpdb, $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }

        $upload = wp_upload_dir();
        $uploads_dir = trailingslashit( $upload['basedir'] ) . 'swpanel-migrator/';
        if ( ! file_exists( $uploads_dir ) ) {
            wp_mkdir_p( $uploads_dir );
        }
        $htaccess_file = $uploads_dir . '.htaccess';
        if ( ! file_exists( $htaccess_file ) ) {
            $rules  = "Options -Indexes\n";
            $rules .= "<FilesMatch \"\.(sql|php|phtml|php5)$\">\n";
            // $rules .= "Deny from all\n";
            $rules .= "Require all denied\n";
            $rules .= "</FilesMatch>";
            file_put_contents( $htaccess_file, $rules );
        }
        $random_name = wp_generate_uuid4() . '.sql';
        update_option('swpanel_migrator_sql_uuid', $random_name);
        $dump_file = $uploads_dir . $random_name;

        $wp_filesystem->put_contents($dump_file, '', FS_CHMOD_FILE);
        
        $tables   = $wpdb->get_results('SHOW TABLES', ARRAY_N);

        foreach ($tables as $table)
        {
            $create_table = $wpdb->get_row("SHOW CREATE TABLE `{$table[0]}`", ARRAY_N);

            $sql = $create_table[1].";\n";
            file_put_contents($dump_file, $sql, FILE_APPEND | LOCK_EX);

            $rows = $wpdb->get_results("SELECT * FROM `{$table[0]}`", ARRAY_A);

            foreach ($rows as $row)
            {
                if (
                    $table[0] === $wpdb->options &&
                    isset($row['option_name']) &&
                    $row['option_name'] === 'swpanel_migrator_maintenance'
                ) {
                    continue;
                }
                $escaped_values = array_map(function ($value) use ($wpdb) {
                    if (is_null($value))
                    {
                        return 'NULL';
                    }
                    $search  = ["\\", "\x00", "\n", "\r", "'", '"', "\x1a"];
                    $replace = ["\\\\", "\\0", "\\n", "\\r", "\\'", '\\"', "\\Z"];
                    return "'" . str_replace( $search, $replace, $value ) . "'";
                }, $row);

                $sql_line = "INSERT IGNORE INTO `{$table[0]}` VALUES (".implode(", ", $escaped_values).");\n";
                file_put_contents($dump_file, $sql_line, FILE_APPEND | LOCK_EX);
            }
        }
    } catch (Exception $e) {
        throw new \Exception(esc_html__('An unexpected error occurred while generating the SQL dump. Please try again.', 'swpanel-migrator'));
    }
}
function swpanel_migrator_send_chunk_to_api($payload_array) {
    $attempt = 0;
    $max_attempts = 5;
    $old_status = get_option('swpanel_migrator_status');
    while ($attempt < $max_attempts){
        try {
            $body = [
                'files'   => $payload_array,
            ];
            swpanel_migrator_wp_remote_api_request('POST', $body, '/upload');
            break;
        } catch (SWPM_ApiException $e) {
            $attempt++;
            swpanel_migrator_update_migration_status([
                'status' => 'api-error',
                'percent' => $old_status['current']['percent'],
                'message' => $old_status['current']['message'],
                /* translators: %1$d is the current attempt number, %2$d is the total number of attempts */
                'details' => $old_status['current']['details'] . '<br>' . sprintf(__('Sending file failed. Retrying attempt %1$d of %2$d.', 'swpanel-migrator'), $attempt, $max_attempts)
            ]);
            sleep(5);
            if ($attempt >= $max_attempts) {
                throw new SWPM_ApiException();
            }
        }
    }
}
function swpanel_migrator_execute_dest_sql() {
    try {
        $upload = wp_upload_dir();
        $uploads_dir = trailingslashit( $upload['basedir'] ) . 'swpanel-migrator/';
        $dump_file = $uploads_dir . get_option('swpanel_migrator_sql_uuid', '');
        $body = [
            'file' => str_replace( ABSPATH, '', $dump_file )
        ];
        swpanel_migrator_wp_remote_api_request('POST', $body, '/sql');
    } catch (SWPM_ApiException $e) {
        throw new \Exception(esc_html__('Failed to create the database on the destination server. Please verify your database credentials and try again.', 'swpanel-migrator'));
    } catch (SWPM_AuthException $e) {
        throw new \Exception(esc_html__("The token you provided is invalid or expired. Please request a new one and try again.", 'swpanel-migrator'));
    } catch (SWPM_ParamsException $e) {
        throw new \Exception(esc_html__("Required parameters are missing.", 'swpanel-migrator'));
    } catch (SWPM_Exception $e) {
        throw new \Exception(esc_html__('Could not connect to the migration service. Please check your connection or try again later.', 'swpanel-migrator'));
    } catch (Exception $e) {
        throw new \Exception(esc_html__('Failed to create the database on the destination server. Please verify your database credentials and try again.', 'swpanel-migrator'));
    }
}

function swpanel_migrator_update_migration_status($data = []) {
    $defaults = [
        'status'  => 'waiting',
        'steps'   => [
            ['label' => __('Waiting for process to start', 'swpanel-migrator'), 'status' => 'running'],
            ['label' => __('Setting up destination site', 'swpanel-migrator'), 'status' => 'running'],
            ['label' => __('Creating SQL dump', 'swpanel-migrator'), 'status' => 'waiting'],
            ['label' => __('Uploading files', 'swpanel-migrator'), 'status' => 'waiting'],
            ['label' => __('Executing SQL destination', 'swpanel-migrator'), 'status' => 'waiting'],
            ['label' => __('Updating WordPress destination', 'swpanel-migrator'), 'status' => 'waiting'],
        ],
        'current' => [
            'message' => '',
            'percent' => 0,
            'details' => '',
        ]
    ];

    $status = get_option('swpanel_migrator_status', $defaults);

    if (isset($data['status'])) {
        $status['status'] = $data['status'];
    }

    if (isset($data['step_index'])) {
        $step_index = intval($data['step_index']);
        $steps_count = count($status['steps']);

        foreach ($status['steps'] as $i => &$step) {
            if ($i < $step_index) {
                $step['status'] = 'done';
            } elseif ($i === $step_index) {
                $step['status'] = 'running';
            } else {
                $step['status'] = 'waiting';
            }
        }
        $status['current']['step_index'] = $step_index;
    }

    $status['current'] = array_merge(
        $status['current'],
        [
            'message' => $data['message'] ?? '',
            'percent' => $data['percent'] ?? 0,
            'details' => $data['details'] ?? ''
        ]
    );
    update_option('swpanel_migrator_status', $status);
}
add_action('wp_ajax_swpanel_migrator_migration_status', function () {
    $status = get_option('swpanel_migrator_status', [
        'status'  => 'waiting',
        'steps'   => [
            ['label' => __('Waiting for process to start', 'swpanel-migrator'), 'status' => 'running'],
            ['label' => __('Creating SQL dump', 'swpanel-migrator'), 'status' => 'waiting'],
            ['label' => __('Uploading files', 'swpanel-migrator'), 'status' => 'waiting'],
            ['label' => __('Executing SQL destination', 'swpanel-migrator'), 'status' => 'waiting'],
            ['label' => __('Updating WordPress destination', 'swpanel-migrator'), 'status' => 'waiting'],
        ],
        'current' => [
            'message' => '',
            'percent' => 0,
            'details' => '',
        ]
    ]);
    wp_send_json_success($status);
});

function swpanel_migrator_maintenance_mode() {
    $is_maintenance = get_option('swpanel_migrator_maintenance', 0);
    if ( ! $is_maintenance ) {
        return;
    }

    $current_page = isset($_GET['page']) ? sanitize_key($_GET['page']) : '';
    if ( wp_doing_ajax() ) {
        return;
    }
    if ( defined('DOING_CRON') && DOING_CRON ) {
        return;
    }
    if ( $current_page === 'swpanel_migrator' ) {
        return;
    }
    if ( !is_admin() ) {
        wp_die(
            '<h1>Site Under Maintenance</h1><p>Migration in progress. Please check back later.</p>',
            'Maintenance Mode',
            ['response' => 503]
        );
    }
    if ( is_admin() && $current_page !== 'swpanel_migrator' ) {
        wp_safe_redirect(admin_url('admin.php?page=swpanel_migrator&migration=started'));
        exit;
    }
}
add_action('admin_init', 'swpanel_migrator_maintenance_mode');
add_action('template_redirect', 'swpanel_migrator_maintenance_mode');

register_uninstall_hook(__FILE__, 'swpanel_migrator_uninstall');
function swpanel_migrator_uninstall() {
    global $wpdb;
    $options = $wpdb->get_col(
        "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE '%swpanel_migrator%'"
    );
    foreach ($options as $option_name) {
        delete_option($option_name);
    }
}

function swpanel_migrator_wp_remote_api_request($method, $body = [], $url_end = '/') {
    $token = get_option('swpanel_migrator_token_url');
    $url = get_option('swpanel_migrator_url');
    $response = wp_remote_post("https://api.swpanel.com$url$url_end", [
        'method'    => $method,
        'headers' => [
            'Content-Type'  => 'application/json',
            'Authorization' => "Bearer $token",
        ],
        'body'    => wp_json_encode($body),
        'timeout' => 300,
    ]);
    $http_code = wp_remote_retrieve_response_code($response);
    if ($http_code < 200 || $http_code >= 300) {
        if ($http_code == 401){
            throw new SWPM_AuthException();
        }
        if ($http_code == 422){
            throw new SWPM_ParamsException();
        }
        throw new SWPM_ApiException();
    }
    if (is_wp_error($response)) {
        throw new SWPM_Exception();
    }
}

add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), 'swpanel_migrator_action_link' );
function swpanel_migrator_action_link($links) {
    $custom['swpanel-migrator-settings'] = sprintf(
        '<a href="%s" aria-label="%s">%s</a>',
        esc_url( admin_url('admin.php?page=swpanel_migrator' )),
        esc_attr__( 'Go to SWPanel Migrator Settings page', 'swpanel-migrator' ),
        esc_html__( 'Settings', 'swpanel-migrator' )
    );
    return array_merge( $custom, (array) $links );
}

register_activation_hook( __FILE__, function() {
    if ( ! wp_next_scheduled( 'swpanel_migrator_cleanup_hook' ) ) {
        wp_schedule_event( time(), 'daily', 'swpanel_migrator_cleanup_hook' );
    }
});

register_deactivation_hook( __FILE__, function() {
    $timestamp = wp_next_scheduled( 'swpanel_migrator_cleanup_hook' );
    if ( $timestamp ) {
        wp_unschedule_event( $timestamp, 'swpanel_migrator_cleanup_hook' );
    }
});

add_action( 'swpanel_migrator_cleanup_hook', function () {
    $upload = wp_upload_dir();
    $backup_dir = trailingslashit( $upload['basedir'] ) . 'swpanel-migrator/';
    if ( ! is_dir( $backup_dir ) ) {
        return;
    }
    $files = glob( $backup_dir . '*' );
    $max_age = HOUR_IN_SECONDS;
    foreach ( $files as $file ) {
        if ( is_file( $file ) 
            && basename($file) !== '.htaccess' 
            && filemtime( $file ) < time() - $max_age ) {
            wp_delete_file( $file );
        }
    }
});
