<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the dashboard.
 *
 * @since      1.0.0
 *
 * @package    sb_bar
 * @subpackage sb_bar/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, dashboard-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    sb_bar
 * @subpackage sb_bar/includes
 * @author     Goran Jakovljevic <goranefbl@gmail.com>
 */
class sb_bar {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      sb_bar_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $sb_bar    The string used to uniquely identify this plugin.
	 */
	protected $sb_bar;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the Dashboard and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		$this->sb_bar = 'sb_bar';
		$this->version = '1.2.11';

		$this->load_dependencies();
		$this->define_global_hooks(); // Call to new method that will show for both front and back
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - sb_bar_Loader. Orchestrates the hooks of the plugin.
	 * - sb_bar_i18n. Defines internationalization functionality.
	 * - sb_bar_Admin. Defines all hooks for the dashboard.
	 * - sb_bar_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-sb-bar-loader.php';

		/**
		 * The class responsible for defining all actions that occur in the Dashboard.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-sb-bar-admin.php';

		/**
		 * The class responsible for defining all Settings.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/settings/class-sb-bar-settings.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/settings/class-sb-bar-enable-settings.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-sb-bar-public.php';

		/**
		 * The class responsible for defining all actions that occur in the global-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-sb-bar-global.php';

		/**
		 * The class responsible for storing and providing shared count info.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-sb-bar-social.php';

		$this->loader = new sb_bar_Loader();

	}

	/**
	 * Register all of the hooks related to the dashboard functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new sb_bar_Admin( $this->get_sb_bar(), $this->get_version() );
		$settings_init_general = new sb_bar_Settings( $this->get_sb_bar() );
		$settings_init_enable = new sb_bar_Enable_Settings( $this->get_sb_bar() );

		$this->loader->add_action( 'admin_menu', $plugin_admin, 'sb_bar_admin_menu' );
		$this->loader->add_action( 'admin_init', $settings_init_general, 'settings_api_init' );
		$this->loader->add_action( 'admin_init', $settings_init_enable, 'settings_api_init' );
		$this->loader->add_filter( 'plugin_action_links_swifty-bar/sb-bar.php', $plugin_admin, 'add_settings_link' );

	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new sb_bar_Public( $this->get_sb_bar(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

		$this->loader->add_action( 'wp_footer', $plugin_public, 'front_end' );
		$this->loader->add_filter( 'the_content', $plugin_public, 'ttr_container' );

		$this->loader->add_action('wp_ajax_nopriv_delete_transient', $plugin_public, 'delete_post_transient');
		$this->loader->add_action('wp_ajax_delete_transient', $plugin_public, 'delete_post_transient');
	}

	/**
	 * Register all of the hooks related to every page of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_global_hooks() {
	    $plugin_global = new sb_bar_Global( $this->get_sb_bar(), $this->get_version() );

	    $this->loader->add_action( 'init', $plugin_global, 'add_image_size' );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_sb_bar() {
		return $this->sb_bar;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    sb_bar_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
