jQuery(document).ready(function ($) {
  // Store timers and data in a closure
  var sliderInstances = {};

  function initCustomSliderAutoplayOnly($slider, $wrapper, $slides, $prev, $next, $pagination, settings) {
    var sliderId = 'sa_auto_' + Math.random().toString(36).slice(2);
    var loopEnabled = settings.loop === 'yes';
    
    // Store instance data
    sliderInstances[sliderId] = {
      timer: null,
      resizeTimer: null,
      animationTimer: null,
      isAnimating: false,
      virtualIndex: 0, // Virtual index that keeps increasing
      loopEnabled: loopEnabled,
      totalRealSlides: $slides.length,
      clonesAdded: false,
      slideWidth: 0,
      slidesPerView: 1,
      wrapperWidth: 0,
      totalSlidesWithClones: 0
    };

    var baseSlidesPerView = parseInt(settings.slidesPerView, 10) || 3;
    var autoplay = settings.autoplay === 'yes';
    var autoplaySpeed = parseInt(settings.autoplaySpeed, 10) || 3000;
    var pauseOnHover = settings.pauseOnHover === 'yes';

    var totalRealSlides = $slides.length;
    var allSlides = $slides; // Will include clones

    function computeSlidesPerView() {
      var w = window.innerWidth;
      if (w <= 767) return 1;
      if (w <= 1024) return Math.min(2, baseSlidesPerView);
      return baseSlidesPerView;
    }

    function maxIndex() {
      return Math.max(0, totalRealSlides - sliderInstances[sliderId].slidesPerView);
    }

    function applySizes() {
      sliderInstances[sliderId].slidesPerView = computeSlidesPerView();
      sliderInstances[sliderId].slideWidth = $slider[0].clientWidth / sliderInstances[sliderId].slidesPerView;
      var slideWidth = sliderInstances[sliderId].slideWidth;

      // Set slide widths for all slides
      allSlides.css({
        width: slideWidth + 'px',
        minWidth: slideWidth + 'px',
        maxWidth: slideWidth + 'px',
        flex: '0 0 ' + slideWidth + 'px',
        boxSizing: 'border-box'
      });

      if (loopEnabled && !sliderInstances[sliderId].clonesAdded) {
        setupInfiniteLoop();
      }

      var totalWidth = loopEnabled ? 
        slideWidth * allSlides.length : 
        slideWidth * totalRealSlides;
      
      sliderInstances[sliderId].wrapperWidth = totalWidth;
      sliderInstances[sliderId].totalSlidesWithClones = allSlides.length;
        
      $wrapper.css({
        width: totalWidth + 'px',
        display: 'flex',
        flexWrap: 'nowrap'
      });

      // For non-loop mode, clamp the virtual index
      if (!loopEnabled) {
        sliderInstances[sliderId].virtualIndex = Math.min(
          sliderInstances[sliderId].virtualIndex, 
          maxIndex()
        );
      }
      
      render(false);
    }

    function setupInfiniteLoop() {
      var slideWidth = sliderInstances[sliderId].slideWidth;
      var slidesPerView = sliderInstances[sliderId].slidesPerView;
      
      // Remove any existing clones first
      $wrapper.find('.sa-clone').remove();
      
      // Store original slides
      var originalSlides = allSlides.toArray();
      
      // For truly continuous sliding, we need MANY clones
      // Add enough clones to make it feel infinite without reset
      var clonesNeeded = Math.max(slidesPerView * 4, totalRealSlides * 2);
      
      // Add clones at the END (duplicate all slides multiple times)
      for (var i = 0; i < clonesNeeded; i++) {
        var cloneIndex = i % totalRealSlides;
        var clone = $(originalSlides[cloneIndex]).clone(true);
        clone.addClass('sa-clone sa-clone-end');
        $wrapper.append(clone);
      }
      
      // Update all slides reference
      allSlides = $wrapper.find('.sa-slide');
      sliderInstances[sliderId].clonesAdded = true;
      sliderInstances[sliderId].totalSlidesWithClones = allSlides.length;
      
      // Start at position 0 (showing first real slides)
      sliderInstances[sliderId].virtualIndex = 0;
    }

    function render(animate) {
      var slideWidth = sliderInstances[sliderId].slideWidth;
      var translateX = -sliderInstances[sliderId].virtualIndex * slideWidth;
      
      if (animate) {
        // Normal animation
        $wrapper.css({
          transition: 'transform 0.6s ease',
          transform: 'translate3d(' + translateX + 'px,0,0)'
        });
        
        sliderInstances[sliderId].isAnimating = true;
        
        // For infinite loop, check if we're running out of clones
        if (loopEnabled) {
          var totalSlides = sliderInstances[sliderId].totalSlidesWithClones;
          var currentIndex = sliderInstances[sliderId].virtualIndex;
          var slidesPerView = sliderInstances[sliderId].slidesPerView;
          
          // If we're getting close to the end of cloned slides
          // We need to seamlessly extend the wrapper with more clones
          if (currentIndex >= totalSlides - (slidesPerView * 3)) {
            // Add more clones before we run out
            setTimeout(function() {
              if (sliderInstances[sliderId].isAnimating) {
                addMoreClones();
              }
            }, 300);
          }
        }
        
        // Set a timeout to mark animation as complete
        clearTimeout(sliderInstances[sliderId].animationTimer);
        sliderInstances[sliderId].animationTimer = setTimeout(function() {
          sliderInstances[sliderId].isAnimating = false;
        }, 600);
      } else {
        // No animation
        $wrapper.css({
          transition: 'none',
          transform: 'translate3d(' + translateX + 'px,0,0)'
        });
      }
      
      updateDots();
    }

    function addMoreClones() {
      if (!loopEnabled || !sliderInstances[sliderId].clonesAdded) return;
      
      var slideWidth = sliderInstances[sliderId].slideWidth;
      var totalRealSlides = sliderInstances[sliderId].totalRealSlides;
      var currentSlides = allSlides.toArray();
      var currentTotal = currentSlides.length;
      
      // Add another set of real slides as clones
      var clonesToAdd = totalRealSlides;
      for (var i = 0; i < clonesToAdd; i++) {
        var cloneIndex = i % totalRealSlides;
        var originalSlide = currentSlides[cloneIndex]; // Get original slide
        var clone = $(originalSlide).clone(true);
        if (!clone.hasClass('sa-clone')) {
          clone.addClass('sa-clone sa-clone-end');
        }
        $wrapper.append(clone);
      }
      
      // Update references
      allSlides = $wrapper.find('.sa-slide');
      sliderInstances[sliderId].totalSlidesWithClones = allSlides.length;
      
      // Update wrapper width
      var newWidth = slideWidth * allSlides.length;
      $wrapper.css('width', newWidth + 'px');
      sliderInstances[sliderId].wrapperWidth = newWidth;
    }

    function goTo(index, animate) {
      if (loopEnabled) {
        // For infinite loop, just set the virtual index
        // We can go as high as we want because we'll keep adding clones
        sliderInstances[sliderId].virtualIndex = index;
      } else {
        // For non-loop, clamp to valid range
        var last = maxIndex();
        sliderInstances[sliderId].virtualIndex = Math.max(0, Math.min(index, last));
      }
      
      render(animate);
    }

    function startAutoplay() {
      if (!autoplay || sliderInstances[sliderId].timer) return;

      sliderInstances[sliderId].timer = setInterval(function () {
        if (sliderInstances[sliderId].isAnimating) return;

        if (loopEnabled) {
          // For infinite loop - always move forward by 1
          // We NEVER reset, just keep moving forward
          goTo(sliderInstances[sliderId].virtualIndex + 1, true);
        } else {
          // For non-loop mode
          var last = maxIndex();
          
          if (sliderInstances[sliderId].virtualIndex >= last) {
            // At the end, reset to beginning
            goTo(0, true);
          } else {
            // Normal step forward
            goTo(sliderInstances[sliderId].virtualIndex + 1, true);
          }
        }
      }, autoplaySpeed);
    }

    function stopAutoplay() {
      if (!sliderInstances[sliderId].timer) return;
      clearInterval(sliderInstances[sliderId].timer);
      sliderInstances[sliderId].timer = null;
    }

    function initDots() {
      if (!$pagination || !$pagination.length) return;
      $pagination.empty();
      
      var dotCount = loopEnabled ? totalRealSlides : (maxIndex() + 1);
      
      for (var i = 0; i < dotCount; i++) {
        $pagination.append('<span class="sa-dot"></span>');
      }
      updateDots();
    }

    function updateDots() {
      if (!$pagination || !$pagination.length) return;
      
      var dotIndex;
      if (loopEnabled) {
        // Calculate which real slide should be active
        var currentIndex = sliderInstances[sliderId].virtualIndex;
        dotIndex = currentIndex % totalRealSlides;
      } else {
        dotIndex = Math.max(0, Math.min(sliderInstances[sliderId].virtualIndex, totalRealSlides - 1));
      }
      
      $pagination.find('.sa-dot').removeClass('active').eq(dotIndex).addClass('active');
    }

    // Navigation
    function handlePrevClick(e) {
      if (e) e.preventDefault();
      stopAutoplay();
      
      if (loopEnabled) {
        // For infinite loop, we can go backward indefinitely
        goTo(sliderInstances[sliderId].virtualIndex - 1, true);
      } else {
        goTo(sliderInstances[sliderId].virtualIndex - 1, true);
      }
      
      startAutoplay();
    }

    function handleNextClick(e) {
      if (e) e.preventDefault();
      stopAutoplay();
      goTo(sliderInstances[sliderId].virtualIndex + 1, true);
      startAutoplay();
    }

    $prev.off('click.' + sliderId).on('click.' + sliderId, handlePrevClick);
    $next.off('click.' + sliderId).on('click.' + sliderId, handleNextClick);

    if ($pagination && $pagination.length) {
      $pagination.off('click.' + sliderId).on('click.' + sliderId, '.sa-dot', function () {
        stopAutoplay();
        
        var dotIndex = $(this).index();
        if (loopEnabled) {
          // For infinite loop, find the nearest occurrence of this slide
          var currentIndex = sliderInstances[sliderId].virtualIndex;
          var currentDot = currentIndex % totalRealSlides;
          var diff = dotIndex - currentDot;
          
          // Always move forward in infinite mode for simplicity
          if (diff < 0) diff += totalRealSlides;
          
          goTo(currentIndex + diff, true);
        } else {
          goTo(dotIndex, true);
        }
        
        startAutoplay();
      });
    }

    if (pauseOnHover) {
      $slider.off('mouseenter.' + sliderId + ' mouseleave.' + sliderId)
        .on('mouseenter.' + sliderId, stopAutoplay)
        .on('mouseleave.' + sliderId, startAutoplay);
    }

    function handleResize() {
      if (sliderInstances[sliderId].resizeTimer) {
        clearTimeout(sliderInstances[sliderId].resizeTimer);
      }
      
      sliderInstances[sliderId].resizeTimer = setTimeout(function () {
        var wasRunning = !!sliderInstances[sliderId].timer;
        stopAutoplay();
        applySizes();
        initDots();
        if (wasRunning) startAutoplay();
      }, 200);
    }

    $(window).off('resize.' + sliderId).on('resize.' + sliderId, handleResize);

    // Cleanup function
    function destroySlider() {
      stopAutoplay();
      clearTimeout(sliderInstances[sliderId].animationTimer);
      clearTimeout(sliderInstances[sliderId].resizeTimer);
      
      $wrapper.off('transitionend.' + sliderId);
      $prev.off('click.' + sliderId);
      $next.off('click.' + sliderId);
      $pagination.off('click.' + sliderId);
      $slider.off('mouseenter.' + sliderId + ' mouseleave.' + sliderId);
      $(window).off('resize.' + sliderId);
      
      // Remove cloned slides if they exist
      if (loopEnabled && sliderInstances[sliderId].clonesAdded) {
        $wrapper.find('.sa-clone').remove();
      }
      
      delete sliderInstances[sliderId];
    }

    // Store destroy method
    $slider.data('destroySlider', destroySlider);

    // Initialize
    applySizes();
    initDots();
    startAutoplay();
  }

  // Helper to prevent duplicate initialization
  var initializedSliders = new WeakSet();

  // Elementor hooks
  elementorFrontend.hooks.addAction('frontend/element_ready/testimonial_widget.default', function ($scope) {
    var $slider = $scope.find('.sa-testimonial-slider');
    if (!$slider.length || initializedSliders.has($slider[0])) return;
    
    initializedSliders.add($slider[0]);
    
    initCustomSliderAutoplayOnly(
      $slider,
      $slider.find('.sa-slider-wrapper'),
      $slider.find('.sa-slide'),
      $slider.find('.sa-prev'),
      $slider.find('.sa-next'),
      $slider.find('.sa-pagination'),
      {
        slidesPerView: $slider.data('slides-per-view'),
        autoplay: $slider.data('autoplay'),
        autoplaySpeed: $slider.data('autoplay-speed'),
        pauseOnHover: $slider.data('pause-on-hover'),
        loop: $slider.data('loop')
      }
    );
  });

  elementorFrontend.hooks.addAction('frontend/element_ready/sa-team-member.default', function ($scope) {
    var $slider = $scope.find('.sa-team-slider');
    if (!$slider.length || initializedSliders.has($slider[0])) return;
    
    initializedSliders.add($slider[0]);
    
    initCustomSliderAutoplayOnly(
      $slider,
      $slider.find('.sa-slider-wrapper'),
      $slider.find('.sa-slide'),
      $slider.find('.sa-prev'),
      $slider.find('.sa-next'),
      $slider.find('.sa-pagination'),
      {
        slidesPerView: $slider.data('slides-per-view'),
        autoplay: $slider.data('autoplay'),
        autoplaySpeed: $slider.data('autoplay-speed'),
        pauseOnHover: $slider.data('pause-on-hover'),
        loop: $slider.data('loop')
      }
    );
  });

  // Cleanup on page unload
  $(window).on('beforeunload', function() {
    Object.keys(sliderInstances).forEach(function(sliderId) {
      var $slider = $('[data-destroy-slider]');
      if ($slider.length) {
        var destroyFn = $slider.data('destroySlider');
        if (destroyFn && typeof destroyFn === 'function') {
          destroyFn();
        }
      }
    });
  });
});