<?php
/**
 * Helper Functions
 *
 * @package SwiftOffload
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get plugin settings
 *
 * @param string $key Setting key
 * @param mixed  $default Default value
 * @return mixed
 */
function swift_offload_get_setting( $key, $default = null ) {
	$settings = get_option( 'swift_offload_settings', array() );
	return isset( $settings[ $key ] ) ? $settings[ $key ] : $default;
}

/**
 * Update plugin setting
 *
 * @param string $key Setting key
 * @param mixed  $value Setting value
 * @return bool
 */
function swift_offload_update_setting( $key, $value ) {
	$settings         = get_option( 'swift_offload_settings', array() );
	$settings[ $key ] = $value;
	return update_option( 'swift_offload_settings', $settings );
}

/**
 * Get storage configuration
 *
 * @param string $key Config key
 * @param mixed  $default Default value
 * @return mixed
 */
function swift_offload_get_storage_config( $key = null, $default = null ) {
	$config = get_option( 'swift_offload_storage_config', array() );

	if ( null === $key ) {
		return $config;
	}

	return isset( $config[ $key ] ) ? $config[ $key ] : $default;
}

/**
 * Get CDN configuration
 *
 * @param string $key Config key
 * @param mixed  $default Default value
 * @return mixed
 */
function swift_offload_get_cdn_config( $key = null, $default = null ) {
	$config = get_option( 'swift_offload_cdn_config', array() );

	if ( null === $key ) {
		return $config;
	}

	return isset( $config[ $key ] ) ? $config[ $key ] : $default;
}

/**
 * Check if attachment is offloaded
 *
 * @param int $attachment_id Attachment ID
 * @return bool
 */
function swift_offload_is_attachment_offloaded( $attachment_id ) {
	global $wpdb;

	$table = $wpdb->prefix . 'swift_offload_items';
	$count = $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COUNT(*) FROM {$table} WHERE attachment_id = %d",
			$attachment_id
		)
	);

	return $count > 0;
}

/**
 * Get attachment S3 data
 *
 * @param int $attachment_id Attachment ID
 * @return array|null
 */
function swift_offload_get_attachment_s3_data( $attachment_id ) {
	global $wpdb;

	$table = $wpdb->prefix . 'swift_offload_items';

	return $wpdb->get_row(
		$wpdb->prepare(
			"SELECT * FROM {$table} WHERE attachment_id = %d",
			$attachment_id
		),
		ARRAY_A
	);
}

/**
 * Generate object key for attachment
 *
 * @param int    $attachment_id Attachment ID
 * @param string $filename Original filename
 * @return string
 */
function swift_offload_generate_object_key( $attachment_id, $filename ) {
	$format      = swift_offload_get_setting( 'object_key_format', '{year}/{month}/{filename}' );
	$upload_date = get_the_date( 'Y-m-d H:i:s', $attachment_id );
	$date        = new \DateTime( $upload_date );

	$replacements = array(
		'{year}'          => $date->format( 'Y' ),
		'{month}'         => $date->format( 'm' ),
		'{day}'           => $date->format( 'd' ),
		'{hour}'          => $date->format( 'H' ),
		'{minute}'        => $date->format( 'i' ),
		'{filename}'      => $filename,
		'{attachment_id}' => $attachment_id,
		'{user_id}'       => get_current_user_id(),
	);

	$object_key = str_replace( array_keys( $replacements ), array_values( $replacements ), $format );

	// Ensure no double slashes and clean path
	$object_key = trim( preg_replace( '/\/+/', '/', $object_key ), '/' );

	return apply_filters( 'swift_offload_object_key', $object_key, $attachment_id, $filename );
}

/**
 * Get S3 URL for attachment
 *
 * @param int    $attachment_id Attachment ID
 * @param string $size Image size
 * @return string|false
 */
function swift_offload_get_attachment_s3_url( $attachment_id, $size = 'full' ) {
	$s3_data = swift_offload_get_attachment_s3_data( $attachment_id );

	if ( ! $s3_data ) {
		return false;
	}

	$storage_config = swift_offload_get_storage_config();
	$cdn_config     = swift_offload_get_cdn_config();

	// Use CDN domain if enabled
	if ( $cdn_config['enabled'] && ! empty( $cdn_config['distribution_domain'] ) ) {
		$base_url = 'https://' . $cdn_config['distribution_domain'];
	} else {
		// Use S3 URL
		$endpoint = ! empty( $storage_config['endpoint'] ) ? $storage_config['endpoint'] : 's3.amazonaws.com';
		$base_url = 'https://' . $s3_data['bucket'] . '.' . $endpoint;
	}

	$object_key = $s3_data['object_key'];

	// Handle image sizes
	if ( $size !== 'full' && ! empty( $s3_data['size_data'] ) ) {
		$size_data = maybe_unserialize( $s3_data['size_data'] );
		if ( isset( $size_data[ $size ] ) ) {
			$object_key = $size_data[ $size ]['object_key'];
		}
	}

	$url = $base_url . '/' . $object_key;

	return apply_filters( 'swift_offload_attachment_url', $url, $attachment_id, $size );
}

/**
 * Format file size
 *
 * @param int $bytes File size in bytes
 * @return string
 */
function swift_offload_format_file_size( $bytes ) {
	$units = array( 'B', 'KB', 'MB', 'GB', 'TB' );

	for ( $i = 0; $bytes > 1024; $i++ ) {
		$bytes /= 1024;
	}

	return round( $bytes, 2 ) . ' ' . $units[ $i ];
}

/**
 * Check if current user can manage Swift Offload
 *
 * @return bool
 */
function swift_offload_current_user_can_manage() {
	return current_user_can( 'manage_options' );
}

/**
 * Sanitize S3 bucket name
 *
 * @param string $bucket_name Bucket name
 * @return string
 */
function swift_offload_sanitize_bucket_name( $bucket_name ) {
	// S3 bucket naming rules
	$bucket_name = strtolower( $bucket_name );
	$bucket_name = preg_replace( '/[^a-z0-9\-\.]/', '', $bucket_name );
	$bucket_name = trim( $bucket_name, '-.' );

	return $bucket_name;
}

/**
 * Get plugin instance
 *
 * @return SwiftOffload\Plugin|null
 */
function swift_offload() {
	return \SwiftOffload\Plugin::get_instance();
}

/**
 * Log message
 *
 * @param string $message Log message
 * @param string $level Log level
 * @param array  $context Additional context
 */
function swift_offload_log( $message, $level = 'info', $context = array() ) {
	if ( class_exists( 'SwiftOffload\\Utils\\Logger' ) ) {
		\SwiftOffload\Utils\Logger::log( $message, $level, $context );
	}
}

/**
 * Check if WordPress is in debug mode
 *
 * @return bool
 */
function swift_offload_is_debug_mode() {
	return defined( 'WP_DEBUG' ) && WP_DEBUG;
}

/**
 * Get mime type groups for filtering
 *
 * @return array
 */
function swift_offload_get_mime_type_groups() {
	return array(
		'images'    => array(
			'image/jpeg',
			'image/png',
			'image/gif',
			'image/webp',
			'image/svg+xml',
		),
		'videos'    => array(
			'video/mp4',
			'video/mpeg',
			'video/quicktime',
			'video/webm',
			'video/ogg',
		),
		'audio'     => array(
			'audio/mpeg',
			'audio/wav',
			'audio/ogg',
			'audio/mp4',
		),
		'documents' => array(
			'application/pdf',
			'application/msword',
			'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
			'application/vnd.ms-excel',
			'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
		),
	);
}
