<?php
/**
 * Validators Class
 *
 * @package SwiftOffload\Utils
 */

namespace SwiftOffload\Utils;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Validators class provides validation utilities
 */
class Validators {

	/**
	 * Validate S3 bucket name
	 *
	 * @param string $bucket_name Bucket name to validate
	 * @return array
	 */
	public static function validate_bucket_name( $bucket_name ) {
		$errors = array();

		// Basic checks
		if ( empty( $bucket_name ) ) {
			$errors[] = __( 'Bucket name is required', 'swift-offload' );
			return array(
				'valid'  => false,
				'errors' => $errors,
			);
		}

		// Length check
		if ( strlen( $bucket_name ) < 3 || strlen( $bucket_name ) > 63 ) {
			$errors[] = __( 'Bucket name must be between 3 and 63 characters', 'swift-offload' );
		}

		// Character check
		if ( ! preg_match( '/^[a-z0-9\-\.]+$/', $bucket_name ) ) {
			$errors[] = __( 'Bucket name can only contain lowercase letters, numbers, hyphens, and periods', 'swift-offload' );
		}

		// Must start and end with letter or number
		if ( ! preg_match( '/^[a-z0-9].*[a-z0-9]$/', $bucket_name ) ) {
			$errors[] = __( 'Bucket name must start and end with a letter or number', 'swift-offload' );
		}

		// Cannot look like IP address
		if ( preg_match( '/^\d+\.\d+\.\d+\.\d+$/', $bucket_name ) ) {
			$errors[] = __( 'Bucket name cannot be formatted as an IP address', 'swift-offload' );
		}

		// Cannot contain consecutive periods
		if ( strpos( $bucket_name, '..' ) !== false ) {
			$errors[] = __( 'Bucket name cannot contain consecutive periods', 'swift-offload' );
		}

		// Cannot contain period next to hyphen
		if ( preg_match( '/\.-|-\./', $bucket_name ) ) {
			$errors[] = __( 'Bucket name cannot contain periods next to hyphens', 'swift-offload' );
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}

	/**
	 * Validate AWS region
	 *
	 * @param string $region Region to validate
	 * @return array
	 */
	public static function validate_aws_region( $region ) {
		$valid_regions = array(
			'us-east-1',
			'us-east-2',
			'us-west-1',
			'us-west-2',
			'eu-west-1',
			'eu-west-2',
			'eu-west-3',
			'eu-central-1',
			'eu-north-1',
			'ap-southeast-1',
			'ap-southeast-2',
			'ap-northeast-1',
			'ap-northeast-2',
			'ap-south-1',
			'ca-central-1',
			'sa-east-1',
			'af-south-1',
			'ap-east-1',
			'ap-southeast-3',
			'me-south-1',
		);

		$errors = array();

		if ( empty( $region ) ) {
			$errors[] = __( 'AWS region is required', 'swift-offload' );
		} elseif ( ! in_array( $region, $valid_regions, true ) ) {
			$errors[] = __( 'Invalid AWS region', 'swift-offload' );
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}

	/**
	 * Validate AWS access key
	 *
	 * @param string $access_key Access key to validate
	 * @return array
	 */
	public static function validate_access_key( $access_key ) {
		$errors = array();

		if ( empty( $access_key ) ) {
			$errors[] = __( 'AWS Access Key ID is required', 'swift-offload' );
		} elseif ( ! preg_match( '/^[A-Z0-9]{16,128}$/', $access_key ) ) {
			$errors[] = __( 'AWS Access Key ID format is invalid', 'swift-offload' );
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}

	/**
	 * Validate AWS secret key
	 *
	 * @param string $secret_key Secret key to validate
	 * @return array
	 */
	public static function validate_secret_key( $secret_key ) {
		$errors = array();

		if ( empty( $secret_key ) ) {
			$errors[] = __( 'AWS Secret Access Key is required', 'swift-offload' );
		} elseif ( strlen( $secret_key ) < 32 ) {
			$errors[] = __( 'AWS Secret Access Key must be at least 32 characters', 'swift-offload' );
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}

	/**
	 * Validate object key format
	 *
	 * @param string $format Object key format template
	 * @return array
	 */
	public static function validate_object_key_format( $format ) {
		$errors = array();

		if ( empty( $format ) ) {
			$errors[] = __( 'Object key format is required', 'swift-offload' );
			return array(
				'valid'  => false,
				'errors' => $errors,
			);
		}

		// Check for {filename} placeholder
		if ( strpos( $format, '{filename}' ) === false ) {
			$errors[] = __( 'Object key format must include {filename} placeholder', 'swift-offload' );
		}

		// Check for invalid characters
		if ( preg_match( '/[<>:"|?*\\\\]/', $format ) ) {
			$errors[] = __( 'Object key format contains invalid characters', 'swift-offload' );
		}

		// Check for valid placeholders
		$valid_placeholders = array( '{year}', '{month}', '{day}', '{hour}', '{minute}', '{filename}', '{attachment_id}', '{user_id}' );
		$used_placeholders  = array();
		preg_match_all( '/\{[^}]+\}/', $format, $matches );

		foreach ( $matches[0] as $placeholder ) {
			if ( ! in_array( $placeholder, $valid_placeholders, true ) ) {
				$errors[] = sprintf( __( 'Invalid placeholder: %s', 'swift-offload' ), $placeholder );
			} else {
				$used_placeholders[] = $placeholder;
			}
		}

		return array(
			'valid'        => empty( $errors ),
			'errors'       => $errors,
			'placeholders' => $used_placeholders,
		);
	}

	/**
	 * Validate CDN domain
	 *
	 * @param string $domain Domain to validate
	 * @return array
	 */
	public static function validate_cdn_domain( $domain ) {
		$errors = array();

		if ( empty( $domain ) ) {
			return array(
				'valid'  => true,
				'errors' => array(),
			); // Optional field
		}

		// Remove protocol if present
		$domain = preg_replace( '/^https?:\/\//', '', $domain );
		$domain = rtrim( $domain, '/' );

		// Basic domain validation
		if ( ! filter_var( 'https://' . $domain, FILTER_VALIDATE_URL ) ) {
			$errors[] = __( 'Invalid domain format', 'swift-offload' );
		}

		// Check if it's a valid CloudFront domain
		if ( strpos( $domain, '.cloudfront.net' ) !== false ) {
			if ( ! preg_match( '/^[a-z0-9\-]+\.cloudfront\.net$/', $domain ) ) {
				$errors[] = __( 'Invalid CloudFront domain format', 'swift-offload' );
			}
		}

		return array(
			'valid'      => empty( $errors ),
			'errors'     => $errors,
			'normalized' => $domain,
		);
	}

	/**
	 * Validate file path for security
	 *
	 * @param string $file_path File path to validate
	 * @return array
	 */
	public static function validate_file_path( $file_path ) {
		$errors = array();

		if ( empty( $file_path ) ) {
			$errors[] = __( 'File path is required', 'swift-offload' );
			return array(
				'valid'  => false,
				'errors' => $errors,
			);
		}

		// Check for directory traversal attempts
		if ( strpos( $file_path, '..' ) !== false ) {
			$errors[] = __( 'File path contains directory traversal attempts', 'swift-offload' );
		}

		// Check for absolute paths (should be relative to uploads)
		if ( strpos( $file_path, '/' ) === 0 || preg_match( '/^[a-zA-Z]:/', $file_path ) ) {
			$errors[] = __( 'File path must be relative to uploads directory', 'swift-offload' );
		}

		// Check for null bytes
		if ( strpos( $file_path, "\0" ) !== false ) {
			$errors[] = __( 'File path contains null bytes', 'swift-offload' );
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}

	/**
	 * Validate attachment ID
	 *
	 * @param mixed $attachment_id Attachment ID to validate
	 * @return array
	 */
	public static function validate_attachment_id( $attachment_id ) {
		$errors = array();

		if ( ! is_numeric( $attachment_id ) || $attachment_id <= 0 ) {
			$errors[] = __( 'Invalid attachment ID', 'swift-offload' );
			return array(
				'valid'  => false,
				'errors' => $errors,
			);
		}

		$attachment_id = (int) $attachment_id;

		// Check if attachment exists
		$post = get_post( $attachment_id );
		if ( ! $post || $post->post_type !== 'attachment' ) {
			$errors[] = __( 'Attachment does not exist', 'swift-offload' );
		}

		return array(
			'valid'         => empty( $errors ),
			'errors'        => $errors,
			'attachment_id' => $attachment_id,
		);
	}

	/**
	 * Validate job data
	 *
	 * @param array $job_data Job data to validate
	 * @return array
	 */
	public static function validate_job_data( $job_data ) {
		$errors = array();

		if ( ! is_array( $job_data ) ) {
			$errors[] = __( 'Job data must be an array', 'swift-offload' );
			return array(
				'valid'  => false,
				'errors' => $errors,
			);
		}

		// Validate attachment IDs if present
		if ( isset( $job_data['attachment_ids'] ) ) {
			if ( ! is_array( $job_data['attachment_ids'] ) ) {
				$errors[] = __( 'Attachment IDs must be an array', 'swift-offload' );
			} else {
				foreach ( $job_data['attachment_ids'] as $id ) {
					$validation = self::validate_attachment_id( $id );
					if ( ! $validation['valid'] ) {
						$errors[] = sprintf( __( 'Invalid attachment ID: %s', 'swift-offload' ), $id );
					}
				}
			}
		}

		// Validate batch size
		if ( isset( $job_data['batch_size'] ) ) {
			$batch_size = (int) $job_data['batch_size'];
			if ( $batch_size < 1 || $batch_size > 100 ) {
				$errors[] = __( 'Batch size must be between 1 and 100', 'swift-offload' );
			}
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}

	/**
	 * Sanitize storage configuration
	 *
	 * @param array $config Configuration to sanitize
	 * @return array
	 */
	public static function sanitize_storage_config( $config ) {
		$sanitized = array();

		// Sanitize each field
		$sanitized['provider']       = sanitize_text_field( $config['provider'] ?? 's3' );
		$sanitized['access_key']     = sanitize_text_field( $config['access_key'] ?? '' );
		$sanitized['secret_key']     = sanitize_text_field( $config['secret_key'] ?? '' );
		$sanitized['region']         = sanitize_text_field( $config['region'] ?? 'us-east-1' );
		$sanitized['bucket']         = swift_offload_sanitize_bucket_name( $config['bucket'] ?? '' );
		$sanitized['endpoint']       = esc_url_raw( $config['endpoint'] ?? '' );
		$sanitized['use_path_style'] = (bool) ( $config['use_path_style'] ?? false );

		return $sanitized;
	}

	/**
	 * Sanitize CDN configuration
	 *
	 * @param array $config Configuration to sanitize
	 * @return array
	 */
	public static function sanitize_cdn_config( $config ) {
		$sanitized = array();

		$sanitized['enabled']             = (bool) ( $config['enabled'] ?? false );
		$sanitized['distribution_domain'] = sanitize_text_field( $config['distribution_domain'] ?? '' );
		$sanitized['custom_domain']       = sanitize_text_field( $config['custom_domain'] ?? '' );
		$sanitized['signed_urls']         = (bool) ( $config['signed_urls'] ?? false );
		$sanitized['query_string_auth']   = (bool) ( $config['query_string_auth'] ?? false );
		$sanitized['private_key_path']    = sanitize_text_field( $config['private_key_path'] ?? '' );
		$sanitized['key_pair_id']         = sanitize_text_field( $config['key_pair_id'] ?? '' );

		return $sanitized;
	}

	/**
	 * Validate and sanitize general settings
	 *
	 * @param array $settings Settings to validate
	 * @return array
	 */
	public static function validate_general_settings( $settings ) {
		$sanitized = array();
		$errors    = array();

		// Offload on upload
		$sanitized['offload_on_upload'] = (bool) ( $settings['offload_on_upload'] ?? true );

		// Remove local files
		$sanitized['remove_local_files'] = (bool) ( $settings['remove_local_files'] ?? false );

		// Object key format
		$object_key_format = $settings['object_key_format'] ?? '{year}/{month}/{filename}';
		$format_validation = self::validate_object_key_format( $object_key_format );

		if ( $format_validation['valid'] ) {
			$sanitized['object_key_format'] = $object_key_format;
		} else {
			$errors                         = array_merge( $errors, $format_validation['errors'] );
			$sanitized['object_key_format'] = '{year}/{month}/{filename}'; // Default
		}

		// Storage class
		$valid_storage_classes = array( 'STANDARD', 'REDUCED_REDUNDANCY', 'STANDARD_IA', 'ONEZONE_IA', 'INTELLIGENT_TIERING', 'GLACIER', 'DEEP_ARCHIVE' );
		$storage_class         = $settings['storage_class'] ?? 'STANDARD';

		if ( in_array( $storage_class, $valid_storage_classes, true ) ) {
			$sanitized['storage_class'] = $storage_class;
		} else {
			$errors[]                   = __( 'Invalid storage class', 'swift-offload' );
			$sanitized['storage_class'] = 'STANDARD';
		}

		// Object ACL
		$valid_acls = array( 'private', 'public-read', 'public-read-write', 'authenticated-read' );
		$object_acl = $settings['object_acl'] ?? 'private';

		if ( in_array( $object_acl, $valid_acls, true ) ) {
			$sanitized['object_acl'] = $object_acl;
		} else {
			$errors[]                = __( 'Invalid object ACL', 'swift-offload' );
			$sanitized['object_acl'] = 'private';
		}

		// File size limits
		if ( isset( $settings['min_file_size'] ) ) {
			$sanitized['min_file_size'] = max( 0, (int) $settings['min_file_size'] );
		}

		if ( isset( $settings['max_file_size'] ) ) {
			$sanitized['max_file_size'] = max( 0, (int) $settings['max_file_size'] );
		}

		// Allowed mime types
		if ( isset( $settings['allowed_mime_types'] ) && is_array( $settings['allowed_mime_types'] ) ) {
			$sanitized['allowed_mime_types'] = array_map( 'sanitize_mime_type', $settings['allowed_mime_types'] );
		}

		return array(
			'valid'     => empty( $errors ),
			'errors'    => $errors,
			'sanitized' => $sanitized,
		);
	}

	/**
	 * Check if current request is valid
	 *
	 * @param string $action Action being performed
	 * @param string $nonce_action Nonce action name
	 * @return bool
	 */
	public static function verify_request( $action = '', $nonce_action = 'swift_offload_nonce' ) {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		// Check nonce for admin requests
		if ( is_admin() && ! empty( $_REQUEST['_wpnonce'] ) ) {
			if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), $nonce_action ) ) {
				return false;
			}
		}

		// Check referrer
		if ( ! empty( $_SERVER['HTTP_REFERER'] ) ) {
			$referrer_host = wp_parse_url( sanitize_url( wp_unslash( $_SERVER['HTTP_REFERER'] ) ), PHP_URL_HOST );
			$site_host     = wp_parse_url( home_url(), PHP_URL_HOST );

			if ( $referrer_host !== $site_host ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Rate limit check
	 *
	 * @param string $action Action being performed
	 * @param int    $limit Number of requests per minute
	 * @return bool
	 */
	public static function check_rate_limit( $action, $limit = 60 ) {
		$user_id = get_current_user_id();
		$key     = "swift_offload_rate_limit_{$action}_{$user_id}";

		$count = get_transient( $key );

		if ( $count === false ) {
			set_transient( $key, 1, MINUTE_IN_SECONDS );
			return true;
		}

		if ( $count >= $limit ) {
			return false;
		}

		set_transient( $key, $count + 1, MINUTE_IN_SECONDS );
		return true;
	}

	/**
	 * Sanitize file upload data
	 *
	 * @param array $upload_data Upload data
	 * @return array
	 */
	public static function sanitize_upload_data( $upload_data ) {
		$sanitized = array();

		if ( isset( $upload_data['file'] ) ) {
			$file_validation   = self::validate_file_path( $upload_data['file'] );
			$sanitized['file'] = $file_validation['valid'] ? $upload_data['file'] : '';
		}

		if ( isset( $upload_data['url'] ) ) {
			$sanitized['url'] = esc_url_raw( $upload_data['url'] );
		}

		if ( isset( $upload_data['type'] ) ) {
			$sanitized['type'] = sanitize_mime_type( $upload_data['type'] );
		}

		return $sanitized;
	}

	/**
	 * Check if IP address is allowed
	 *
	 * @param string $ip_address IP address to check
	 * @param array  $allowed_ips Array of allowed IP addresses/ranges
	 * @return bool
	 */
	public static function is_ip_allowed( $ip_address, $allowed_ips = array() ) {
		if ( empty( $allowed_ips ) ) {
			return true; // No restrictions
		}

		foreach ( $allowed_ips as $allowed_ip ) {
			if ( strpos( $allowed_ip, '/' ) !== false ) {
				// CIDR notation
				if ( self::ip_in_range( $ip_address, $allowed_ip ) ) {
					return true;
				}
			} else {
				// Exact match
				if ( $ip_address === $allowed_ip ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Check if IP is in CIDR range
	 *
	 * @param string $ip IP address
	 * @param string $range CIDR range
	 * @return bool
	 */
	private static function ip_in_range( $ip, $range ) {
		list($subnet, $bits) = explode( '/', $range );

		if ( $bits === null ) {
			$bits = 32;
		}

		$ip      = ip2long( $ip );
		$subnet  = ip2long( $subnet );
		$mask    = -1 << ( 32 - $bits );
		$subnet &= $mask;

		return ( $ip & $mask ) === $subnet;
	}
}
