<?php
/**
 * Logger Class
 *
 * @package SwiftOffload\Utils
 */

namespace SwiftOffload\Utils;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Logger class provides logging functionality
 */
class Logger {

	/**
	 * Log levels
	 */
	const EMERGENCY = 'emergency';
	const ALERT     = 'alert';
	const CRITICAL  = 'critical';
	const ERROR     = 'error';
	const WARNING   = 'warning';
	const NOTICE    = 'notice';
	const INFO      = 'info';
	const DEBUG     = 'debug';

	/**
	 * Log level priorities
	 *
	 * @var array
	 */
	private static $level_priorities = array(
		self::EMERGENCY => 8,
		self::ALERT     => 7,
		self::CRITICAL  => 6,
		self::ERROR     => 5,
		self::WARNING   => 4,
		self::NOTICE    => 3,
		self::INFO      => 2,
		self::DEBUG     => 1,
	);

	/**
	 * Maximum log file size (5MB)
	 */
	const MAX_LOG_SIZE = 5 * 1024 * 1024;

	/**
	 * Log a message
	 *
	 * @param string $message Log message
	 * @param string $level Log level
	 * @param array  $context Additional context
	 */
	public static function log( $message, $level = self::INFO, $context = array() ) {
		// Check if logging is enabled
		if ( ! self::is_logging_enabled() ) {
			return;
		}

		// Check log level
		$min_level = self::get_min_log_level();
		if ( self::$level_priorities[ $level ] < self::$level_priorities[ $min_level ] ) {
			return;
		}

		// Prepare log entry
		$log_entry = self::format_log_entry( $message, $level, $context );

		// Write to log file
		self::write_to_log_file( $log_entry );

		// Also log to WordPress debug log if WP_DEBUG is enabled
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( "[SwiftOffload] [{$level}] {$message}" );
		}
	}

	/**
	 * Log emergency message
	 *
	 * @param string $message Log message
	 * @param array  $context Additional context
	 */
	public static function emergency( $message, $context = array() ) {
		self::log( $message, self::EMERGENCY, $context );
	}

	/**
	 * Log alert message
	 *
	 * @param string $message Log message
	 * @param array  $context Additional context
	 */
	public static function alert( $message, $context = array() ) {
		self::log( $message, self::ALERT, $context );
	}

	/**
	 * Log critical message
	 *
	 * @param string $message Log message
	 * @param array  $context Additional context
	 */
	public static function critical( $message, $context = array() ) {
		self::log( $message, self::CRITICAL, $context );
	}

	/**
	 * Log error message
	 *
	 * @param string $message Log message
	 * @param array  $context Additional context
	 */
	public static function error( $message, $context = array() ) {
		self::log( $message, self::ERROR, $context );
	}

	/**
	 * Log warning message
	 *
	 * @param string $message Log message
	 * @param array  $context Additional context
	 */
	public static function warning( $message, $context = array() ) {
		self::log( $message, self::WARNING, $context );
	}

	/**
	 * Log notice message
	 *
	 * @param string $message Log message
	 * @param array  $context Additional context
	 */
	public static function notice( $message, $context = array() ) {
		self::log( $message, self::NOTICE, $context );
	}

	/**
	 * Log info message
	 *
	 * @param string $message Log message
	 * @param array  $context Additional context
	 */
	public static function info( $message, $context = array() ) {
		self::log( $message, self::INFO, $context );
	}

	/**
	 * Log debug message
	 *
	 * @param string $message Log message
	 * @param array  $context Additional context
	 */
	public static function debug( $message, $context = array() ) {
		self::log( $message, self::DEBUG, $context );
	}

	/**
	 * Get log file path
	 *
	 * @return string
	 */
	public static function get_log_file_path() {
		$upload_dir = wp_upload_dir();
		$log_dir    = $upload_dir['basedir'] . '/swift-offload-logs';

		// Create log directory if it doesn't exist
		if ( ! file_exists( $log_dir ) ) {
			wp_mkdir_p( $log_dir );

			// Add .htaccess to prevent direct access
			$htaccess_content = "Order deny,allow\nDeny from all\n";
			file_put_contents( $log_dir . '/.htaccess', $htaccess_content );
		}

		return $log_dir . '/swift-offload-' . date( 'Y-m-d' ) . '.log';
	}

	/**
	 * Check if logging is enabled
	 *
	 * @return bool
	 */
	private static function is_logging_enabled() {
		return swift_offload_get_setting( 'enable_logging', true );
	}

	/**
	 * Get minimum log level
	 *
	 * @return string
	 */
	private static function get_min_log_level() {
		return swift_offload_get_setting( 'log_level', self::INFO );
	}

	/**
	 * Format log entry
	 *
	 * @param string $message Log message
	 * @param string $level Log level
	 * @param array  $context Additional context
	 * @return string
	 */
	private static function format_log_entry( $message, $level, $context ) {
		$timestamp = current_time( 'Y-m-d H:i:s' );
		$level     = strtoupper( $level );

		// Add context if provided
		$context_str = '';
		if ( ! empty( $context ) ) {
			$context_str = ' | Context: ' . wp_json_encode( $context );
		}

		// Add memory usage
		$memory_usage = round( memory_get_usage() / 1024 / 1024, 2 );

		return "[{$timestamp}] [{$level}] {$message} | Memory: {$memory_usage}MB{$context_str}" . PHP_EOL;
	}

	/**
	 * Write to log file
	 *
	 * @param string $log_entry Log entry to write
	 */
	private static function write_to_log_file( $log_entry ) {
		$log_file = self::get_log_file_path();

		// Check file size and rotate if necessary
		if ( file_exists( $log_file ) && filesize( $log_file ) > self::MAX_LOG_SIZE ) {
			self::rotate_log_file( $log_file );
		}

		// Write to log file
		file_put_contents( $log_file, $log_entry, FILE_APPEND | LOCK_EX );
	}

	/**
	 * Rotate log file
	 *
	 * @param string $log_file Log file path
	 */
	private static function rotate_log_file( $log_file ) {
		$backup_file = $log_file . '.1';

		// Remove old backup if exists
		if ( file_exists( $backup_file ) ) {
			wp_delete_file( $backup_file );
		}

		// Move current log to backup.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Atomic file operation required for log rotation.
		rename( $log_file, $backup_file );
	}

	/**
	 * Get recent log entries
	 *
	 * @param int    $lines Number of lines to retrieve
	 * @param string $level Minimum log level
	 * @return array
	 */
	public static function get_recent_logs( $lines = 100, $level = self::DEBUG ) {
		$log_file = self::get_log_file_path();

		if ( ! file_exists( $log_file ) ) {
			return array();
		}

		// Get last N lines from log file
		$log_lines   = self::tail_file( $log_file, $lines );
		$parsed_logs = array();

		foreach ( $log_lines as $line ) {
			$parsed = self::parse_log_line( $line );
			if ( $parsed && self::$level_priorities[ $parsed['level'] ] >= self::$level_priorities[ $level ] ) {
				$parsed_logs[] = $parsed;
			}
		}

		return array_reverse( $parsed_logs ); // Most recent first
	}

	/**
	 * Parse log line
	 *
	 * @param string $line Log line
	 * @return array|null
	 */
	private static function parse_log_line( $line ) {
		if ( preg_match( '/^\[([\d\-\s:]+)\]\s+\[([^\]]+)\]\s+(.+)$/', $line, $matches ) ) {
			$message_parts = explode( ' | ', $matches[3] );
			$message       = $message_parts[0];

			return array(
				'timestamp' => $matches[1],
				'level'     => strtolower( $matches[2] ),
				'message'   => $message,
				'raw'       => $line,
			);
		}

		return null;
	}

	/**
	 * Get last N lines from file
	 *
	 * @param string $file File path
	 * @param int    $lines Number of lines
	 * @return array
	 */
	private static function tail_file( $file, $lines ) {
		$handle = fopen( $file, 'r' );
		if ( ! $handle ) {
			return array();
		}

		$line_array = array();

		while ( ! feof( $handle ) ) {
			$line = fgets( $handle );
			if ( $line !== false ) {
				$line_array[] = rtrim( $line );

				// Keep only the last N lines
				if ( count( $line_array ) > $lines ) {
					array_shift( $line_array );
				}
			}
		}

		fclose( $handle );
		return $line_array;
	}

	/**
	 * Clear log files
	 *
	 * @return bool
	 */
	public static function clear_logs() {
		$log_file = self::get_log_file_path();
		$log_dir  = dirname( $log_file );

		$cleared = 0;

		// Remove all log files in the directory
		$log_files = glob( $log_dir . '/swift-offload-*.log*' );
		foreach ( $log_files as $file ) {
			wp_delete_file( $file );
			if ( ! file_exists( $file ) ) {
				++$cleared;
			}
		}

		self::info( "Cleared {$cleared} log files" );

		return $cleared > 0;
	}

	/**
	 * Get log file size
	 *
	 * @return int
	 */
	public static function get_log_file_size() {
		$log_file = self::get_log_file_path();
		return file_exists( $log_file ) ? filesize( $log_file ) : 0;
	}

	/**
	 * Export logs as JSON
	 *
	 * @param int    $lines Number of lines to export
	 * @param string $level Minimum log level
	 * @return string
	 */
	public static function export_logs_json( $lines = 1000, $level = self::DEBUG ) {
		$logs = self::get_recent_logs( $lines, $level );

		return wp_json_encode(
			array(
				'exported_at'   => current_time( 'c' ),
				'total_entries' => count( $logs ),
				'min_level'     => $level,
				'logs'          => $logs,
			),
			JSON_PRETTY_PRINT
		);
	}
}
