<?php
/**
 * Storage Interfaces
 *
 * @package SwiftOffload\Storage
 */

namespace SwiftOffload\Storage;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Storage Provider Interface
 */
interface Storage_Interface {

	/**
	 * Initialize the storage provider
	 *
	 * @param array $config Configuration array
	 * @return bool
	 */
	public function init( $config );

	/**
	 * Test connection to storage provider
	 *
	 * @return Upload_Result
	 */
	public function test_connection();

	/**
	 * Upload file to storage
	 *
	 * @param string $local_path Local file path
	 * @param string $object_key Object key/path in storage
	 * @param array  $options Upload options
	 * @return Upload_Result
	 */
	public function upload_file( $local_path, $object_key, $options = array() );

	/**
	 * Delete object from storage
	 *
	 * @param string $object_key Object key to delete
	 * @return bool
	 */
	public function delete_object( $object_key );

	/**
	 * Check if object exists
	 *
	 * @param string $object_key Object key to check
	 * @return bool
	 */
	public function object_exists( $object_key );

	/**
	 * Get object URL
	 *
	 * @param string $object_key Object key
	 * @param array  $options URL options
	 * @return string
	 */
	public function get_object_url( $object_key, $options = array() );

	/**
	 * Get signed URL for private objects
	 *
	 * @param string $object_key Object key
	 * @param int    $expires Expiration time in seconds
	 * @param array  $options Additional options
	 * @return string
	 */
	public function get_signed_url( $object_key, $expires = 3600, $options = array() );

	/**
	 * List objects in bucket
	 *
	 * @param string $prefix Object key prefix
	 * @param int    $max_keys Maximum number of keys to return
	 * @return array
	 */
	public function list_objects( $prefix = '', $max_keys = 1000 );

	/**
	 * Get object metadata
	 *
	 * @param string $object_key Object key
	 * @return array|false
	 */
	public function get_object_metadata( $object_key );

	/**
	 * Copy object to new location
	 *
	 * @param string $source_key Source object key
	 * @param string $dest_key Destination object key
	 * @param array  $options Copy options
	 * @return bool
	 */
	public function copy_object( $source_key, $dest_key, $options = array() );
}

/**
 * Upload Result Class
 */
class Upload_Result {

	/**
	 * Success status
	 *
	 * @var bool
	 */
	public $success;

	/**
	 * Object key
	 *
	 * @var string
	 */
	public $object_key;

	/**
	 * Object URL
	 *
	 * @var string
	 */
	public $url;

	/**
	 * File size
	 *
	 * @var int
	 */
	public $size;

	/**
	 * ETag
	 *
	 * @var string
	 */
	public $etag;

	/**
	 * Error message
	 *
	 * @var string
	 */
	public $error;

	/**
	 * Additional metadata
	 *
	 * @var array
	 */
	public $metadata;

	/**
	 * Constructor
	 *
	 * @param bool  $success Success status
	 * @param array $data Result data
	 */
	public function __construct( $success = false, $data = array() ) {
		$this->success = $success;

		foreach ( $data as $key => $value ) {
			if ( property_exists( $this, $key ) ) {
				$this->$key = $value;
			}
		}
	}

	/**
	 * Check if upload was successful
	 *
	 * @return bool
	 */
	public function is_success() {
		return $this->success;
	}

	/**
	 * Get error message
	 *
	 * @return string
	 */
	public function get_error() {
		return $this->error ?: '';
	}

	/**
	 * Convert to array
	 *
	 * @return array
	 */
	public function to_array() {
		return array(
			'success'    => $this->success,
			'object_key' => $this->object_key,
			'url'        => $this->url,
			'size'       => $this->size,
			'etag'       => $this->etag,
			'error'      => $this->error,
			'metadata'   => $this->metadata,
		);
	}
}

/**
 * Signed URL Options Class
 */
class Signed_Url_Options {

	/**
	 * Expiration time in seconds
	 *
	 * @var int
	 */
	public $expires = 3600;

	/**
	 * HTTP method
	 *
	 * @var string
	 */
	public $method = 'GET';

	/**
	 * Content type
	 *
	 * @var string
	 */
	public $content_type;

	/**
	 * Custom headers
	 *
	 * @var array
	 */
	public $headers = array();

	/**
	 * Constructor
	 *
	 * @param array $options Options array
	 */
	public function __construct( $options = array() ) {
		foreach ( $options as $key => $value ) {
			if ( property_exists( $this, $key ) ) {
				$this->$key = $value;
			}
		}
	}
}
