<?php
/**
 * Storage Interface
 *
 * @package SwiftOffload\Storage
 */

namespace SwiftOffload\Storage;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Storage interface for cloud storage providers
 */
interface Storage_Interface {

	/**
	 * Upload a file to storage
	 *
	 * @param string $local_path Local file path
	 * @param string $object_key Remote object key
	 * @param array  $metadata   File metadata
	 * @return Storage_Result
	 */
	public function upload_file( $local_path, $object_key, $metadata = array() );

	/**
	 * Download a file from storage
	 *
	 * @param string $object_key Remote object key
	 * @param string $local_path Local file path
	 * @return Storage_Result
	 */
	public function download_file( $object_key, $local_path );

	/**
	 * Delete a file from storage
	 *
	 * @param string $object_key Remote object key
	 * @return Storage_Result
	 */
	public function delete_file( $object_key );

	/**
	 * Check if an object exists in storage
	 *
	 * @param string $object_key Remote object key
	 * @return bool
	 */
	public function object_exists( $object_key );

	/**
	 * Get object metadata
	 *
	 * @param string $object_key Remote object key
	 * @return Storage_Result
	 */
	public function get_object_metadata( $object_key );

	/**
	 * Generate a presigned URL for an object
	 *
	 * @param string $object_key Remote object key
	 * @param int    $expires    Expiration time in seconds
	 * @param string $method     HTTP method (GET, PUT, etc.)
	 * @return string|false
	 */
	public function get_presigned_url( $object_key, $expires = 3600, $method = 'GET' );

	/**
	 * Get the public URL for an object
	 *
	 * @param string $object_key Remote object key
	 * @return string
	 */
	public function get_object_url( $object_key );

	/**
	 * List objects in storage
	 *
	 * @param string $prefix Object key prefix
	 * @param int    $limit  Maximum number of objects to return
	 * @return Storage_Result
	 */
	public function list_objects( $prefix = '', $limit = 1000 );

	/**
	 * Copy an object within storage
	 *
	 * @param string $source_key      Source object key
	 * @param string $destination_key Destination object key
	 * @return Storage_Result
	 */
	public function copy_object( $source_key, $destination_key );

	/**
	 * Initiate multipart upload
	 *
	 * @param string $object_key Remote object key
	 * @param array  $metadata   File metadata
	 * @return Storage_Result
	 */
	public function initiate_multipart_upload( $object_key, $metadata = array() );

	/**
	 * Upload a part for multipart upload
	 *
	 * @param string $upload_id   Upload ID from initiate_multipart_upload
	 * @param string $object_key  Remote object key
	 * @param int    $part_number Part number (1-based)
	 * @param string $data        Part data
	 * @return Storage_Result
	 */
	public function upload_part( $upload_id, $object_key, $part_number, $data );

	/**
	 * Complete multipart upload
	 *
	 * @param string $upload_id Upload ID
	 * @param string $object_key Remote object key
	 * @param array  $parts     Array of part information
	 * @return Storage_Result
	 */
	public function complete_multipart_upload( $upload_id, $object_key, $parts );

	/**
	 * Abort multipart upload
	 *
	 * @param string $upload_id Upload ID
	 * @param string $object_key Remote object key
	 * @return Storage_Result
	 */
	public function abort_multipart_upload( $upload_id, $object_key );

	/**
	 * Test the storage connection
	 *
	 * @return Storage_Result
	 */
	public function test_connection();

	/**
	 * Get storage information (used space, available space, etc.)
	 *
	 * @return Storage_Result
	 */
	public function get_storage_info();

	/**
	 * Set object ACL
	 *
	 * @param string $object_key Remote object key
	 * @param string $acl        ACL setting
	 * @return Storage_Result
	 */
	public function set_object_acl( $object_key, $acl );

	/**
	 * Get object ACL
	 *
	 * @param string $object_key Remote object key
	 * @return Storage_Result
	 */
	public function get_object_acl( $object_key );
}
