<?php
/**
 * S3 Storage Provider
 *
 * @package SwiftOffload\Storage
 */

namespace SwiftOffload\Storage;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * S3 Storage Provider Class
 */
class Storage_S3 implements Storage_Interface {

	/**
	 * S3 client configuration
	 *
	 * @var array
	 */
	private $config = array();

	/**
	 * AWS credentials
	 *
	 * @var array
	 */
	private $credentials = array();

	/**
	 * Bucket name
	 *
	 * @var string
	 */
	private $bucket;

	/**
	 * Region
	 *
	 * @var string
	 */
	private $region;

	/**
	 * Custom endpoint
	 *
	 * @var string
	 */
	private $endpoint;

	/**
	 * Use path style URLs
	 *
	 * @var bool
	 */
	private $use_path_style = false;

	/**
	 * Initialize the storage provider
	 *
	 * @param array $config Configuration array
	 * @return bool
	 */
	public function init( $config ) {
		$this->config = $config;

		// Required fields
		$required = array( 'access_key', 'secret_key', 'region', 'bucket' );
		foreach ( $required as $field ) {
			if ( empty( $config[ $field ] ) ) {
				return false;
			}
		}

		$this->credentials = array(
			'key'    => $config['access_key'],
			'secret' => $config['secret_key'],
		);

		$this->bucket         = $config['bucket'];
		$this->region         = $config['region'];
		$this->endpoint       = $config['endpoint'] ?? '';
		$this->use_path_style = ! empty( $config['use_path_style'] );

		return true;
	}

	/**
	 * Test connection to S3
	 *
	 * @return Upload_Result
	 */
	public function test_connection() {
		try {
			$response = $this->make_s3_request(
				'HEAD',
				'/',
				array(
					'headers' => array(
						'Authorization' => $this->get_authorization_header( 'HEAD', '/' ),
						'Host'          => $this->get_host(),
						'X-Amz-Date'    => $this->get_amz_date(),
					),
				)
			);

			if ( wp_remote_retrieve_response_code( $response ) === 200 ) {
				return new Upload_Result(
					true,
					array(
						'message' => __( 'Connection successful', 'swift-offload' ),
					)
				);
			} else {
				return new Upload_Result(
					false,
					array(
						'error' => __( 'Connection failed: Invalid credentials or bucket access', 'swift-offload' ),
					)
				);
			}
		} catch ( \Exception $e ) {
			return new Upload_Result(
				false,
				array(
					'error' => sprintf( __( 'Connection failed: %s', 'swift-offload' ), $e->getMessage() ),
				)
			);
		}
	}

	/**
	 * Upload file to S3
	 *
	 * @param string $local_path Local file path
	 * @param string $object_key Object key/path in storage
	 * @param array  $options Upload options
	 * @return Upload_Result
	 */
	public function upload_file( $local_path, $object_key, $options = array() ) {
		if ( ! file_exists( $local_path ) ) {
			return new Upload_Result(
				false,
				array(
					'error' => __( 'Local file does not exist', 'swift-offload' ),
				)
			);
		}

		$file_size           = filesize( $local_path );
		$multipart_threshold = 100 * 1024 * 1024; // 100MB

		// Use multipart upload for large files
		if ( $file_size > $multipart_threshold ) {
			return $this->multipart_upload( $local_path, $object_key, $options );
		} else {
			return $this->simple_upload( $local_path, $object_key, $options );
		}
	}

	/**
	 * Simple upload for smaller files
	 *
	 * @param string $local_path Local file path
	 * @param string $object_key Object key
	 * @param array  $options Upload options
	 * @return Upload_Result
	 */
	private function simple_upload( $local_path, $object_key, $options ) {
		try {
			$file_content  = file_get_contents( $local_path );
			$content_type  = $options['content_type'] ?? $this->get_content_type( $local_path );
			$acl           = $options['acl'] ?? 'private';
			$storage_class = $options['storage_class'] ?? 'STANDARD';

			$headers = array(
				'Content-Type'        => $content_type,
				'Content-Length'      => strlen( $file_content ),
				'Host'                => $this->get_host(),
				'X-Amz-Date'          => $this->get_amz_date(),
				'x-amz-acl'           => $acl,
				'x-amz-storage-class' => $storage_class,
			);

			// Add content MD5 for integrity
			$headers['Content-MD5'] = base64_encode( md5( $file_content, true ) );

			// Add authorization header
			$headers['Authorization'] = $this->get_authorization_header( 'PUT', '/' . $object_key, $headers );

			$response = wp_remote_request(
				$this->get_url( $object_key ),
				array(
					'method'  => 'PUT',
					'headers' => $headers,
					'body'    => $file_content,
					'timeout' => 300,
				)
			);

			if ( is_wp_error( $response ) ) {
				return new Upload_Result(
					false,
					array(
						'error' => $response->get_error_message(),
					)
				);
			}

			$response_code = wp_remote_retrieve_response_code( $response );

			if ( $response_code === 200 || $response_code === 201 ) {
				$etag = wp_remote_retrieve_header( $response, 'etag' );
				$etag = trim( $etag, '"' );

				return new Upload_Result(
					true,
					array(
						'object_key' => $object_key,
						'url'        => $this->get_object_url( $object_key ),
						'size'       => filesize( $local_path ),
						'etag'       => $etag,
						'metadata'   => array(
							'content_type'  => $content_type,
							'acl'           => $acl,
							'storage_class' => $storage_class,
						),
					)
				);
			} else {
				$error_body = wp_remote_retrieve_body( $response );
				return new Upload_Result(
					false,
					array(
						'error' => sprintf( __( 'Upload failed with status %1$d: %2$s', 'swift-offload' ), $response_code, $error_body ),
					)
				);
			}
		} catch ( \Exception $e ) {
			return new Upload_Result(
				false,
				array(
					'error' => sprintf( __( 'Upload failed: %s', 'swift-offload' ), $e->getMessage() ),
				)
			);
		}
	}

	/**
	 * Multipart upload for large files
	 *
	 * @param string $local_path Local file path
	 * @param string $object_key Object key
	 * @param array  $options Upload options
	 * @return Upload_Result
	 */
	private function multipart_upload( $local_path, $object_key, $options ) {
		$chunk_size  = 5 * 1024 * 1024; // 5MB chunks
		$file_size   = filesize( $local_path );
		$total_parts = ceil( $file_size / $chunk_size );

		try {
			// Initiate multipart upload
			$upload_id = $this->initiate_multipart_upload( $object_key, $options );
			if ( ! $upload_id ) {
				return new Upload_Result(
					false,
					array(
						'error' => __( 'Failed to initiate multipart upload', 'swift-offload' ),
					)
				);
			}

			$parts       = array();
			$file_handle = fopen( $local_path, 'rb' );

			// Upload parts
			for ( $part_number = 1; $part_number <= $total_parts; $part_number++ ) {
				$chunk_data = fread( $file_handle, $chunk_size );

				$part_result = $this->upload_part( $upload_id, $object_key, $part_number, $chunk_data );
				if ( ! $part_result ) {
					fclose( $file_handle );
					$this->abort_multipart_upload( $upload_id, $object_key );
					return new Upload_Result(
						false,
						array(
							'error' => sprintf( __( 'Failed to upload part %d', 'swift-offload' ), $part_number ),
						)
					);
				}

				$parts[] = array(
					'PartNumber' => $part_number,
					'ETag'       => $part_result,
				);
			}

			fclose( $file_handle );

			// Complete multipart upload
			$etag = $this->complete_multipart_upload( $upload_id, $object_key, $parts );
			if ( ! $etag ) {
				return new Upload_Result(
					false,
					array(
						'error' => __( 'Failed to complete multipart upload', 'swift-offload' ),
					)
				);
			}

			return new Upload_Result(
				true,
				array(
					'object_key' => $object_key,
					'url'        => $this->get_object_url( $object_key ),
					'size'       => $file_size,
					'etag'       => $etag,
					'metadata'   => array(
						'multipart' => true,
						'parts'     => count( $parts ),
					),
				)
			);

		} catch ( \Exception $e ) {
			return new Upload_Result(
				false,
				array(
					'error' => sprintf( __( 'Multipart upload failed: %s', 'swift-offload' ), $e->getMessage() ),
				)
			);
		}
	}

	/**
	 * Initiate multipart upload
	 */
	public function initiate_multipart_upload( $object_key, $metadata = array() ) {
		$headers = array(
			'Host'       => $this->get_host(),
			'X-Amz-Date' => $this->get_amz_date(),
			'x-amz-acl'  => $metadata['acl'] ?? 'private',
		);

		$headers['Authorization'] = $this->get_authorization_header( 'POST', '/' . $object_key . '?uploads', $headers );

		$response = wp_remote_post(
			$this->get_url( $object_key ) . '?uploads',
			array(
				'headers' => $headers,
				'timeout' => 60,
			)
		);

		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
			return false;
		}

		$body = wp_remote_retrieve_body( $response );
		$xml  = simplexml_load_string( $body );

		return (string) $xml->UploadId;
	}

	/**
	 * Upload a single part
	 */
	public function upload_part( $upload_id, $object_key, $part_number, $data ) {
		$headers = array(
			'Content-Length' => strlen( $data ),
			'Content-MD5'    => base64_encode( md5( $data, true ) ),
			'Host'           => $this->get_host(),
			'X-Amz-Date'     => $this->get_amz_date(),
		);

		$query                    = "partNumber={$part_number}&uploadId={$upload_id}";
		$headers['Authorization'] = $this->get_authorization_header( 'PUT', '/' . $object_key . '?' . $query, $headers );

		$response = wp_remote_request(
			$this->get_url( $object_key ) . '?' . $query,
			array(
				'method'  => 'PUT',
				'headers' => $headers,
				'body'    => $data,
				'timeout' => 300,
			)
		);

		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
			return false;
		}

		return trim( wp_remote_retrieve_header( $response, 'etag' ), '"' );
	}

	/**
	 * Complete multipart upload
	 */
	public function complete_multipart_upload( $upload_id, $object_key, $parts ) {
		$xml_parts = '';
		foreach ( $parts as $part ) {
			$xml_parts .= "<Part><PartNumber>{$part['PartNumber']}</PartNumber><ETag>{$part['ETag']}</ETag></Part>";
		}

		$xml_body = "<CompleteMultipartUpload>{$xml_parts}</CompleteMultipartUpload>";

		$headers = array(
			'Content-Type'   => 'application/xml',
			'Content-Length' => strlen( $xml_body ),
			'Host'           => $this->get_host(),
			'X-Amz-Date'     => $this->get_amz_date(),
		);

		$query                    = "uploadId={$upload_id}";
		$headers['Authorization'] = $this->get_authorization_header( 'POST', '/' . $object_key . '?' . $query, $headers );

		$response = wp_remote_post(
			$this->get_url( $object_key ) . '?' . $query,
			array(
				'headers' => $headers,
				'body'    => $xml_body,
				'timeout' => 300,
			)
		);

		if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
			return false;
		}

		$body = wp_remote_retrieve_body( $response );
		$xml  = simplexml_load_string( $body );

		return trim( (string) $xml->ETag, '"' );
	}

	/**
	 * Abort multipart upload
	 */
	public function abort_multipart_upload( $upload_id, $object_key ) {
		$headers = array(
			'Host'       => $this->get_host(),
			'X-Amz-Date' => $this->get_amz_date(),
		);

		$query                    = "uploadId={$upload_id}";
		$headers['Authorization'] = $this->get_authorization_header( 'DELETE', '/' . $object_key . '?' . $query, $headers );

		wp_remote_request(
			$this->get_url( $object_key ) . '?' . $query,
			array(
				'method'  => 'DELETE',
				'headers' => $headers,
				'timeout' => 60,
			)
		);
	}

	/**
	 * Delete object from S3
	 *
	 * @param string $object_key Object key to delete
	 * @return bool
	 */
	public function delete_object( $object_key ) {
		try {
			$headers = array(
				'Host'       => $this->get_host(),
				'X-Amz-Date' => $this->get_amz_date(),
			);

			$headers['Authorization'] = $this->get_authorization_header( 'DELETE', '/' . $object_key, $headers );

			$response = wp_remote_request(
				$this->get_url( $object_key ),
				array(
					'method'  => 'DELETE',
					'headers' => $headers,
					'timeout' => 60,
				)
			);

			$response_code = wp_remote_retrieve_response_code( $response );
			return $response_code === 204 || $response_code === 200;

		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * Check if object exists
	 *
	 * @param string $object_key Object key to check
	 * @return bool
	 */
	public function object_exists( $object_key ) {
		try {
			$headers = array(
				'Host'       => $this->get_host(),
				'X-Amz-Date' => $this->get_amz_date(),
			);

			$headers['Authorization'] = $this->get_authorization_header( 'HEAD', '/' . $object_key, $headers );

			$response = wp_remote_head(
				$this->get_url( $object_key ),
				array(
					'headers' => $headers,
					'timeout' => 30,
				)
			);

			return wp_remote_retrieve_response_code( $response ) === 200;

		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * Get object URL
	 *
	 * @param string $object_key Object key
	 * @param array  $options URL options
	 * @return string
	 */
	public function get_object_url( $object_key, $options = array() ) {
		$cdn_config = swift_offload_get_cdn_config();

		// Use CDN domain if configured
		if ( ! empty( $cdn_config['enabled'] ) && ! empty( $cdn_config['distribution_domain'] ) ) {
			return 'https://' . $cdn_config['distribution_domain'] . '/' . $object_key;
		}

		// Use custom CDN domain if configured
		if ( ! empty( $cdn_config['custom_domain'] ) ) {
			return 'https://' . $cdn_config['custom_domain'] . '/' . $object_key;
		}

		// Use S3 URL
		return $this->get_url( $object_key );
	}

	/**
	 * Get signed URL for private objects
	 *
	 * @param string $object_key Object key
	 * @param int    $expires Expiration time in seconds
	 * @param array  $options Additional options
	 * @return string
	 */
	public function get_signed_url( $object_key, $expires = 3600, $options = array() ) {
		$expiration = time() + $expires;
		$method     = $options['method'] ?? 'GET';

		$string_to_sign = $method . "\n\n\n" . $expiration . "\n/" . $this->bucket . '/' . $object_key;
		$signature      = $this->sign_string( $string_to_sign );

		$query_params = array(
			'AWSAccessKeyId' => $this->credentials['key'],
			'Expires'        => $expiration,
			'Signature'      => $signature,
		);

		return $this->get_url( $object_key ) . '?' . http_build_query( $query_params );
	}

	/**
	 * List objects in bucket
	 *
	 * @param string $prefix Object key prefix
	 * @param int    $max_keys Maximum number of keys to return
	 * @return array
	 */
	public function list_objects( $prefix = '', $max_keys = 1000 ) {
		try {
			$query_params = array(
				'max-keys' => $max_keys,
			);

			if ( ! empty( $prefix ) ) {
				$query_params['prefix'] = $prefix;
			}

			$query_string = http_build_query( $query_params );
			$path         = '/?' . $query_string;

			$headers = array(
				'Host'       => $this->get_host(),
				'X-Amz-Date' => $this->get_amz_date(),
			);

			$headers['Authorization'] = $this->get_authorization_header( 'GET', $path, $headers );

			$response = wp_remote_get(
				$this->get_url( '' ) . '?' . $query_string,
				array(
					'headers' => $headers,
					'timeout' => 60,
				)
			);

			if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
				return array();
			}

			$body = wp_remote_retrieve_body( $response );
			$xml  = simplexml_load_string( $body );

			$objects = array();
			if ( isset( $xml->Contents ) ) {
				foreach ( $xml->Contents as $object ) {
					$objects[] = array(
						'key'           => (string) $object->Key,
						'size'          => (int) $object->Size,
						'last_modified' => (string) $object->LastModified,
						'etag'          => trim( (string) $object->ETag, '"' ),
					);
				}
			}

			return $objects;

		} catch ( \Exception $e ) {
			return array();
		}
	}

	/**
	 * Get object metadata
	 *
	 * @param string $object_key Object key
	 * @return array|false
	 */
	public function get_object_metadata( $object_key ) {
		try {
			$headers = array(
				'Host'       => $this->get_host(),
				'X-Amz-Date' => $this->get_amz_date(),
			);

			$headers['Authorization'] = $this->get_authorization_header( 'HEAD', '/' . $object_key, $headers );

			$response = wp_remote_head(
				$this->get_url( $object_key ),
				array(
					'headers' => $headers,
					'timeout' => 30,
				)
			);

			if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
				return false;
			}

			$headers = wp_remote_retrieve_headers( $response );

			return array(
				'content_type'   => $headers['content-type'] ?? '',
				'content_length' => (int) ( $headers['content-length'] ?? 0 ),
				'etag'           => trim( $headers['etag'] ?? '', '"' ),
				'last_modified'  => $headers['last-modified'] ?? '',
				'storage_class'  => $headers['x-amz-storage-class'] ?? 'STANDARD',
			);

		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * Copy object to new location
	 *
	 * @param string $source_key Source object key
	 * @param string $dest_key Destination object key
	 * @param array  $options Copy options
	 * @return bool
	 */
	public function copy_object( $source_key, $dest_key, $options = array() ) {
		try {
			$copy_source = $this->bucket . '/' . $source_key;

			$headers = array(
				'x-amz-copy-source' => $copy_source,
				'Host'              => $this->get_host(),
				'X-Amz-Date'        => $this->get_amz_date(),
			);

			$headers['Authorization'] = $this->get_authorization_header( 'PUT', '/' . $dest_key, $headers );

			$response = wp_remote_request(
				$this->get_url( $dest_key ),
				array(
					'method'  => 'PUT',
					'headers' => $headers,
					'timeout' => 120,
				)
			);

			return wp_remote_retrieve_response_code( $response ) === 200;

		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * Get S3 host
	 */
	private function get_host() {
		if ( ! empty( $this->endpoint ) ) {
			return wp_parse_url( $this->endpoint, PHP_URL_HOST );
		}

		if ( $this->use_path_style ) {
			return "s3.{$this->region}.amazonaws.com";
		}

		return "{$this->bucket}.s3.{$this->region}.amazonaws.com";
	}

	/**
	 * Get object URL
	 */
	private function get_url( $object_key = '' ) {
		$protocol = 'https://';
		$host     = $this->get_host();

		if ( $this->use_path_style || ! empty( $this->endpoint ) ) {
			return $protocol . $host . '/' . $this->bucket . ( $object_key ? '/' . $object_key : '' );
		}

		return $protocol . $host . ( $object_key ? '/' . $object_key : '' );
	}

	/**
	 * Get AMZ date
	 */
	private function get_amz_date() {
		return gmdate( 'Ymd\THis\Z' );
	}

	/**
	 * Get content type
	 */
	private function get_content_type( $file_path ) {
		$mime_type = wp_get_mime_type( $file_path );
		return $mime_type ?: 'binary/octet-stream';
	}

	/**
	 * Make S3 request
	 */
	private function make_s3_request( $method, $path, $options = array() ) {
		$url = $this->get_url( ltrim( $path, '/' ) );

		$args = array_merge(
			array(
				'method'  => $method,
				'timeout' => 30,
			),
			$options
		);

		return wp_remote_request( $url, $args );
	}

	/**
	 * Get authorization header (simplified AWS Signature Version 4)
	 */
	private function get_authorization_header( $method, $uri, $headers = array() ) {
		// This is a simplified implementation
		// In production, you'd want to use AWS SDK or more robust signing

		$access_key = $this->credentials['key'];
		$secret_key = $this->credentials['secret'];
		$date       = gmdate( 'Ymd' );
		$time       = gmdate( 'Ymd\THis\Z' );

		$credential = $access_key . '/' . $date . '/' . $this->region . '/s3/aws4_request';

		// Create canonical request (simplified)
		$canonical_headers = '';
		foreach ( $headers as $key => $value ) {
			if ( strtolower( $key ) !== 'authorization' ) {
				$canonical_headers .= strtolower( $key ) . ':' . trim( $value ) . "\n";
			}
		}

		$signed_headers = implode( ';', array_map( 'strtolower', array_keys( $headers ) ) );

		// For simplicity, using basic signing - in production use AWS SDK
		$string_to_sign = $method . "\n" . $uri . "\n\n" . $canonical_headers . "\n" . $signed_headers . "\nUNSIGNED-PAYLOAD";
		$signature      = hash_hmac( 'sha256', $string_to_sign, $secret_key );

		return "AWS4-HMAC-SHA256 Credential={$credential}, SignedHeaders={$signed_headers}, Signature={$signature}";
	}

	/**
	 * Sign string with secret key
	 */
	private function sign_string( $string_to_sign ) {
		return base64_encode( hash_hmac( 'sha1', $string_to_sign, $this->credentials['secret'], true ) );
	}

	/**
	 * Download a file from storage
	 *
	 * @param string $object_key Remote object key
	 * @param string $local_path Local file path
	 * @return Storage_Result
	 */
	public function download_file( $object_key, $local_path ) {
		$url = $this->get_object_url( $object_key );

		$response = wp_remote_get(
			$url,
			array(
				'timeout'  => 300,
				'stream'   => true,
				'filename' => $local_path,
			)
		);

		if ( is_wp_error( $response ) ) {
			return Storage_Result::error( $response->get_error_message() );
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		if ( $status_code !== 200 ) {
			return Storage_Result::error( "Download failed with status: {$status_code}", $status_code );
		}

		return Storage_Result::success( array( 'local_path' => $local_path ) );
	}

	/**
	 * Delete a file from storage (alias for delete_object)
	 *
	 * @param string $object_key Remote object key
	 * @return Storage_Result
	 */
	public function delete_file( $object_key ) {
		return $this->delete_object( $object_key );
	}

	/**
	 * Generate a presigned URL for an object (alias for get_signed_url)
	 *
	 * @param string $object_key Remote object key
	 * @param int    $expires    Expiration time in seconds
	 * @param string $method     HTTP method (GET, PUT, etc.)
	 * @return string|false
	 */
	public function get_presigned_url( $object_key, $expires = 3600, $method = 'GET' ) {
		return $this->get_signed_url( $object_key, $expires, array( 'method' => $method ) );
	}


	/**
	 * Get storage information
	 *
	 * @return Storage_Result
	 */
	public function get_storage_info() {
		$response = $this->make_s3_request( 'GET', '/' );

		if ( is_wp_error( $response ) ) {
			return Storage_Result::error( $response->get_error_message() );
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		if ( $status_code !== 200 ) {
			return Storage_Result::error( "Failed to get storage info: {$status_code}", $status_code );
		}

		return Storage_Result::success(
			array(
				'bucket'   => $this->bucket,
				'region'   => $this->region,
				'endpoint' => $this->endpoint,
			)
		);
	}

	/**
	 * Set object ACL
	 *
	 * @param string $object_key Remote object key
	 * @param string $acl        ACL setting
	 * @return Storage_Result
	 */
	public function set_object_acl( $object_key, $acl ) {
		$headers = array(
			'Host'       => $this->get_host(),
			'X-Amz-Date' => $this->get_amz_date(),
			'x-amz-acl'  => $acl,
		);

		$headers['Authorization'] = $this->get_authorization_header( 'PUT', '/' . $object_key . '?acl', $headers );

		$response = wp_remote_request(
			$this->get_url( $object_key ) . '?acl',
			array(
				'method'  => 'PUT',
				'headers' => $headers,
				'timeout' => 60,
			)
		);

		if ( is_wp_error( $response ) ) {
			return Storage_Result::error( $response->get_error_message() );
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		if ( $status_code !== 200 ) {
			return Storage_Result::error( "Failed to set ACL: {$status_code}", $status_code );
		}

		return Storage_Result::success( array( 'acl' => $acl ) );
	}

	/**
	 * Get object ACL
	 *
	 * @param string $object_key Remote object key
	 * @return Storage_Result
	 */
	public function get_object_acl( $object_key ) {
		$headers = array(
			'Host'       => $this->get_host(),
			'X-Amz-Date' => $this->get_amz_date(),
		);

		$headers['Authorization'] = $this->get_authorization_header( 'GET', '/' . $object_key . '?acl', $headers );

		$response = wp_remote_get(
			$this->get_url( $object_key ) . '?acl',
			array(
				'headers' => $headers,
				'timeout' => 60,
			)
		);

		if ( is_wp_error( $response ) ) {
			return Storage_Result::error( $response->get_error_message() );
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		if ( $status_code !== 200 ) {
			return Storage_Result::error( "Failed to get ACL: {$status_code}", $status_code );
		}

		$body = wp_remote_retrieve_body( $response );
		return Storage_Result::success( array( 'acl_xml' => $body ) );
	}
}
